function createElementWithString(htmlString){
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = htmlString;
    return tempDiv.firstElementChild;
}

function div$2(options) {
    return create$1("div", options);
}

function create$1(tag, options) {
    const elem = document.createElement(tag);
    if (options) {
        if (options.class) {
            elem.classList.add(options.class);
        }
        if (options.id) {
            elem.id = options.id;
        }
        if(options.style) {
            applyStyle$1(elem, options.style);
        }
    }
    return elem;
}

function hide$1(elem) {
    const cssStyle = getComputedStyle(elem);
    if(cssStyle.display !== "none") {
        elem._initialDisplay = cssStyle.display;
    }
    elem.style.display = "none";
}

function show$1(elem) {
    //const currentDisplay = getComputedStyle(elem).display;
    //if (currentDisplay === "none") {
        const d = elem._initialDisplay || "block";
        elem.style.display = d;
   // }
}

function pageCoordinates(e) {

    if (e.type.startsWith("touch")) {
        const touch = e.touches[0];
        return {x: touch.pageX, y: touch.pageY};
    } else {
        return {x: e.pageX, y: e.pageY}
    }
}

function applyStyle$1(elem, style) {
    for (let key of Object.keys(style)) {
        elem.style[key] = style[key];
    }
}

function guid$2  () {
    return ("0000" + (Math.random() * Math.pow(36, 4) << 0).toString(36)).slice(-4);
}

let getMouseXY = (domElement, { clientX, clientY }) => {

    // DOMRect object with eight properties: left, top, right, bottom, x, y, width, height
    const { left, top, width, height } = domElement.getBoundingClientRect();

    const x = clientX - left;
    const y = clientY - top;
    return { x, y, xNormalized: x/width, yNormalized: y/height, width, height };

};

/**
 * Translate the mouse coordinates for the event to the coordinates for the given target element
 * @param event
 * @param domElement
 * @returns {{x: number, y: number}}
 */
function translateMouseCoordinates(event, domElement) {

    const { clientX, clientY } = event;
    return getMouseXY(domElement, { clientX, clientY });

}

/**
 * Generic container for UI components
 */
class Panel {

    constructor() {
        this.elem = create$1('div', { class: 'igv-ui-panel-column' });
    }

    add(component) {

        if(component instanceof Node) {
            this.elem.appendChild(component);
        }
        else if(typeof component === 'object') {
            this.elem.appendChild(component.elem);
        }
        else {
            // Assuming a string, possibly html
            const wrapper = div$2();
            wrapper.innerHTML = component;
            this.elem.appendChild(wrapper);
            this.html = wrapper;
        }
    }


}

function createCheckbox$1(name, initialState) {
    const container = div$2({class: 'igv-ui-trackgear-popover-check-container'});
    const svg = iconMarkup$1('check', (true === initialState ? '#444' : 'transparent'));
    svg.style.borderColor = 'gray';
    svg.style.borderWidth = '1px';
    svg.style.borderStyle = 'solid';

    container.appendChild(svg);
    let label = div$2(); //{ class: 'igv-some-label-class' });
    label.textContent = name;
    container.appendChild(label);

    return container;
}

function createIcon$1(name, color) {
    return iconMarkup$1(name, color);
}

function iconMarkup$1(name, color) {
    color = color || "currentColor";
    let icon = icons$1[name];
    if (!icon) {
        console.error(`No icon named: ${name}`);
        icon = icons$1["question"];
    }

    const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
    svg.setAttributeNS(null,'viewBox', '0 0 ' + icon[0] + ' ' + icon[1]);
    const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
    path.setAttributeNS(null,'fill',  color );
    path.setAttributeNS(null,'d', icon[4]);
    svg.appendChild(path);
    return svg;
}

const icons$1 = {
    "check": [512, 512, [], "f00c", "M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"],
    "cog": [512, 512, [], "f013", "M444.788 291.1l42.616 24.599c4.867 2.809 7.126 8.618 5.459 13.985-11.07 35.642-29.97 67.842-54.689 94.586a12.016 12.016 0 0 1-14.832 2.254l-42.584-24.595a191.577 191.577 0 0 1-60.759 35.13v49.182a12.01 12.01 0 0 1-9.377 11.718c-34.956 7.85-72.499 8.256-109.219.007-5.49-1.233-9.403-6.096-9.403-11.723v-49.184a191.555 191.555 0 0 1-60.759-35.13l-42.584 24.595a12.016 12.016 0 0 1-14.832-2.254c-24.718-26.744-43.619-58.944-54.689-94.586-1.667-5.366.592-11.175 5.459-13.985L67.212 291.1a193.48 193.48 0 0 1 0-70.199l-42.616-24.599c-4.867-2.809-7.126-8.618-5.459-13.985 11.07-35.642 29.97-67.842 54.689-94.586a12.016 12.016 0 0 1 14.832-2.254l42.584 24.595a191.577 191.577 0 0 1 60.759-35.13V25.759a12.01 12.01 0 0 1 9.377-11.718c34.956-7.85 72.499-8.256 109.219-.007 5.49 1.233 9.403 6.096 9.403 11.723v49.184a191.555 191.555 0 0 1 60.759 35.13l42.584-24.595a12.016 12.016 0 0 1 14.832 2.254c24.718 26.744 43.619 58.944 54.689 94.586 1.667 5.366-.592 11.175-5.459 13.985L444.788 220.9a193.485 193.485 0 0 1 0 70.2zM336 256c0-44.112-35.888-80-80-80s-80 35.888-80 80 35.888 80 80 80 80-35.888 80-80z"],
    "exclamation": [192, 512, [], "f12a", "M176 432c0 44.112-35.888 80-80 80s-80-35.888-80-80 35.888-80 80-80 80 35.888 80 80zM25.26 25.199l13.6 272C39.499 309.972 50.041 320 62.83 320h66.34c12.789 0 23.331-10.028 23.97-22.801l13.6-272C167.425 11.49 156.496 0 142.77 0H49.23C35.504 0 24.575 11.49 25.26 25.199z"],
    "exclamation-circle": [512, 512, [], "f06a", "M504 256c0 136.997-111.043 248-248 248S8 392.997 8 256C8 119.083 119.043 8 256 8s248 111.083 248 248zm-248 50c-25.405 0-46 20.595-46 46s20.595 46 46 46 46-20.595 46-46-20.595-46-46-46zm-43.673-165.346l7.418 136c.347 6.364 5.609 11.346 11.982 11.346h48.546c6.373 0 11.635-4.982 11.982-11.346l7.418-136c.375-6.874-5.098-12.654-11.982-12.654h-63.383c-6.884 0-12.356 5.78-11.981 12.654z"],
    "exclamation-triangle": [576, 512, [], "f071", "M569.517 440.013C587.975 472.007 564.806 512 527.94 512H48.054c-36.937 0-59.999-40.055-41.577-71.987L246.423 23.985c18.467-32.009 64.72-31.951 83.154 0l239.94 416.028zM288 354c-25.405 0-46 20.595-46 46s20.595 46 46 46 46-20.595 46-46-20.595-46-46-46zm-43.673-165.346l7.418 136c.347 6.364 5.609 11.346 11.982 11.346h48.546c6.373 0 11.635-4.982 11.982-11.346l7.418-136c.375-6.874-5.098-12.654-11.982-12.654h-63.383c-6.884 0-12.356 5.78-11.981 12.654z"],
    "minus": [448, 512, [], "f068", "M424 318.2c13.3 0 24-10.7 24-24v-76.4c0-13.3-10.7-24-24-24H24c-13.3 0-24 10.7-24 24v76.4c0 13.3 10.7 24 24 24h400z"],
    "minus-circle": [512, 512, [], "f056", "M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zM124 296c-6.6 0-12-5.4-12-12v-56c0-6.6 5.4-12 12-12h264c6.6 0 12 5.4 12 12v56c0 6.6-5.4 12-12 12H124z"],
    "minus-square": [448, 512, [], "f146", "M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM92 296c-6.6 0-12-5.4-12-12v-56c0-6.6 5.4-12 12-12h264c6.6 0 12 5.4 12 12v56c0 6.6-5.4 12-12 12H92z"],
    "plus": [448, 512, [], "f067", "M448 294.2v-76.4c0-13.3-10.7-24-24-24H286.2V56c0-13.3-10.7-24-24-24h-76.4c-13.3 0-24 10.7-24 24v137.8H24c-13.3 0-24 10.7-24 24v76.4c0 13.3 10.7 24 24 24h137.8V456c0 13.3 10.7 24 24 24h76.4c13.3 0 24-10.7 24-24V318.2H424c13.3 0 24-10.7 24-24z"],
    "plus-circle": [512, 512, [], "f055", "M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm144 276c0 6.6-5.4 12-12 12h-92v92c0 6.6-5.4 12-12 12h-56c-6.6 0-12-5.4-12-12v-92h-92c-6.6 0-12-5.4-12-12v-56c0-6.6 5.4-12 12-12h92v-92c0-6.6 5.4-12 12-12h56c6.6 0 12 5.4 12 12v92h92c6.6 0 12 5.4 12 12v56z"],
    "plus-square": [448, 512, [], "f0fe", "M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zm-32 252c0 6.6-5.4 12-12 12h-92v92c0 6.6-5.4 12-12 12h-56c-6.6 0-12-5.4-12-12v-92H92c-6.6 0-12-5.4-12-12v-56c0-6.6 5.4-12 12-12h92v-92c0-6.6 5.4-12 12-12h56c6.6 0 12 5.4 12 12v92h92c6.6 0 12 5.4 12 12v56z"],
    "question": [384, 512, [], "f128", "M202.021 0C122.202 0 70.503 32.703 29.914 91.026c-7.363 10.58-5.093 25.086 5.178 32.874l43.138 32.709c10.373 7.865 25.132 6.026 33.253-4.148 25.049-31.381 43.63-49.449 82.757-49.449 30.764 0 68.816 19.799 68.816 49.631 0 22.552-18.617 34.134-48.993 51.164-35.423 19.86-82.299 44.576-82.299 106.405V320c0 13.255 10.745 24 24 24h72.471c13.255 0 24-10.745 24-24v-5.773c0-42.86 125.268-44.645 125.268-160.627C377.504 66.256 286.902 0 202.021 0zM192 373.459c-38.196 0-69.271 31.075-69.271 69.271 0 38.195 31.075 69.27 69.271 69.27s69.271-31.075 69.271-69.271-31.075-69.27-69.271-69.27z"],
    "save": [448, 512, [], "f0c7", "M433.941 129.941l-83.882-83.882A48 48 0 0 0 316.118 32H48C21.49 32 0 53.49 0 80v352c0 26.51 21.49 48 48 48h352c26.51 0 48-21.49 48-48V163.882a48 48 0 0 0-14.059-33.941zM224 416c-35.346 0-64-28.654-64-64 0-35.346 28.654-64 64-64s64 28.654 64 64c0 35.346-28.654 64-64 64zm96-304.52V212c0 6.627-5.373 12-12 12H76c-6.627 0-12-5.373-12-12V108c0-6.627 5.373-12 12-12h228.52c3.183 0 6.235 1.264 8.485 3.515l3.48 3.48A11.996 11.996 0 0 1 320 111.48z"],
    "search": [512, 512, [], "f002", "M505 442.7L405.3 343c-4.5-4.5-10.6-7-17-7H372c27.6-35.3 44-79.7 44-128C416 93.1 322.9 0 208 0S0 93.1 0 208s93.1 208 208 208c48.3 0 92.7-16.4 128-44v16.3c0 6.4 2.5 12.5 7 17l99.7 99.7c9.4 9.4 24.6 9.4 33.9 0l28.3-28.3c9.4-9.4 9.4-24.6.1-34zM208 336c-70.7 0-128-57.2-128-128 0-70.7 57.2-128 128-128 70.7 0 128 57.2 128 128 0 70.7-57.2 128-128 128z"],
    "share": [512, 512, [], "f064", "M503.691 189.836L327.687 37.851C312.281 24.546 288 35.347 288 56.015v80.053C127.371 137.907 0 170.1 0 322.326c0 61.441 39.581 122.309 83.333 154.132 13.653 9.931 33.111-2.533 28.077-18.631C66.066 312.814 132.917 274.316 288 272.085V360c0 20.7 24.3 31.453 39.687 18.164l176.004-152c11.071-9.562 11.086-26.753 0-36.328z"],
    "spinner": [512, 512, [], "f110", "M304 48c0 26.51-21.49 48-48 48s-48-21.49-48-48 21.49-48 48-48 48 21.49 48 48zm-48 368c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zm208-208c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zM96 256c0-26.51-21.49-48-48-48S0 229.49 0 256s21.49 48 48 48 48-21.49 48-48zm12.922 99.078c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.491-48-48-48zm294.156 0c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.49-48-48-48zM108.922 60.922c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.491-48-48-48z"],
    "square": [448, 512, [], "f0c8", "M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48z"],
    "square-full": [512, 512, [], "f45c", "M512 512H0V0h512v512z"],
    "times": [384, 512, [], "f00d", "M323.1 441l53.9-53.9c9.4-9.4 9.4-24.5 0-33.9L279.8 256l97.2-97.2c9.4-9.4 9.4-24.5 0-33.9L323.1 71c-9.4-9.4-24.5-9.4-33.9 0L192 168.2 94.8 71c-9.4-9.4-24.5-9.4-33.9 0L7 124.9c-9.4 9.4-9.4 24.5 0 33.9l97.2 97.2L7 353.2c-9.4 9.4-9.4 24.5 0 33.9L60.9 441c9.4 9.4 24.5 9.4 33.9 0l97.2-97.2 97.2 97.2c9.3 9.3 24.5 9.3 33.9 0z"],
    "times-circle": [512, 512, [], "f057", "M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm121.6 313.1c4.7 4.7 4.7 12.3 0 17L338 377.6c-4.7 4.7-12.3 4.7-17 0L256 312l-65.1 65.6c-4.7 4.7-12.3 4.7-17 0L134.4 338c-4.7-4.7-4.7-12.3 0-17l65.6-65-65.6-65.1c-4.7-4.7-4.7-12.3 0-17l39.6-39.6c4.7-4.7 12.3-4.7 17 0l65 65.7 65.1-65.6c4.7-4.7 12.3-4.7 17 0l39.6 39.6c4.7 4.7 4.7 12.3 0 17L312 256l65.6 65.1z"],
    "wrench": [512, 512, [], "f0ad", "M481.156 200c9.3 0 15.12 10.155 10.325 18.124C466.295 259.992 420.419 288 368 288c-79.222 0-143.501-63.974-143.997-143.079C223.505 65.469 288.548-.001 368.002 0c52.362.001 98.196 27.949 123.4 69.743C496.24 77.766 490.523 88 481.154 88H376l-40 56 40 56h105.156zm-171.649 93.003L109.255 493.255c-24.994 24.993-65.515 24.994-90.51 0-24.993-24.994-24.993-65.516 0-90.51L218.991 202.5c16.16 41.197 49.303 74.335 90.516 90.503zM104 432c0-13.255-10.745-24-24-24s-24 10.745-24 24 10.745 24 24 24 24-10.745 24-24z"],
};

function attachDialogCloseHandlerWithParent$1(parent, closeHandler) {

    var container = document.createElement("div");
    parent.appendChild(container);
    container.appendChild(createIcon$1("times"));
    container.addEventListener('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        closeHandler();
    });
}

/**
 * Make the target element movable by clicking and dragging on the handle.  This is not a general purprose function,
 * it makes several options specific to igv dialogs, the primary one being that the
 * target is absolutely positioned in pixel coordinates

 */

let dragData$1;  // Its assumed we are only dragging one element at a time.

function makeDraggable$1(target, handle, constraint) {

    handle.addEventListener('mousedown', dragStart.bind(target));

    function dragStart(event) {

        event.stopPropagation();
        event.preventDefault();

        const dragFunction = drag$1.bind(this);
        const dragEndFunction = dragEnd$1.bind(this);
        const computedStyle = getComputedStyle(this);


        const boundingClientRect = this.getBoundingClientRect();
        dragData$1 =
            {
                constraint,
                dragFunction,
                dragEndFunction,
                screenX: event.screenX,
                screenY: event.screenY,
                minDy: -boundingClientRect.top,   // Don't slide upwards more than this
                minDx: -boundingClientRect.left,
                top: parseInt(computedStyle.top.replace("px", "")),
                left: parseInt(computedStyle.left.replace("px", ""))
            };

        document.addEventListener('mousemove', dragFunction);
        document.addEventListener('mouseup', dragEndFunction);
        document.addEventListener('mouseleave', dragEndFunction);
        document.addEventListener('mouseexit', dragEndFunction);
    }
}

function drag$1(event) {

    if (!dragData$1) {
        console.error("No drag data!");
        return
    }
    event.stopPropagation();
    event.preventDefault();
    const dx = Math.max(dragData$1.minDx, event.screenX - dragData$1.screenX);
    const dy = Math.max(dragData$1.minDy, event.screenY - dragData$1.screenY);
    const left = dragData$1.left + dx;
    const top = dragData$1.top + dy;

    this.style.left = `${left}px`;
    this.style.top = `${top}px`;
}

function dragEnd$1(event) {

    if (!dragData$1) {
        console.error("No drag data!");
        return
    }
    event.stopPropagation();
    event.preventDefault();

    const dragFunction = dragData$1.dragFunction;
    const dragEndFunction = dragData$1.dragEndFunction;
    document.removeEventListener('mousemove', dragFunction);
    document.removeEventListener('mouseup', dragEndFunction);
    document.removeEventListener('mouseleave', dragEndFunction);
    document.removeEventListener('mouseexit', dragEndFunction);
    dragData$1 = undefined;
}

class Dialog {

    constructor({parent, label, content, okHandler, cancelHandler}) {

        this.parent = parent;

        const cancel = () => {
            this.elem.style.display = 'none';
            if (typeof cancelHandler === 'function') {
                cancelHandler(this);
            }
        };

        // dialog container
        this.elem = div$2();
        this.elem.classList.add('igv-ui-generic-dialog-container', 'igv-ui-center-fixed');

        // dialog header
        const header = div$2({class: 'igv-ui-generic-dialog-header'});
        this.elem.appendChild(header);

        attachDialogCloseHandlerWithParent$1(header, cancel);

        // dialog label
        if(label) {
            const labelDiv = div$2({class: 'igv-ui-dialog-one-liner'});
            this.elem.appendChild(labelDiv);
            labelDiv.innerHTML = label;
        }

        // input container
        content.elem.style.margin = '16px';
        this.elem.appendChild(content.elem);

        this.content = content;

        // ok | cancel
        const buttons = div$2({class: 'igv-ui-generic-dialog-ok-cancel'});
        this.elem.appendChild(buttons);

        // ok
        this.ok = div$2();
        buttons.appendChild(this.ok);
        this.ok.textContent = 'OK';

        // cancel
        this.cancel = div$2();
        buttons.appendChild(this.cancel);
        this.cancel.textContent = 'Cancel';

        this.callback = undefined;

        this.ok.addEventListener('click',  e => {
            this.elem.style.display = 'none';
            if (typeof okHandler === 'function') {
                okHandler(this);
            } else if (this.callback && typeof this.callback === 'function') {
                this.callback(this);
            }
        });

        this.cancel.addEventListener('click', cancel);

        makeDraggable$1(this.elem, header);

        // Consume all clicks in component
        this.elem.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
        });

    }

    present(options, e) {

        if (options.label && this.label) {
            this.label.textContent = options.label;
        }

        if (options.html) {
            const div = this.content.html;
            div.innerHTML = options.html;
        }

        if (options.text) {
            const div = this.content.html;
            div.innerText = options.text;
        }

        if (options.value && this.input) {
            this.input.value = options.value;
        }

        if (options.callback) {
            this.callback = options.callback;
        }

        const { top} = e.currentTarget.parentElement.getBoundingClientRect();
        this.elem.style.top = `${ top }px`;

        this.elem.style.display = 'flex';
    }
}

/**
 * Covers string literals and String objects
 * @param x
 * @returns {boolean}
 */
function isString$3(x) {
    return typeof x === "string" || x instanceof String
}


// StackOverflow: http://stackoverflow.com/a/10810674/116169
function numberFormatter$1(rawNumber) {

    var dec = String(rawNumber).split(/[.,]/),
        sep = ',',
        decsep = '.';

    return dec[0].split('').reverse().reduce(function (prev, now, i) {
        return i % 3 === 0 ? prev + sep + now : prev + now;
    }).split('').reverse().join('') + (dec[1] ? decsep + dec[1] : '');
}

const splitLines$3 = function (string) {
    return string.split(/\n|\r\n|\r/g);
};


function splitStringRespectingQuotes(string, delim) {

    var tokens = [],
        len = string.length,
        i,
        n = 0,
        quote = false,
        c;

    if (len > 0) {

        tokens[n] = string.charAt(0);
        for (i = 1; i < len; i++) {
            c = string.charAt(i);
            if (c === '"') {
                quote = !quote;
            } else if (!quote && c === delim) {
                n++;
                tokens[n] = "";
            } else {
                tokens[n] += c;
            }
        }
    }
    return tokens;
}

function stripQuotes$2(str) {
    if(str === undefined) {
        return str;
    }
    if(str.startsWith("'") || str.startsWith('"')) {
        str = str.substring(1);
    }
    if (str.endsWith("'") || str.endsWith('"')) {
        str = str.substring(0, str.length - 1);
    }
    return str;
}

function capitalize(str) {
    return str.length > 0 ? str.charAt(0).toUpperCase() + str.slice(1) : str;
}


/**
 * Parse a locus string and return a range object.  Locus string is of the form chr:start-end.  End is optional
 *
 */
function parseLocusString$1(string) {

    const t1 = string.split(":");
    const t2 = t1[1].split("-");

    const range = {
        chr: t1[0],
        start: Number.parseInt(t2[0].replace(/,/g, '')) - 1
    };

    if (t2.length > 1) {
        range.end = Number.parseInt(t2[1].replace(/,/g, ''));
    } else {
        range.end = range.start + 1;
    }

    return range;
}

/**
 * Return the filename from the path.   Example
 *   https://foo.com/bar.bed?param=2   => bar.bed
 * @param urlOrFile
 */

function getFilename$2(urlOrFile) {

    if (urlOrFile.name !== undefined) {
        return urlOrFile.name
    } else if (isString$3(urlOrFile)) {

        let index = urlOrFile.lastIndexOf("/");
        let filename = index < 0 ? urlOrFile : urlOrFile.substr(index + 1);

        //Strip parameters -- handle local files later
        index = filename.indexOf("?");
        if (index > 0) {
            filename = filename.substr(0, index);
        }
        return filename
    } else {
        throw Error(`Expected File or string, got ${typeof urlOrFile}`)
    }
}


/**
 * Test if object is a File or File-like object.
 *
 * @param object
 */
function isFile$1(object) {
    if(!object) {
        return false;
    }
    return typeof object !== 'function' &&
        (object instanceof File ||
            (object.hasOwnProperty("name") && typeof object.slice === 'function' && typeof object.arrayBuffer === 'function'))
}

function download$1(filename, data) {

    const element = document.createElement('a');
    element.setAttribute('href', data);
    element.setAttribute('download', filename);
    element.style.display = 'none';
    document.body.appendChild(element);
    element.click();
    document.body.removeChild(element);
}

if (typeof process === 'object' && typeof window === 'undefined') {
    global.atob = function (str) {
        return Buffer.from(str, 'base64').toString('binary');
    };
}


function parseUri$1(str) {

    var o = options$1,
        m = o.parser["loose"].exec(str),
        uri = {},
        i = 14;

    while (i--) uri[o.key[i]] = m[i] || "";

    uri[o.q.name] = {};
    uri[o.key[12]].replace(o.q.parser, function ($0, $1, $2) {
        if ($1) uri[o.q.name][$1] = $2;
    });

    return uri;
}

const options$1 = {
    strictMode: false,
    key: ["source", "protocol", "authority", "userInfo", "user", "password", "host", "port", "relative", "path", "directory", "file", "query", "anchor"],
    q: {
        name: "queryKey",
        parser: /(?:^|&)([^&=]*)=?([^&]*)/g
    },
    parser: {
        strict: /^(?:([^:\/?#]+):)?(?:\/\/((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?))?((((?:[^?#\/]*\/)*)([^?#]*))(?:\?([^#]*))?(?:#(.*))?)/,
        loose: /^(?:(?![^:@]+:[^:@\/]*@)([^:\/?#.]+):)?(?:\/\/)?((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/
    }
};

/**
 * Resolve a url, which might be a string, function (that returns a string or Promse), or Promise (that resolves to a string)
 *
 * @param url
 * @returns {Promise<*>}
 */
async function resolveURL(url) {
    return (typeof url === 'function')  ?  url() :  url;
}

/*! pako 2.1.0 https://github.com/nodeca/pako @license (MIT AND Zlib) */
// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

/* eslint-disable space-unary-ops */

/* Public constants ==========================================================*/
/* ===========================================================================*/


//const Z_FILTERED          = 1;
//const Z_HUFFMAN_ONLY      = 2;
//const Z_RLE               = 3;
const Z_FIXED$1$1               = 4;
//const Z_DEFAULT_STRATEGY  = 0;

/* Possible values of the data_type field (though see inflate()) */
const Z_BINARY$1              = 0;
const Z_TEXT$1                = 1;
//const Z_ASCII             = 1; // = Z_TEXT
const Z_UNKNOWN$1$1             = 2;

/*============================================================================*/


function zero$1$1(buf) { let len = buf.length; while (--len >= 0) { buf[len] = 0; } }

// From zutil.h

const STORED_BLOCK$1 = 0;
const STATIC_TREES$1 = 1;
const DYN_TREES$1    = 2;
/* The three kinds of block type */

const MIN_MATCH$1$1    = 3;
const MAX_MATCH$1$1    = 258;
/* The minimum and maximum match lengths */

// From deflate.h
/* ===========================================================================
 * Internal compression state.
 */

const LENGTH_CODES$1$1  = 29;
/* number of length codes, not counting the special END_BLOCK code */

const LITERALS$1$1      = 256;
/* number of literal bytes 0..255 */

const L_CODES$1$1       = LITERALS$1$1 + 1 + LENGTH_CODES$1$1;
/* number of Literal or Length codes, including the END_BLOCK code */

const D_CODES$1$1       = 30;
/* number of distance codes */

const BL_CODES$1$1      = 19;
/* number of codes used to transfer the bit lengths */

const HEAP_SIZE$1$1     = 2 * L_CODES$1$1 + 1;
/* maximum heap size */

const MAX_BITS$1$1      = 15;
/* All codes must not exceed MAX_BITS bits */

const Buf_size$1      = 16;
/* size of bit buffer in bi_buf */


/* ===========================================================================
 * Constants
 */

const MAX_BL_BITS$1 = 7;
/* Bit length codes must not exceed MAX_BL_BITS bits */

const END_BLOCK$1   = 256;
/* end of block literal code */

const REP_3_6$1     = 16;
/* repeat previous bit length 3-6 times (2 bits of repeat count) */

const REPZ_3_10$1   = 17;
/* repeat a zero length 3-10 times  (3 bits of repeat count) */

const REPZ_11_138$1 = 18;
/* repeat a zero length 11-138 times  (7 bits of repeat count) */

/* eslint-disable comma-spacing,array-bracket-spacing */
const extra_lbits$1 =   /* extra bits for each length code */
  new Uint8Array([0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0]);

const extra_dbits$1 =   /* extra bits for each distance code */
  new Uint8Array([0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13]);

const extra_blbits$1 =  /* extra bits for each bit length code */
  new Uint8Array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7]);

const bl_order$1 =
  new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]);
/* eslint-enable comma-spacing,array-bracket-spacing */

/* The lengths of the bit length codes are sent in order of decreasing
 * probability, to avoid transmitting the lengths for unused bit length codes.
 */

/* ===========================================================================
 * Local data. These are initialized only once.
 */

// We pre-fill arrays with 0 to avoid uninitialized gaps

const DIST_CODE_LEN$1 = 512; /* see definition of array dist_code below */

// !!!! Use flat array instead of structure, Freq = i*2, Len = i*2+1
const static_ltree$1  = new Array((L_CODES$1$1 + 2) * 2);
zero$1$1(static_ltree$1);
/* The static literal tree. Since the bit lengths are imposed, there is no
 * need for the L_CODES extra codes used during heap construction. However
 * The codes 286 and 287 are needed to build a canonical tree (see _tr_init
 * below).
 */

const static_dtree$1  = new Array(D_CODES$1$1 * 2);
zero$1$1(static_dtree$1);
/* The static distance tree. (Actually a trivial tree since all codes use
 * 5 bits.)
 */

const _dist_code$1    = new Array(DIST_CODE_LEN$1);
zero$1$1(_dist_code$1);
/* Distance codes. The first 256 values correspond to the distances
 * 3 .. 258, the last 256 values correspond to the top 8 bits of
 * the 15 bit distances.
 */

const _length_code$1  = new Array(MAX_MATCH$1$1 - MIN_MATCH$1$1 + 1);
zero$1$1(_length_code$1);
/* length code for each normalized match length (0 == MIN_MATCH) */

const base_length$1   = new Array(LENGTH_CODES$1$1);
zero$1$1(base_length$1);
/* First normalized length for each code (0 = MIN_MATCH) */

const base_dist$1     = new Array(D_CODES$1$1);
zero$1$1(base_dist$1);
/* First normalized distance for each code (0 = distance of 1) */


function StaticTreeDesc$1(static_tree, extra_bits, extra_base, elems, max_length) {

  this.static_tree  = static_tree;  /* static tree or NULL */
  this.extra_bits   = extra_bits;   /* extra bits for each code or NULL */
  this.extra_base   = extra_base;   /* base index for extra_bits */
  this.elems        = elems;        /* max number of elements in the tree */
  this.max_length   = max_length;   /* max bit length for the codes */

  // show if `static_tree` has data or dummy - needed for monomorphic objects
  this.has_stree    = static_tree && static_tree.length;
}


let static_l_desc$1;
let static_d_desc$1;
let static_bl_desc$1;


function TreeDesc$1(dyn_tree, stat_desc) {
  this.dyn_tree = dyn_tree;     /* the dynamic tree */
  this.max_code = 0;            /* largest code with non zero frequency */
  this.stat_desc = stat_desc;   /* the corresponding static tree */
}



const d_code$1 = (dist) => {

  return dist < 256 ? _dist_code$1[dist] : _dist_code$1[256 + (dist >>> 7)];
};


/* ===========================================================================
 * Output a short LSB first on the stream.
 * IN assertion: there is enough room in pendingBuf.
 */
const put_short$1 = (s, w) => {
//    put_byte(s, (uch)((w) & 0xff));
//    put_byte(s, (uch)((ush)(w) >> 8));
  s.pending_buf[s.pending++] = (w) & 0xff;
  s.pending_buf[s.pending++] = (w >>> 8) & 0xff;
};


/* ===========================================================================
 * Send a value on a given number of bits.
 * IN assertion: length <= 16 and value fits in length bits.
 */
const send_bits$1 = (s, value, length) => {

  if (s.bi_valid > (Buf_size$1 - length)) {
    s.bi_buf |= (value << s.bi_valid) & 0xffff;
    put_short$1(s, s.bi_buf);
    s.bi_buf = value >> (Buf_size$1 - s.bi_valid);
    s.bi_valid += length - Buf_size$1;
  } else {
    s.bi_buf |= (value << s.bi_valid) & 0xffff;
    s.bi_valid += length;
  }
};


const send_code$1 = (s, c, tree) => {

  send_bits$1(s, tree[c * 2]/*.Code*/, tree[c * 2 + 1]/*.Len*/);
};


/* ===========================================================================
 * Reverse the first len bits of a code, using straightforward code (a faster
 * method would use a table)
 * IN assertion: 1 <= len <= 15
 */
const bi_reverse$1 = (code, len) => {

  let res = 0;
  do {
    res |= code & 1;
    code >>>= 1;
    res <<= 1;
  } while (--len > 0);
  return res >>> 1;
};


/* ===========================================================================
 * Flush the bit buffer, keeping at most 7 bits in it.
 */
const bi_flush$1 = (s) => {

  if (s.bi_valid === 16) {
    put_short$1(s, s.bi_buf);
    s.bi_buf = 0;
    s.bi_valid = 0;

  } else if (s.bi_valid >= 8) {
    s.pending_buf[s.pending++] = s.bi_buf & 0xff;
    s.bi_buf >>= 8;
    s.bi_valid -= 8;
  }
};


/* ===========================================================================
 * Compute the optimal bit lengths for a tree and update the total bit length
 * for the current block.
 * IN assertion: the fields freq and dad are set, heap[heap_max] and
 *    above are the tree nodes sorted by increasing frequency.
 * OUT assertions: the field len is set to the optimal bit length, the
 *     array bl_count contains the frequencies for each bit length.
 *     The length opt_len is updated; static_len is also updated if stree is
 *     not null.
 */
const gen_bitlen$1 = (s, desc) => {
//    deflate_state *s;
//    tree_desc *desc;    /* the tree descriptor */

  const tree            = desc.dyn_tree;
  const max_code        = desc.max_code;
  const stree           = desc.stat_desc.static_tree;
  const has_stree       = desc.stat_desc.has_stree;
  const extra           = desc.stat_desc.extra_bits;
  const base            = desc.stat_desc.extra_base;
  const max_length      = desc.stat_desc.max_length;
  let h;              /* heap index */
  let n, m;           /* iterate over the tree elements */
  let bits;           /* bit length */
  let xbits;          /* extra bits */
  let f;              /* frequency */
  let overflow = 0;   /* number of elements with bit length too large */

  for (bits = 0; bits <= MAX_BITS$1$1; bits++) {
    s.bl_count[bits] = 0;
  }

  /* In a first pass, compute the optimal bit lengths (which may
   * overflow in the case of the bit length tree).
   */
  tree[s.heap[s.heap_max] * 2 + 1]/*.Len*/ = 0; /* root of the heap */

  for (h = s.heap_max + 1; h < HEAP_SIZE$1$1; h++) {
    n = s.heap[h];
    bits = tree[tree[n * 2 + 1]/*.Dad*/ * 2 + 1]/*.Len*/ + 1;
    if (bits > max_length) {
      bits = max_length;
      overflow++;
    }
    tree[n * 2 + 1]/*.Len*/ = bits;
    /* We overwrite tree[n].Dad which is no longer needed */

    if (n > max_code) { continue; } /* not a leaf node */

    s.bl_count[bits]++;
    xbits = 0;
    if (n >= base) {
      xbits = extra[n - base];
    }
    f = tree[n * 2]/*.Freq*/;
    s.opt_len += f * (bits + xbits);
    if (has_stree) {
      s.static_len += f * (stree[n * 2 + 1]/*.Len*/ + xbits);
    }
  }
  if (overflow === 0) { return; }

  // Tracev((stderr,"\nbit length overflow\n"));
  /* This happens for example on obj2 and pic of the Calgary corpus */

  /* Find the first bit length which could increase: */
  do {
    bits = max_length - 1;
    while (s.bl_count[bits] === 0) { bits--; }
    s.bl_count[bits]--;      /* move one leaf down the tree */
    s.bl_count[bits + 1] += 2; /* move one overflow item as its brother */
    s.bl_count[max_length]--;
    /* The brother of the overflow item also moves one step up,
     * but this does not affect bl_count[max_length]
     */
    overflow -= 2;
  } while (overflow > 0);

  /* Now recompute all bit lengths, scanning in increasing frequency.
   * h is still equal to HEAP_SIZE. (It is simpler to reconstruct all
   * lengths instead of fixing only the wrong ones. This idea is taken
   * from 'ar' written by Haruhiko Okumura.)
   */
  for (bits = max_length; bits !== 0; bits--) {
    n = s.bl_count[bits];
    while (n !== 0) {
      m = s.heap[--h];
      if (m > max_code) { continue; }
      if (tree[m * 2 + 1]/*.Len*/ !== bits) {
        // Tracev((stderr,"code %d bits %d->%d\n", m, tree[m].Len, bits));
        s.opt_len += (bits - tree[m * 2 + 1]/*.Len*/) * tree[m * 2]/*.Freq*/;
        tree[m * 2 + 1]/*.Len*/ = bits;
      }
      n--;
    }
  }
};


/* ===========================================================================
 * Generate the codes for a given tree and bit counts (which need not be
 * optimal).
 * IN assertion: the array bl_count contains the bit length statistics for
 * the given tree and the field len is set for all tree elements.
 * OUT assertion: the field code is set for all tree elements of non
 *     zero code length.
 */
const gen_codes$1 = (tree, max_code, bl_count) => {
//    ct_data *tree;             /* the tree to decorate */
//    int max_code;              /* largest code with non zero frequency */
//    ushf *bl_count;            /* number of codes at each bit length */

  const next_code = new Array(MAX_BITS$1$1 + 1); /* next code value for each bit length */
  let code = 0;              /* running code value */
  let bits;                  /* bit index */
  let n;                     /* code index */

  /* The distribution counts are first used to generate the code values
   * without bit reversal.
   */
  for (bits = 1; bits <= MAX_BITS$1$1; bits++) {
    code = (code + bl_count[bits - 1]) << 1;
    next_code[bits] = code;
  }
  /* Check that the bit counts in bl_count are consistent. The last code
   * must be all ones.
   */
  //Assert (code + bl_count[MAX_BITS]-1 == (1<<MAX_BITS)-1,
  //        "inconsistent bit counts");
  //Tracev((stderr,"\ngen_codes: max_code %d ", max_code));

  for (n = 0;  n <= max_code; n++) {
    let len = tree[n * 2 + 1]/*.Len*/;
    if (len === 0) { continue; }
    /* Now reverse the bits */
    tree[n * 2]/*.Code*/ = bi_reverse$1(next_code[len]++, len);

    //Tracecv(tree != static_ltree, (stderr,"\nn %3d %c l %2d c %4x (%x) ",
    //     n, (isgraph(n) ? n : ' '), len, tree[n].Code, next_code[len]-1));
  }
};


/* ===========================================================================
 * Initialize the various 'constant' tables.
 */
const tr_static_init$1 = () => {

  let n;        /* iterates over tree elements */
  let bits;     /* bit counter */
  let length;   /* length value */
  let code;     /* code value */
  let dist;     /* distance index */
  const bl_count = new Array(MAX_BITS$1$1 + 1);
  /* number of codes at each bit length for an optimal tree */

  // do check in _tr_init()
  //if (static_init_done) return;

  /* For some embedded targets, global variables are not initialized: */
/*#ifdef NO_INIT_GLOBAL_POINTERS
  static_l_desc.static_tree = static_ltree;
  static_l_desc.extra_bits = extra_lbits;
  static_d_desc.static_tree = static_dtree;
  static_d_desc.extra_bits = extra_dbits;
  static_bl_desc.extra_bits = extra_blbits;
#endif*/

  /* Initialize the mapping length (0..255) -> length code (0..28) */
  length = 0;
  for (code = 0; code < LENGTH_CODES$1$1 - 1; code++) {
    base_length$1[code] = length;
    for (n = 0; n < (1 << extra_lbits$1[code]); n++) {
      _length_code$1[length++] = code;
    }
  }
  //Assert (length == 256, "tr_static_init: length != 256");
  /* Note that the length 255 (match length 258) can be represented
   * in two different ways: code 284 + 5 bits or code 285, so we
   * overwrite length_code[255] to use the best encoding:
   */
  _length_code$1[length - 1] = code;

  /* Initialize the mapping dist (0..32K) -> dist code (0..29) */
  dist = 0;
  for (code = 0; code < 16; code++) {
    base_dist$1[code] = dist;
    for (n = 0; n < (1 << extra_dbits$1[code]); n++) {
      _dist_code$1[dist++] = code;
    }
  }
  //Assert (dist == 256, "tr_static_init: dist != 256");
  dist >>= 7; /* from now on, all distances are divided by 128 */
  for (; code < D_CODES$1$1; code++) {
    base_dist$1[code] = dist << 7;
    for (n = 0; n < (1 << (extra_dbits$1[code] - 7)); n++) {
      _dist_code$1[256 + dist++] = code;
    }
  }
  //Assert (dist == 256, "tr_static_init: 256+dist != 512");

  /* Construct the codes of the static literal tree */
  for (bits = 0; bits <= MAX_BITS$1$1; bits++) {
    bl_count[bits] = 0;
  }

  n = 0;
  while (n <= 143) {
    static_ltree$1[n * 2 + 1]/*.Len*/ = 8;
    n++;
    bl_count[8]++;
  }
  while (n <= 255) {
    static_ltree$1[n * 2 + 1]/*.Len*/ = 9;
    n++;
    bl_count[9]++;
  }
  while (n <= 279) {
    static_ltree$1[n * 2 + 1]/*.Len*/ = 7;
    n++;
    bl_count[7]++;
  }
  while (n <= 287) {
    static_ltree$1[n * 2 + 1]/*.Len*/ = 8;
    n++;
    bl_count[8]++;
  }
  /* Codes 286 and 287 do not exist, but we must include them in the
   * tree construction to get a canonical Huffman tree (longest code
   * all ones)
   */
  gen_codes$1(static_ltree$1, L_CODES$1$1 + 1, bl_count);

  /* The static distance tree is trivial: */
  for (n = 0; n < D_CODES$1$1; n++) {
    static_dtree$1[n * 2 + 1]/*.Len*/ = 5;
    static_dtree$1[n * 2]/*.Code*/ = bi_reverse$1(n, 5);
  }

  // Now data ready and we can init static trees
  static_l_desc$1 = new StaticTreeDesc$1(static_ltree$1, extra_lbits$1, LITERALS$1$1 + 1, L_CODES$1$1, MAX_BITS$1$1);
  static_d_desc$1 = new StaticTreeDesc$1(static_dtree$1, extra_dbits$1, 0,          D_CODES$1$1, MAX_BITS$1$1);
  static_bl_desc$1 = new StaticTreeDesc$1(new Array(0), extra_blbits$1, 0,         BL_CODES$1$1, MAX_BL_BITS$1);

  //static_init_done = true;
};


/* ===========================================================================
 * Initialize a new block.
 */
const init_block$1 = (s) => {

  let n; /* iterates over tree elements */

  /* Initialize the trees. */
  for (n = 0; n < L_CODES$1$1;  n++) { s.dyn_ltree[n * 2]/*.Freq*/ = 0; }
  for (n = 0; n < D_CODES$1$1;  n++) { s.dyn_dtree[n * 2]/*.Freq*/ = 0; }
  for (n = 0; n < BL_CODES$1$1; n++) { s.bl_tree[n * 2]/*.Freq*/ = 0; }

  s.dyn_ltree[END_BLOCK$1 * 2]/*.Freq*/ = 1;
  s.opt_len = s.static_len = 0;
  s.sym_next = s.matches = 0;
};


/* ===========================================================================
 * Flush the bit buffer and align the output on a byte boundary
 */
const bi_windup$1 = (s) =>
{
  if (s.bi_valid > 8) {
    put_short$1(s, s.bi_buf);
  } else if (s.bi_valid > 0) {
    //put_byte(s, (Byte)s->bi_buf);
    s.pending_buf[s.pending++] = s.bi_buf;
  }
  s.bi_buf = 0;
  s.bi_valid = 0;
};

/* ===========================================================================
 * Compares to subtrees, using the tree depth as tie breaker when
 * the subtrees have equal frequency. This minimizes the worst case length.
 */
const smaller$1 = (tree, n, m, depth) => {

  const _n2 = n * 2;
  const _m2 = m * 2;
  return (tree[_n2]/*.Freq*/ < tree[_m2]/*.Freq*/ ||
         (tree[_n2]/*.Freq*/ === tree[_m2]/*.Freq*/ && depth[n] <= depth[m]));
};

/* ===========================================================================
 * Restore the heap property by moving down the tree starting at node k,
 * exchanging a node with the smallest of its two sons if necessary, stopping
 * when the heap property is re-established (each father smaller than its
 * two sons).
 */
const pqdownheap$1 = (s, tree, k) => {
//    deflate_state *s;
//    ct_data *tree;  /* the tree to restore */
//    int k;               /* node to move down */

  const v = s.heap[k];
  let j = k << 1;  /* left son of k */
  while (j <= s.heap_len) {
    /* Set j to the smallest of the two sons: */
    if (j < s.heap_len &&
      smaller$1(tree, s.heap[j + 1], s.heap[j], s.depth)) {
      j++;
    }
    /* Exit if v is smaller than both sons */
    if (smaller$1(tree, v, s.heap[j], s.depth)) { break; }

    /* Exchange v with the smallest son */
    s.heap[k] = s.heap[j];
    k = j;

    /* And continue down the tree, setting j to the left son of k */
    j <<= 1;
  }
  s.heap[k] = v;
};


// inlined manually
// const SMALLEST = 1;

/* ===========================================================================
 * Send the block data compressed using the given Huffman trees
 */
const compress_block$1 = (s, ltree, dtree) => {
//    deflate_state *s;
//    const ct_data *ltree; /* literal tree */
//    const ct_data *dtree; /* distance tree */

  let dist;           /* distance of matched string */
  let lc;             /* match length or unmatched char (if dist == 0) */
  let sx = 0;         /* running index in sym_buf */
  let code;           /* the code to send */
  let extra;          /* number of extra bits to send */

  if (s.sym_next !== 0) {
    do {
      dist = s.pending_buf[s.sym_buf + sx++] & 0xff;
      dist += (s.pending_buf[s.sym_buf + sx++] & 0xff) << 8;
      lc = s.pending_buf[s.sym_buf + sx++];
      if (dist === 0) {
        send_code$1(s, lc, ltree); /* send a literal byte */
        //Tracecv(isgraph(lc), (stderr," '%c' ", lc));
      } else {
        /* Here, lc is the match length - MIN_MATCH */
        code = _length_code$1[lc];
        send_code$1(s, code + LITERALS$1$1 + 1, ltree); /* send the length code */
        extra = extra_lbits$1[code];
        if (extra !== 0) {
          lc -= base_length$1[code];
          send_bits$1(s, lc, extra);       /* send the extra length bits */
        }
        dist--; /* dist is now the match distance - 1 */
        code = d_code$1(dist);
        //Assert (code < D_CODES, "bad d_code");

        send_code$1(s, code, dtree);       /* send the distance code */
        extra = extra_dbits$1[code];
        if (extra !== 0) {
          dist -= base_dist$1[code];
          send_bits$1(s, dist, extra);   /* send the extra distance bits */
        }
      } /* literal or match pair ? */

      /* Check that the overlay between pending_buf and sym_buf is ok: */
      //Assert(s->pending < s->lit_bufsize + sx, "pendingBuf overflow");

    } while (sx < s.sym_next);
  }

  send_code$1(s, END_BLOCK$1, ltree);
};


/* ===========================================================================
 * Construct one Huffman tree and assigns the code bit strings and lengths.
 * Update the total bit length for the current block.
 * IN assertion: the field freq is set for all tree elements.
 * OUT assertions: the fields len and code are set to the optimal bit length
 *     and corresponding code. The length opt_len is updated; static_len is
 *     also updated if stree is not null. The field max_code is set.
 */
const build_tree$1 = (s, desc) => {
//    deflate_state *s;
//    tree_desc *desc; /* the tree descriptor */

  const tree     = desc.dyn_tree;
  const stree    = desc.stat_desc.static_tree;
  const has_stree = desc.stat_desc.has_stree;
  const elems    = desc.stat_desc.elems;
  let n, m;          /* iterate over heap elements */
  let max_code = -1; /* largest code with non zero frequency */
  let node;          /* new node being created */

  /* Construct the initial heap, with least frequent element in
   * heap[SMALLEST]. The sons of heap[n] are heap[2*n] and heap[2*n+1].
   * heap[0] is not used.
   */
  s.heap_len = 0;
  s.heap_max = HEAP_SIZE$1$1;

  for (n = 0; n < elems; n++) {
    if (tree[n * 2]/*.Freq*/ !== 0) {
      s.heap[++s.heap_len] = max_code = n;
      s.depth[n] = 0;

    } else {
      tree[n * 2 + 1]/*.Len*/ = 0;
    }
  }

  /* The pkzip format requires that at least one distance code exists,
   * and that at least one bit should be sent even if there is only one
   * possible code. So to avoid special checks later on we force at least
   * two codes of non zero frequency.
   */
  while (s.heap_len < 2) {
    node = s.heap[++s.heap_len] = (max_code < 2 ? ++max_code : 0);
    tree[node * 2]/*.Freq*/ = 1;
    s.depth[node] = 0;
    s.opt_len--;

    if (has_stree) {
      s.static_len -= stree[node * 2 + 1]/*.Len*/;
    }
    /* node is 0 or 1 so it does not have extra bits */
  }
  desc.max_code = max_code;

  /* The elements heap[heap_len/2+1 .. heap_len] are leaves of the tree,
   * establish sub-heaps of increasing lengths:
   */
  for (n = (s.heap_len >> 1/*int /2*/); n >= 1; n--) { pqdownheap$1(s, tree, n); }

  /* Construct the Huffman tree by repeatedly combining the least two
   * frequent nodes.
   */
  node = elems;              /* next internal node of the tree */
  do {
    //pqremove(s, tree, n);  /* n = node of least frequency */
    /*** pqremove ***/
    n = s.heap[1/*SMALLEST*/];
    s.heap[1/*SMALLEST*/] = s.heap[s.heap_len--];
    pqdownheap$1(s, tree, 1/*SMALLEST*/);
    /***/

    m = s.heap[1/*SMALLEST*/]; /* m = node of next least frequency */

    s.heap[--s.heap_max] = n; /* keep the nodes sorted by frequency */
    s.heap[--s.heap_max] = m;

    /* Create a new node father of n and m */
    tree[node * 2]/*.Freq*/ = tree[n * 2]/*.Freq*/ + tree[m * 2]/*.Freq*/;
    s.depth[node] = (s.depth[n] >= s.depth[m] ? s.depth[n] : s.depth[m]) + 1;
    tree[n * 2 + 1]/*.Dad*/ = tree[m * 2 + 1]/*.Dad*/ = node;

    /* and insert the new node in the heap */
    s.heap[1/*SMALLEST*/] = node++;
    pqdownheap$1(s, tree, 1/*SMALLEST*/);

  } while (s.heap_len >= 2);

  s.heap[--s.heap_max] = s.heap[1/*SMALLEST*/];

  /* At this point, the fields freq and dad are set. We can now
   * generate the bit lengths.
   */
  gen_bitlen$1(s, desc);

  /* The field len is now set, we can generate the bit codes */
  gen_codes$1(tree, max_code, s.bl_count);
};


/* ===========================================================================
 * Scan a literal or distance tree to determine the frequencies of the codes
 * in the bit length tree.
 */
const scan_tree$1 = (s, tree, max_code) => {
//    deflate_state *s;
//    ct_data *tree;   /* the tree to be scanned */
//    int max_code;    /* and its largest code of non zero frequency */

  let n;                     /* iterates over all tree elements */
  let prevlen = -1;          /* last emitted length */
  let curlen;                /* length of current code */

  let nextlen = tree[0 * 2 + 1]/*.Len*/; /* length of next code */

  let count = 0;             /* repeat count of the current code */
  let max_count = 7;         /* max repeat count */
  let min_count = 4;         /* min repeat count */

  if (nextlen === 0) {
    max_count = 138;
    min_count = 3;
  }
  tree[(max_code + 1) * 2 + 1]/*.Len*/ = 0xffff; /* guard */

  for (n = 0; n <= max_code; n++) {
    curlen = nextlen;
    nextlen = tree[(n + 1) * 2 + 1]/*.Len*/;

    if (++count < max_count && curlen === nextlen) {
      continue;

    } else if (count < min_count) {
      s.bl_tree[curlen * 2]/*.Freq*/ += count;

    } else if (curlen !== 0) {

      if (curlen !== prevlen) { s.bl_tree[curlen * 2]/*.Freq*/++; }
      s.bl_tree[REP_3_6$1 * 2]/*.Freq*/++;

    } else if (count <= 10) {
      s.bl_tree[REPZ_3_10$1 * 2]/*.Freq*/++;

    } else {
      s.bl_tree[REPZ_11_138$1 * 2]/*.Freq*/++;
    }

    count = 0;
    prevlen = curlen;

    if (nextlen === 0) {
      max_count = 138;
      min_count = 3;

    } else if (curlen === nextlen) {
      max_count = 6;
      min_count = 3;

    } else {
      max_count = 7;
      min_count = 4;
    }
  }
};


/* ===========================================================================
 * Send a literal or distance tree in compressed form, using the codes in
 * bl_tree.
 */
const send_tree$1 = (s, tree, max_code) => {
//    deflate_state *s;
//    ct_data *tree; /* the tree to be scanned */
//    int max_code;       /* and its largest code of non zero frequency */

  let n;                     /* iterates over all tree elements */
  let prevlen = -1;          /* last emitted length */
  let curlen;                /* length of current code */

  let nextlen = tree[0 * 2 + 1]/*.Len*/; /* length of next code */

  let count = 0;             /* repeat count of the current code */
  let max_count = 7;         /* max repeat count */
  let min_count = 4;         /* min repeat count */

  /* tree[max_code+1].Len = -1; */  /* guard already set */
  if (nextlen === 0) {
    max_count = 138;
    min_count = 3;
  }

  for (n = 0; n <= max_code; n++) {
    curlen = nextlen;
    nextlen = tree[(n + 1) * 2 + 1]/*.Len*/;

    if (++count < max_count && curlen === nextlen) {
      continue;

    } else if (count < min_count) {
      do { send_code$1(s, curlen, s.bl_tree); } while (--count !== 0);

    } else if (curlen !== 0) {
      if (curlen !== prevlen) {
        send_code$1(s, curlen, s.bl_tree);
        count--;
      }
      //Assert(count >= 3 && count <= 6, " 3_6?");
      send_code$1(s, REP_3_6$1, s.bl_tree);
      send_bits$1(s, count - 3, 2);

    } else if (count <= 10) {
      send_code$1(s, REPZ_3_10$1, s.bl_tree);
      send_bits$1(s, count - 3, 3);

    } else {
      send_code$1(s, REPZ_11_138$1, s.bl_tree);
      send_bits$1(s, count - 11, 7);
    }

    count = 0;
    prevlen = curlen;
    if (nextlen === 0) {
      max_count = 138;
      min_count = 3;

    } else if (curlen === nextlen) {
      max_count = 6;
      min_count = 3;

    } else {
      max_count = 7;
      min_count = 4;
    }
  }
};


/* ===========================================================================
 * Construct the Huffman tree for the bit lengths and return the index in
 * bl_order of the last bit length code to send.
 */
const build_bl_tree$1 = (s) => {

  let max_blindex;  /* index of last bit length code of non zero freq */

  /* Determine the bit length frequencies for literal and distance trees */
  scan_tree$1(s, s.dyn_ltree, s.l_desc.max_code);
  scan_tree$1(s, s.dyn_dtree, s.d_desc.max_code);

  /* Build the bit length tree: */
  build_tree$1(s, s.bl_desc);
  /* opt_len now includes the length of the tree representations, except
   * the lengths of the bit lengths codes and the 5+5+4 bits for the counts.
   */

  /* Determine the number of bit length codes to send. The pkzip format
   * requires that at least 4 bit length codes be sent. (appnote.txt says
   * 3 but the actual value used is 4.)
   */
  for (max_blindex = BL_CODES$1$1 - 1; max_blindex >= 3; max_blindex--) {
    if (s.bl_tree[bl_order$1[max_blindex] * 2 + 1]/*.Len*/ !== 0) {
      break;
    }
  }
  /* Update opt_len to include the bit length tree and counts */
  s.opt_len += 3 * (max_blindex + 1) + 5 + 5 + 4;
  //Tracev((stderr, "\ndyn trees: dyn %ld, stat %ld",
  //        s->opt_len, s->static_len));

  return max_blindex;
};


/* ===========================================================================
 * Send the header for a block using dynamic Huffman trees: the counts, the
 * lengths of the bit length codes, the literal tree and the distance tree.
 * IN assertion: lcodes >= 257, dcodes >= 1, blcodes >= 4.
 */
const send_all_trees$1 = (s, lcodes, dcodes, blcodes) => {
//    deflate_state *s;
//    int lcodes, dcodes, blcodes; /* number of codes for each tree */

  let rank;                    /* index in bl_order */

  //Assert (lcodes >= 257 && dcodes >= 1 && blcodes >= 4, "not enough codes");
  //Assert (lcodes <= L_CODES && dcodes <= D_CODES && blcodes <= BL_CODES,
  //        "too many codes");
  //Tracev((stderr, "\nbl counts: "));
  send_bits$1(s, lcodes - 257, 5); /* not +255 as stated in appnote.txt */
  send_bits$1(s, dcodes - 1,   5);
  send_bits$1(s, blcodes - 4,  4); /* not -3 as stated in appnote.txt */
  for (rank = 0; rank < blcodes; rank++) {
    //Tracev((stderr, "\nbl code %2d ", bl_order[rank]));
    send_bits$1(s, s.bl_tree[bl_order$1[rank] * 2 + 1]/*.Len*/, 3);
  }
  //Tracev((stderr, "\nbl tree: sent %ld", s->bits_sent));

  send_tree$1(s, s.dyn_ltree, lcodes - 1); /* literal tree */
  //Tracev((stderr, "\nlit tree: sent %ld", s->bits_sent));

  send_tree$1(s, s.dyn_dtree, dcodes - 1); /* distance tree */
  //Tracev((stderr, "\ndist tree: sent %ld", s->bits_sent));
};


/* ===========================================================================
 * Check if the data type is TEXT or BINARY, using the following algorithm:
 * - TEXT if the two conditions below are satisfied:
 *    a) There are no non-portable control characters belonging to the
 *       "block list" (0..6, 14..25, 28..31).
 *    b) There is at least one printable character belonging to the
 *       "allow list" (9 {TAB}, 10 {LF}, 13 {CR}, 32..255).
 * - BINARY otherwise.
 * - The following partially-portable control characters form a
 *   "gray list" that is ignored in this detection algorithm:
 *   (7 {BEL}, 8 {BS}, 11 {VT}, 12 {FF}, 26 {SUB}, 27 {ESC}).
 * IN assertion: the fields Freq of dyn_ltree are set.
 */
const detect_data_type$1 = (s) => {
  /* block_mask is the bit mask of block-listed bytes
   * set bits 0..6, 14..25, and 28..31
   * 0xf3ffc07f = binary 11110011111111111100000001111111
   */
  let block_mask = 0xf3ffc07f;
  let n;

  /* Check for non-textual ("block-listed") bytes. */
  for (n = 0; n <= 31; n++, block_mask >>>= 1) {
    if ((block_mask & 1) && (s.dyn_ltree[n * 2]/*.Freq*/ !== 0)) {
      return Z_BINARY$1;
    }
  }

  /* Check for textual ("allow-listed") bytes. */
  if (s.dyn_ltree[9 * 2]/*.Freq*/ !== 0 || s.dyn_ltree[10 * 2]/*.Freq*/ !== 0 ||
      s.dyn_ltree[13 * 2]/*.Freq*/ !== 0) {
    return Z_TEXT$1;
  }
  for (n = 32; n < LITERALS$1$1; n++) {
    if (s.dyn_ltree[n * 2]/*.Freq*/ !== 0) {
      return Z_TEXT$1;
    }
  }

  /* There are no "block-listed" or "allow-listed" bytes:
   * this stream either is empty or has tolerated ("gray-listed") bytes only.
   */
  return Z_BINARY$1;
};


let static_init_done$1 = false;

/* ===========================================================================
 * Initialize the tree data structures for a new zlib stream.
 */
const _tr_init$1$1 = (s) =>
{

  if (!static_init_done$1) {
    tr_static_init$1();
    static_init_done$1 = true;
  }

  s.l_desc  = new TreeDesc$1(s.dyn_ltree, static_l_desc$1);
  s.d_desc  = new TreeDesc$1(s.dyn_dtree, static_d_desc$1);
  s.bl_desc = new TreeDesc$1(s.bl_tree, static_bl_desc$1);

  s.bi_buf = 0;
  s.bi_valid = 0;

  /* Initialize the first block of the first file: */
  init_block$1(s);
};


/* ===========================================================================
 * Send a stored block
 */
const _tr_stored_block$1$1 = (s, buf, stored_len, last) => {
//DeflateState *s;
//charf *buf;       /* input block */
//ulg stored_len;   /* length of input block */
//int last;         /* one if this is the last block for a file */

  send_bits$1(s, (STORED_BLOCK$1 << 1) + (last ? 1 : 0), 3);    /* send block type */
  bi_windup$1(s);        /* align on byte boundary */
  put_short$1(s, stored_len);
  put_short$1(s, ~stored_len);
  if (stored_len) {
    s.pending_buf.set(s.window.subarray(buf, buf + stored_len), s.pending);
  }
  s.pending += stored_len;
};


/* ===========================================================================
 * Send one empty static block to give enough lookahead for inflate.
 * This takes 10 bits, of which 7 may remain in the bit buffer.
 */
const _tr_align$1$1 = (s) => {
  send_bits$1(s, STATIC_TREES$1 << 1, 3);
  send_code$1(s, END_BLOCK$1, static_ltree$1);
  bi_flush$1(s);
};


/* ===========================================================================
 * Determine the best encoding for the current block: dynamic trees, static
 * trees or store, and write out the encoded block.
 */
const _tr_flush_block$1$1 = (s, buf, stored_len, last) => {
//DeflateState *s;
//charf *buf;       /* input block, or NULL if too old */
//ulg stored_len;   /* length of input block */
//int last;         /* one if this is the last block for a file */

  let opt_lenb, static_lenb;  /* opt_len and static_len in bytes */
  let max_blindex = 0;        /* index of last bit length code of non zero freq */

  /* Build the Huffman trees unless a stored block is forced */
  if (s.level > 0) {

    /* Check if the file is binary or text */
    if (s.strm.data_type === Z_UNKNOWN$1$1) {
      s.strm.data_type = detect_data_type$1(s);
    }

    /* Construct the literal and distance trees */
    build_tree$1(s, s.l_desc);
    // Tracev((stderr, "\nlit data: dyn %ld, stat %ld", s->opt_len,
    //        s->static_len));

    build_tree$1(s, s.d_desc);
    // Tracev((stderr, "\ndist data: dyn %ld, stat %ld", s->opt_len,
    //        s->static_len));
    /* At this point, opt_len and static_len are the total bit lengths of
     * the compressed block data, excluding the tree representations.
     */

    /* Build the bit length tree for the above two trees, and get the index
     * in bl_order of the last bit length code to send.
     */
    max_blindex = build_bl_tree$1(s);

    /* Determine the best encoding. Compute the block lengths in bytes. */
    opt_lenb = (s.opt_len + 3 + 7) >>> 3;
    static_lenb = (s.static_len + 3 + 7) >>> 3;

    // Tracev((stderr, "\nopt %lu(%lu) stat %lu(%lu) stored %lu lit %u ",
    //        opt_lenb, s->opt_len, static_lenb, s->static_len, stored_len,
    //        s->sym_next / 3));

    if (static_lenb <= opt_lenb) { opt_lenb = static_lenb; }

  } else {
    // Assert(buf != (char*)0, "lost buf");
    opt_lenb = static_lenb = stored_len + 5; /* force a stored block */
  }

  if ((stored_len + 4 <= opt_lenb) && (buf !== -1)) {
    /* 4: two words for the lengths */

    /* The test buf != NULL is only necessary if LIT_BUFSIZE > WSIZE.
     * Otherwise we can't have processed more than WSIZE input bytes since
     * the last block flush, because compression would have been
     * successful. If LIT_BUFSIZE <= WSIZE, it is never too late to
     * transform a block into a stored block.
     */
    _tr_stored_block$1$1(s, buf, stored_len, last);

  } else if (s.strategy === Z_FIXED$1$1 || static_lenb === opt_lenb) {

    send_bits$1(s, (STATIC_TREES$1 << 1) + (last ? 1 : 0), 3);
    compress_block$1(s, static_ltree$1, static_dtree$1);

  } else {
    send_bits$1(s, (DYN_TREES$1 << 1) + (last ? 1 : 0), 3);
    send_all_trees$1(s, s.l_desc.max_code + 1, s.d_desc.max_code + 1, max_blindex + 1);
    compress_block$1(s, s.dyn_ltree, s.dyn_dtree);
  }
  // Assert (s->compressed_len == s->bits_sent, "bad compressed size");
  /* The above check is made mod 2^32, for files larger than 512 MB
   * and uLong implemented on 32 bits.
   */
  init_block$1(s);

  if (last) {
    bi_windup$1(s);
  }
  // Tracev((stderr,"\ncomprlen %lu(%lu) ", s->compressed_len>>3,
  //       s->compressed_len-7*last));
};

/* ===========================================================================
 * Save the match info and tally the frequency counts. Return true if
 * the current block must be flushed.
 */
const _tr_tally$1$1 = (s, dist, lc) => {
//    deflate_state *s;
//    unsigned dist;  /* distance of matched string */
//    unsigned lc;    /* match length-MIN_MATCH or unmatched char (if dist==0) */

  s.pending_buf[s.sym_buf + s.sym_next++] = dist;
  s.pending_buf[s.sym_buf + s.sym_next++] = dist >> 8;
  s.pending_buf[s.sym_buf + s.sym_next++] = lc;
  if (dist === 0) {
    /* lc is the unmatched char */
    s.dyn_ltree[lc * 2]/*.Freq*/++;
  } else {
    s.matches++;
    /* Here, lc is the match length - MIN_MATCH */
    dist--;             /* dist = match distance - 1 */
    //Assert((ush)dist < (ush)MAX_DIST(s) &&
    //       (ush)lc <= (ush)(MAX_MATCH-MIN_MATCH) &&
    //       (ush)d_code(dist) < (ush)D_CODES,  "_tr_tally: bad match");

    s.dyn_ltree[(_length_code$1[lc] + LITERALS$1$1 + 1) * 2]/*.Freq*/++;
    s.dyn_dtree[d_code$1(dist) * 2]/*.Freq*/++;
  }

  return (s.sym_next === s.sym_end);
};

var _tr_init_1$1  = _tr_init$1$1;
var _tr_stored_block_1$1 = _tr_stored_block$1$1;
var _tr_flush_block_1$1  = _tr_flush_block$1$1;
var _tr_tally_1$1 = _tr_tally$1$1;
var _tr_align_1$1 = _tr_align$1$1;

var trees$1 = {
	_tr_init: _tr_init_1$1,
	_tr_stored_block: _tr_stored_block_1$1,
	_tr_flush_block: _tr_flush_block_1$1,
	_tr_tally: _tr_tally_1$1,
	_tr_align: _tr_align_1$1
};

// Note: adler32 takes 12% for level 0 and 2% for level 6.
// It isn't worth it to make additional optimizations as in original.
// Small size is preferable.

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

const adler32$1 = (adler, buf, len, pos) => {
  let s1 = (adler & 0xffff) |0,
      s2 = ((adler >>> 16) & 0xffff) |0,
      n = 0;

  while (len !== 0) {
    // Set limit ~ twice less than 5552, to keep
    // s2 in 31-bits, because we force signed ints.
    // in other case %= will fail.
    n = len > 2000 ? 2000 : len;
    len -= n;

    do {
      s1 = (s1 + buf[pos++]) |0;
      s2 = (s2 + s1) |0;
    } while (--n);

    s1 %= 65521;
    s2 %= 65521;
  }

  return (s1 | (s2 << 16)) |0;
};


var adler32_1$1 = adler32$1;

// Note: we can't get significant speed boost here.
// So write code to minimize size - no pregenerated tables
// and array tools dependencies.

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

// Use ordinary array, since untyped makes no boost here
const makeTable$1 = () => {
  let c, table = [];

  for (var n = 0; n < 256; n++) {
    c = n;
    for (var k = 0; k < 8; k++) {
      c = ((c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1));
    }
    table[n] = c;
  }

  return table;
};

// Create table on load. Just 255 signed longs. Not a problem.
const crcTable$1 = new Uint32Array(makeTable$1());


const crc32$1 = (crc, buf, len, pos) => {
  const t = crcTable$1;
  const end = pos + len;

  crc ^= -1;

  for (let i = pos; i < end; i++) {
    crc = (crc >>> 8) ^ t[(crc ^ buf[i]) & 0xFF];
  }

  return (crc ^ (-1)); // >>> 0;
};


var crc32_1$1 = crc32$1;

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

var messages$1 = {
  2:      'need dictionary',     /* Z_NEED_DICT       2  */
  1:      'stream end',          /* Z_STREAM_END      1  */
  0:      '',                    /* Z_OK              0  */
  '-1':   'file error',          /* Z_ERRNO         (-1) */
  '-2':   'stream error',        /* Z_STREAM_ERROR  (-2) */
  '-3':   'data error',          /* Z_DATA_ERROR    (-3) */
  '-4':   'insufficient memory', /* Z_MEM_ERROR     (-4) */
  '-5':   'buffer error',        /* Z_BUF_ERROR     (-5) */
  '-6':   'incompatible version' /* Z_VERSION_ERROR (-6) */
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

var constants$2$1 = {

  /* Allowed flush values; see deflate() and inflate() below for details */
  Z_NO_FLUSH:         0,
  Z_PARTIAL_FLUSH:    1,
  Z_SYNC_FLUSH:       2,
  Z_FULL_FLUSH:       3,
  Z_FINISH:           4,
  Z_BLOCK:            5,
  Z_TREES:            6,

  /* Return codes for the compression/decompression functions. Negative values
  * are errors, positive values are used for special but normal events.
  */
  Z_OK:               0,
  Z_STREAM_END:       1,
  Z_NEED_DICT:        2,
  Z_ERRNO:           -1,
  Z_STREAM_ERROR:    -2,
  Z_DATA_ERROR:      -3,
  Z_MEM_ERROR:       -4,
  Z_BUF_ERROR:       -5,
  //Z_VERSION_ERROR: -6,

  /* compression levels */
  Z_NO_COMPRESSION:         0,
  Z_BEST_SPEED:             1,
  Z_BEST_COMPRESSION:       9,
  Z_DEFAULT_COMPRESSION:   -1,


  Z_FILTERED:               1,
  Z_HUFFMAN_ONLY:           2,
  Z_RLE:                    3,
  Z_FIXED:                  4,
  Z_DEFAULT_STRATEGY:       0,

  /* Possible values of the data_type field (though see inflate()) */
  Z_BINARY:                 0,
  Z_TEXT:                   1,
  //Z_ASCII:                1, // = Z_TEXT (deprecated)
  Z_UNKNOWN:                2,

  /* The deflate compression method */
  Z_DEFLATED:               8
  //Z_NULL:                 null // Use -1 or null inline, depending on var type
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

const { _tr_init: _tr_init$2, _tr_stored_block: _tr_stored_block$2, _tr_flush_block: _tr_flush_block$2, _tr_tally: _tr_tally$2, _tr_align: _tr_align$2 } = trees$1;




/* Public constants ==========================================================*/
/* ===========================================================================*/

const {
  Z_NO_FLUSH: Z_NO_FLUSH$2$1, Z_PARTIAL_FLUSH: Z_PARTIAL_FLUSH$1, Z_FULL_FLUSH: Z_FULL_FLUSH$1$1, Z_FINISH: Z_FINISH$3$1, Z_BLOCK: Z_BLOCK$1$1,
  Z_OK: Z_OK$3$1, Z_STREAM_END: Z_STREAM_END$3$1, Z_STREAM_ERROR: Z_STREAM_ERROR$2$1, Z_DATA_ERROR: Z_DATA_ERROR$2$1, Z_BUF_ERROR: Z_BUF_ERROR$1$1,
  Z_DEFAULT_COMPRESSION: Z_DEFAULT_COMPRESSION$1$1,
  Z_FILTERED: Z_FILTERED$1, Z_HUFFMAN_ONLY: Z_HUFFMAN_ONLY$1, Z_RLE: Z_RLE$1, Z_FIXED: Z_FIXED$2, Z_DEFAULT_STRATEGY: Z_DEFAULT_STRATEGY$1$1,
  Z_UNKNOWN: Z_UNKNOWN$2,
  Z_DEFLATED: Z_DEFLATED$2$1
} = constants$2$1;

/*============================================================================*/


const MAX_MEM_LEVEL$1 = 9;
/* Maximum value for memLevel in deflateInit2 */
const MAX_WBITS$1$1 = 15;
/* 32K LZ77 window */
const DEF_MEM_LEVEL$1 = 8;


const LENGTH_CODES$2  = 29;
/* number of length codes, not counting the special END_BLOCK code */
const LITERALS$2      = 256;
/* number of literal bytes 0..255 */
const L_CODES$2       = LITERALS$2 + 1 + LENGTH_CODES$2;
/* number of Literal or Length codes, including the END_BLOCK code */
const D_CODES$2       = 30;
/* number of distance codes */
const BL_CODES$2      = 19;
/* number of codes used to transfer the bit lengths */
const HEAP_SIZE$2     = 2 * L_CODES$2 + 1;
/* maximum heap size */
const MAX_BITS$2  = 15;
/* All codes must not exceed MAX_BITS bits */

const MIN_MATCH$2 = 3;
const MAX_MATCH$2 = 258;
const MIN_LOOKAHEAD$1 = (MAX_MATCH$2 + MIN_MATCH$2 + 1);

const PRESET_DICT$1 = 0x20;

const INIT_STATE$1    =  42;    /* zlib header -> BUSY_STATE */
//#ifdef GZIP
const GZIP_STATE$1    =  57;    /* gzip header -> BUSY_STATE | EXTRA_STATE */
//#endif
const EXTRA_STATE$1   =  69;    /* gzip extra block -> NAME_STATE */
const NAME_STATE$1    =  73;    /* gzip file name -> COMMENT_STATE */
const COMMENT_STATE$1 =  91;    /* gzip comment -> HCRC_STATE */
const HCRC_STATE$1    = 103;    /* gzip header CRC -> BUSY_STATE */
const BUSY_STATE$1    = 113;    /* deflate -> FINISH_STATE */
const FINISH_STATE$1  = 666;    /* stream complete */

const BS_NEED_MORE$1      = 1; /* block not completed, need more input or more output */
const BS_BLOCK_DONE$1     = 2; /* block flush performed */
const BS_FINISH_STARTED$1 = 3; /* finish started, need only more output at next deflate */
const BS_FINISH_DONE$1    = 4; /* finish done, accept no more input or output */

const OS_CODE$1 = 0x03; // Unix :) . Don't detect, use this default.

const err$1 = (strm, errorCode) => {
  strm.msg = messages$1[errorCode];
  return errorCode;
};

const rank$1 = (f) => {
  return ((f) * 2) - ((f) > 4 ? 9 : 0);
};

const zero$2 = (buf) => {
  let len = buf.length; while (--len >= 0) { buf[len] = 0; }
};

/* ===========================================================================
 * Slide the hash table when sliding the window down (could be avoided with 32
 * bit values at the expense of memory usage). We slide even when level == 0 to
 * keep the hash table consistent if we switch back to level > 0 later.
 */
const slide_hash$1 = (s) => {
  let n, m;
  let p;
  let wsize = s.w_size;

  n = s.hash_size;
  p = n;
  do {
    m = s.head[--p];
    s.head[p] = (m >= wsize ? m - wsize : 0);
  } while (--n);
  n = wsize;
//#ifndef FASTEST
  p = n;
  do {
    m = s.prev[--p];
    s.prev[p] = (m >= wsize ? m - wsize : 0);
    /* If n is not on any hash chain, prev[n] is garbage but
     * its value will never be used.
     */
  } while (--n);
//#endif
};

/* eslint-disable new-cap */
let HASH_ZLIB$1 = (s, prev, data) => ((prev << s.hash_shift) ^ data) & s.hash_mask;
// This hash causes less collisions, https://github.com/nodeca/pako/issues/135
// But breaks binary compatibility
//let HASH_FAST = (s, prev, data) => ((prev << 8) + (prev >> 8) + (data << 4)) & s.hash_mask;
let HASH$1 = HASH_ZLIB$1;


/* =========================================================================
 * Flush as much pending output as possible. All deflate() output, except for
 * some deflate_stored() output, goes through this function so some
 * applications may wish to modify it to avoid allocating a large
 * strm->next_out buffer and copying into it. (See also read_buf()).
 */
const flush_pending$1 = (strm) => {
  const s = strm.state;

  //_tr_flush_bits(s);
  let len = s.pending;
  if (len > strm.avail_out) {
    len = strm.avail_out;
  }
  if (len === 0) { return; }

  strm.output.set(s.pending_buf.subarray(s.pending_out, s.pending_out + len), strm.next_out);
  strm.next_out  += len;
  s.pending_out  += len;
  strm.total_out += len;
  strm.avail_out -= len;
  s.pending      -= len;
  if (s.pending === 0) {
    s.pending_out = 0;
  }
};


const flush_block_only$1 = (s, last) => {
  _tr_flush_block$2(s, (s.block_start >= 0 ? s.block_start : -1), s.strstart - s.block_start, last);
  s.block_start = s.strstart;
  flush_pending$1(s.strm);
};


const put_byte$1 = (s, b) => {
  s.pending_buf[s.pending++] = b;
};


/* =========================================================================
 * Put a short in the pending buffer. The 16-bit value is put in MSB order.
 * IN assertion: the stream state is correct and there is enough room in
 * pending_buf.
 */
const putShortMSB$1 = (s, b) => {

  //  put_byte(s, (Byte)(b >> 8));
//  put_byte(s, (Byte)(b & 0xff));
  s.pending_buf[s.pending++] = (b >>> 8) & 0xff;
  s.pending_buf[s.pending++] = b & 0xff;
};


/* ===========================================================================
 * Read a new buffer from the current input stream, update the adler32
 * and total number of bytes read.  All deflate() input goes through
 * this function so some applications may wish to modify it to avoid
 * allocating a large strm->input buffer and copying from it.
 * (See also flush_pending()).
 */
const read_buf$1 = (strm, buf, start, size) => {

  let len = strm.avail_in;

  if (len > size) { len = size; }
  if (len === 0) { return 0; }

  strm.avail_in -= len;

  // zmemcpy(buf, strm->next_in, len);
  buf.set(strm.input.subarray(strm.next_in, strm.next_in + len), start);
  if (strm.state.wrap === 1) {
    strm.adler = adler32_1$1(strm.adler, buf, len, start);
  }

  else if (strm.state.wrap === 2) {
    strm.adler = crc32_1$1(strm.adler, buf, len, start);
  }

  strm.next_in += len;
  strm.total_in += len;

  return len;
};


/* ===========================================================================
 * Set match_start to the longest match starting at the given string and
 * return its length. Matches shorter or equal to prev_length are discarded,
 * in which case the result is equal to prev_length and match_start is
 * garbage.
 * IN assertions: cur_match is the head of the hash chain for the current
 *   string (strstart) and its distance is <= MAX_DIST, and prev_length >= 1
 * OUT assertion: the match length is not greater than s->lookahead.
 */
const longest_match$1 = (s, cur_match) => {

  let chain_length = s.max_chain_length;      /* max hash chain length */
  let scan = s.strstart; /* current string */
  let match;                       /* matched string */
  let len;                           /* length of current match */
  let best_len = s.prev_length;              /* best match length so far */
  let nice_match = s.nice_match;             /* stop if match long enough */
  const limit = (s.strstart > (s.w_size - MIN_LOOKAHEAD$1)) ?
      s.strstart - (s.w_size - MIN_LOOKAHEAD$1) : 0/*NIL*/;

  const _win = s.window; // shortcut

  const wmask = s.w_mask;
  const prev  = s.prev;

  /* Stop when cur_match becomes <= limit. To simplify the code,
   * we prevent matches with the string of window index 0.
   */

  const strend = s.strstart + MAX_MATCH$2;
  let scan_end1  = _win[scan + best_len - 1];
  let scan_end   = _win[scan + best_len];

  /* The code is optimized for HASH_BITS >= 8 and MAX_MATCH-2 multiple of 16.
   * It is easy to get rid of this optimization if necessary.
   */
  // Assert(s->hash_bits >= 8 && MAX_MATCH == 258, "Code too clever");

  /* Do not waste too much time if we already have a good match: */
  if (s.prev_length >= s.good_match) {
    chain_length >>= 2;
  }
  /* Do not look for matches beyond the end of the input. This is necessary
   * to make deflate deterministic.
   */
  if (nice_match > s.lookahead) { nice_match = s.lookahead; }

  // Assert((ulg)s->strstart <= s->window_size-MIN_LOOKAHEAD, "need lookahead");

  do {
    // Assert(cur_match < s->strstart, "no future");
    match = cur_match;

    /* Skip to next match if the match length cannot increase
     * or if the match length is less than 2.  Note that the checks below
     * for insufficient lookahead only occur occasionally for performance
     * reasons.  Therefore uninitialized memory will be accessed, and
     * conditional jumps will be made that depend on those values.
     * However the length of the match is limited to the lookahead, so
     * the output of deflate is not affected by the uninitialized values.
     */

    if (_win[match + best_len]     !== scan_end  ||
        _win[match + best_len - 1] !== scan_end1 ||
        _win[match]                !== _win[scan] ||
        _win[++match]              !== _win[scan + 1]) {
      continue;
    }

    /* The check at best_len-1 can be removed because it will be made
     * again later. (This heuristic is not always a win.)
     * It is not necessary to compare scan[2] and match[2] since they
     * are always equal when the other bytes match, given that
     * the hash keys are equal and that HASH_BITS >= 8.
     */
    scan += 2;
    match++;
    // Assert(*scan == *match, "match[2]?");

    /* We check for insufficient lookahead only every 8th comparison;
     * the 256th check will be made at strstart+258.
     */
    do {
      /*jshint noempty:false*/
    } while (_win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
             _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
             _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
             _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
             scan < strend);

    // Assert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");

    len = MAX_MATCH$2 - (strend - scan);
    scan = strend - MAX_MATCH$2;

    if (len > best_len) {
      s.match_start = cur_match;
      best_len = len;
      if (len >= nice_match) {
        break;
      }
      scan_end1  = _win[scan + best_len - 1];
      scan_end   = _win[scan + best_len];
    }
  } while ((cur_match = prev[cur_match & wmask]) > limit && --chain_length !== 0);

  if (best_len <= s.lookahead) {
    return best_len;
  }
  return s.lookahead;
};


/* ===========================================================================
 * Fill the window when the lookahead becomes insufficient.
 * Updates strstart and lookahead.
 *
 * IN assertion: lookahead < MIN_LOOKAHEAD
 * OUT assertions: strstart <= window_size-MIN_LOOKAHEAD
 *    At least one byte has been read, or avail_in == 0; reads are
 *    performed for at least two bytes (required for the zip translate_eol
 *    option -- not supported here).
 */
const fill_window$1 = (s) => {

  const _w_size = s.w_size;
  let n, more, str;

  //Assert(s->lookahead < MIN_LOOKAHEAD, "already enough lookahead");

  do {
    more = s.window_size - s.lookahead - s.strstart;

    // JS ints have 32 bit, block below not needed
    /* Deal with !@#$% 64K limit: */
    //if (sizeof(int) <= 2) {
    //    if (more == 0 && s->strstart == 0 && s->lookahead == 0) {
    //        more = wsize;
    //
    //  } else if (more == (unsigned)(-1)) {
    //        /* Very unlikely, but possible on 16 bit machine if
    //         * strstart == 0 && lookahead == 1 (input done a byte at time)
    //         */
    //        more--;
    //    }
    //}


    /* If the window is almost full and there is insufficient lookahead,
     * move the upper half to the lower one to make room in the upper half.
     */
    if (s.strstart >= _w_size + (_w_size - MIN_LOOKAHEAD$1)) {

      s.window.set(s.window.subarray(_w_size, _w_size + _w_size - more), 0);
      s.match_start -= _w_size;
      s.strstart -= _w_size;
      /* we now have strstart >= MAX_DIST */
      s.block_start -= _w_size;
      if (s.insert > s.strstart) {
        s.insert = s.strstart;
      }
      slide_hash$1(s);
      more += _w_size;
    }
    if (s.strm.avail_in === 0) {
      break;
    }

    /* If there was no sliding:
     *    strstart <= WSIZE+MAX_DIST-1 && lookahead <= MIN_LOOKAHEAD - 1 &&
     *    more == window_size - lookahead - strstart
     * => more >= window_size - (MIN_LOOKAHEAD-1 + WSIZE + MAX_DIST-1)
     * => more >= window_size - 2*WSIZE + 2
     * In the BIG_MEM or MMAP case (not yet supported),
     *   window_size == input_size + MIN_LOOKAHEAD  &&
     *   strstart + s->lookahead <= input_size => more >= MIN_LOOKAHEAD.
     * Otherwise, window_size == 2*WSIZE so more >= 2.
     * If there was sliding, more >= WSIZE. So in all cases, more >= 2.
     */
    //Assert(more >= 2, "more < 2");
    n = read_buf$1(s.strm, s.window, s.strstart + s.lookahead, more);
    s.lookahead += n;

    /* Initialize the hash value now that we have some input: */
    if (s.lookahead + s.insert >= MIN_MATCH$2) {
      str = s.strstart - s.insert;
      s.ins_h = s.window[str];

      /* UPDATE_HASH(s, s->ins_h, s->window[str + 1]); */
      s.ins_h = HASH$1(s, s.ins_h, s.window[str + 1]);
//#if MIN_MATCH != 3
//        Call update_hash() MIN_MATCH-3 more times
//#endif
      while (s.insert) {
        /* UPDATE_HASH(s, s->ins_h, s->window[str + MIN_MATCH-1]); */
        s.ins_h = HASH$1(s, s.ins_h, s.window[str + MIN_MATCH$2 - 1]);

        s.prev[str & s.w_mask] = s.head[s.ins_h];
        s.head[s.ins_h] = str;
        str++;
        s.insert--;
        if (s.lookahead + s.insert < MIN_MATCH$2) {
          break;
        }
      }
    }
    /* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,
     * but this is not important since only literal bytes will be emitted.
     */

  } while (s.lookahead < MIN_LOOKAHEAD$1 && s.strm.avail_in !== 0);

  /* If the WIN_INIT bytes after the end of the current data have never been
   * written, then zero those bytes in order to avoid memory check reports of
   * the use of uninitialized (or uninitialised as Julian writes) bytes by
   * the longest match routines.  Update the high water mark for the next
   * time through here.  WIN_INIT is set to MAX_MATCH since the longest match
   * routines allow scanning to strstart + MAX_MATCH, ignoring lookahead.
   */
//  if (s.high_water < s.window_size) {
//    const curr = s.strstart + s.lookahead;
//    let init = 0;
//
//    if (s.high_water < curr) {
//      /* Previous high water mark below current data -- zero WIN_INIT
//       * bytes or up to end of window, whichever is less.
//       */
//      init = s.window_size - curr;
//      if (init > WIN_INIT)
//        init = WIN_INIT;
//      zmemzero(s->window + curr, (unsigned)init);
//      s->high_water = curr + init;
//    }
//    else if (s->high_water < (ulg)curr + WIN_INIT) {
//      /* High water mark at or above current data, but below current data
//       * plus WIN_INIT -- zero out to current data plus WIN_INIT, or up
//       * to end of window, whichever is less.
//       */
//      init = (ulg)curr + WIN_INIT - s->high_water;
//      if (init > s->window_size - s->high_water)
//        init = s->window_size - s->high_water;
//      zmemzero(s->window + s->high_water, (unsigned)init);
//      s->high_water += init;
//    }
//  }
//
//  Assert((ulg)s->strstart <= s->window_size - MIN_LOOKAHEAD,
//    "not enough room for search");
};

/* ===========================================================================
 * Copy without compression as much as possible from the input stream, return
 * the current block state.
 *
 * In case deflateParams() is used to later switch to a non-zero compression
 * level, s->matches (otherwise unused when storing) keeps track of the number
 * of hash table slides to perform. If s->matches is 1, then one hash table
 * slide will be done when switching. If s->matches is 2, the maximum value
 * allowed here, then the hash table will be cleared, since two or more slides
 * is the same as a clear.
 *
 * deflate_stored() is written to minimize the number of times an input byte is
 * copied. It is most efficient with large input and output buffers, which
 * maximizes the opportunites to have a single copy from next_in to next_out.
 */
const deflate_stored$1 = (s, flush) => {

  /* Smallest worthy block size when not flushing or finishing. By default
   * this is 32K. This can be as small as 507 bytes for memLevel == 1. For
   * large input and output buffers, the stored block size will be larger.
   */
  let min_block = s.pending_buf_size - 5 > s.w_size ? s.w_size : s.pending_buf_size - 5;

  /* Copy as many min_block or larger stored blocks directly to next_out as
   * possible. If flushing, copy the remaining available input to next_out as
   * stored blocks, if there is enough space.
   */
  let len, left, have, last = 0;
  let used = s.strm.avail_in;
  do {
    /* Set len to the maximum size block that we can copy directly with the
     * available input data and output space. Set left to how much of that
     * would be copied from what's left in the window.
     */
    len = 65535/* MAX_STORED */;     /* maximum deflate stored block length */
    have = (s.bi_valid + 42) >> 3;     /* number of header bytes */
    if (s.strm.avail_out < have) {         /* need room for header */
      break;
    }
      /* maximum stored block length that will fit in avail_out: */
    have = s.strm.avail_out - have;
    left = s.strstart - s.block_start;  /* bytes left in window */
    if (len > left + s.strm.avail_in) {
      len = left + s.strm.avail_in;   /* limit len to the input */
    }
    if (len > have) {
      len = have;             /* limit len to the output */
    }

    /* If the stored block would be less than min_block in length, or if
     * unable to copy all of the available input when flushing, then try
     * copying to the window and the pending buffer instead. Also don't
     * write an empty block when flushing -- deflate() does that.
     */
    if (len < min_block && ((len === 0 && flush !== Z_FINISH$3$1) ||
                        flush === Z_NO_FLUSH$2$1 ||
                        len !== left + s.strm.avail_in)) {
      break;
    }

    /* Make a dummy stored block in pending to get the header bytes,
     * including any pending bits. This also updates the debugging counts.
     */
    last = flush === Z_FINISH$3$1 && len === left + s.strm.avail_in ? 1 : 0;
    _tr_stored_block$2(s, 0, 0, last);

    /* Replace the lengths in the dummy stored block with len. */
    s.pending_buf[s.pending - 4] = len;
    s.pending_buf[s.pending - 3] = len >> 8;
    s.pending_buf[s.pending - 2] = ~len;
    s.pending_buf[s.pending - 1] = ~len >> 8;

    /* Write the stored block header bytes. */
    flush_pending$1(s.strm);

//#ifdef ZLIB_DEBUG
//    /* Update debugging counts for the data about to be copied. */
//    s->compressed_len += len << 3;
//    s->bits_sent += len << 3;
//#endif

    /* Copy uncompressed bytes from the window to next_out. */
    if (left) {
      if (left > len) {
        left = len;
      }
      //zmemcpy(s->strm->next_out, s->window + s->block_start, left);
      s.strm.output.set(s.window.subarray(s.block_start, s.block_start + left), s.strm.next_out);
      s.strm.next_out += left;
      s.strm.avail_out -= left;
      s.strm.total_out += left;
      s.block_start += left;
      len -= left;
    }

    /* Copy uncompressed bytes directly from next_in to next_out, updating
     * the check value.
     */
    if (len) {
      read_buf$1(s.strm, s.strm.output, s.strm.next_out, len);
      s.strm.next_out += len;
      s.strm.avail_out -= len;
      s.strm.total_out += len;
    }
  } while (last === 0);

  /* Update the sliding window with the last s->w_size bytes of the copied
   * data, or append all of the copied data to the existing window if less
   * than s->w_size bytes were copied. Also update the number of bytes to
   * insert in the hash tables, in the event that deflateParams() switches to
   * a non-zero compression level.
   */
  used -= s.strm.avail_in;    /* number of input bytes directly copied */
  if (used) {
    /* If any input was used, then no unused input remains in the window,
     * therefore s->block_start == s->strstart.
     */
    if (used >= s.w_size) {  /* supplant the previous history */
      s.matches = 2;     /* clear hash */
      //zmemcpy(s->window, s->strm->next_in - s->w_size, s->w_size);
      s.window.set(s.strm.input.subarray(s.strm.next_in - s.w_size, s.strm.next_in), 0);
      s.strstart = s.w_size;
      s.insert = s.strstart;
    }
    else {
      if (s.window_size - s.strstart <= used) {
        /* Slide the window down. */
        s.strstart -= s.w_size;
        //zmemcpy(s->window, s->window + s->w_size, s->strstart);
        s.window.set(s.window.subarray(s.w_size, s.w_size + s.strstart), 0);
        if (s.matches < 2) {
          s.matches++;   /* add a pending slide_hash() */
        }
        if (s.insert > s.strstart) {
          s.insert = s.strstart;
        }
      }
      //zmemcpy(s->window + s->strstart, s->strm->next_in - used, used);
      s.window.set(s.strm.input.subarray(s.strm.next_in - used, s.strm.next_in), s.strstart);
      s.strstart += used;
      s.insert += used > s.w_size - s.insert ? s.w_size - s.insert : used;
    }
    s.block_start = s.strstart;
  }
  if (s.high_water < s.strstart) {
    s.high_water = s.strstart;
  }

  /* If the last block was written to next_out, then done. */
  if (last) {
    return BS_FINISH_DONE$1;
  }

  /* If flushing and all input has been consumed, then done. */
  if (flush !== Z_NO_FLUSH$2$1 && flush !== Z_FINISH$3$1 &&
    s.strm.avail_in === 0 && s.strstart === s.block_start) {
    return BS_BLOCK_DONE$1;
  }

  /* Fill the window with any remaining input. */
  have = s.window_size - s.strstart;
  if (s.strm.avail_in > have && s.block_start >= s.w_size) {
    /* Slide the window down. */
    s.block_start -= s.w_size;
    s.strstart -= s.w_size;
    //zmemcpy(s->window, s->window + s->w_size, s->strstart);
    s.window.set(s.window.subarray(s.w_size, s.w_size + s.strstart), 0);
    if (s.matches < 2) {
      s.matches++;       /* add a pending slide_hash() */
    }
    have += s.w_size;      /* more space now */
    if (s.insert > s.strstart) {
      s.insert = s.strstart;
    }
  }
  if (have > s.strm.avail_in) {
    have = s.strm.avail_in;
  }
  if (have) {
    read_buf$1(s.strm, s.window, s.strstart, have);
    s.strstart += have;
    s.insert += have > s.w_size - s.insert ? s.w_size - s.insert : have;
  }
  if (s.high_water < s.strstart) {
    s.high_water = s.strstart;
  }

  /* There was not enough avail_out to write a complete worthy or flushed
   * stored block to next_out. Write a stored block to pending instead, if we
   * have enough input for a worthy block, or if flushing and there is enough
   * room for the remaining input as a stored block in the pending buffer.
   */
  have = (s.bi_valid + 42) >> 3;     /* number of header bytes */
    /* maximum stored block length that will fit in pending: */
  have = s.pending_buf_size - have > 65535/* MAX_STORED */ ? 65535/* MAX_STORED */ : s.pending_buf_size - have;
  min_block = have > s.w_size ? s.w_size : have;
  left = s.strstart - s.block_start;
  if (left >= min_block ||
     ((left || flush === Z_FINISH$3$1) && flush !== Z_NO_FLUSH$2$1 &&
     s.strm.avail_in === 0 && left <= have)) {
    len = left > have ? have : left;
    last = flush === Z_FINISH$3$1 && s.strm.avail_in === 0 &&
         len === left ? 1 : 0;
    _tr_stored_block$2(s, s.block_start, len, last);
    s.block_start += len;
    flush_pending$1(s.strm);
  }

  /* We've done all we can with the available input and output. */
  return last ? BS_FINISH_STARTED$1 : BS_NEED_MORE$1;
};


/* ===========================================================================
 * Compress as much as possible from the input stream, return the current
 * block state.
 * This function does not perform lazy evaluation of matches and inserts
 * new strings in the dictionary only for unmatched strings or for short
 * matches. It is used only for the fast compression options.
 */
const deflate_fast$1 = (s, flush) => {

  let hash_head;        /* head of the hash chain */
  let bflush;           /* set if current block must be flushed */

  for (;;) {
    /* Make sure that we always have enough lookahead, except
     * at the end of the input file. We need MAX_MATCH bytes
     * for the next match, plus MIN_MATCH bytes to insert the
     * string following the next match.
     */
    if (s.lookahead < MIN_LOOKAHEAD$1) {
      fill_window$1(s);
      if (s.lookahead < MIN_LOOKAHEAD$1 && flush === Z_NO_FLUSH$2$1) {
        return BS_NEED_MORE$1;
      }
      if (s.lookahead === 0) {
        break; /* flush the current block */
      }
    }

    /* Insert the string window[strstart .. strstart+2] in the
     * dictionary, and set hash_head to the head of the hash chain:
     */
    hash_head = 0/*NIL*/;
    if (s.lookahead >= MIN_MATCH$2) {
      /*** INSERT_STRING(s, s.strstart, hash_head); ***/
      s.ins_h = HASH$1(s, s.ins_h, s.window[s.strstart + MIN_MATCH$2 - 1]);
      hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
      s.head[s.ins_h] = s.strstart;
      /***/
    }

    /* Find the longest match, discarding those <= prev_length.
     * At this point we have always match_length < MIN_MATCH
     */
    if (hash_head !== 0/*NIL*/ && ((s.strstart - hash_head) <= (s.w_size - MIN_LOOKAHEAD$1))) {
      /* To simplify the code, we prevent matches with the string
       * of window index 0 (in particular we have to avoid a match
       * of the string with itself at the start of the input file).
       */
      s.match_length = longest_match$1(s, hash_head);
      /* longest_match() sets match_start */
    }
    if (s.match_length >= MIN_MATCH$2) {
      // check_match(s, s.strstart, s.match_start, s.match_length); // for debug only

      /*** _tr_tally_dist(s, s.strstart - s.match_start,
                     s.match_length - MIN_MATCH, bflush); ***/
      bflush = _tr_tally$2(s, s.strstart - s.match_start, s.match_length - MIN_MATCH$2);

      s.lookahead -= s.match_length;

      /* Insert new strings in the hash table only if the match length
       * is not too large. This saves time but degrades compression.
       */
      if (s.match_length <= s.max_lazy_match/*max_insert_length*/ && s.lookahead >= MIN_MATCH$2) {
        s.match_length--; /* string at strstart already in table */
        do {
          s.strstart++;
          /*** INSERT_STRING(s, s.strstart, hash_head); ***/
          s.ins_h = HASH$1(s, s.ins_h, s.window[s.strstart + MIN_MATCH$2 - 1]);
          hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
          s.head[s.ins_h] = s.strstart;
          /***/
          /* strstart never exceeds WSIZE-MAX_MATCH, so there are
           * always MIN_MATCH bytes ahead.
           */
        } while (--s.match_length !== 0);
        s.strstart++;
      } else
      {
        s.strstart += s.match_length;
        s.match_length = 0;
        s.ins_h = s.window[s.strstart];
        /* UPDATE_HASH(s, s.ins_h, s.window[s.strstart+1]); */
        s.ins_h = HASH$1(s, s.ins_h, s.window[s.strstart + 1]);

//#if MIN_MATCH != 3
//                Call UPDATE_HASH() MIN_MATCH-3 more times
//#endif
        /* If lookahead < MIN_MATCH, ins_h is garbage, but it does not
         * matter since it will be recomputed at next deflate call.
         */
      }
    } else {
      /* No match, output a literal byte */
      //Tracevv((stderr,"%c", s.window[s.strstart]));
      /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
      bflush = _tr_tally$2(s, 0, s.window[s.strstart]);

      s.lookahead--;
      s.strstart++;
    }
    if (bflush) {
      /*** FLUSH_BLOCK(s, 0); ***/
      flush_block_only$1(s, false);
      if (s.strm.avail_out === 0) {
        return BS_NEED_MORE$1;
      }
      /***/
    }
  }
  s.insert = ((s.strstart < (MIN_MATCH$2 - 1)) ? s.strstart : MIN_MATCH$2 - 1);
  if (flush === Z_FINISH$3$1) {
    /*** FLUSH_BLOCK(s, 1); ***/
    flush_block_only$1(s, true);
    if (s.strm.avail_out === 0) {
      return BS_FINISH_STARTED$1;
    }
    /***/
    return BS_FINISH_DONE$1;
  }
  if (s.sym_next) {
    /*** FLUSH_BLOCK(s, 0); ***/
    flush_block_only$1(s, false);
    if (s.strm.avail_out === 0) {
      return BS_NEED_MORE$1;
    }
    /***/
  }
  return BS_BLOCK_DONE$1;
};

/* ===========================================================================
 * Same as above, but achieves better compression. We use a lazy
 * evaluation for matches: a match is finally adopted only if there is
 * no better match at the next window position.
 */
const deflate_slow$1 = (s, flush) => {

  let hash_head;          /* head of hash chain */
  let bflush;              /* set if current block must be flushed */

  let max_insert;

  /* Process the input block. */
  for (;;) {
    /* Make sure that we always have enough lookahead, except
     * at the end of the input file. We need MAX_MATCH bytes
     * for the next match, plus MIN_MATCH bytes to insert the
     * string following the next match.
     */
    if (s.lookahead < MIN_LOOKAHEAD$1) {
      fill_window$1(s);
      if (s.lookahead < MIN_LOOKAHEAD$1 && flush === Z_NO_FLUSH$2$1) {
        return BS_NEED_MORE$1;
      }
      if (s.lookahead === 0) { break; } /* flush the current block */
    }

    /* Insert the string window[strstart .. strstart+2] in the
     * dictionary, and set hash_head to the head of the hash chain:
     */
    hash_head = 0/*NIL*/;
    if (s.lookahead >= MIN_MATCH$2) {
      /*** INSERT_STRING(s, s.strstart, hash_head); ***/
      s.ins_h = HASH$1(s, s.ins_h, s.window[s.strstart + MIN_MATCH$2 - 1]);
      hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
      s.head[s.ins_h] = s.strstart;
      /***/
    }

    /* Find the longest match, discarding those <= prev_length.
     */
    s.prev_length = s.match_length;
    s.prev_match = s.match_start;
    s.match_length = MIN_MATCH$2 - 1;

    if (hash_head !== 0/*NIL*/ && s.prev_length < s.max_lazy_match &&
        s.strstart - hash_head <= (s.w_size - MIN_LOOKAHEAD$1)/*MAX_DIST(s)*/) {
      /* To simplify the code, we prevent matches with the string
       * of window index 0 (in particular we have to avoid a match
       * of the string with itself at the start of the input file).
       */
      s.match_length = longest_match$1(s, hash_head);
      /* longest_match() sets match_start */

      if (s.match_length <= 5 &&
         (s.strategy === Z_FILTERED$1 || (s.match_length === MIN_MATCH$2 && s.strstart - s.match_start > 4096/*TOO_FAR*/))) {

        /* If prev_match is also MIN_MATCH, match_start is garbage
         * but we will ignore the current match anyway.
         */
        s.match_length = MIN_MATCH$2 - 1;
      }
    }
    /* If there was a match at the previous step and the current
     * match is not better, output the previous match:
     */
    if (s.prev_length >= MIN_MATCH$2 && s.match_length <= s.prev_length) {
      max_insert = s.strstart + s.lookahead - MIN_MATCH$2;
      /* Do not insert strings in hash table beyond this. */

      //check_match(s, s.strstart-1, s.prev_match, s.prev_length);

      /***_tr_tally_dist(s, s.strstart - 1 - s.prev_match,
                     s.prev_length - MIN_MATCH, bflush);***/
      bflush = _tr_tally$2(s, s.strstart - 1 - s.prev_match, s.prev_length - MIN_MATCH$2);
      /* Insert in hash table all strings up to the end of the match.
       * strstart-1 and strstart are already inserted. If there is not
       * enough lookahead, the last two strings are not inserted in
       * the hash table.
       */
      s.lookahead -= s.prev_length - 1;
      s.prev_length -= 2;
      do {
        if (++s.strstart <= max_insert) {
          /*** INSERT_STRING(s, s.strstart, hash_head); ***/
          s.ins_h = HASH$1(s, s.ins_h, s.window[s.strstart + MIN_MATCH$2 - 1]);
          hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
          s.head[s.ins_h] = s.strstart;
          /***/
        }
      } while (--s.prev_length !== 0);
      s.match_available = 0;
      s.match_length = MIN_MATCH$2 - 1;
      s.strstart++;

      if (bflush) {
        /*** FLUSH_BLOCK(s, 0); ***/
        flush_block_only$1(s, false);
        if (s.strm.avail_out === 0) {
          return BS_NEED_MORE$1;
        }
        /***/
      }

    } else if (s.match_available) {
      /* If there was no match at the previous position, output a
       * single literal. If there was a match but the current match
       * is longer, truncate the previous match to a single literal.
       */
      //Tracevv((stderr,"%c", s->window[s->strstart-1]));
      /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
      bflush = _tr_tally$2(s, 0, s.window[s.strstart - 1]);

      if (bflush) {
        /*** FLUSH_BLOCK_ONLY(s, 0) ***/
        flush_block_only$1(s, false);
        /***/
      }
      s.strstart++;
      s.lookahead--;
      if (s.strm.avail_out === 0) {
        return BS_NEED_MORE$1;
      }
    } else {
      /* There is no previous match to compare with, wait for
       * the next step to decide.
       */
      s.match_available = 1;
      s.strstart++;
      s.lookahead--;
    }
  }
  //Assert (flush != Z_NO_FLUSH, "no flush?");
  if (s.match_available) {
    //Tracevv((stderr,"%c", s->window[s->strstart-1]));
    /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
    bflush = _tr_tally$2(s, 0, s.window[s.strstart - 1]);

    s.match_available = 0;
  }
  s.insert = s.strstart < MIN_MATCH$2 - 1 ? s.strstart : MIN_MATCH$2 - 1;
  if (flush === Z_FINISH$3$1) {
    /*** FLUSH_BLOCK(s, 1); ***/
    flush_block_only$1(s, true);
    if (s.strm.avail_out === 0) {
      return BS_FINISH_STARTED$1;
    }
    /***/
    return BS_FINISH_DONE$1;
  }
  if (s.sym_next) {
    /*** FLUSH_BLOCK(s, 0); ***/
    flush_block_only$1(s, false);
    if (s.strm.avail_out === 0) {
      return BS_NEED_MORE$1;
    }
    /***/
  }

  return BS_BLOCK_DONE$1;
};


/* ===========================================================================
 * For Z_RLE, simply look for runs of bytes, generate matches only of distance
 * one.  Do not maintain a hash table.  (It will be regenerated if this run of
 * deflate switches away from Z_RLE.)
 */
const deflate_rle$1 = (s, flush) => {

  let bflush;            /* set if current block must be flushed */
  let prev;              /* byte at distance one to match */
  let scan, strend;      /* scan goes up to strend for length of run */

  const _win = s.window;

  for (;;) {
    /* Make sure that we always have enough lookahead, except
     * at the end of the input file. We need MAX_MATCH bytes
     * for the longest run, plus one for the unrolled loop.
     */
    if (s.lookahead <= MAX_MATCH$2) {
      fill_window$1(s);
      if (s.lookahead <= MAX_MATCH$2 && flush === Z_NO_FLUSH$2$1) {
        return BS_NEED_MORE$1;
      }
      if (s.lookahead === 0) { break; } /* flush the current block */
    }

    /* See how many times the previous byte repeats */
    s.match_length = 0;
    if (s.lookahead >= MIN_MATCH$2 && s.strstart > 0) {
      scan = s.strstart - 1;
      prev = _win[scan];
      if (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan]) {
        strend = s.strstart + MAX_MATCH$2;
        do {
          /*jshint noempty:false*/
        } while (prev === _win[++scan] && prev === _win[++scan] &&
                 prev === _win[++scan] && prev === _win[++scan] &&
                 prev === _win[++scan] && prev === _win[++scan] &&
                 prev === _win[++scan] && prev === _win[++scan] &&
                 scan < strend);
        s.match_length = MAX_MATCH$2 - (strend - scan);
        if (s.match_length > s.lookahead) {
          s.match_length = s.lookahead;
        }
      }
      //Assert(scan <= s->window+(uInt)(s->window_size-1), "wild scan");
    }

    /* Emit match if have run of MIN_MATCH or longer, else emit literal */
    if (s.match_length >= MIN_MATCH$2) {
      //check_match(s, s.strstart, s.strstart - 1, s.match_length);

      /*** _tr_tally_dist(s, 1, s.match_length - MIN_MATCH, bflush); ***/
      bflush = _tr_tally$2(s, 1, s.match_length - MIN_MATCH$2);

      s.lookahead -= s.match_length;
      s.strstart += s.match_length;
      s.match_length = 0;
    } else {
      /* No match, output a literal byte */
      //Tracevv((stderr,"%c", s->window[s->strstart]));
      /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
      bflush = _tr_tally$2(s, 0, s.window[s.strstart]);

      s.lookahead--;
      s.strstart++;
    }
    if (bflush) {
      /*** FLUSH_BLOCK(s, 0); ***/
      flush_block_only$1(s, false);
      if (s.strm.avail_out === 0) {
        return BS_NEED_MORE$1;
      }
      /***/
    }
  }
  s.insert = 0;
  if (flush === Z_FINISH$3$1) {
    /*** FLUSH_BLOCK(s, 1); ***/
    flush_block_only$1(s, true);
    if (s.strm.avail_out === 0) {
      return BS_FINISH_STARTED$1;
    }
    /***/
    return BS_FINISH_DONE$1;
  }
  if (s.sym_next) {
    /*** FLUSH_BLOCK(s, 0); ***/
    flush_block_only$1(s, false);
    if (s.strm.avail_out === 0) {
      return BS_NEED_MORE$1;
    }
    /***/
  }
  return BS_BLOCK_DONE$1;
};

/* ===========================================================================
 * For Z_HUFFMAN_ONLY, do not look for matches.  Do not maintain a hash table.
 * (It will be regenerated if this run of deflate switches away from Huffman.)
 */
const deflate_huff$1 = (s, flush) => {

  let bflush;             /* set if current block must be flushed */

  for (;;) {
    /* Make sure that we have a literal to write. */
    if (s.lookahead === 0) {
      fill_window$1(s);
      if (s.lookahead === 0) {
        if (flush === Z_NO_FLUSH$2$1) {
          return BS_NEED_MORE$1;
        }
        break;      /* flush the current block */
      }
    }

    /* Output a literal byte */
    s.match_length = 0;
    //Tracevv((stderr,"%c", s->window[s->strstart]));
    /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
    bflush = _tr_tally$2(s, 0, s.window[s.strstart]);
    s.lookahead--;
    s.strstart++;
    if (bflush) {
      /*** FLUSH_BLOCK(s, 0); ***/
      flush_block_only$1(s, false);
      if (s.strm.avail_out === 0) {
        return BS_NEED_MORE$1;
      }
      /***/
    }
  }
  s.insert = 0;
  if (flush === Z_FINISH$3$1) {
    /*** FLUSH_BLOCK(s, 1); ***/
    flush_block_only$1(s, true);
    if (s.strm.avail_out === 0) {
      return BS_FINISH_STARTED$1;
    }
    /***/
    return BS_FINISH_DONE$1;
  }
  if (s.sym_next) {
    /*** FLUSH_BLOCK(s, 0); ***/
    flush_block_only$1(s, false);
    if (s.strm.avail_out === 0) {
      return BS_NEED_MORE$1;
    }
    /***/
  }
  return BS_BLOCK_DONE$1;
};

/* Values for max_lazy_match, good_match and max_chain_length, depending on
 * the desired pack level (0..9). The values given below have been tuned to
 * exclude worst case performance for pathological files. Better values may be
 * found for specific files.
 */
function Config$1(good_length, max_lazy, nice_length, max_chain, func) {

  this.good_length = good_length;
  this.max_lazy = max_lazy;
  this.nice_length = nice_length;
  this.max_chain = max_chain;
  this.func = func;
}

const configuration_table$1 = [
  /*      good lazy nice chain */
  new Config$1(0, 0, 0, 0, deflate_stored$1),          /* 0 store only */
  new Config$1(4, 4, 8, 4, deflate_fast$1),            /* 1 max speed, no lazy matches */
  new Config$1(4, 5, 16, 8, deflate_fast$1),           /* 2 */
  new Config$1(4, 6, 32, 32, deflate_fast$1),          /* 3 */

  new Config$1(4, 4, 16, 16, deflate_slow$1),          /* 4 lazy matches */
  new Config$1(8, 16, 32, 32, deflate_slow$1),         /* 5 */
  new Config$1(8, 16, 128, 128, deflate_slow$1),       /* 6 */
  new Config$1(8, 32, 128, 256, deflate_slow$1),       /* 7 */
  new Config$1(32, 128, 258, 1024, deflate_slow$1),    /* 8 */
  new Config$1(32, 258, 258, 4096, deflate_slow$1)     /* 9 max compression */
];


/* ===========================================================================
 * Initialize the "longest match" routines for a new zlib stream
 */
const lm_init$1 = (s) => {

  s.window_size = 2 * s.w_size;

  /*** CLEAR_HASH(s); ***/
  zero$2(s.head); // Fill with NIL (= 0);

  /* Set the default configuration parameters:
   */
  s.max_lazy_match = configuration_table$1[s.level].max_lazy;
  s.good_match = configuration_table$1[s.level].good_length;
  s.nice_match = configuration_table$1[s.level].nice_length;
  s.max_chain_length = configuration_table$1[s.level].max_chain;

  s.strstart = 0;
  s.block_start = 0;
  s.lookahead = 0;
  s.insert = 0;
  s.match_length = s.prev_length = MIN_MATCH$2 - 1;
  s.match_available = 0;
  s.ins_h = 0;
};


function DeflateState$1() {
  this.strm = null;            /* pointer back to this zlib stream */
  this.status = 0;            /* as the name implies */
  this.pending_buf = null;      /* output still pending */
  this.pending_buf_size = 0;  /* size of pending_buf */
  this.pending_out = 0;       /* next pending byte to output to the stream */
  this.pending = 0;           /* nb of bytes in the pending buffer */
  this.wrap = 0;              /* bit 0 true for zlib, bit 1 true for gzip */
  this.gzhead = null;         /* gzip header information to write */
  this.gzindex = 0;           /* where in extra, name, or comment */
  this.method = Z_DEFLATED$2$1; /* can only be DEFLATED */
  this.last_flush = -1;   /* value of flush param for previous deflate call */

  this.w_size = 0;  /* LZ77 window size (32K by default) */
  this.w_bits = 0;  /* log2(w_size)  (8..16) */
  this.w_mask = 0;  /* w_size - 1 */

  this.window = null;
  /* Sliding window. Input bytes are read into the second half of the window,
   * and move to the first half later to keep a dictionary of at least wSize
   * bytes. With this organization, matches are limited to a distance of
   * wSize-MAX_MATCH bytes, but this ensures that IO is always
   * performed with a length multiple of the block size.
   */

  this.window_size = 0;
  /* Actual size of window: 2*wSize, except when the user input buffer
   * is directly used as sliding window.
   */

  this.prev = null;
  /* Link to older string with same hash index. To limit the size of this
   * array to 64K, this link is maintained only for the last 32K strings.
   * An index in this array is thus a window index modulo 32K.
   */

  this.head = null;   /* Heads of the hash chains or NIL. */

  this.ins_h = 0;       /* hash index of string to be inserted */
  this.hash_size = 0;   /* number of elements in hash table */
  this.hash_bits = 0;   /* log2(hash_size) */
  this.hash_mask = 0;   /* hash_size-1 */

  this.hash_shift = 0;
  /* Number of bits by which ins_h must be shifted at each input
   * step. It must be such that after MIN_MATCH steps, the oldest
   * byte no longer takes part in the hash key, that is:
   *   hash_shift * MIN_MATCH >= hash_bits
   */

  this.block_start = 0;
  /* Window position at the beginning of the current output block. Gets
   * negative when the window is moved backwards.
   */

  this.match_length = 0;      /* length of best match */
  this.prev_match = 0;        /* previous match */
  this.match_available = 0;   /* set if previous match exists */
  this.strstart = 0;          /* start of string to insert */
  this.match_start = 0;       /* start of matching string */
  this.lookahead = 0;         /* number of valid bytes ahead in window */

  this.prev_length = 0;
  /* Length of the best match at previous step. Matches not greater than this
   * are discarded. This is used in the lazy match evaluation.
   */

  this.max_chain_length = 0;
  /* To speed up deflation, hash chains are never searched beyond this
   * length.  A higher limit improves compression ratio but degrades the
   * speed.
   */

  this.max_lazy_match = 0;
  /* Attempt to find a better match only when the current match is strictly
   * smaller than this value. This mechanism is used only for compression
   * levels >= 4.
   */
  // That's alias to max_lazy_match, don't use directly
  //this.max_insert_length = 0;
  /* Insert new strings in the hash table only if the match length is not
   * greater than this length. This saves time but degrades compression.
   * max_insert_length is used only for compression levels <= 3.
   */

  this.level = 0;     /* compression level (1..9) */
  this.strategy = 0;  /* favor or force Huffman coding*/

  this.good_match = 0;
  /* Use a faster search when the previous match is longer than this */

  this.nice_match = 0; /* Stop searching when current match exceeds this */

              /* used by trees.c: */

  /* Didn't use ct_data typedef below to suppress compiler warning */

  // struct ct_data_s dyn_ltree[HEAP_SIZE];   /* literal and length tree */
  // struct ct_data_s dyn_dtree[2*D_CODES+1]; /* distance tree */
  // struct ct_data_s bl_tree[2*BL_CODES+1];  /* Huffman tree for bit lengths */

  // Use flat array of DOUBLE size, with interleaved fata,
  // because JS does not support effective
  this.dyn_ltree  = new Uint16Array(HEAP_SIZE$2 * 2);
  this.dyn_dtree  = new Uint16Array((2 * D_CODES$2 + 1) * 2);
  this.bl_tree    = new Uint16Array((2 * BL_CODES$2 + 1) * 2);
  zero$2(this.dyn_ltree);
  zero$2(this.dyn_dtree);
  zero$2(this.bl_tree);

  this.l_desc   = null;         /* desc. for literal tree */
  this.d_desc   = null;         /* desc. for distance tree */
  this.bl_desc  = null;         /* desc. for bit length tree */

  //ush bl_count[MAX_BITS+1];
  this.bl_count = new Uint16Array(MAX_BITS$2 + 1);
  /* number of codes at each bit length for an optimal tree */

  //int heap[2*L_CODES+1];      /* heap used to build the Huffman trees */
  this.heap = new Uint16Array(2 * L_CODES$2 + 1);  /* heap used to build the Huffman trees */
  zero$2(this.heap);

  this.heap_len = 0;               /* number of elements in the heap */
  this.heap_max = 0;               /* element of largest frequency */
  /* The sons of heap[n] are heap[2*n] and heap[2*n+1]. heap[0] is not used.
   * The same heap array is used to build all trees.
   */

  this.depth = new Uint16Array(2 * L_CODES$2 + 1); //uch depth[2*L_CODES+1];
  zero$2(this.depth);
  /* Depth of each subtree used as tie breaker for trees of equal frequency
   */

  this.sym_buf = 0;        /* buffer for distances and literals/lengths */

  this.lit_bufsize = 0;
  /* Size of match buffer for literals/lengths.  There are 4 reasons for
   * limiting lit_bufsize to 64K:
   *   - frequencies can be kept in 16 bit counters
   *   - if compression is not successful for the first block, all input
   *     data is still in the window so we can still emit a stored block even
   *     when input comes from standard input.  (This can also be done for
   *     all blocks if lit_bufsize is not greater than 32K.)
   *   - if compression is not successful for a file smaller than 64K, we can
   *     even emit a stored file instead of a stored block (saving 5 bytes).
   *     This is applicable only for zip (not gzip or zlib).
   *   - creating new Huffman trees less frequently may not provide fast
   *     adaptation to changes in the input data statistics. (Take for
   *     example a binary file with poorly compressible code followed by
   *     a highly compressible string table.) Smaller buffer sizes give
   *     fast adaptation but have of course the overhead of transmitting
   *     trees more frequently.
   *   - I can't count above 4
   */

  this.sym_next = 0;      /* running index in sym_buf */
  this.sym_end = 0;       /* symbol table full when sym_next reaches this */

  this.opt_len = 0;       /* bit length of current block with optimal trees */
  this.static_len = 0;    /* bit length of current block with static trees */
  this.matches = 0;       /* number of string matches in current block */
  this.insert = 0;        /* bytes at end of window left to insert */


  this.bi_buf = 0;
  /* Output buffer. bits are inserted starting at the bottom (least
   * significant bits).
   */
  this.bi_valid = 0;
  /* Number of valid bits in bi_buf.  All bits above the last valid bit
   * are always zero.
   */

  // Used for window memory init. We safely ignore it for JS. That makes
  // sense only for pointers and memory check tools.
  //this.high_water = 0;
  /* High water mark offset in window for initialized bytes -- bytes above
   * this are set to zero in order to avoid memory check warnings when
   * longest match routines access bytes past the input.  This is then
   * updated to the new high water mark.
   */
}


/* =========================================================================
 * Check for a valid deflate stream state. Return 0 if ok, 1 if not.
 */
const deflateStateCheck$1 = (strm) => {

  if (!strm) {
    return 1;
  }
  const s = strm.state;
  if (!s || s.strm !== strm || (s.status !== INIT_STATE$1 &&
//#ifdef GZIP
                                s.status !== GZIP_STATE$1 &&
//#endif
                                s.status !== EXTRA_STATE$1 &&
                                s.status !== NAME_STATE$1 &&
                                s.status !== COMMENT_STATE$1 &&
                                s.status !== HCRC_STATE$1 &&
                                s.status !== BUSY_STATE$1 &&
                                s.status !== FINISH_STATE$1)) {
    return 1;
  }
  return 0;
};


const deflateResetKeep$1 = (strm) => {

  if (deflateStateCheck$1(strm)) {
    return err$1(strm, Z_STREAM_ERROR$2$1);
  }

  strm.total_in = strm.total_out = 0;
  strm.data_type = Z_UNKNOWN$2;

  const s = strm.state;
  s.pending = 0;
  s.pending_out = 0;

  if (s.wrap < 0) {
    s.wrap = -s.wrap;
    /* was made negative by deflate(..., Z_FINISH); */
  }
  s.status =
//#ifdef GZIP
    s.wrap === 2 ? GZIP_STATE$1 :
//#endif
    s.wrap ? INIT_STATE$1 : BUSY_STATE$1;
  strm.adler = (s.wrap === 2) ?
    0  // crc32(0, Z_NULL, 0)
  :
    1; // adler32(0, Z_NULL, 0)
  s.last_flush = -2;
  _tr_init$2(s);
  return Z_OK$3$1;
};


const deflateReset$1 = (strm) => {

  const ret = deflateResetKeep$1(strm);
  if (ret === Z_OK$3$1) {
    lm_init$1(strm.state);
  }
  return ret;
};


const deflateSetHeader$1 = (strm, head) => {

  if (deflateStateCheck$1(strm) || strm.state.wrap !== 2) {
    return Z_STREAM_ERROR$2$1;
  }
  strm.state.gzhead = head;
  return Z_OK$3$1;
};


const deflateInit2$1 = (strm, level, method, windowBits, memLevel, strategy) => {

  if (!strm) { // === Z_NULL
    return Z_STREAM_ERROR$2$1;
  }
  let wrap = 1;

  if (level === Z_DEFAULT_COMPRESSION$1$1) {
    level = 6;
  }

  if (windowBits < 0) { /* suppress zlib wrapper */
    wrap = 0;
    windowBits = -windowBits;
  }

  else if (windowBits > 15) {
    wrap = 2;           /* write gzip wrapper instead */
    windowBits -= 16;
  }


  if (memLevel < 1 || memLevel > MAX_MEM_LEVEL$1 || method !== Z_DEFLATED$2$1 ||
    windowBits < 8 || windowBits > 15 || level < 0 || level > 9 ||
    strategy < 0 || strategy > Z_FIXED$2 || (windowBits === 8 && wrap !== 1)) {
    return err$1(strm, Z_STREAM_ERROR$2$1);
  }


  if (windowBits === 8) {
    windowBits = 9;
  }
  /* until 256-byte window bug fixed */

  const s = new DeflateState$1();

  strm.state = s;
  s.strm = strm;
  s.status = INIT_STATE$1;     /* to pass state test in deflateReset() */

  s.wrap = wrap;
  s.gzhead = null;
  s.w_bits = windowBits;
  s.w_size = 1 << s.w_bits;
  s.w_mask = s.w_size - 1;

  s.hash_bits = memLevel + 7;
  s.hash_size = 1 << s.hash_bits;
  s.hash_mask = s.hash_size - 1;
  s.hash_shift = ~~((s.hash_bits + MIN_MATCH$2 - 1) / MIN_MATCH$2);

  s.window = new Uint8Array(s.w_size * 2);
  s.head = new Uint16Array(s.hash_size);
  s.prev = new Uint16Array(s.w_size);

  // Don't need mem init magic for JS.
  //s.high_water = 0;  /* nothing written to s->window yet */

  s.lit_bufsize = 1 << (memLevel + 6); /* 16K elements by default */

  /* We overlay pending_buf and sym_buf. This works since the average size
   * for length/distance pairs over any compressed block is assured to be 31
   * bits or less.
   *
   * Analysis: The longest fixed codes are a length code of 8 bits plus 5
   * extra bits, for lengths 131 to 257. The longest fixed distance codes are
   * 5 bits plus 13 extra bits, for distances 16385 to 32768. The longest
   * possible fixed-codes length/distance pair is then 31 bits total.
   *
   * sym_buf starts one-fourth of the way into pending_buf. So there are
   * three bytes in sym_buf for every four bytes in pending_buf. Each symbol
   * in sym_buf is three bytes -- two for the distance and one for the
   * literal/length. As each symbol is consumed, the pointer to the next
   * sym_buf value to read moves forward three bytes. From that symbol, up to
   * 31 bits are written to pending_buf. The closest the written pending_buf
   * bits gets to the next sym_buf symbol to read is just before the last
   * code is written. At that time, 31*(n-2) bits have been written, just
   * after 24*(n-2) bits have been consumed from sym_buf. sym_buf starts at
   * 8*n bits into pending_buf. (Note that the symbol buffer fills when n-1
   * symbols are written.) The closest the writing gets to what is unread is
   * then n+14 bits. Here n is lit_bufsize, which is 16384 by default, and
   * can range from 128 to 32768.
   *
   * Therefore, at a minimum, there are 142 bits of space between what is
   * written and what is read in the overlain buffers, so the symbols cannot
   * be overwritten by the compressed data. That space is actually 139 bits,
   * due to the three-bit fixed-code block header.
   *
   * That covers the case where either Z_FIXED is specified, forcing fixed
   * codes, or when the use of fixed codes is chosen, because that choice
   * results in a smaller compressed block than dynamic codes. That latter
   * condition then assures that the above analysis also covers all dynamic
   * blocks. A dynamic-code block will only be chosen to be emitted if it has
   * fewer bits than a fixed-code block would for the same set of symbols.
   * Therefore its average symbol length is assured to be less than 31. So
   * the compressed data for a dynamic block also cannot overwrite the
   * symbols from which it is being constructed.
   */

  s.pending_buf_size = s.lit_bufsize * 4;
  s.pending_buf = new Uint8Array(s.pending_buf_size);

  // It is offset from `s.pending_buf` (size is `s.lit_bufsize * 2`)
  //s->sym_buf = s->pending_buf + s->lit_bufsize;
  s.sym_buf = s.lit_bufsize;

  //s->sym_end = (s->lit_bufsize - 1) * 3;
  s.sym_end = (s.lit_bufsize - 1) * 3;
  /* We avoid equality with lit_bufsize*3 because of wraparound at 64K
   * on 16 bit machines and because stored blocks are restricted to
   * 64K-1 bytes.
   */

  s.level = level;
  s.strategy = strategy;
  s.method = method;

  return deflateReset$1(strm);
};

const deflateInit$1 = (strm, level) => {

  return deflateInit2$1(strm, level, Z_DEFLATED$2$1, MAX_WBITS$1$1, DEF_MEM_LEVEL$1, Z_DEFAULT_STRATEGY$1$1);
};


/* ========================================================================= */
const deflate$2$1 = (strm, flush) => {

  if (deflateStateCheck$1(strm) || flush > Z_BLOCK$1$1 || flush < 0) {
    return strm ? err$1(strm, Z_STREAM_ERROR$2$1) : Z_STREAM_ERROR$2$1;
  }

  const s = strm.state;

  if (!strm.output ||
      (strm.avail_in !== 0 && !strm.input) ||
      (s.status === FINISH_STATE$1 && flush !== Z_FINISH$3$1)) {
    return err$1(strm, (strm.avail_out === 0) ? Z_BUF_ERROR$1$1 : Z_STREAM_ERROR$2$1);
  }

  const old_flush = s.last_flush;
  s.last_flush = flush;

  /* Flush as much pending output as possible */
  if (s.pending !== 0) {
    flush_pending$1(strm);
    if (strm.avail_out === 0) {
      /* Since avail_out is 0, deflate will be called again with
       * more output space, but possibly with both pending and
       * avail_in equal to zero. There won't be anything to do,
       * but this is not an error situation so make sure we
       * return OK instead of BUF_ERROR at next call of deflate:
       */
      s.last_flush = -1;
      return Z_OK$3$1;
    }

    /* Make sure there is something to do and avoid duplicate consecutive
     * flushes. For repeated and useless calls with Z_FINISH, we keep
     * returning Z_STREAM_END instead of Z_BUF_ERROR.
     */
  } else if (strm.avail_in === 0 && rank$1(flush) <= rank$1(old_flush) &&
    flush !== Z_FINISH$3$1) {
    return err$1(strm, Z_BUF_ERROR$1$1);
  }

  /* User must not provide more input after the first FINISH: */
  if (s.status === FINISH_STATE$1 && strm.avail_in !== 0) {
    return err$1(strm, Z_BUF_ERROR$1$1);
  }

  /* Write the header */
  if (s.status === INIT_STATE$1 && s.wrap === 0) {
    s.status = BUSY_STATE$1;
  }
  if (s.status === INIT_STATE$1) {
    /* zlib header */
    let header = (Z_DEFLATED$2$1 + ((s.w_bits - 8) << 4)) << 8;
    let level_flags = -1;

    if (s.strategy >= Z_HUFFMAN_ONLY$1 || s.level < 2) {
      level_flags = 0;
    } else if (s.level < 6) {
      level_flags = 1;
    } else if (s.level === 6) {
      level_flags = 2;
    } else {
      level_flags = 3;
    }
    header |= (level_flags << 6);
    if (s.strstart !== 0) { header |= PRESET_DICT$1; }
    header += 31 - (header % 31);

    putShortMSB$1(s, header);

    /* Save the adler32 of the preset dictionary: */
    if (s.strstart !== 0) {
      putShortMSB$1(s, strm.adler >>> 16);
      putShortMSB$1(s, strm.adler & 0xffff);
    }
    strm.adler = 1; // adler32(0L, Z_NULL, 0);
    s.status = BUSY_STATE$1;

    /* Compression must start with an empty pending buffer */
    flush_pending$1(strm);
    if (s.pending !== 0) {
      s.last_flush = -1;
      return Z_OK$3$1;
    }
  }
//#ifdef GZIP
  if (s.status === GZIP_STATE$1) {
    /* gzip header */
    strm.adler = 0;  //crc32(0L, Z_NULL, 0);
    put_byte$1(s, 31);
    put_byte$1(s, 139);
    put_byte$1(s, 8);
    if (!s.gzhead) { // s->gzhead == Z_NULL
      put_byte$1(s, 0);
      put_byte$1(s, 0);
      put_byte$1(s, 0);
      put_byte$1(s, 0);
      put_byte$1(s, 0);
      put_byte$1(s, s.level === 9 ? 2 :
                  (s.strategy >= Z_HUFFMAN_ONLY$1 || s.level < 2 ?
                   4 : 0));
      put_byte$1(s, OS_CODE$1);
      s.status = BUSY_STATE$1;

      /* Compression must start with an empty pending buffer */
      flush_pending$1(strm);
      if (s.pending !== 0) {
        s.last_flush = -1;
        return Z_OK$3$1;
      }
    }
    else {
      put_byte$1(s, (s.gzhead.text ? 1 : 0) +
                  (s.gzhead.hcrc ? 2 : 0) +
                  (!s.gzhead.extra ? 0 : 4) +
                  (!s.gzhead.name ? 0 : 8) +
                  (!s.gzhead.comment ? 0 : 16)
      );
      put_byte$1(s, s.gzhead.time & 0xff);
      put_byte$1(s, (s.gzhead.time >> 8) & 0xff);
      put_byte$1(s, (s.gzhead.time >> 16) & 0xff);
      put_byte$1(s, (s.gzhead.time >> 24) & 0xff);
      put_byte$1(s, s.level === 9 ? 2 :
                  (s.strategy >= Z_HUFFMAN_ONLY$1 || s.level < 2 ?
                   4 : 0));
      put_byte$1(s, s.gzhead.os & 0xff);
      if (s.gzhead.extra && s.gzhead.extra.length) {
        put_byte$1(s, s.gzhead.extra.length & 0xff);
        put_byte$1(s, (s.gzhead.extra.length >> 8) & 0xff);
      }
      if (s.gzhead.hcrc) {
        strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending, 0);
      }
      s.gzindex = 0;
      s.status = EXTRA_STATE$1;
    }
  }
  if (s.status === EXTRA_STATE$1) {
    if (s.gzhead.extra/* != Z_NULL*/) {
      let beg = s.pending;   /* start of bytes to update crc */
      let left = (s.gzhead.extra.length & 0xffff) - s.gzindex;
      while (s.pending + left > s.pending_buf_size) {
        let copy = s.pending_buf_size - s.pending;
        // zmemcpy(s.pending_buf + s.pending,
        //    s.gzhead.extra + s.gzindex, copy);
        s.pending_buf.set(s.gzhead.extra.subarray(s.gzindex, s.gzindex + copy), s.pending);
        s.pending = s.pending_buf_size;
        //--- HCRC_UPDATE(beg) ---//
        if (s.gzhead.hcrc && s.pending > beg) {
          strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
        }
        //---//
        s.gzindex += copy;
        flush_pending$1(strm);
        if (s.pending !== 0) {
          s.last_flush = -1;
          return Z_OK$3$1;
        }
        beg = 0;
        left -= copy;
      }
      // JS specific: s.gzhead.extra may be TypedArray or Array for backward compatibility
      //              TypedArray.slice and TypedArray.from don't exist in IE10-IE11
      let gzhead_extra = new Uint8Array(s.gzhead.extra);
      // zmemcpy(s->pending_buf + s->pending,
      //     s->gzhead->extra + s->gzindex, left);
      s.pending_buf.set(gzhead_extra.subarray(s.gzindex, s.gzindex + left), s.pending);
      s.pending += left;
      //--- HCRC_UPDATE(beg) ---//
      if (s.gzhead.hcrc && s.pending > beg) {
        strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
      }
      //---//
      s.gzindex = 0;
    }
    s.status = NAME_STATE$1;
  }
  if (s.status === NAME_STATE$1) {
    if (s.gzhead.name/* != Z_NULL*/) {
      let beg = s.pending;   /* start of bytes to update crc */
      let val;
      do {
        if (s.pending === s.pending_buf_size) {
          //--- HCRC_UPDATE(beg) ---//
          if (s.gzhead.hcrc && s.pending > beg) {
            strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
          }
          //---//
          flush_pending$1(strm);
          if (s.pending !== 0) {
            s.last_flush = -1;
            return Z_OK$3$1;
          }
          beg = 0;
        }
        // JS specific: little magic to add zero terminator to end of string
        if (s.gzindex < s.gzhead.name.length) {
          val = s.gzhead.name.charCodeAt(s.gzindex++) & 0xff;
        } else {
          val = 0;
        }
        put_byte$1(s, val);
      } while (val !== 0);
      //--- HCRC_UPDATE(beg) ---//
      if (s.gzhead.hcrc && s.pending > beg) {
        strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
      }
      //---//
      s.gzindex = 0;
    }
    s.status = COMMENT_STATE$1;
  }
  if (s.status === COMMENT_STATE$1) {
    if (s.gzhead.comment/* != Z_NULL*/) {
      let beg = s.pending;   /* start of bytes to update crc */
      let val;
      do {
        if (s.pending === s.pending_buf_size) {
          //--- HCRC_UPDATE(beg) ---//
          if (s.gzhead.hcrc && s.pending > beg) {
            strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
          }
          //---//
          flush_pending$1(strm);
          if (s.pending !== 0) {
            s.last_flush = -1;
            return Z_OK$3$1;
          }
          beg = 0;
        }
        // JS specific: little magic to add zero terminator to end of string
        if (s.gzindex < s.gzhead.comment.length) {
          val = s.gzhead.comment.charCodeAt(s.gzindex++) & 0xff;
        } else {
          val = 0;
        }
        put_byte$1(s, val);
      } while (val !== 0);
      //--- HCRC_UPDATE(beg) ---//
      if (s.gzhead.hcrc && s.pending > beg) {
        strm.adler = crc32_1$1(strm.adler, s.pending_buf, s.pending - beg, beg);
      }
      //---//
    }
    s.status = HCRC_STATE$1;
  }
  if (s.status === HCRC_STATE$1) {
    if (s.gzhead.hcrc) {
      if (s.pending + 2 > s.pending_buf_size) {
        flush_pending$1(strm);
        if (s.pending !== 0) {
          s.last_flush = -1;
          return Z_OK$3$1;
        }
      }
      put_byte$1(s, strm.adler & 0xff);
      put_byte$1(s, (strm.adler >> 8) & 0xff);
      strm.adler = 0; //crc32(0L, Z_NULL, 0);
    }
    s.status = BUSY_STATE$1;

    /* Compression must start with an empty pending buffer */
    flush_pending$1(strm);
    if (s.pending !== 0) {
      s.last_flush = -1;
      return Z_OK$3$1;
    }
  }
//#endif

  /* Start a new block or continue the current one.
   */
  if (strm.avail_in !== 0 || s.lookahead !== 0 ||
    (flush !== Z_NO_FLUSH$2$1 && s.status !== FINISH_STATE$1)) {
    let bstate = s.level === 0 ? deflate_stored$1(s, flush) :
                 s.strategy === Z_HUFFMAN_ONLY$1 ? deflate_huff$1(s, flush) :
                 s.strategy === Z_RLE$1 ? deflate_rle$1(s, flush) :
                 configuration_table$1[s.level].func(s, flush);

    if (bstate === BS_FINISH_STARTED$1 || bstate === BS_FINISH_DONE$1) {
      s.status = FINISH_STATE$1;
    }
    if (bstate === BS_NEED_MORE$1 || bstate === BS_FINISH_STARTED$1) {
      if (strm.avail_out === 0) {
        s.last_flush = -1;
        /* avoid BUF_ERROR next call, see above */
      }
      return Z_OK$3$1;
      /* If flush != Z_NO_FLUSH && avail_out == 0, the next call
       * of deflate should use the same flush parameter to make sure
       * that the flush is complete. So we don't have to output an
       * empty block here, this will be done at next call. This also
       * ensures that for a very small output buffer, we emit at most
       * one empty block.
       */
    }
    if (bstate === BS_BLOCK_DONE$1) {
      if (flush === Z_PARTIAL_FLUSH$1) {
        _tr_align$2(s);
      }
      else if (flush !== Z_BLOCK$1$1) { /* FULL_FLUSH or SYNC_FLUSH */

        _tr_stored_block$2(s, 0, 0, false);
        /* For a full flush, this empty block will be recognized
         * as a special marker by inflate_sync().
         */
        if (flush === Z_FULL_FLUSH$1$1) {
          /*** CLEAR_HASH(s); ***/             /* forget history */
          zero$2(s.head); // Fill with NIL (= 0);

          if (s.lookahead === 0) {
            s.strstart = 0;
            s.block_start = 0;
            s.insert = 0;
          }
        }
      }
      flush_pending$1(strm);
      if (strm.avail_out === 0) {
        s.last_flush = -1; /* avoid BUF_ERROR at next call, see above */
        return Z_OK$3$1;
      }
    }
  }

  if (flush !== Z_FINISH$3$1) { return Z_OK$3$1; }
  if (s.wrap <= 0) { return Z_STREAM_END$3$1; }

  /* Write the trailer */
  if (s.wrap === 2) {
    put_byte$1(s, strm.adler & 0xff);
    put_byte$1(s, (strm.adler >> 8) & 0xff);
    put_byte$1(s, (strm.adler >> 16) & 0xff);
    put_byte$1(s, (strm.adler >> 24) & 0xff);
    put_byte$1(s, strm.total_in & 0xff);
    put_byte$1(s, (strm.total_in >> 8) & 0xff);
    put_byte$1(s, (strm.total_in >> 16) & 0xff);
    put_byte$1(s, (strm.total_in >> 24) & 0xff);
  }
  else
  {
    putShortMSB$1(s, strm.adler >>> 16);
    putShortMSB$1(s, strm.adler & 0xffff);
  }

  flush_pending$1(strm);
  /* If avail_out is zero, the application will call deflate again
   * to flush the rest.
   */
  if (s.wrap > 0) { s.wrap = -s.wrap; }
  /* write the trailer only once! */
  return s.pending !== 0 ? Z_OK$3$1 : Z_STREAM_END$3$1;
};


const deflateEnd$1 = (strm) => {

  if (deflateStateCheck$1(strm)) {
    return Z_STREAM_ERROR$2$1;
  }

  const status = strm.state.status;

  strm.state = null;

  return status === BUSY_STATE$1 ? err$1(strm, Z_DATA_ERROR$2$1) : Z_OK$3$1;
};


/* =========================================================================
 * Initializes the compression dictionary from the given byte
 * sequence without producing any compressed output.
 */
const deflateSetDictionary$1 = (strm, dictionary) => {

  let dictLength = dictionary.length;

  if (deflateStateCheck$1(strm)) {
    return Z_STREAM_ERROR$2$1;
  }

  const s = strm.state;
  const wrap = s.wrap;

  if (wrap === 2 || (wrap === 1 && s.status !== INIT_STATE$1) || s.lookahead) {
    return Z_STREAM_ERROR$2$1;
  }

  /* when using zlib wrappers, compute Adler-32 for provided dictionary */
  if (wrap === 1) {
    /* adler32(strm->adler, dictionary, dictLength); */
    strm.adler = adler32_1$1(strm.adler, dictionary, dictLength, 0);
  }

  s.wrap = 0;   /* avoid computing Adler-32 in read_buf */

  /* if dictionary would fill window, just replace the history */
  if (dictLength >= s.w_size) {
    if (wrap === 0) {            /* already empty otherwise */
      /*** CLEAR_HASH(s); ***/
      zero$2(s.head); // Fill with NIL (= 0);
      s.strstart = 0;
      s.block_start = 0;
      s.insert = 0;
    }
    /* use the tail */
    // dictionary = dictionary.slice(dictLength - s.w_size);
    let tmpDict = new Uint8Array(s.w_size);
    tmpDict.set(dictionary.subarray(dictLength - s.w_size, dictLength), 0);
    dictionary = tmpDict;
    dictLength = s.w_size;
  }
  /* insert dictionary into window and hash */
  const avail = strm.avail_in;
  const next = strm.next_in;
  const input = strm.input;
  strm.avail_in = dictLength;
  strm.next_in = 0;
  strm.input = dictionary;
  fill_window$1(s);
  while (s.lookahead >= MIN_MATCH$2) {
    let str = s.strstart;
    let n = s.lookahead - (MIN_MATCH$2 - 1);
    do {
      /* UPDATE_HASH(s, s->ins_h, s->window[str + MIN_MATCH-1]); */
      s.ins_h = HASH$1(s, s.ins_h, s.window[str + MIN_MATCH$2 - 1]);

      s.prev[str & s.w_mask] = s.head[s.ins_h];

      s.head[s.ins_h] = str;
      str++;
    } while (--n);
    s.strstart = str;
    s.lookahead = MIN_MATCH$2 - 1;
    fill_window$1(s);
  }
  s.strstart += s.lookahead;
  s.block_start = s.strstart;
  s.insert = s.lookahead;
  s.lookahead = 0;
  s.match_length = s.prev_length = MIN_MATCH$2 - 1;
  s.match_available = 0;
  strm.next_in = next;
  strm.input = input;
  strm.avail_in = avail;
  s.wrap = wrap;
  return Z_OK$3$1;
};


var deflateInit_1$1 = deflateInit$1;
var deflateInit2_1$1 = deflateInit2$1;
var deflateReset_1$1 = deflateReset$1;
var deflateResetKeep_1$1 = deflateResetKeep$1;
var deflateSetHeader_1$1 = deflateSetHeader$1;
var deflate_2$1$1 = deflate$2$1;
var deflateEnd_1$1 = deflateEnd$1;
var deflateSetDictionary_1$1 = deflateSetDictionary$1;
var deflateInfo$1 = 'pako deflate (from Nodeca project)';

/* Not implemented
module.exports.deflateBound = deflateBound;
module.exports.deflateCopy = deflateCopy;
module.exports.deflateGetDictionary = deflateGetDictionary;
module.exports.deflateParams = deflateParams;
module.exports.deflatePending = deflatePending;
module.exports.deflatePrime = deflatePrime;
module.exports.deflateTune = deflateTune;
*/

var deflate_1$2$1 = {
	deflateInit: deflateInit_1$1,
	deflateInit2: deflateInit2_1$1,
	deflateReset: deflateReset_1$1,
	deflateResetKeep: deflateResetKeep_1$1,
	deflateSetHeader: deflateSetHeader_1$1,
	deflate: deflate_2$1$1,
	deflateEnd: deflateEnd_1$1,
	deflateSetDictionary: deflateSetDictionary_1$1,
	deflateInfo: deflateInfo$1
};

const _has$1 = (obj, key) => {
  return Object.prototype.hasOwnProperty.call(obj, key);
};

var assign$1 = function (obj /*from1, from2, from3, ...*/) {
  const sources = Array.prototype.slice.call(arguments, 1);
  while (sources.length) {
    const source = sources.shift();
    if (!source) { continue; }

    if (typeof source !== 'object') {
      throw new TypeError(source + 'must be non-object');
    }

    for (const p in source) {
      if (_has$1(source, p)) {
        obj[p] = source[p];
      }
    }
  }

  return obj;
};


// Join array of chunks to single array.
var flattenChunks$1 = (chunks) => {
  // calculate data length
  let len = 0;

  for (let i = 0, l = chunks.length; i < l; i++) {
    len += chunks[i].length;
  }

  // join chunks
  const result = new Uint8Array(len);

  for (let i = 0, pos = 0, l = chunks.length; i < l; i++) {
    let chunk = chunks[i];
    result.set(chunk, pos);
    pos += chunk.length;
  }

  return result;
};

var common$1 = {
	assign: assign$1,
	flattenChunks: flattenChunks$1
};

// String encode/decode helpers


// Quick check if we can use fast array to bin string conversion
//
// - apply(Array) can fail on Android 2.2
// - apply(Uint8Array) can fail on iOS 5.1 Safari
//
let STR_APPLY_UIA_OK$1 = true;

try { String.fromCharCode.apply(null, new Uint8Array(1)); } catch (__) { STR_APPLY_UIA_OK$1 = false; }


// Table with utf8 lengths (calculated by first byte of sequence)
// Note, that 5 & 6-byte values and some 4-byte values can not be represented in JS,
// because max possible codepoint is 0x10ffff
const _utf8len$1 = new Uint8Array(256);
for (let q = 0; q < 256; q++) {
  _utf8len$1[q] = (q >= 252 ? 6 : q >= 248 ? 5 : q >= 240 ? 4 : q >= 224 ? 3 : q >= 192 ? 2 : 1);
}
_utf8len$1[254] = _utf8len$1[254] = 1; // Invalid sequence start


// convert string to array (typed, when possible)
var string2buf$1 = (str) => {
  if (typeof TextEncoder === 'function' && TextEncoder.prototype.encode) {
    return new TextEncoder().encode(str);
  }

  let buf, c, c2, m_pos, i, str_len = str.length, buf_len = 0;

  // count binary size
  for (m_pos = 0; m_pos < str_len; m_pos++) {
    c = str.charCodeAt(m_pos);
    if ((c & 0xfc00) === 0xd800 && (m_pos + 1 < str_len)) {
      c2 = str.charCodeAt(m_pos + 1);
      if ((c2 & 0xfc00) === 0xdc00) {
        c = 0x10000 + ((c - 0xd800) << 10) + (c2 - 0xdc00);
        m_pos++;
      }
    }
    buf_len += c < 0x80 ? 1 : c < 0x800 ? 2 : c < 0x10000 ? 3 : 4;
  }

  // allocate buffer
  buf = new Uint8Array(buf_len);

  // convert
  for (i = 0, m_pos = 0; i < buf_len; m_pos++) {
    c = str.charCodeAt(m_pos);
    if ((c & 0xfc00) === 0xd800 && (m_pos + 1 < str_len)) {
      c2 = str.charCodeAt(m_pos + 1);
      if ((c2 & 0xfc00) === 0xdc00) {
        c = 0x10000 + ((c - 0xd800) << 10) + (c2 - 0xdc00);
        m_pos++;
      }
    }
    if (c < 0x80) {
      /* one byte */
      buf[i++] = c;
    } else if (c < 0x800) {
      /* two bytes */
      buf[i++] = 0xC0 | (c >>> 6);
      buf[i++] = 0x80 | (c & 0x3f);
    } else if (c < 0x10000) {
      /* three bytes */
      buf[i++] = 0xE0 | (c >>> 12);
      buf[i++] = 0x80 | (c >>> 6 & 0x3f);
      buf[i++] = 0x80 | (c & 0x3f);
    } else {
      /* four bytes */
      buf[i++] = 0xf0 | (c >>> 18);
      buf[i++] = 0x80 | (c >>> 12 & 0x3f);
      buf[i++] = 0x80 | (c >>> 6 & 0x3f);
      buf[i++] = 0x80 | (c & 0x3f);
    }
  }

  return buf;
};

// Helper
const buf2binstring$1 = (buf, len) => {
  // On Chrome, the arguments in a function call that are allowed is `65534`.
  // If the length of the buffer is smaller than that, we can use this optimization,
  // otherwise we will take a slower path.
  if (len < 65534) {
    if (buf.subarray && STR_APPLY_UIA_OK$1) {
      return String.fromCharCode.apply(null, buf.length === len ? buf : buf.subarray(0, len));
    }
  }

  let result = '';
  for (let i = 0; i < len; i++) {
    result += String.fromCharCode(buf[i]);
  }
  return result;
};


// convert array to string
var buf2string$1 = (buf, max) => {
  const len = max || buf.length;

  if (typeof TextDecoder === 'function' && TextDecoder.prototype.decode) {
    return new TextDecoder().decode(buf.subarray(0, max));
  }

  let i, out;

  // Reserve max possible length (2 words per char)
  // NB: by unknown reasons, Array is significantly faster for
  //     String.fromCharCode.apply than Uint16Array.
  const utf16buf = new Array(len * 2);

  for (out = 0, i = 0; i < len;) {
    let c = buf[i++];
    // quick process ascii
    if (c < 0x80) { utf16buf[out++] = c; continue; }

    let c_len = _utf8len$1[c];
    // skip 5 & 6 byte codes
    if (c_len > 4) { utf16buf[out++] = 0xfffd; i += c_len - 1; continue; }

    // apply mask on first byte
    c &= c_len === 2 ? 0x1f : c_len === 3 ? 0x0f : 0x07;
    // join the rest
    while (c_len > 1 && i < len) {
      c = (c << 6) | (buf[i++] & 0x3f);
      c_len--;
    }

    // terminated by end of string?
    if (c_len > 1) { utf16buf[out++] = 0xfffd; continue; }

    if (c < 0x10000) {
      utf16buf[out++] = c;
    } else {
      c -= 0x10000;
      utf16buf[out++] = 0xd800 | ((c >> 10) & 0x3ff);
      utf16buf[out++] = 0xdc00 | (c & 0x3ff);
    }
  }

  return buf2binstring$1(utf16buf, out);
};


// Calculate max possible position in utf8 buffer,
// that will not break sequence. If that's not possible
// - (very small limits) return max size as is.
//
// buf[] - utf8 bytes array
// max   - length limit (mandatory);
var utf8border$1 = (buf, max) => {

  max = max || buf.length;
  if (max > buf.length) { max = buf.length; }

  // go back from last position, until start of sequence found
  let pos = max - 1;
  while (pos >= 0 && (buf[pos] & 0xC0) === 0x80) { pos--; }

  // Very small and broken sequence,
  // return max, because we should return something anyway.
  if (pos < 0) { return max; }

  // If we came to start of buffer - that means buffer is too small,
  // return max too.
  if (pos === 0) { return max; }

  return (pos + _utf8len$1[buf[pos]] > max) ? pos : max;
};

var strings$1 = {
	string2buf: string2buf$1,
	buf2string: buf2string$1,
	utf8border: utf8border$1
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

function ZStream$1() {
  /* next input byte */
  this.input = null; // JS specific, because we have no pointers
  this.next_in = 0;
  /* number of bytes available at input */
  this.avail_in = 0;
  /* total number of input bytes read so far */
  this.total_in = 0;
  /* next output byte should be put there */
  this.output = null; // JS specific, because we have no pointers
  this.next_out = 0;
  /* remaining free space at output */
  this.avail_out = 0;
  /* total number of bytes output so far */
  this.total_out = 0;
  /* last error message, NULL if no error */
  this.msg = ''/*Z_NULL*/;
  /* not visible by applications */
  this.state = null;
  /* best guess about the data type: binary or text */
  this.data_type = 2/*Z_UNKNOWN*/;
  /* adler32 value of the uncompressed data */
  this.adler = 0;
}

var zstream$1 = ZStream$1;

const toString$1$1 = Object.prototype.toString;

/* Public constants ==========================================================*/
/* ===========================================================================*/

const {
  Z_NO_FLUSH: Z_NO_FLUSH$1$1, Z_SYNC_FLUSH: Z_SYNC_FLUSH$1, Z_FULL_FLUSH: Z_FULL_FLUSH$2, Z_FINISH: Z_FINISH$2$1,
  Z_OK: Z_OK$2$1, Z_STREAM_END: Z_STREAM_END$2$1,
  Z_DEFAULT_COMPRESSION: Z_DEFAULT_COMPRESSION$2,
  Z_DEFAULT_STRATEGY: Z_DEFAULT_STRATEGY$2,
  Z_DEFLATED: Z_DEFLATED$1$1
} = constants$2$1;

/* ===========================================================================*/


/**
 * class Deflate
 *
 * Generic JS-style wrapper for zlib calls. If you don't need
 * streaming behaviour - use more simple functions: [[deflate]],
 * [[deflateRaw]] and [[gzip]].
 **/

/* internal
 * Deflate.chunks -> Array
 *
 * Chunks of output data, if [[Deflate#onData]] not overridden.
 **/

/**
 * Deflate.result -> Uint8Array
 *
 * Compressed result, generated by default [[Deflate#onData]]
 * and [[Deflate#onEnd]] handlers. Filled after you push last chunk
 * (call [[Deflate#push]] with `Z_FINISH` / `true` param).
 **/

/**
 * Deflate.err -> Number
 *
 * Error code after deflate finished. 0 (Z_OK) on success.
 * You will not need it in real life, because deflate errors
 * are possible only on wrong options or bad `onData` / `onEnd`
 * custom handlers.
 **/

/**
 * Deflate.msg -> String
 *
 * Error message, if [[Deflate.err]] != 0
 **/


/**
 * new Deflate(options)
 * - options (Object): zlib deflate options.
 *
 * Creates new deflator instance with specified params. Throws exception
 * on bad params. Supported options:
 *
 * - `level`
 * - `windowBits`
 * - `memLevel`
 * - `strategy`
 * - `dictionary`
 *
 * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
 * for more information on these.
 *
 * Additional options, for internal needs:
 *
 * - `chunkSize` - size of generated data chunks (16K by default)
 * - `raw` (Boolean) - do raw deflate
 * - `gzip` (Boolean) - create gzip wrapper
 * - `header` (Object) - custom header for gzip
 *   - `text` (Boolean) - true if compressed data believed to be text
 *   - `time` (Number) - modification time, unix timestamp
 *   - `os` (Number) - operation system code
 *   - `extra` (Array) - array of bytes with extra data (max 65536)
 *   - `name` (String) - file name (binary string)
 *   - `comment` (String) - comment (binary string)
 *   - `hcrc` (Boolean) - true if header crc should be added
 *
 * ##### Example:
 *
 * ```javascript
 * const pako = require('pako')
 *   , chunk1 = new Uint8Array([1,2,3,4,5,6,7,8,9])
 *   , chunk2 = new Uint8Array([10,11,12,13,14,15,16,17,18,19]);
 *
 * const deflate = new pako.Deflate({ level: 3});
 *
 * deflate.push(chunk1, false);
 * deflate.push(chunk2, true);  // true -> last chunk
 *
 * if (deflate.err) { throw new Error(deflate.err); }
 *
 * console.log(deflate.result);
 * ```
 **/
function Deflate$1$1(options) {
  this.options = common$1.assign({
    level: Z_DEFAULT_COMPRESSION$2,
    method: Z_DEFLATED$1$1,
    chunkSize: 16384,
    windowBits: 15,
    memLevel: 8,
    strategy: Z_DEFAULT_STRATEGY$2
  }, options || {});

  let opt = this.options;

  if (opt.raw && (opt.windowBits > 0)) {
    opt.windowBits = -opt.windowBits;
  }

  else if (opt.gzip && (opt.windowBits > 0) && (opt.windowBits < 16)) {
    opt.windowBits += 16;
  }

  this.err    = 0;      // error code, if happens (0 = Z_OK)
  this.msg    = '';     // error message
  this.ended  = false;  // used to avoid multiple onEnd() calls
  this.chunks = [];     // chunks of compressed data

  this.strm = new zstream$1();
  this.strm.avail_out = 0;

  let status = deflate_1$2$1.deflateInit2(
    this.strm,
    opt.level,
    opt.method,
    opt.windowBits,
    opt.memLevel,
    opt.strategy
  );

  if (status !== Z_OK$2$1) {
    throw new Error(messages$1[status]);
  }

  if (opt.header) {
    deflate_1$2$1.deflateSetHeader(this.strm, opt.header);
  }

  if (opt.dictionary) {
    let dict;
    // Convert data if needed
    if (typeof opt.dictionary === 'string') {
      // If we need to compress text, change encoding to utf8.
      dict = strings$1.string2buf(opt.dictionary);
    } else if (toString$1$1.call(opt.dictionary) === '[object ArrayBuffer]') {
      dict = new Uint8Array(opt.dictionary);
    } else {
      dict = opt.dictionary;
    }

    status = deflate_1$2$1.deflateSetDictionary(this.strm, dict);

    if (status !== Z_OK$2$1) {
      throw new Error(messages$1[status]);
    }

    this._dict_set = true;
  }
}

/**
 * Deflate#push(data[, flush_mode]) -> Boolean
 * - data (Uint8Array|ArrayBuffer|String): input data. Strings will be
 *   converted to utf8 byte sequence.
 * - flush_mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
 *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` means Z_FINISH.
 *
 * Sends input data to deflate pipe, generating [[Deflate#onData]] calls with
 * new compressed chunks. Returns `true` on success. The last data block must
 * have `flush_mode` Z_FINISH (or `true`). That will flush internal pending
 * buffers and call [[Deflate#onEnd]].
 *
 * On fail call [[Deflate#onEnd]] with error code and return false.
 *
 * ##### Example
 *
 * ```javascript
 * push(chunk, false); // push one of data chunks
 * ...
 * push(chunk, true);  // push last chunk
 * ```
 **/
Deflate$1$1.prototype.push = function (data, flush_mode) {
  const strm = this.strm;
  const chunkSize = this.options.chunkSize;
  let status, _flush_mode;

  if (this.ended) { return false; }

  if (flush_mode === ~~flush_mode) _flush_mode = flush_mode;
  else _flush_mode = flush_mode === true ? Z_FINISH$2$1 : Z_NO_FLUSH$1$1;

  // Convert data if needed
  if (typeof data === 'string') {
    // If we need to compress text, change encoding to utf8.
    strm.input = strings$1.string2buf(data);
  } else if (toString$1$1.call(data) === '[object ArrayBuffer]') {
    strm.input = new Uint8Array(data);
  } else {
    strm.input = data;
  }

  strm.next_in = 0;
  strm.avail_in = strm.input.length;

  for (;;) {
    if (strm.avail_out === 0) {
      strm.output = new Uint8Array(chunkSize);
      strm.next_out = 0;
      strm.avail_out = chunkSize;
    }

    // Make sure avail_out > 6 to avoid repeating markers
    if ((_flush_mode === Z_SYNC_FLUSH$1 || _flush_mode === Z_FULL_FLUSH$2) && strm.avail_out <= 6) {
      this.onData(strm.output.subarray(0, strm.next_out));
      strm.avail_out = 0;
      continue;
    }

    status = deflate_1$2$1.deflate(strm, _flush_mode);

    // Ended => flush and finish
    if (status === Z_STREAM_END$2$1) {
      if (strm.next_out > 0) {
        this.onData(strm.output.subarray(0, strm.next_out));
      }
      status = deflate_1$2$1.deflateEnd(this.strm);
      this.onEnd(status);
      this.ended = true;
      return status === Z_OK$2$1;
    }

    // Flush if out buffer full
    if (strm.avail_out === 0) {
      this.onData(strm.output);
      continue;
    }

    // Flush if requested and has data
    if (_flush_mode > 0 && strm.next_out > 0) {
      this.onData(strm.output.subarray(0, strm.next_out));
      strm.avail_out = 0;
      continue;
    }

    if (strm.avail_in === 0) break;
  }

  return true;
};


/**
 * Deflate#onData(chunk) -> Void
 * - chunk (Uint8Array): output data.
 *
 * By default, stores data blocks in `chunks[]` property and glue
 * those in `onEnd`. Override this handler, if you need another behaviour.
 **/
Deflate$1$1.prototype.onData = function (chunk) {
  this.chunks.push(chunk);
};


/**
 * Deflate#onEnd(status) -> Void
 * - status (Number): deflate status. 0 (Z_OK) on success,
 *   other if not.
 *
 * Called once after you tell deflate that the input stream is
 * complete (Z_FINISH). By default - join collected chunks,
 * free memory and fill `results` / `err` properties.
 **/
Deflate$1$1.prototype.onEnd = function (status) {
  // On success - join
  if (status === Z_OK$2$1) {
    this.result = common$1.flattenChunks(this.chunks);
  }
  this.chunks = [];
  this.err = status;
  this.msg = this.strm.msg;
};


/**
 * deflate(data[, options]) -> Uint8Array
 * - data (Uint8Array|ArrayBuffer|String): input data to compress.
 * - options (Object): zlib deflate options.
 *
 * Compress `data` with deflate algorithm and `options`.
 *
 * Supported options are:
 *
 * - level
 * - windowBits
 * - memLevel
 * - strategy
 * - dictionary
 *
 * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
 * for more information on these.
 *
 * Sugar (options):
 *
 * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
 *   negative windowBits implicitly.
 *
 * ##### Example:
 *
 * ```javascript
 * const pako = require('pako')
 * const data = new Uint8Array([1,2,3,4,5,6,7,8,9]);
 *
 * console.log(pako.deflate(data));
 * ```
 **/
function deflate$1(input, options) {
  const deflator = new Deflate$1$1(options);

  deflator.push(input, true);

  // That will never happens, if you don't cheat with options :)
  if (deflator.err) { throw deflator.msg || messages$1[deflator.err]; }

  return deflator.result;
}


/**
 * deflateRaw(data[, options]) -> Uint8Array
 * - data (Uint8Array|ArrayBuffer|String): input data to compress.
 * - options (Object): zlib deflate options.
 *
 * The same as [[deflate]], but creates raw data, without wrapper
 * (header and adler32 crc).
 **/
function deflateRaw$1(input, options) {
  options = options || {};
  options.raw = true;
  return deflate$1(input, options);
}


/**
 * gzip(data[, options]) -> Uint8Array
 * - data (Uint8Array|ArrayBuffer|String): input data to compress.
 * - options (Object): zlib deflate options.
 *
 * The same as [[deflate]], but create gzip wrapper instead of
 * deflate one.
 **/
function gzip$1(input, options) {
  options = options || {};
  options.gzip = true;
  return deflate$1(input, options);
}


var Deflate_1$1 = Deflate$1$1;
var deflate_2 = deflate$1;
var deflateRaw_1$1 = deflateRaw$1;
var gzip_1$1 = gzip$1;
var constants$1 = constants$2$1;

var deflate_1$1 = {
	Deflate: Deflate_1$1,
	deflate: deflate_2,
	deflateRaw: deflateRaw_1$1,
	gzip: gzip_1$1,
	constants: constants$1
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

// See state defs from inflate.js
const BAD$1$1 = 16209;       /* got a data error -- remain here until reset */
const TYPE$1$1 = 16191;      /* i: waiting for type bits, including last-flag bit */

/*
   Decode literal, length, and distance codes and write out the resulting
   literal and match bytes until either not enough input or output is
   available, an end-of-block is encountered, or a data error is encountered.
   When large enough input and output buffers are supplied to inflate(), for
   example, a 16K input buffer and a 64K output buffer, more than 95% of the
   inflate execution time is spent in this routine.

   Entry assumptions:

        state.mode === LEN
        strm.avail_in >= 6
        strm.avail_out >= 258
        start >= strm.avail_out
        state.bits < 8

   On return, state.mode is one of:

        LEN -- ran out of enough output space or enough available input
        TYPE -- reached end of block code, inflate() to interpret next block
        BAD -- error in block data

   Notes:

    - The maximum input bits used by a length/distance pair is 15 bits for the
      length code, 5 bits for the length extra, 15 bits for the distance code,
      and 13 bits for the distance extra.  This totals 48 bits, or six bytes.
      Therefore if strm.avail_in >= 6, then there is enough input to avoid
      checking for available input while decoding.

    - The maximum bytes that a single length/distance pair can output is 258
      bytes, which is the maximum length that can be coded.  inflate_fast()
      requires strm.avail_out >= 258 for each loop to avoid checking for
      output space.
 */
var inffast$1 = function inflate_fast(strm, start) {
  let _in;                    /* local strm.input */
  let last;                   /* have enough input while in < last */
  let _out;                   /* local strm.output */
  let beg;                    /* inflate()'s initial strm.output */
  let end;                    /* while out < end, enough space available */
//#ifdef INFLATE_STRICT
  let dmax;                   /* maximum distance from zlib header */
//#endif
  let wsize;                  /* window size or zero if not using window */
  let whave;                  /* valid bytes in the window */
  let wnext;                  /* window write index */
  // Use `s_window` instead `window`, avoid conflict with instrumentation tools
  let s_window;               /* allocated sliding window, if wsize != 0 */
  let hold;                   /* local strm.hold */
  let bits;                   /* local strm.bits */
  let lcode;                  /* local strm.lencode */
  let dcode;                  /* local strm.distcode */
  let lmask;                  /* mask for first level of length codes */
  let dmask;                  /* mask for first level of distance codes */
  let here;                   /* retrieved table entry */
  let op;                     /* code bits, operation, extra bits, or */
                              /*  window position, window bytes to copy */
  let len;                    /* match length, unused bytes */
  let dist;                   /* match distance */
  let from;                   /* where to copy match from */
  let from_source;


  let input, output; // JS specific, because we have no pointers

  /* copy state to local variables */
  const state = strm.state;
  //here = state.here;
  _in = strm.next_in;
  input = strm.input;
  last = _in + (strm.avail_in - 5);
  _out = strm.next_out;
  output = strm.output;
  beg = _out - (start - strm.avail_out);
  end = _out + (strm.avail_out - 257);
//#ifdef INFLATE_STRICT
  dmax = state.dmax;
//#endif
  wsize = state.wsize;
  whave = state.whave;
  wnext = state.wnext;
  s_window = state.window;
  hold = state.hold;
  bits = state.bits;
  lcode = state.lencode;
  dcode = state.distcode;
  lmask = (1 << state.lenbits) - 1;
  dmask = (1 << state.distbits) - 1;


  /* decode literals and length/distances until end-of-block or not enough
     input data or output space */

  top:
  do {
    if (bits < 15) {
      hold += input[_in++] << bits;
      bits += 8;
      hold += input[_in++] << bits;
      bits += 8;
    }

    here = lcode[hold & lmask];

    dolen:
    for (;;) { // Goto emulation
      op = here >>> 24/*here.bits*/;
      hold >>>= op;
      bits -= op;
      op = (here >>> 16) & 0xff/*here.op*/;
      if (op === 0) {                          /* literal */
        //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
        //        "inflate:         literal '%c'\n" :
        //        "inflate:         literal 0x%02x\n", here.val));
        output[_out++] = here & 0xffff/*here.val*/;
      }
      else if (op & 16) {                     /* length base */
        len = here & 0xffff/*here.val*/;
        op &= 15;                           /* number of extra bits */
        if (op) {
          if (bits < op) {
            hold += input[_in++] << bits;
            bits += 8;
          }
          len += hold & ((1 << op) - 1);
          hold >>>= op;
          bits -= op;
        }
        //Tracevv((stderr, "inflate:         length %u\n", len));
        if (bits < 15) {
          hold += input[_in++] << bits;
          bits += 8;
          hold += input[_in++] << bits;
          bits += 8;
        }
        here = dcode[hold & dmask];

        dodist:
        for (;;) { // goto emulation
          op = here >>> 24/*here.bits*/;
          hold >>>= op;
          bits -= op;
          op = (here >>> 16) & 0xff/*here.op*/;

          if (op & 16) {                      /* distance base */
            dist = here & 0xffff/*here.val*/;
            op &= 15;                       /* number of extra bits */
            if (bits < op) {
              hold += input[_in++] << bits;
              bits += 8;
              if (bits < op) {
                hold += input[_in++] << bits;
                bits += 8;
              }
            }
            dist += hold & ((1 << op) - 1);
//#ifdef INFLATE_STRICT
            if (dist > dmax) {
              strm.msg = 'invalid distance too far back';
              state.mode = BAD$1$1;
              break top;
            }
//#endif
            hold >>>= op;
            bits -= op;
            //Tracevv((stderr, "inflate:         distance %u\n", dist));
            op = _out - beg;                /* max distance in output */
            if (dist > op) {                /* see if copy from window */
              op = dist - op;               /* distance back in window */
              if (op > whave) {
                if (state.sane) {
                  strm.msg = 'invalid distance too far back';
                  state.mode = BAD$1$1;
                  break top;
                }

// (!) This block is disabled in zlib defaults,
// don't enable it for binary compatibility
//#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
//                if (len <= op - whave) {
//                  do {
//                    output[_out++] = 0;
//                  } while (--len);
//                  continue top;
//                }
//                len -= op - whave;
//                do {
//                  output[_out++] = 0;
//                } while (--op > whave);
//                if (op === 0) {
//                  from = _out - dist;
//                  do {
//                    output[_out++] = output[from++];
//                  } while (--len);
//                  continue top;
//                }
//#endif
              }
              from = 0; // window index
              from_source = s_window;
              if (wnext === 0) {           /* very common case */
                from += wsize - op;
                if (op < len) {         /* some from window */
                  len -= op;
                  do {
                    output[_out++] = s_window[from++];
                  } while (--op);
                  from = _out - dist;  /* rest from output */
                  from_source = output;
                }
              }
              else if (wnext < op) {      /* wrap around window */
                from += wsize + wnext - op;
                op -= wnext;
                if (op < len) {         /* some from end of window */
                  len -= op;
                  do {
                    output[_out++] = s_window[from++];
                  } while (--op);
                  from = 0;
                  if (wnext < len) {  /* some from start of window */
                    op = wnext;
                    len -= op;
                    do {
                      output[_out++] = s_window[from++];
                    } while (--op);
                    from = _out - dist;      /* rest from output */
                    from_source = output;
                  }
                }
              }
              else {                      /* contiguous in window */
                from += wnext - op;
                if (op < len) {         /* some from window */
                  len -= op;
                  do {
                    output[_out++] = s_window[from++];
                  } while (--op);
                  from = _out - dist;  /* rest from output */
                  from_source = output;
                }
              }
              while (len > 2) {
                output[_out++] = from_source[from++];
                output[_out++] = from_source[from++];
                output[_out++] = from_source[from++];
                len -= 3;
              }
              if (len) {
                output[_out++] = from_source[from++];
                if (len > 1) {
                  output[_out++] = from_source[from++];
                }
              }
            }
            else {
              from = _out - dist;          /* copy direct from output */
              do {                        /* minimum length is three */
                output[_out++] = output[from++];
                output[_out++] = output[from++];
                output[_out++] = output[from++];
                len -= 3;
              } while (len > 2);
              if (len) {
                output[_out++] = output[from++];
                if (len > 1) {
                  output[_out++] = output[from++];
                }
              }
            }
          }
          else if ((op & 64) === 0) {          /* 2nd level distance code */
            here = dcode[(here & 0xffff)/*here.val*/ + (hold & ((1 << op) - 1))];
            continue dodist;
          }
          else {
            strm.msg = 'invalid distance code';
            state.mode = BAD$1$1;
            break top;
          }

          break; // need to emulate goto via "continue"
        }
      }
      else if ((op & 64) === 0) {              /* 2nd level length code */
        here = lcode[(here & 0xffff)/*here.val*/ + (hold & ((1 << op) - 1))];
        continue dolen;
      }
      else if (op & 32) {                     /* end-of-block */
        //Tracevv((stderr, "inflate:         end of block\n"));
        state.mode = TYPE$1$1;
        break top;
      }
      else {
        strm.msg = 'invalid literal/length code';
        state.mode = BAD$1$1;
        break top;
      }

      break; // need to emulate goto via "continue"
    }
  } while (_in < last && _out < end);

  /* return unused bytes (on entry, bits < 8, so in won't go too far back) */
  len = bits >> 3;
  _in -= len;
  bits -= len << 3;
  hold &= (1 << bits) - 1;

  /* update state and return */
  strm.next_in = _in;
  strm.next_out = _out;
  strm.avail_in = (_in < last ? 5 + (last - _in) : 5 - (_in - last));
  strm.avail_out = (_out < end ? 257 + (end - _out) : 257 - (_out - end));
  state.hold = hold;
  state.bits = bits;
  return;
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

const MAXBITS$1 = 15;
const ENOUGH_LENS$1$1 = 852;
const ENOUGH_DISTS$1$1 = 592;
//const ENOUGH = (ENOUGH_LENS+ENOUGH_DISTS);

const CODES$1$1 = 0;
const LENS$1$1 = 1;
const DISTS$1$1 = 2;

const lbase$1 = new Uint16Array([ /* Length codes 257..285 base */
  3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31,
  35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0
]);

const lext$1 = new Uint8Array([ /* Length codes 257..285 extra */
  16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18,
  19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78
]);

const dbase$1 = new Uint16Array([ /* Distance codes 0..29 base */
  1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193,
  257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145,
  8193, 12289, 16385, 24577, 0, 0
]);

const dext$1 = new Uint8Array([ /* Distance codes 0..29 extra */
  16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22,
  23, 23, 24, 24, 25, 25, 26, 26, 27, 27,
  28, 28, 29, 29, 64, 64
]);

const inflate_table$1 = (type, lens, lens_index, codes, table, table_index, work, opts) =>
{
  const bits = opts.bits;
      //here = opts.here; /* table entry for duplication */

  let len = 0;               /* a code's length in bits */
  let sym = 0;               /* index of code symbols */
  let min = 0, max = 0;          /* minimum and maximum code lengths */
  let root = 0;              /* number of index bits for root table */
  let curr = 0;              /* number of index bits for current table */
  let drop = 0;              /* code bits to drop for sub-table */
  let left = 0;                   /* number of prefix codes available */
  let used = 0;              /* code entries in table used */
  let huff = 0;              /* Huffman code */
  let incr;              /* for incrementing code, index */
  let fill;              /* index for replicating entries */
  let low;               /* low bits for current root entry */
  let mask;              /* mask for low root bits */
  let next;             /* next available space in table */
  let base = null;     /* base value table to use */
//  let shoextra;    /* extra bits table to use */
  let match;                  /* use base and extra for symbol >= match */
  const count = new Uint16Array(MAXBITS$1 + 1); //[MAXBITS+1];    /* number of codes of each length */
  const offs = new Uint16Array(MAXBITS$1 + 1); //[MAXBITS+1];     /* offsets in table for each length */
  let extra = null;

  let here_bits, here_op, here_val;

  /*
   Process a set of code lengths to create a canonical Huffman code.  The
   code lengths are lens[0..codes-1].  Each length corresponds to the
   symbols 0..codes-1.  The Huffman code is generated by first sorting the
   symbols by length from short to long, and retaining the symbol order
   for codes with equal lengths.  Then the code starts with all zero bits
   for the first code of the shortest length, and the codes are integer
   increments for the same length, and zeros are appended as the length
   increases.  For the deflate format, these bits are stored backwards
   from their more natural integer increment ordering, and so when the
   decoding tables are built in the large loop below, the integer codes
   are incremented backwards.

   This routine assumes, but does not check, that all of the entries in
   lens[] are in the range 0..MAXBITS.  The caller must assure this.
   1..MAXBITS is interpreted as that code length.  zero means that that
   symbol does not occur in this code.

   The codes are sorted by computing a count of codes for each length,
   creating from that a table of starting indices for each length in the
   sorted table, and then entering the symbols in order in the sorted
   table.  The sorted table is work[], with that space being provided by
   the caller.

   The length counts are used for other purposes as well, i.e. finding
   the minimum and maximum length codes, determining if there are any
   codes at all, checking for a valid set of lengths, and looking ahead
   at length counts to determine sub-table sizes when building the
   decoding tables.
   */

  /* accumulate lengths for codes (assumes lens[] all in 0..MAXBITS) */
  for (len = 0; len <= MAXBITS$1; len++) {
    count[len] = 0;
  }
  for (sym = 0; sym < codes; sym++) {
    count[lens[lens_index + sym]]++;
  }

  /* bound code lengths, force root to be within code lengths */
  root = bits;
  for (max = MAXBITS$1; max >= 1; max--) {
    if (count[max] !== 0) { break; }
  }
  if (root > max) {
    root = max;
  }
  if (max === 0) {                     /* no symbols to code at all */
    //table.op[opts.table_index] = 64;  //here.op = (var char)64;    /* invalid code marker */
    //table.bits[opts.table_index] = 1;   //here.bits = (var char)1;
    //table.val[opts.table_index++] = 0;   //here.val = (var short)0;
    table[table_index++] = (1 << 24) | (64 << 16) | 0;


    //table.op[opts.table_index] = 64;
    //table.bits[opts.table_index] = 1;
    //table.val[opts.table_index++] = 0;
    table[table_index++] = (1 << 24) | (64 << 16) | 0;

    opts.bits = 1;
    return 0;     /* no symbols, but wait for decoding to report error */
  }
  for (min = 1; min < max; min++) {
    if (count[min] !== 0) { break; }
  }
  if (root < min) {
    root = min;
  }

  /* check for an over-subscribed or incomplete set of lengths */
  left = 1;
  for (len = 1; len <= MAXBITS$1; len++) {
    left <<= 1;
    left -= count[len];
    if (left < 0) {
      return -1;
    }        /* over-subscribed */
  }
  if (left > 0 && (type === CODES$1$1 || max !== 1)) {
    return -1;                      /* incomplete set */
  }

  /* generate offsets into symbol table for each length for sorting */
  offs[1] = 0;
  for (len = 1; len < MAXBITS$1; len++) {
    offs[len + 1] = offs[len] + count[len];
  }

  /* sort symbols by length, by symbol order within each length */
  for (sym = 0; sym < codes; sym++) {
    if (lens[lens_index + sym] !== 0) {
      work[offs[lens[lens_index + sym]]++] = sym;
    }
  }

  /*
   Create and fill in decoding tables.  In this loop, the table being
   filled is at next and has curr index bits.  The code being used is huff
   with length len.  That code is converted to an index by dropping drop
   bits off of the bottom.  For codes where len is less than drop + curr,
   those top drop + curr - len bits are incremented through all values to
   fill the table with replicated entries.

   root is the number of index bits for the root table.  When len exceeds
   root, sub-tables are created pointed to by the root entry with an index
   of the low root bits of huff.  This is saved in low to check for when a
   new sub-table should be started.  drop is zero when the root table is
   being filled, and drop is root when sub-tables are being filled.

   When a new sub-table is needed, it is necessary to look ahead in the
   code lengths to determine what size sub-table is needed.  The length
   counts are used for this, and so count[] is decremented as codes are
   entered in the tables.

   used keeps track of how many table entries have been allocated from the
   provided *table space.  It is checked for LENS and DIST tables against
   the constants ENOUGH_LENS and ENOUGH_DISTS to guard against changes in
   the initial root table size constants.  See the comments in inftrees.h
   for more information.

   sym increments through all symbols, and the loop terminates when
   all codes of length max, i.e. all codes, have been processed.  This
   routine permits incomplete codes, so another loop after this one fills
   in the rest of the decoding tables with invalid code markers.
   */

  /* set up for code type */
  // poor man optimization - use if-else instead of switch,
  // to avoid deopts in old v8
  if (type === CODES$1$1) {
    base = extra = work;    /* dummy value--not used */
    match = 20;

  } else if (type === LENS$1$1) {
    base = lbase$1;
    extra = lext$1;
    match = 257;

  } else {                    /* DISTS */
    base = dbase$1;
    extra = dext$1;
    match = 0;
  }

  /* initialize opts for loop */
  huff = 0;                   /* starting code */
  sym = 0;                    /* starting code symbol */
  len = min;                  /* starting code length */
  next = table_index;              /* current table to fill in */
  curr = root;                /* current table index bits */
  drop = 0;                   /* current bits to drop from code for index */
  low = -1;                   /* trigger new sub-table when len > root */
  used = 1 << root;          /* use root table entries */
  mask = used - 1;            /* mask for comparing low */

  /* check available table space */
  if ((type === LENS$1$1 && used > ENOUGH_LENS$1$1) ||
    (type === DISTS$1$1 && used > ENOUGH_DISTS$1$1)) {
    return 1;
  }

  /* process all codes and make table entries */
  for (;;) {
    /* create table entry */
    here_bits = len - drop;
    if (work[sym] + 1 < match) {
      here_op = 0;
      here_val = work[sym];
    }
    else if (work[sym] >= match) {
      here_op = extra[work[sym] - match];
      here_val = base[work[sym] - match];
    }
    else {
      here_op = 32 + 64;         /* end of block */
      here_val = 0;
    }

    /* replicate for those indices with low len bits equal to huff */
    incr = 1 << (len - drop);
    fill = 1 << curr;
    min = fill;                 /* save offset to next table */
    do {
      fill -= incr;
      table[next + (huff >> drop) + fill] = (here_bits << 24) | (here_op << 16) | here_val |0;
    } while (fill !== 0);

    /* backwards increment the len-bit code huff */
    incr = 1 << (len - 1);
    while (huff & incr) {
      incr >>= 1;
    }
    if (incr !== 0) {
      huff &= incr - 1;
      huff += incr;
    } else {
      huff = 0;
    }

    /* go to next symbol, update count, len */
    sym++;
    if (--count[len] === 0) {
      if (len === max) { break; }
      len = lens[lens_index + work[sym]];
    }

    /* create new sub-table if needed */
    if (len > root && (huff & mask) !== low) {
      /* if first time, transition to sub-tables */
      if (drop === 0) {
        drop = root;
      }

      /* increment past last table */
      next += min;            /* here min is 1 << curr */

      /* determine length of next table */
      curr = len - drop;
      left = 1 << curr;
      while (curr + drop < max) {
        left -= count[curr + drop];
        if (left <= 0) { break; }
        curr++;
        left <<= 1;
      }

      /* check for enough space */
      used += 1 << curr;
      if ((type === LENS$1$1 && used > ENOUGH_LENS$1$1) ||
        (type === DISTS$1$1 && used > ENOUGH_DISTS$1$1)) {
        return 1;
      }

      /* point entry in root table to sub-table */
      low = huff & mask;
      /*table.op[low] = curr;
      table.bits[low] = root;
      table.val[low] = next - opts.table_index;*/
      table[low] = (root << 24) | (curr << 16) | (next - table_index) |0;
    }
  }

  /* fill in remaining table entry if code is incomplete (guaranteed to have
   at most one remaining entry, since if the code is incomplete, the
   maximum code length that was allowed to get this far is one bit) */
  if (huff !== 0) {
    //table.op[next + huff] = 64;            /* invalid code marker */
    //table.bits[next + huff] = len - drop;
    //table.val[next + huff] = 0;
    table[next + huff] = ((len - drop) << 24) | (64 << 16) |0;
  }

  /* set return parameters */
  //opts.table_index += used;
  opts.bits = root;
  return 0;
};


var inftrees$1 = inflate_table$1;

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.






const CODES$2 = 0;
const LENS$2 = 1;
const DISTS$2 = 2;

/* Public constants ==========================================================*/
/* ===========================================================================*/

const {
  Z_FINISH: Z_FINISH$1$1, Z_BLOCK: Z_BLOCK$2, Z_TREES: Z_TREES$1,
  Z_OK: Z_OK$1$1, Z_STREAM_END: Z_STREAM_END$1$1, Z_NEED_DICT: Z_NEED_DICT$1$1, Z_STREAM_ERROR: Z_STREAM_ERROR$1$1, Z_DATA_ERROR: Z_DATA_ERROR$1$1, Z_MEM_ERROR: Z_MEM_ERROR$1$1, Z_BUF_ERROR: Z_BUF_ERROR$2,
  Z_DEFLATED: Z_DEFLATED$3
} = constants$2$1;


/* STATES ====================================================================*/
/* ===========================================================================*/


const    HEAD$1 = 16180;       /* i: waiting for magic header */
const    FLAGS$1 = 16181;      /* i: waiting for method and flags (gzip) */
const    TIME$1 = 16182;       /* i: waiting for modification time (gzip) */
const    OS$1 = 16183;         /* i: waiting for extra flags and operating system (gzip) */
const    EXLEN$1 = 16184;      /* i: waiting for extra length (gzip) */
const    EXTRA$1 = 16185;      /* i: waiting for extra bytes (gzip) */
const    NAME$1 = 16186;       /* i: waiting for end of file name (gzip) */
const    COMMENT$1 = 16187;    /* i: waiting for end of comment (gzip) */
const    HCRC$1 = 16188;       /* i: waiting for header crc (gzip) */
const    DICTID$1 = 16189;    /* i: waiting for dictionary check value */
const    DICT$1 = 16190;      /* waiting for inflateSetDictionary() call */
const        TYPE$2 = 16191;      /* i: waiting for type bits, including last-flag bit */
const        TYPEDO$1 = 16192;    /* i: same, but skip check to exit inflate on new block */
const        STORED$1 = 16193;    /* i: waiting for stored size (length and complement) */
const        COPY_$1 = 16194;     /* i/o: same as COPY below, but only first time in */
const        COPY$1 = 16195;      /* i/o: waiting for input or output to copy stored block */
const        TABLE$1 = 16196;     /* i: waiting for dynamic block table lengths */
const        LENLENS$1 = 16197;   /* i: waiting for code length code lengths */
const        CODELENS$1 = 16198;  /* i: waiting for length/lit and distance code lengths */
const            LEN_$1 = 16199;      /* i: same as LEN below, but only first time in */
const            LEN$1 = 16200;       /* i: waiting for length/lit/eob code */
const            LENEXT$1 = 16201;    /* i: waiting for length extra bits */
const            DIST$1 = 16202;      /* i: waiting for distance code */
const            DISTEXT$1 = 16203;   /* i: waiting for distance extra bits */
const            MATCH$1 = 16204;     /* o: waiting for output space to copy string */
const            LIT$1 = 16205;       /* o: waiting for output space to write literal */
const    CHECK$1 = 16206;     /* i: waiting for 32-bit check value */
const    LENGTH$1 = 16207;    /* i: waiting for 32-bit length (gzip) */
const    DONE$1 = 16208;      /* finished check, done -- remain here until reset */
const    BAD$2 = 16209;       /* got a data error -- remain here until reset */
const    MEM$1 = 16210;       /* got an inflate() memory error -- remain here until reset */
const    SYNC$1 = 16211;      /* looking for synchronization bytes to restart inflate() */

/* ===========================================================================*/



const ENOUGH_LENS$2 = 852;
const ENOUGH_DISTS$2 = 592;
//const ENOUGH =  (ENOUGH_LENS+ENOUGH_DISTS);

const MAX_WBITS$2 = 15;
/* 32K LZ77 window */
const DEF_WBITS$1 = MAX_WBITS$2;


const zswap32$1 = (q) => {

  return  (((q >>> 24) & 0xff) +
          ((q >>> 8) & 0xff00) +
          ((q & 0xff00) << 8) +
          ((q & 0xff) << 24));
};


function InflateState$1() {
  this.strm = null;           /* pointer back to this zlib stream */
  this.mode = 0;              /* current inflate mode */
  this.last = false;          /* true if processing last block */
  this.wrap = 0;              /* bit 0 true for zlib, bit 1 true for gzip,
                                 bit 2 true to validate check value */
  this.havedict = false;      /* true if dictionary provided */
  this.flags = 0;             /* gzip header method and flags (0 if zlib), or
                                 -1 if raw or no header yet */
  this.dmax = 0;              /* zlib header max distance (INFLATE_STRICT) */
  this.check = 0;             /* protected copy of check value */
  this.total = 0;             /* protected copy of output count */
  // TODO: may be {}
  this.head = null;           /* where to save gzip header information */

  /* sliding window */
  this.wbits = 0;             /* log base 2 of requested window size */
  this.wsize = 0;             /* window size or zero if not using window */
  this.whave = 0;             /* valid bytes in the window */
  this.wnext = 0;             /* window write index */
  this.window = null;         /* allocated sliding window, if needed */

  /* bit accumulator */
  this.hold = 0;              /* input bit accumulator */
  this.bits = 0;              /* number of bits in "in" */

  /* for string and stored block copying */
  this.length = 0;            /* literal or length of data to copy */
  this.offset = 0;            /* distance back to copy string from */

  /* for table and code decoding */
  this.extra = 0;             /* extra bits needed */

  /* fixed and dynamic code tables */
  this.lencode = null;          /* starting table for length/literal codes */
  this.distcode = null;         /* starting table for distance codes */
  this.lenbits = 0;           /* index bits for lencode */
  this.distbits = 0;          /* index bits for distcode */

  /* dynamic table building */
  this.ncode = 0;             /* number of code length code lengths */
  this.nlen = 0;              /* number of length code lengths */
  this.ndist = 0;             /* number of distance code lengths */
  this.have = 0;              /* number of code lengths in lens[] */
  this.next = null;              /* next available space in codes[] */

  this.lens = new Uint16Array(320); /* temporary storage for code lengths */
  this.work = new Uint16Array(288); /* work area for code table building */

  /*
   because we don't have pointers in js, we use lencode and distcode directly
   as buffers so we don't need codes
  */
  //this.codes = new Int32Array(ENOUGH);       /* space for code tables */
  this.lendyn = null;              /* dynamic table for length/literal codes (JS specific) */
  this.distdyn = null;             /* dynamic table for distance codes (JS specific) */
  this.sane = 0;                   /* if false, allow invalid distance too far */
  this.back = 0;                   /* bits back of last unprocessed length/lit */
  this.was = 0;                    /* initial length of match */
}


const inflateStateCheck$1 = (strm) => {

  if (!strm) {
    return 1;
  }
  const state = strm.state;
  if (!state || state.strm !== strm ||
    state.mode < HEAD$1 || state.mode > SYNC$1) {
    return 1;
  }
  return 0;
};


const inflateResetKeep$1 = (strm) => {

  if (inflateStateCheck$1(strm)) { return Z_STREAM_ERROR$1$1; }
  const state = strm.state;
  strm.total_in = strm.total_out = state.total = 0;
  strm.msg = ''; /*Z_NULL*/
  if (state.wrap) {       /* to support ill-conceived Java test suite */
    strm.adler = state.wrap & 1;
  }
  state.mode = HEAD$1;
  state.last = 0;
  state.havedict = 0;
  state.flags = -1;
  state.dmax = 32768;
  state.head = null/*Z_NULL*/;
  state.hold = 0;
  state.bits = 0;
  //state.lencode = state.distcode = state.next = state.codes;
  state.lencode = state.lendyn = new Int32Array(ENOUGH_LENS$2);
  state.distcode = state.distdyn = new Int32Array(ENOUGH_DISTS$2);

  state.sane = 1;
  state.back = -1;
  //Tracev((stderr, "inflate: reset\n"));
  return Z_OK$1$1;
};


const inflateReset$1 = (strm) => {

  if (inflateStateCheck$1(strm)) { return Z_STREAM_ERROR$1$1; }
  const state = strm.state;
  state.wsize = 0;
  state.whave = 0;
  state.wnext = 0;
  return inflateResetKeep$1(strm);

};


const inflateReset2$1 = (strm, windowBits) => {
  let wrap;

  /* get the state */
  if (inflateStateCheck$1(strm)) { return Z_STREAM_ERROR$1$1; }
  const state = strm.state;

  /* extract wrap request from windowBits parameter */
  if (windowBits < 0) {
    wrap = 0;
    windowBits = -windowBits;
  }
  else {
    wrap = (windowBits >> 4) + 5;
    if (windowBits < 48) {
      windowBits &= 15;
    }
  }

  /* set number of window bits, free window if different */
  if (windowBits && (windowBits < 8 || windowBits > 15)) {
    return Z_STREAM_ERROR$1$1;
  }
  if (state.window !== null && state.wbits !== windowBits) {
    state.window = null;
  }

  /* update state and reset the rest of it */
  state.wrap = wrap;
  state.wbits = windowBits;
  return inflateReset$1(strm);
};


const inflateInit2$1 = (strm, windowBits) => {

  if (!strm) { return Z_STREAM_ERROR$1$1; }
  //strm.msg = Z_NULL;                 /* in case we return an error */

  const state = new InflateState$1();

  //if (state === Z_NULL) return Z_MEM_ERROR;
  //Tracev((stderr, "inflate: allocated\n"));
  strm.state = state;
  state.strm = strm;
  state.window = null/*Z_NULL*/;
  state.mode = HEAD$1;     /* to pass state test in inflateReset2() */
  const ret = inflateReset2$1(strm, windowBits);
  if (ret !== Z_OK$1$1) {
    strm.state = null/*Z_NULL*/;
  }
  return ret;
};


const inflateInit$1 = (strm) => {

  return inflateInit2$1(strm, DEF_WBITS$1);
};


/*
 Return state with length and distance decoding tables and index sizes set to
 fixed code decoding.  Normally this returns fixed tables from inffixed.h.
 If BUILDFIXED is defined, then instead this routine builds the tables the
 first time it's called, and returns those tables the first time and
 thereafter.  This reduces the size of the code by about 2K bytes, in
 exchange for a little execution time.  However, BUILDFIXED should not be
 used for threaded applications, since the rewriting of the tables and virgin
 may not be thread-safe.
 */
let virgin$1 = true;

let lenfix$1, distfix$1; // We have no pointers in JS, so keep tables separate


const fixedtables$1 = (state) => {

  /* build fixed huffman tables if first call (may not be thread safe) */
  if (virgin$1) {
    lenfix$1 = new Int32Array(512);
    distfix$1 = new Int32Array(32);

    /* literal/length table */
    let sym = 0;
    while (sym < 144) { state.lens[sym++] = 8; }
    while (sym < 256) { state.lens[sym++] = 9; }
    while (sym < 280) { state.lens[sym++] = 7; }
    while (sym < 288) { state.lens[sym++] = 8; }

    inftrees$1(LENS$2,  state.lens, 0, 288, lenfix$1,   0, state.work, { bits: 9 });

    /* distance table */
    sym = 0;
    while (sym < 32) { state.lens[sym++] = 5; }

    inftrees$1(DISTS$2, state.lens, 0, 32,   distfix$1, 0, state.work, { bits: 5 });

    /* do this just once */
    virgin$1 = false;
  }

  state.lencode = lenfix$1;
  state.lenbits = 9;
  state.distcode = distfix$1;
  state.distbits = 5;
};


/*
 Update the window with the last wsize (normally 32K) bytes written before
 returning.  If window does not exist yet, create it.  This is only called
 when a window is already in use, or when output has been written during this
 inflate call, but the end of the deflate stream has not been reached yet.
 It is also called to create a window for dictionary data when a dictionary
 is loaded.

 Providing output buffers larger than 32K to inflate() should provide a speed
 advantage, since only the last 32K of output is copied to the sliding window
 upon return from inflate(), and since all distances after the first 32K of
 output will fall in the output data, making match copies simpler and faster.
 The advantage may be dependent on the size of the processor's data caches.
 */
const updatewindow$1 = (strm, src, end, copy) => {

  let dist;
  const state = strm.state;

  /* if it hasn't been done already, allocate space for the window */
  if (state.window === null) {
    state.wsize = 1 << state.wbits;
    state.wnext = 0;
    state.whave = 0;

    state.window = new Uint8Array(state.wsize);
  }

  /* copy state->wsize or less output bytes into the circular window */
  if (copy >= state.wsize) {
    state.window.set(src.subarray(end - state.wsize, end), 0);
    state.wnext = 0;
    state.whave = state.wsize;
  }
  else {
    dist = state.wsize - state.wnext;
    if (dist > copy) {
      dist = copy;
    }
    //zmemcpy(state->window + state->wnext, end - copy, dist);
    state.window.set(src.subarray(end - copy, end - copy + dist), state.wnext);
    copy -= dist;
    if (copy) {
      //zmemcpy(state->window, end - copy, copy);
      state.window.set(src.subarray(end - copy, end), 0);
      state.wnext = copy;
      state.whave = state.wsize;
    }
    else {
      state.wnext += dist;
      if (state.wnext === state.wsize) { state.wnext = 0; }
      if (state.whave < state.wsize) { state.whave += dist; }
    }
  }
  return 0;
};


const inflate$2$1 = (strm, flush) => {

  let state;
  let input, output;          // input/output buffers
  let next;                   /* next input INDEX */
  let put;                    /* next output INDEX */
  let have, left;             /* available input and output */
  let hold;                   /* bit buffer */
  let bits;                   /* bits in bit buffer */
  let _in, _out;              /* save starting available input and output */
  let copy;                   /* number of stored or match bytes to copy */
  let from;                   /* where to copy match bytes from */
  let from_source;
  let here = 0;               /* current decoding table entry */
  let here_bits, here_op, here_val; // paked "here" denormalized (JS specific)
  //let last;                   /* parent table entry */
  let last_bits, last_op, last_val; // paked "last" denormalized (JS specific)
  let len;                    /* length to copy for repeats, bits to drop */
  let ret;                    /* return code */
  const hbuf = new Uint8Array(4);    /* buffer for gzip header crc calculation */
  let opts;

  let n; // temporary variable for NEED_BITS

  const order = /* permutation of code lengths */
    new Uint8Array([ 16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15 ]);


  if (inflateStateCheck$1(strm) || !strm.output ||
      (!strm.input && strm.avail_in !== 0)) {
    return Z_STREAM_ERROR$1$1;
  }

  state = strm.state;
  if (state.mode === TYPE$2) { state.mode = TYPEDO$1; }    /* skip check */


  //--- LOAD() ---
  put = strm.next_out;
  output = strm.output;
  left = strm.avail_out;
  next = strm.next_in;
  input = strm.input;
  have = strm.avail_in;
  hold = state.hold;
  bits = state.bits;
  //---

  _in = have;
  _out = left;
  ret = Z_OK$1$1;

  inf_leave: // goto emulation
  for (;;) {
    switch (state.mode) {
      case HEAD$1:
        if (state.wrap === 0) {
          state.mode = TYPEDO$1;
          break;
        }
        //=== NEEDBITS(16);
        while (bits < 16) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        if ((state.wrap & 2) && hold === 0x8b1f) {  /* gzip header */
          if (state.wbits === 0) {
            state.wbits = 15;
          }
          state.check = 0/*crc32(0L, Z_NULL, 0)*/;
          //=== CRC2(state.check, hold);
          hbuf[0] = hold & 0xff;
          hbuf[1] = (hold >>> 8) & 0xff;
          state.check = crc32_1$1(state.check, hbuf, 2, 0);
          //===//

          //=== INITBITS();
          hold = 0;
          bits = 0;
          //===//
          state.mode = FLAGS$1;
          break;
        }
        if (state.head) {
          state.head.done = false;
        }
        if (!(state.wrap & 1) ||   /* check if zlib header allowed */
          (((hold & 0xff)/*BITS(8)*/ << 8) + (hold >> 8)) % 31) {
          strm.msg = 'incorrect header check';
          state.mode = BAD$2;
          break;
        }
        if ((hold & 0x0f)/*BITS(4)*/ !== Z_DEFLATED$3) {
          strm.msg = 'unknown compression method';
          state.mode = BAD$2;
          break;
        }
        //--- DROPBITS(4) ---//
        hold >>>= 4;
        bits -= 4;
        //---//
        len = (hold & 0x0f)/*BITS(4)*/ + 8;
        if (state.wbits === 0) {
          state.wbits = len;
        }
        if (len > 15 || len > state.wbits) {
          strm.msg = 'invalid window size';
          state.mode = BAD$2;
          break;
        }

        // !!! pako patch. Force use `options.windowBits` if passed.
        // Required to always use max window size by default.
        state.dmax = 1 << state.wbits;
        //state.dmax = 1 << len;

        state.flags = 0;               /* indicate zlib header */
        //Tracev((stderr, "inflate:   zlib header ok\n"));
        strm.adler = state.check = 1/*adler32(0L, Z_NULL, 0)*/;
        state.mode = hold & 0x200 ? DICTID$1 : TYPE$2;
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        break;
      case FLAGS$1:
        //=== NEEDBITS(16); */
        while (bits < 16) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        state.flags = hold;
        if ((state.flags & 0xff) !== Z_DEFLATED$3) {
          strm.msg = 'unknown compression method';
          state.mode = BAD$2;
          break;
        }
        if (state.flags & 0xe000) {
          strm.msg = 'unknown header flags set';
          state.mode = BAD$2;
          break;
        }
        if (state.head) {
          state.head.text = ((hold >> 8) & 1);
        }
        if ((state.flags & 0x0200) && (state.wrap & 4)) {
          //=== CRC2(state.check, hold);
          hbuf[0] = hold & 0xff;
          hbuf[1] = (hold >>> 8) & 0xff;
          state.check = crc32_1$1(state.check, hbuf, 2, 0);
          //===//
        }
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        state.mode = TIME$1;
        /* falls through */
      case TIME$1:
        //=== NEEDBITS(32); */
        while (bits < 32) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        if (state.head) {
          state.head.time = hold;
        }
        if ((state.flags & 0x0200) && (state.wrap & 4)) {
          //=== CRC4(state.check, hold)
          hbuf[0] = hold & 0xff;
          hbuf[1] = (hold >>> 8) & 0xff;
          hbuf[2] = (hold >>> 16) & 0xff;
          hbuf[3] = (hold >>> 24) & 0xff;
          state.check = crc32_1$1(state.check, hbuf, 4, 0);
          //===
        }
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        state.mode = OS$1;
        /* falls through */
      case OS$1:
        //=== NEEDBITS(16); */
        while (bits < 16) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        if (state.head) {
          state.head.xflags = (hold & 0xff);
          state.head.os = (hold >> 8);
        }
        if ((state.flags & 0x0200) && (state.wrap & 4)) {
          //=== CRC2(state.check, hold);
          hbuf[0] = hold & 0xff;
          hbuf[1] = (hold >>> 8) & 0xff;
          state.check = crc32_1$1(state.check, hbuf, 2, 0);
          //===//
        }
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        state.mode = EXLEN$1;
        /* falls through */
      case EXLEN$1:
        if (state.flags & 0x0400) {
          //=== NEEDBITS(16); */
          while (bits < 16) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          state.length = hold;
          if (state.head) {
            state.head.extra_len = hold;
          }
          if ((state.flags & 0x0200) && (state.wrap & 4)) {
            //=== CRC2(state.check, hold);
            hbuf[0] = hold & 0xff;
            hbuf[1] = (hold >>> 8) & 0xff;
            state.check = crc32_1$1(state.check, hbuf, 2, 0);
            //===//
          }
          //=== INITBITS();
          hold = 0;
          bits = 0;
          //===//
        }
        else if (state.head) {
          state.head.extra = null/*Z_NULL*/;
        }
        state.mode = EXTRA$1;
        /* falls through */
      case EXTRA$1:
        if (state.flags & 0x0400) {
          copy = state.length;
          if (copy > have) { copy = have; }
          if (copy) {
            if (state.head) {
              len = state.head.extra_len - state.length;
              if (!state.head.extra) {
                // Use untyped array for more convenient processing later
                state.head.extra = new Uint8Array(state.head.extra_len);
              }
              state.head.extra.set(
                input.subarray(
                  next,
                  // extra field is limited to 65536 bytes
                  // - no need for additional size check
                  next + copy
                ),
                /*len + copy > state.head.extra_max - len ? state.head.extra_max : copy,*/
                len
              );
              //zmemcpy(state.head.extra + len, next,
              //        len + copy > state.head.extra_max ?
              //        state.head.extra_max - len : copy);
            }
            if ((state.flags & 0x0200) && (state.wrap & 4)) {
              state.check = crc32_1$1(state.check, input, copy, next);
            }
            have -= copy;
            next += copy;
            state.length -= copy;
          }
          if (state.length) { break inf_leave; }
        }
        state.length = 0;
        state.mode = NAME$1;
        /* falls through */
      case NAME$1:
        if (state.flags & 0x0800) {
          if (have === 0) { break inf_leave; }
          copy = 0;
          do {
            // TODO: 2 or 1 bytes?
            len = input[next + copy++];
            /* use constant limit because in js we should not preallocate memory */
            if (state.head && len &&
                (state.length < 65536 /*state.head.name_max*/)) {
              state.head.name += String.fromCharCode(len);
            }
          } while (len && copy < have);

          if ((state.flags & 0x0200) && (state.wrap & 4)) {
            state.check = crc32_1$1(state.check, input, copy, next);
          }
          have -= copy;
          next += copy;
          if (len) { break inf_leave; }
        }
        else if (state.head) {
          state.head.name = null;
        }
        state.length = 0;
        state.mode = COMMENT$1;
        /* falls through */
      case COMMENT$1:
        if (state.flags & 0x1000) {
          if (have === 0) { break inf_leave; }
          copy = 0;
          do {
            len = input[next + copy++];
            /* use constant limit because in js we should not preallocate memory */
            if (state.head && len &&
                (state.length < 65536 /*state.head.comm_max*/)) {
              state.head.comment += String.fromCharCode(len);
            }
          } while (len && copy < have);
          if ((state.flags & 0x0200) && (state.wrap & 4)) {
            state.check = crc32_1$1(state.check, input, copy, next);
          }
          have -= copy;
          next += copy;
          if (len) { break inf_leave; }
        }
        else if (state.head) {
          state.head.comment = null;
        }
        state.mode = HCRC$1;
        /* falls through */
      case HCRC$1:
        if (state.flags & 0x0200) {
          //=== NEEDBITS(16); */
          while (bits < 16) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          if ((state.wrap & 4) && hold !== (state.check & 0xffff)) {
            strm.msg = 'header crc mismatch';
            state.mode = BAD$2;
            break;
          }
          //=== INITBITS();
          hold = 0;
          bits = 0;
          //===//
        }
        if (state.head) {
          state.head.hcrc = ((state.flags >> 9) & 1);
          state.head.done = true;
        }
        strm.adler = state.check = 0;
        state.mode = TYPE$2;
        break;
      case DICTID$1:
        //=== NEEDBITS(32); */
        while (bits < 32) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        strm.adler = state.check = zswap32$1(hold);
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        state.mode = DICT$1;
        /* falls through */
      case DICT$1:
        if (state.havedict === 0) {
          //--- RESTORE() ---
          strm.next_out = put;
          strm.avail_out = left;
          strm.next_in = next;
          strm.avail_in = have;
          state.hold = hold;
          state.bits = bits;
          //---
          return Z_NEED_DICT$1$1;
        }
        strm.adler = state.check = 1/*adler32(0L, Z_NULL, 0)*/;
        state.mode = TYPE$2;
        /* falls through */
      case TYPE$2:
        if (flush === Z_BLOCK$2 || flush === Z_TREES$1) { break inf_leave; }
        /* falls through */
      case TYPEDO$1:
        if (state.last) {
          //--- BYTEBITS() ---//
          hold >>>= bits & 7;
          bits -= bits & 7;
          //---//
          state.mode = CHECK$1;
          break;
        }
        //=== NEEDBITS(3); */
        while (bits < 3) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        state.last = (hold & 0x01)/*BITS(1)*/;
        //--- DROPBITS(1) ---//
        hold >>>= 1;
        bits -= 1;
        //---//

        switch ((hold & 0x03)/*BITS(2)*/) {
          case 0:                             /* stored block */
            //Tracev((stderr, "inflate:     stored block%s\n",
            //        state.last ? " (last)" : ""));
            state.mode = STORED$1;
            break;
          case 1:                             /* fixed block */
            fixedtables$1(state);
            //Tracev((stderr, "inflate:     fixed codes block%s\n",
            //        state.last ? " (last)" : ""));
            state.mode = LEN_$1;             /* decode codes */
            if (flush === Z_TREES$1) {
              //--- DROPBITS(2) ---//
              hold >>>= 2;
              bits -= 2;
              //---//
              break inf_leave;
            }
            break;
          case 2:                             /* dynamic block */
            //Tracev((stderr, "inflate:     dynamic codes block%s\n",
            //        state.last ? " (last)" : ""));
            state.mode = TABLE$1;
            break;
          case 3:
            strm.msg = 'invalid block type';
            state.mode = BAD$2;
        }
        //--- DROPBITS(2) ---//
        hold >>>= 2;
        bits -= 2;
        //---//
        break;
      case STORED$1:
        //--- BYTEBITS() ---// /* go to byte boundary */
        hold >>>= bits & 7;
        bits -= bits & 7;
        //---//
        //=== NEEDBITS(32); */
        while (bits < 32) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        if ((hold & 0xffff) !== ((hold >>> 16) ^ 0xffff)) {
          strm.msg = 'invalid stored block lengths';
          state.mode = BAD$2;
          break;
        }
        state.length = hold & 0xffff;
        //Tracev((stderr, "inflate:       stored length %u\n",
        //        state.length));
        //=== INITBITS();
        hold = 0;
        bits = 0;
        //===//
        state.mode = COPY_$1;
        if (flush === Z_TREES$1) { break inf_leave; }
        /* falls through */
      case COPY_$1:
        state.mode = COPY$1;
        /* falls through */
      case COPY$1:
        copy = state.length;
        if (copy) {
          if (copy > have) { copy = have; }
          if (copy > left) { copy = left; }
          if (copy === 0) { break inf_leave; }
          //--- zmemcpy(put, next, copy); ---
          output.set(input.subarray(next, next + copy), put);
          //---//
          have -= copy;
          next += copy;
          left -= copy;
          put += copy;
          state.length -= copy;
          break;
        }
        //Tracev((stderr, "inflate:       stored end\n"));
        state.mode = TYPE$2;
        break;
      case TABLE$1:
        //=== NEEDBITS(14); */
        while (bits < 14) {
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
        }
        //===//
        state.nlen = (hold & 0x1f)/*BITS(5)*/ + 257;
        //--- DROPBITS(5) ---//
        hold >>>= 5;
        bits -= 5;
        //---//
        state.ndist = (hold & 0x1f)/*BITS(5)*/ + 1;
        //--- DROPBITS(5) ---//
        hold >>>= 5;
        bits -= 5;
        //---//
        state.ncode = (hold & 0x0f)/*BITS(4)*/ + 4;
        //--- DROPBITS(4) ---//
        hold >>>= 4;
        bits -= 4;
        //---//
//#ifndef PKZIP_BUG_WORKAROUND
        if (state.nlen > 286 || state.ndist > 30) {
          strm.msg = 'too many length or distance symbols';
          state.mode = BAD$2;
          break;
        }
//#endif
        //Tracev((stderr, "inflate:       table sizes ok\n"));
        state.have = 0;
        state.mode = LENLENS$1;
        /* falls through */
      case LENLENS$1:
        while (state.have < state.ncode) {
          //=== NEEDBITS(3);
          while (bits < 3) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          state.lens[order[state.have++]] = (hold & 0x07);//BITS(3);
          //--- DROPBITS(3) ---//
          hold >>>= 3;
          bits -= 3;
          //---//
        }
        while (state.have < 19) {
          state.lens[order[state.have++]] = 0;
        }
        // We have separate tables & no pointers. 2 commented lines below not needed.
        //state.next = state.codes;
        //state.lencode = state.next;
        // Switch to use dynamic table
        state.lencode = state.lendyn;
        state.lenbits = 7;

        opts = { bits: state.lenbits };
        ret = inftrees$1(CODES$2, state.lens, 0, 19, state.lencode, 0, state.work, opts);
        state.lenbits = opts.bits;

        if (ret) {
          strm.msg = 'invalid code lengths set';
          state.mode = BAD$2;
          break;
        }
        //Tracev((stderr, "inflate:       code lengths ok\n"));
        state.have = 0;
        state.mode = CODELENS$1;
        /* falls through */
      case CODELENS$1:
        while (state.have < state.nlen + state.ndist) {
          for (;;) {
            here = state.lencode[hold & ((1 << state.lenbits) - 1)];/*BITS(state.lenbits)*/
            here_bits = here >>> 24;
            here_op = (here >>> 16) & 0xff;
            here_val = here & 0xffff;

            if ((here_bits) <= bits) { break; }
            //--- PULLBYTE() ---//
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
            //---//
          }
          if (here_val < 16) {
            //--- DROPBITS(here.bits) ---//
            hold >>>= here_bits;
            bits -= here_bits;
            //---//
            state.lens[state.have++] = here_val;
          }
          else {
            if (here_val === 16) {
              //=== NEEDBITS(here.bits + 2);
              n = here_bits + 2;
              while (bits < n) {
                if (have === 0) { break inf_leave; }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              //--- DROPBITS(here.bits) ---//
              hold >>>= here_bits;
              bits -= here_bits;
              //---//
              if (state.have === 0) {
                strm.msg = 'invalid bit length repeat';
                state.mode = BAD$2;
                break;
              }
              len = state.lens[state.have - 1];
              copy = 3 + (hold & 0x03);//BITS(2);
              //--- DROPBITS(2) ---//
              hold >>>= 2;
              bits -= 2;
              //---//
            }
            else if (here_val === 17) {
              //=== NEEDBITS(here.bits + 3);
              n = here_bits + 3;
              while (bits < n) {
                if (have === 0) { break inf_leave; }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              //--- DROPBITS(here.bits) ---//
              hold >>>= here_bits;
              bits -= here_bits;
              //---//
              len = 0;
              copy = 3 + (hold & 0x07);//BITS(3);
              //--- DROPBITS(3) ---//
              hold >>>= 3;
              bits -= 3;
              //---//
            }
            else {
              //=== NEEDBITS(here.bits + 7);
              n = here_bits + 7;
              while (bits < n) {
                if (have === 0) { break inf_leave; }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              //--- DROPBITS(here.bits) ---//
              hold >>>= here_bits;
              bits -= here_bits;
              //---//
              len = 0;
              copy = 11 + (hold & 0x7f);//BITS(7);
              //--- DROPBITS(7) ---//
              hold >>>= 7;
              bits -= 7;
              //---//
            }
            if (state.have + copy > state.nlen + state.ndist) {
              strm.msg = 'invalid bit length repeat';
              state.mode = BAD$2;
              break;
            }
            while (copy--) {
              state.lens[state.have++] = len;
            }
          }
        }

        /* handle error breaks in while */
        if (state.mode === BAD$2) { break; }

        /* check for end-of-block code (better have one) */
        if (state.lens[256] === 0) {
          strm.msg = 'invalid code -- missing end-of-block';
          state.mode = BAD$2;
          break;
        }

        /* build code tables -- note: do not change the lenbits or distbits
           values here (9 and 6) without reading the comments in inftrees.h
           concerning the ENOUGH constants, which depend on those values */
        state.lenbits = 9;

        opts = { bits: state.lenbits };
        ret = inftrees$1(LENS$2, state.lens, 0, state.nlen, state.lencode, 0, state.work, opts);
        // We have separate tables & no pointers. 2 commented lines below not needed.
        // state.next_index = opts.table_index;
        state.lenbits = opts.bits;
        // state.lencode = state.next;

        if (ret) {
          strm.msg = 'invalid literal/lengths set';
          state.mode = BAD$2;
          break;
        }

        state.distbits = 6;
        //state.distcode.copy(state.codes);
        // Switch to use dynamic table
        state.distcode = state.distdyn;
        opts = { bits: state.distbits };
        ret = inftrees$1(DISTS$2, state.lens, state.nlen, state.ndist, state.distcode, 0, state.work, opts);
        // We have separate tables & no pointers. 2 commented lines below not needed.
        // state.next_index = opts.table_index;
        state.distbits = opts.bits;
        // state.distcode = state.next;

        if (ret) {
          strm.msg = 'invalid distances set';
          state.mode = BAD$2;
          break;
        }
        //Tracev((stderr, 'inflate:       codes ok\n'));
        state.mode = LEN_$1;
        if (flush === Z_TREES$1) { break inf_leave; }
        /* falls through */
      case LEN_$1:
        state.mode = LEN$1;
        /* falls through */
      case LEN$1:
        if (have >= 6 && left >= 258) {
          //--- RESTORE() ---
          strm.next_out = put;
          strm.avail_out = left;
          strm.next_in = next;
          strm.avail_in = have;
          state.hold = hold;
          state.bits = bits;
          //---
          inffast$1(strm, _out);
          //--- LOAD() ---
          put = strm.next_out;
          output = strm.output;
          left = strm.avail_out;
          next = strm.next_in;
          input = strm.input;
          have = strm.avail_in;
          hold = state.hold;
          bits = state.bits;
          //---

          if (state.mode === TYPE$2) {
            state.back = -1;
          }
          break;
        }
        state.back = 0;
        for (;;) {
          here = state.lencode[hold & ((1 << state.lenbits) - 1)];  /*BITS(state.lenbits)*/
          here_bits = here >>> 24;
          here_op = (here >>> 16) & 0xff;
          here_val = here & 0xffff;

          if (here_bits <= bits) { break; }
          //--- PULLBYTE() ---//
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
          //---//
        }
        if (here_op && (here_op & 0xf0) === 0) {
          last_bits = here_bits;
          last_op = here_op;
          last_val = here_val;
          for (;;) {
            here = state.lencode[last_val +
                    ((hold & ((1 << (last_bits + last_op)) - 1))/*BITS(last.bits + last.op)*/ >> last_bits)];
            here_bits = here >>> 24;
            here_op = (here >>> 16) & 0xff;
            here_val = here & 0xffff;

            if ((last_bits + here_bits) <= bits) { break; }
            //--- PULLBYTE() ---//
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
            //---//
          }
          //--- DROPBITS(last.bits) ---//
          hold >>>= last_bits;
          bits -= last_bits;
          //---//
          state.back += last_bits;
        }
        //--- DROPBITS(here.bits) ---//
        hold >>>= here_bits;
        bits -= here_bits;
        //---//
        state.back += here_bits;
        state.length = here_val;
        if (here_op === 0) {
          //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
          //        "inflate:         literal '%c'\n" :
          //        "inflate:         literal 0x%02x\n", here.val));
          state.mode = LIT$1;
          break;
        }
        if (here_op & 32) {
          //Tracevv((stderr, "inflate:         end of block\n"));
          state.back = -1;
          state.mode = TYPE$2;
          break;
        }
        if (here_op & 64) {
          strm.msg = 'invalid literal/length code';
          state.mode = BAD$2;
          break;
        }
        state.extra = here_op & 15;
        state.mode = LENEXT$1;
        /* falls through */
      case LENEXT$1:
        if (state.extra) {
          //=== NEEDBITS(state.extra);
          n = state.extra;
          while (bits < n) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          state.length += hold & ((1 << state.extra) - 1)/*BITS(state.extra)*/;
          //--- DROPBITS(state.extra) ---//
          hold >>>= state.extra;
          bits -= state.extra;
          //---//
          state.back += state.extra;
        }
        //Tracevv((stderr, "inflate:         length %u\n", state.length));
        state.was = state.length;
        state.mode = DIST$1;
        /* falls through */
      case DIST$1:
        for (;;) {
          here = state.distcode[hold & ((1 << state.distbits) - 1)];/*BITS(state.distbits)*/
          here_bits = here >>> 24;
          here_op = (here >>> 16) & 0xff;
          here_val = here & 0xffff;

          if ((here_bits) <= bits) { break; }
          //--- PULLBYTE() ---//
          if (have === 0) { break inf_leave; }
          have--;
          hold += input[next++] << bits;
          bits += 8;
          //---//
        }
        if ((here_op & 0xf0) === 0) {
          last_bits = here_bits;
          last_op = here_op;
          last_val = here_val;
          for (;;) {
            here = state.distcode[last_val +
                    ((hold & ((1 << (last_bits + last_op)) - 1))/*BITS(last.bits + last.op)*/ >> last_bits)];
            here_bits = here >>> 24;
            here_op = (here >>> 16) & 0xff;
            here_val = here & 0xffff;

            if ((last_bits + here_bits) <= bits) { break; }
            //--- PULLBYTE() ---//
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
            //---//
          }
          //--- DROPBITS(last.bits) ---//
          hold >>>= last_bits;
          bits -= last_bits;
          //---//
          state.back += last_bits;
        }
        //--- DROPBITS(here.bits) ---//
        hold >>>= here_bits;
        bits -= here_bits;
        //---//
        state.back += here_bits;
        if (here_op & 64) {
          strm.msg = 'invalid distance code';
          state.mode = BAD$2;
          break;
        }
        state.offset = here_val;
        state.extra = (here_op) & 15;
        state.mode = DISTEXT$1;
        /* falls through */
      case DISTEXT$1:
        if (state.extra) {
          //=== NEEDBITS(state.extra);
          n = state.extra;
          while (bits < n) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          state.offset += hold & ((1 << state.extra) - 1)/*BITS(state.extra)*/;
          //--- DROPBITS(state.extra) ---//
          hold >>>= state.extra;
          bits -= state.extra;
          //---//
          state.back += state.extra;
        }
//#ifdef INFLATE_STRICT
        if (state.offset > state.dmax) {
          strm.msg = 'invalid distance too far back';
          state.mode = BAD$2;
          break;
        }
//#endif
        //Tracevv((stderr, "inflate:         distance %u\n", state.offset));
        state.mode = MATCH$1;
        /* falls through */
      case MATCH$1:
        if (left === 0) { break inf_leave; }
        copy = _out - left;
        if (state.offset > copy) {         /* copy from window */
          copy = state.offset - copy;
          if (copy > state.whave) {
            if (state.sane) {
              strm.msg = 'invalid distance too far back';
              state.mode = BAD$2;
              break;
            }
// (!) This block is disabled in zlib defaults,
// don't enable it for binary compatibility
//#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
//          Trace((stderr, "inflate.c too far\n"));
//          copy -= state.whave;
//          if (copy > state.length) { copy = state.length; }
//          if (copy > left) { copy = left; }
//          left -= copy;
//          state.length -= copy;
//          do {
//            output[put++] = 0;
//          } while (--copy);
//          if (state.length === 0) { state.mode = LEN; }
//          break;
//#endif
          }
          if (copy > state.wnext) {
            copy -= state.wnext;
            from = state.wsize - copy;
          }
          else {
            from = state.wnext - copy;
          }
          if (copy > state.length) { copy = state.length; }
          from_source = state.window;
        }
        else {                              /* copy from output */
          from_source = output;
          from = put - state.offset;
          copy = state.length;
        }
        if (copy > left) { copy = left; }
        left -= copy;
        state.length -= copy;
        do {
          output[put++] = from_source[from++];
        } while (--copy);
        if (state.length === 0) { state.mode = LEN$1; }
        break;
      case LIT$1:
        if (left === 0) { break inf_leave; }
        output[put++] = state.length;
        left--;
        state.mode = LEN$1;
        break;
      case CHECK$1:
        if (state.wrap) {
          //=== NEEDBITS(32);
          while (bits < 32) {
            if (have === 0) { break inf_leave; }
            have--;
            // Use '|' instead of '+' to make sure that result is signed
            hold |= input[next++] << bits;
            bits += 8;
          }
          //===//
          _out -= left;
          strm.total_out += _out;
          state.total += _out;
          if ((state.wrap & 4) && _out) {
            strm.adler = state.check =
                /*UPDATE_CHECK(state.check, put - _out, _out);*/
                (state.flags ? crc32_1$1(state.check, output, _out, put - _out) : adler32_1$1(state.check, output, _out, put - _out));

          }
          _out = left;
          // NB: crc32 stored as signed 32-bit int, zswap32 returns signed too
          if ((state.wrap & 4) && (state.flags ? hold : zswap32$1(hold)) !== state.check) {
            strm.msg = 'incorrect data check';
            state.mode = BAD$2;
            break;
          }
          //=== INITBITS();
          hold = 0;
          bits = 0;
          //===//
          //Tracev((stderr, "inflate:   check matches trailer\n"));
        }
        state.mode = LENGTH$1;
        /* falls through */
      case LENGTH$1:
        if (state.wrap && state.flags) {
          //=== NEEDBITS(32);
          while (bits < 32) {
            if (have === 0) { break inf_leave; }
            have--;
            hold += input[next++] << bits;
            bits += 8;
          }
          //===//
          if ((state.wrap & 4) && hold !== (state.total & 0xffffffff)) {
            strm.msg = 'incorrect length check';
            state.mode = BAD$2;
            break;
          }
          //=== INITBITS();
          hold = 0;
          bits = 0;
          //===//
          //Tracev((stderr, "inflate:   length matches trailer\n"));
        }
        state.mode = DONE$1;
        /* falls through */
      case DONE$1:
        ret = Z_STREAM_END$1$1;
        break inf_leave;
      case BAD$2:
        ret = Z_DATA_ERROR$1$1;
        break inf_leave;
      case MEM$1:
        return Z_MEM_ERROR$1$1;
      case SYNC$1:
        /* falls through */
      default:
        return Z_STREAM_ERROR$1$1;
    }
  }

  // inf_leave <- here is real place for "goto inf_leave", emulated via "break inf_leave"

  /*
     Return from inflate(), updating the total counts and the check value.
     If there was no progress during the inflate() call, return a buffer
     error.  Call updatewindow() to create and/or update the window state.
     Note: a memory error from inflate() is non-recoverable.
   */

  //--- RESTORE() ---
  strm.next_out = put;
  strm.avail_out = left;
  strm.next_in = next;
  strm.avail_in = have;
  state.hold = hold;
  state.bits = bits;
  //---

  if (state.wsize || (_out !== strm.avail_out && state.mode < BAD$2 &&
                      (state.mode < CHECK$1 || flush !== Z_FINISH$1$1))) {
    if (updatewindow$1(strm, strm.output, strm.next_out, _out - strm.avail_out)) ;
  }
  _in -= strm.avail_in;
  _out -= strm.avail_out;
  strm.total_in += _in;
  strm.total_out += _out;
  state.total += _out;
  if ((state.wrap & 4) && _out) {
    strm.adler = state.check = /*UPDATE_CHECK(state.check, strm.next_out - _out, _out);*/
      (state.flags ? crc32_1$1(state.check, output, _out, strm.next_out - _out) : adler32_1$1(state.check, output, _out, strm.next_out - _out));
  }
  strm.data_type = state.bits + (state.last ? 64 : 0) +
                    (state.mode === TYPE$2 ? 128 : 0) +
                    (state.mode === LEN_$1 || state.mode === COPY_$1 ? 256 : 0);
  if (((_in === 0 && _out === 0) || flush === Z_FINISH$1$1) && ret === Z_OK$1$1) {
    ret = Z_BUF_ERROR$2;
  }
  return ret;
};


const inflateEnd$1 = (strm) => {

  if (inflateStateCheck$1(strm)) {
    return Z_STREAM_ERROR$1$1;
  }

  let state = strm.state;
  if (state.window) {
    state.window = null;
  }
  strm.state = null;
  return Z_OK$1$1;
};


const inflateGetHeader$1 = (strm, head) => {

  /* check state */
  if (inflateStateCheck$1(strm)) { return Z_STREAM_ERROR$1$1; }
  const state = strm.state;
  if ((state.wrap & 2) === 0) { return Z_STREAM_ERROR$1$1; }

  /* save header structure */
  state.head = head;
  head.done = false;
  return Z_OK$1$1;
};


const inflateSetDictionary$1 = (strm, dictionary) => {
  const dictLength = dictionary.length;

  let state;
  let dictid;
  let ret;

  /* check state */
  if (inflateStateCheck$1(strm)) { return Z_STREAM_ERROR$1$1; }
  state = strm.state;

  if (state.wrap !== 0 && state.mode !== DICT$1) {
    return Z_STREAM_ERROR$1$1;
  }

  /* check for correct dictionary identifier */
  if (state.mode === DICT$1) {
    dictid = 1; /* adler32(0, null, 0)*/
    /* dictid = adler32(dictid, dictionary, dictLength); */
    dictid = adler32_1$1(dictid, dictionary, dictLength, 0);
    if (dictid !== state.check) {
      return Z_DATA_ERROR$1$1;
    }
  }
  /* copy dictionary to window using updatewindow(), which will amend the
   existing dictionary if appropriate */
  ret = updatewindow$1(strm, dictionary, dictLength, dictLength);
  if (ret) {
    state.mode = MEM$1;
    return Z_MEM_ERROR$1$1;
  }
  state.havedict = 1;
  // Tracev((stderr, "inflate:   dictionary set\n"));
  return Z_OK$1$1;
};


var inflateReset_1$1 = inflateReset$1;
var inflateReset2_1$1 = inflateReset2$1;
var inflateResetKeep_1$1 = inflateResetKeep$1;
var inflateInit_1$1 = inflateInit$1;
var inflateInit2_1$1 = inflateInit2$1;
var inflate_2$1$1 = inflate$2$1;
var inflateEnd_1$1 = inflateEnd$1;
var inflateGetHeader_1$1 = inflateGetHeader$1;
var inflateSetDictionary_1$1 = inflateSetDictionary$1;
var inflateInfo$1 = 'pako inflate (from Nodeca project)';

/* Not implemented
module.exports.inflateCodesUsed = inflateCodesUsed;
module.exports.inflateCopy = inflateCopy;
module.exports.inflateGetDictionary = inflateGetDictionary;
module.exports.inflateMark = inflateMark;
module.exports.inflatePrime = inflatePrime;
module.exports.inflateSync = inflateSync;
module.exports.inflateSyncPoint = inflateSyncPoint;
module.exports.inflateUndermine = inflateUndermine;
module.exports.inflateValidate = inflateValidate;
*/

var inflate_1$2$1 = {
	inflateReset: inflateReset_1$1,
	inflateReset2: inflateReset2_1$1,
	inflateResetKeep: inflateResetKeep_1$1,
	inflateInit: inflateInit_1$1,
	inflateInit2: inflateInit2_1$1,
	inflate: inflate_2$1$1,
	inflateEnd: inflateEnd_1$1,
	inflateGetHeader: inflateGetHeader_1$1,
	inflateSetDictionary: inflateSetDictionary_1$1,
	inflateInfo: inflateInfo$1
};

// (C) 1995-2013 Jean-loup Gailly and Mark Adler
// (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//   claim that you wrote the original software. If you use this software
//   in a product, an acknowledgment in the product documentation would be
//   appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//   misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.

function GZheader$1() {
  /* true if compressed data believed to be text */
  this.text       = 0;
  /* modification time */
  this.time       = 0;
  /* extra flags (not used when writing a gzip file) */
  this.xflags     = 0;
  /* operating system */
  this.os         = 0;
  /* pointer to extra field or Z_NULL if none */
  this.extra      = null;
  /* extra field length (valid if extra != Z_NULL) */
  this.extra_len  = 0; // Actually, we don't need it in JS,
                       // but leave for few code modifications

  //
  // Setup limits is not necessary because in js we should not preallocate memory
  // for inflate use constant limit in 65536 bytes
  //

  /* space at extra (only when reading header) */
  // this.extra_max  = 0;
  /* pointer to zero-terminated file name or Z_NULL */
  this.name       = '';
  /* space at name (only when reading header) */
  // this.name_max   = 0;
  /* pointer to zero-terminated comment or Z_NULL */
  this.comment    = '';
  /* space at comment (only when reading header) */
  // this.comm_max   = 0;
  /* true if there was or will be a header crc */
  this.hcrc       = 0;
  /* true when done reading gzip header (not used when writing a gzip file) */
  this.done       = false;
}

var gzheader$1 = GZheader$1;

const toString$6 = Object.prototype.toString;

/* Public constants ==========================================================*/
/* ===========================================================================*/

const {
  Z_NO_FLUSH: Z_NO_FLUSH$3, Z_FINISH: Z_FINISH$4,
  Z_OK: Z_OK$4, Z_STREAM_END: Z_STREAM_END$4, Z_NEED_DICT: Z_NEED_DICT$2, Z_STREAM_ERROR: Z_STREAM_ERROR$3, Z_DATA_ERROR: Z_DATA_ERROR$3, Z_MEM_ERROR: Z_MEM_ERROR$2
} = constants$2$1;

/* ===========================================================================*/


/**
 * class Inflate
 *
 * Generic JS-style wrapper for zlib calls. If you don't need
 * streaming behaviour - use more simple functions: [[inflate]]
 * and [[inflateRaw]].
 **/

/* internal
 * inflate.chunks -> Array
 *
 * Chunks of output data, if [[Inflate#onData]] not overridden.
 **/

/**
 * Inflate.result -> Uint8Array|String
 *
 * Uncompressed result, generated by default [[Inflate#onData]]
 * and [[Inflate#onEnd]] handlers. Filled after you push last chunk
 * (call [[Inflate#push]] with `Z_FINISH` / `true` param).
 **/

/**
 * Inflate.err -> Number
 *
 * Error code after inflate finished. 0 (Z_OK) on success.
 * Should be checked if broken data possible.
 **/

/**
 * Inflate.msg -> String
 *
 * Error message, if [[Inflate.err]] != 0
 **/


/**
 * new Inflate(options)
 * - options (Object): zlib inflate options.
 *
 * Creates new inflator instance with specified params. Throws exception
 * on bad params. Supported options:
 *
 * - `windowBits`
 * - `dictionary`
 *
 * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
 * for more information on these.
 *
 * Additional options, for internal needs:
 *
 * - `chunkSize` - size of generated data chunks (16K by default)
 * - `raw` (Boolean) - do raw inflate
 * - `to` (String) - if equal to 'string', then result will be converted
 *   from utf8 to utf16 (javascript) string. When string output requested,
 *   chunk length can differ from `chunkSize`, depending on content.
 *
 * By default, when no options set, autodetect deflate/gzip data format via
 * wrapper header.
 *
 * ##### Example:
 *
 * ```javascript
 * const pako = require('pako')
 * const chunk1 = new Uint8Array([1,2,3,4,5,6,7,8,9])
 * const chunk2 = new Uint8Array([10,11,12,13,14,15,16,17,18,19]);
 *
 * const inflate = new pako.Inflate({ level: 3});
 *
 * inflate.push(chunk1, false);
 * inflate.push(chunk2, true);  // true -> last chunk
 *
 * if (inflate.err) { throw new Error(inflate.err); }
 *
 * console.log(inflate.result);
 * ```
 **/
function Inflate$1$1(options) {
  this.options = common$1.assign({
    chunkSize: 1024 * 64,
    windowBits: 15,
    to: ''
  }, options || {});

  const opt = this.options;

  // Force window size for `raw` data, if not set directly,
  // because we have no header for autodetect.
  if (opt.raw && (opt.windowBits >= 0) && (opt.windowBits < 16)) {
    opt.windowBits = -opt.windowBits;
    if (opt.windowBits === 0) { opt.windowBits = -15; }
  }

  // If `windowBits` not defined (and mode not raw) - set autodetect flag for gzip/deflate
  if ((opt.windowBits >= 0) && (opt.windowBits < 16) &&
      !(options && options.windowBits)) {
    opt.windowBits += 32;
  }

  // Gzip header has no info about windows size, we can do autodetect only
  // for deflate. So, if window size not set, force it to max when gzip possible
  if ((opt.windowBits > 15) && (opt.windowBits < 48)) {
    // bit 3 (16) -> gzipped data
    // bit 4 (32) -> autodetect gzip/deflate
    if ((opt.windowBits & 15) === 0) {
      opt.windowBits |= 15;
    }
  }

  this.err    = 0;      // error code, if happens (0 = Z_OK)
  this.msg    = '';     // error message
  this.ended  = false;  // used to avoid multiple onEnd() calls
  this.chunks = [];     // chunks of compressed data

  this.strm   = new zstream$1();
  this.strm.avail_out = 0;

  let status  = inflate_1$2$1.inflateInit2(
    this.strm,
    opt.windowBits
  );

  if (status !== Z_OK$4) {
    throw new Error(messages$1[status]);
  }

  this.header = new gzheader$1();

  inflate_1$2$1.inflateGetHeader(this.strm, this.header);

  // Setup dictionary
  if (opt.dictionary) {
    // Convert data if needed
    if (typeof opt.dictionary === 'string') {
      opt.dictionary = strings$1.string2buf(opt.dictionary);
    } else if (toString$6.call(opt.dictionary) === '[object ArrayBuffer]') {
      opt.dictionary = new Uint8Array(opt.dictionary);
    }
    if (opt.raw) { //In raw mode we need to set the dictionary early
      status = inflate_1$2$1.inflateSetDictionary(this.strm, opt.dictionary);
      if (status !== Z_OK$4) {
        throw new Error(messages$1[status]);
      }
    }
  }
}

/**
 * Inflate#push(data[, flush_mode]) -> Boolean
 * - data (Uint8Array|ArrayBuffer): input data
 * - flush_mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE
 *   flush modes. See constants. Skipped or `false` means Z_NO_FLUSH,
 *   `true` means Z_FINISH.
 *
 * Sends input data to inflate pipe, generating [[Inflate#onData]] calls with
 * new output chunks. Returns `true` on success. If end of stream detected,
 * [[Inflate#onEnd]] will be called.
 *
 * `flush_mode` is not needed for normal operation, because end of stream
 * detected automatically. You may try to use it for advanced things, but
 * this functionality was not tested.
 *
 * On fail call [[Inflate#onEnd]] with error code and return false.
 *
 * ##### Example
 *
 * ```javascript
 * push(chunk, false); // push one of data chunks
 * ...
 * push(chunk, true);  // push last chunk
 * ```
 **/
Inflate$1$1.prototype.push = function (data, flush_mode) {
  const strm = this.strm;
  const chunkSize = this.options.chunkSize;
  const dictionary = this.options.dictionary;
  let status, _flush_mode, last_avail_out;

  if (this.ended) return false;

  if (flush_mode === ~~flush_mode) _flush_mode = flush_mode;
  else _flush_mode = flush_mode === true ? Z_FINISH$4 : Z_NO_FLUSH$3;

  // Convert data if needed
  if (toString$6.call(data) === '[object ArrayBuffer]') {
    strm.input = new Uint8Array(data);
  } else {
    strm.input = data;
  }

  strm.next_in = 0;
  strm.avail_in = strm.input.length;

  for (;;) {
    if (strm.avail_out === 0) {
      strm.output = new Uint8Array(chunkSize);
      strm.next_out = 0;
      strm.avail_out = chunkSize;
    }

    status = inflate_1$2$1.inflate(strm, _flush_mode);

    if (status === Z_NEED_DICT$2 && dictionary) {
      status = inflate_1$2$1.inflateSetDictionary(strm, dictionary);

      if (status === Z_OK$4) {
        status = inflate_1$2$1.inflate(strm, _flush_mode);
      } else if (status === Z_DATA_ERROR$3) {
        // Replace code with more verbose
        status = Z_NEED_DICT$2;
      }
    }

    // Skip snyc markers if more data follows and not raw mode
    while (strm.avail_in > 0 &&
           status === Z_STREAM_END$4 &&
           strm.state.wrap > 0 &&
           data[strm.next_in] !== 0)
    {
      inflate_1$2$1.inflateReset(strm);
      status = inflate_1$2$1.inflate(strm, _flush_mode);
    }

    switch (status) {
      case Z_STREAM_ERROR$3:
      case Z_DATA_ERROR$3:
      case Z_NEED_DICT$2:
      case Z_MEM_ERROR$2:
        this.onEnd(status);
        this.ended = true;
        return false;
    }

    // Remember real `avail_out` value, because we may patch out buffer content
    // to align utf8 strings boundaries.
    last_avail_out = strm.avail_out;

    if (strm.next_out) {
      if (strm.avail_out === 0 || status === Z_STREAM_END$4) {

        if (this.options.to === 'string') {

          let next_out_utf8 = strings$1.utf8border(strm.output, strm.next_out);

          let tail = strm.next_out - next_out_utf8;
          let utf8str = strings$1.buf2string(strm.output, next_out_utf8);

          // move tail & realign counters
          strm.next_out = tail;
          strm.avail_out = chunkSize - tail;
          if (tail) strm.output.set(strm.output.subarray(next_out_utf8, next_out_utf8 + tail), 0);

          this.onData(utf8str);

        } else {
          this.onData(strm.output.length === strm.next_out ? strm.output : strm.output.subarray(0, strm.next_out));
        }
      }
    }

    // Must repeat iteration if out buffer is full
    if (status === Z_OK$4 && last_avail_out === 0) continue;

    // Finalize if end of stream reached.
    if (status === Z_STREAM_END$4) {
      status = inflate_1$2$1.inflateEnd(this.strm);
      this.onEnd(status);
      this.ended = true;
      return true;
    }

    if (strm.avail_in === 0) break;
  }

  return true;
};


/**
 * Inflate#onData(chunk) -> Void
 * - chunk (Uint8Array|String): output data. When string output requested,
 *   each chunk will be string.
 *
 * By default, stores data blocks in `chunks[]` property and glue
 * those in `onEnd`. Override this handler, if you need another behaviour.
 **/
Inflate$1$1.prototype.onData = function (chunk) {
  this.chunks.push(chunk);
};


/**
 * Inflate#onEnd(status) -> Void
 * - status (Number): inflate status. 0 (Z_OK) on success,
 *   other if not.
 *
 * Called either after you tell inflate that the input stream is
 * complete (Z_FINISH). By default - join collected chunks,
 * free memory and fill `results` / `err` properties.
 **/
Inflate$1$1.prototype.onEnd = function (status) {
  // On success - join
  if (status === Z_OK$4) {
    if (this.options.to === 'string') {
      this.result = this.chunks.join('');
    } else {
      this.result = common$1.flattenChunks(this.chunks);
    }
  }
  this.chunks = [];
  this.err = status;
  this.msg = this.strm.msg;
};


/**
 * inflate(data[, options]) -> Uint8Array|String
 * - data (Uint8Array|ArrayBuffer): input data to decompress.
 * - options (Object): zlib inflate options.
 *
 * Decompress `data` with inflate/ungzip and `options`. Autodetect
 * format via wrapper header by default. That's why we don't provide
 * separate `ungzip` method.
 *
 * Supported options are:
 *
 * - windowBits
 *
 * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
 * for more information.
 *
 * Sugar (options):
 *
 * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
 *   negative windowBits implicitly.
 * - `to` (String) - if equal to 'string', then result will be converted
 *   from utf8 to utf16 (javascript) string. When string output requested,
 *   chunk length can differ from `chunkSize`, depending on content.
 *
 *
 * ##### Example:
 *
 * ```javascript
 * const pako = require('pako');
 * const input = pako.deflate(new Uint8Array([1,2,3,4,5,6,7,8,9]));
 * let output;
 *
 * try {
 *   output = pako.inflate(input);
 * } catch (err) {
 *   console.log(err);
 * }
 * ```
 **/
function inflate$1$1(input, options) {
  const inflator = new Inflate$1$1(options);

  inflator.push(input);

  // That will never happens, if you don't cheat with options :)
  if (inflator.err) throw inflator.msg || messages$1[inflator.err];

  return inflator.result;
}


/**
 * inflateRaw(data[, options]) -> Uint8Array|String
 * - data (Uint8Array|ArrayBuffer): input data to decompress.
 * - options (Object): zlib inflate options.
 *
 * The same as [[inflate]], but creates raw data, without wrapper
 * (header and adler32 crc).
 **/
function inflateRaw$1$1(input, options) {
  options = options || {};
  options.raw = true;
  return inflate$1$1(input, options);
}


/**
 * ungzip(data[, options]) -> Uint8Array|String
 * - data (Uint8Array|ArrayBuffer): input data to decompress.
 * - options (Object): zlib inflate options.
 *
 * Just shortcut to [[inflate]], because it autodetects format
 * by header.content. Done for convenience.
 **/


var Inflate_1$1$1 = Inflate$1$1;
var inflate_2$2 = inflate$1$1;
var inflateRaw_1$1$1 = inflateRaw$1$1;
var ungzip$1$1 = inflate$1$1;
var constants$3 = constants$2$1;

var inflate_1$1$1 = {
	Inflate: Inflate_1$1$1,
	inflate: inflate_2$2,
	inflateRaw: inflateRaw_1$1$1,
	ungzip: ungzip$1$1,
	constants: constants$3
};

const { Deflate, deflate, deflateRaw, gzip } = deflate_1$1;

const { Inflate: Inflate$2, inflate: inflate$3, inflateRaw: inflateRaw$2, ungzip: ungzip$2 } = inflate_1$1$1;
var deflateRaw_1 = deflateRaw;
var inflate_1$3 = inflate$3;
var inflateRaw_1$2 = inflateRaw$2;
var ungzip_1$1 = ungzip$2;

const FEXTRA$1 = 4;  // gzip spec F.EXTRA flag

function isgzipped$1(data) {
    const b = ArrayBuffer.isView(data) ? data : new Uint8Array(data);
    return b[0] ===31 && b[1] === 139;
}

/**
 * Pako does not properly ungzip block compressed files if > 1 block is present.  Test for bgzip and use wrapper.
 */
function ungzip_blocks$1(data) {
    const ba = ArrayBuffer.isView(data) ? data : new Uint8Array(data);
    const b = ba[3] & FEXTRA$1;
    if (b !== 0 && ba[12] === 66 && ba[13] === 67) {
        return unbgzf$1(ba.buffer);
    } else {
        return ungzip_1$1(ba);
    }
}

// Uncompress data,  assumed to be series of bgzipped blocks
function unbgzf$1(data, lim) {

    const oBlockList = [];
    let ptr = 0;
    let totalSize = 0;

    lim = lim || data.byteLength - 18;

    while (ptr < lim) {
        try {
            const ba = ArrayBuffer.isView(data) ? data : new Uint8Array(data, ptr, 18);
            const xlen = (ba[11] << 8) | (ba[10]);
            const flg = ba[3];
            const fextra = flg & FEXTRA$1;
            const si1 = ba[12];
            const si2 = ba[13];
            const slen = (ba[15] << 8) | (ba[14]);
            const bsize = ((ba[17] << 8) | (ba[16])) + 1;
            const start = 12 + xlen + ptr;    // Start of CDATA
            const bytesLeft = data.byteLength - start;
            const cDataSize = bsize - xlen - 19;
            if (bytesLeft < cDataSize || cDataSize <= 0) break;

            const a = new Uint8Array(data, start, cDataSize);
            const unc = inflateRaw_1$2(a);

            // const inflate = new Zlib.RawInflate(a);
            // const unc = inflate.decompress();

            ptr += (cDataSize - 1) + 26; //inflate.ip + 26
            totalSize += unc.byteLength;
            oBlockList.push(unc);
        } catch (e) {
            console.error(e);
            break;
        }
    }

    // Concatenate decompressed blocks
    if (oBlockList.length === 1) {
        return oBlockList[0];
    } else {
        const out = new Uint8Array(totalSize);
        let cursor = 0;
        for (let i = 0; i < oBlockList.length; ++i) {
            var b = new Uint8Array(oBlockList[i]);
            arrayCopy$1(b, 0, out, cursor, b.length);
            cursor += b.length;
        }
        return out;
    }
}

function bgzBlockSize$1(data) {
    const ba = ArrayBuffer.isView(data) ? data : new Uint8Array(data);
    const bsize = (ba[17] << 8 | ba[16]) + 1;
    return bsize;
}

// From Thomas Down's zlib implementation

const testArray$1 = new Uint8Array(1);
const hasSubarray$1 = (typeof testArray$1.subarray === 'function');

function arrayCopy$1(src, srcOffset, dest, destOffset, count) {
    if (count === 0) {
        return;
    }
    if (!src) {
        throw "Undef src";
    } else if (!dest) {
        throw "Undef dest";
    }
    if (srcOffset === 0 && count === src.length) {
        arrayCopy_fast$1(src, dest, destOffset);
    } else if (hasSubarray$1) {
        arrayCopy_fast$1(src.subarray(srcOffset, srcOffset + count), dest, destOffset);
    } else if (src.BYTES_PER_ELEMENT === 1 && count > 100) {
        arrayCopy_fast$1(new Uint8Array(src.buffer, src.byteOffset + srcOffset, count), dest, destOffset);
    } else {
        arrayCopy_slow$1(src, srcOffset, dest, destOffset, count);
    }
}

function arrayCopy_slow$1(src, srcOffset, dest, destOffset, count) {
    for (let i = 0; i < count; ++i) {
        dest[destOffset + i] = src[srcOffset + i];
    }
}

function arrayCopy_fast$1(src, dest, destOffset) {
    dest.set(src, destOffset);
}


/**
 * Compress string and encode in a url safe form
 * @param s
 */
function compressString(str) {

    const bytes = new Uint8Array(str.length);
    for (var i = 0; i < str.length; i++) {
        bytes[i] = str.charCodeAt(i);
    }
    const compressedBytes = new deflateRaw_1(bytes);            // UInt8Arry
    const compressedString = String.fromCharCode.apply(null, compressedBytes);      // Convert to string
    let enc = btoa(compressedString);
    return enc.replace(/\+/g, '.').replace(/\//g, '_').replace(/=/g, '-');   // URL safe
}

/**
 * Uncompress the url-safe encoded compressed string, presumably created by compressString above
 *
 * @param enc
 * @returns {string}
 */
function uncompressString(enc) {

    enc = enc.replace(/\./g, '+').replace(/_/g, '/').replace(/-/g, '=');

    const compressedString = atob(enc);
    const compressedBytes = [];
    for (let i = 0; i < compressedString.length; i++) {
        compressedBytes.push(compressedString.charCodeAt(i));
    }
    //const bytes = new Zlib.RawInflate(compressedBytes).decompress();
    const bytes = inflateRaw_1$2(compressedBytes);

    let str = '';
    for (let b of bytes) {
        str += String.fromCharCode(b);
    }
    return str;
}


/**
 * @param dataURI
 * @returns {Array<number>|Uint8Array}
 */
function decodeDataURI$1(dataURI, gzip) {

    const split = dataURI.split(',');
    const info = split[0].split(':')[1];
    let dataString = split[1];

    if (info.indexOf('base64') >= 0) {
        dataString = atob(dataString);

        const bytes = new Uint8Array(dataString.length);
        for (let i = 0; i < dataString.length; i++) {
            bytes[i] = dataString.charCodeAt(i);
        }

        let plain;
        if (gzip || info.indexOf('gzip') > 0) {
            plain = ungzip_1$1(bytes);
        } else {
            plain = bytes;
        }
        return plain
    } else {
        return decodeURIComponent(dataString);      // URL encoded string -- not currently used or tested
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
const IGVMath = {

    lerp: (v0, v1, t) => {
        return (1 - t) * v0 + t * v1;
    },

    mean: function (array) {

        var t = 0, n = 0,
            i;
        for (i = 0; i < array.length; i++) {
            if (!isNaN(array[i])) {
                t += array[i];
                n++;
            }
        }
        return n > 0 ? t / n : 0;
    },

    meanAndStdev: function (array) {

        var v, t = 0, t2 = 0, n = 0, i;

        for (i = 0; i < array.length; i++) {

            v = array[i];

            if (!isNaN(v)) {
                t += v;
                t2 += v * v;
                n++;
            }
        }
        return n > 0 ? {mean: t / n, stdev: Math.sqrt(t2 - t * t / n)} : {mean: 0, stdev: 0};
    },

    median: function (numbers) {
        // median of [3, 5, 4, 4, 1, 1, 2, 3] = 3
        var median = 0, numsLen = numbers.length;
        numbers.sort();

        if (
            numsLen % 2 === 0 // is even
        ) {
            // average of two middle numbers
            median = (numbers[numsLen / 2 - 1] + numbers[numsLen / 2]) / 2;
        } else { // is odd
            // middle number only
            median = numbers[(numsLen - 1) / 2];
        }

        return median;
    },

    // Fast percentile function for "p" near edges.  This needs profiled for p in middle (e.g. median)
    percentile: function (array, p) {

        if (array.length === 0) return undefined;

        var k = Math.floor(array.length * ((100 - p) / 100));
        if (k === 0) {
            array.sort(function (a, b) {
                return b - a
            });
            return array[k];
        } else {
            return selectElement(array, k);
        }

    },


    clamp: function (value, min, max) {
        return Math.min(Math.max(value, min), max);
    },

    log2: function (x) {
        return Math.log(x) / Math.LN2;
    }

};

function selectElement(array, k) {

    // Credit Steve Hanov http://stevehanov.ca/blog/index.php?id=122
    var heap = new BinaryHeap(),
        i;

    for (i = 0; i < array.length; i++) {

        var item = array[i];

        // If we have not yet found k items, or the current item is larger than
        // the smallest item on the heap, add current item
        if (heap.content.length < k || item > heap.content[0]) {
            // If the heap is full, remove the smallest element on the heap.
            if (heap.content.length === k) {
                heap.pop();
            }
            heap.push(item);
        }
    }

    return heap.content[0];
}


function BinaryHeap() {
    this.content = [];
}

BinaryHeap.prototype = {
    push: function (element) {
        // Add the new element to the end of the array.
        this.content.push(element);
        // Allow it to bubble up.
        this.bubbleUp(this.content.length - 1);
    },

    pop: function () {
        // Store the first element so we can return it later.
        var result = this.content[0];
        // Get the element at the end of the array.
        var end = this.content.pop();
        // If there are any elements left, put the end element at the
        // start, and let it sink down.
        if (this.content.length > 0) {
            this.content[0] = end;
            this.sinkDown(0);
        }
        return result;
    },

    remove: function (node) {
        var length = this.content.length;
        // To remove a value, we must search through the array to find
        // it.
        for (var i = 0; i < length; i++) {
            if (this.content[i] !== node) continue;
            // When it is found, the process seen in 'pop' is repeated
            // to fill up the hole.
            var end = this.content.pop();
            // If the element we popped was the one we needed to remove,
            // we're done.
            if (i === length - 1) break;
            // Otherwise, we replace the removed element with the popped
            // one, and allow it to float up or sink down as appropriate.
            this.content[i] = end;
            this.bubbleUp(i);
            this.sinkDown(i);
            break;
        }
    },

    size: function () {
        return this.content.length;
    },

    bubbleUp: function (n) {
        // Fetch the element that has to be moved.
        var element = this.content[n], score = element;
        // When at 0, an element can not go up any further.
        while (n > 0) {
            // Compute the parent element's index, and fetch it.
            var parentN = Math.floor((n + 1) / 2) - 1,
                parent = this.content[parentN];
            // If the parent has a lesser score, things are in order and we
            // are done.
            if (score >= parent)
                break;

            // Otherwise, swap the parent with the current element and
            // continue.
            this.content[parentN] = element;
            this.content[n] = parent;
            n = parentN;
        }
    },

    sinkDown: function (n) {
        // Look up the target element and its score.
        var length = this.content.length,
            element = this.content[n],
            elemScore = element;

        while (true) {
            // Compute the indices of the child elements.
            var child2N = (n + 1) * 2, child1N = child2N - 1;
            // This is used to store the new position of the element,
            // if any.
            var swap = null;
            // If the first child exists (is inside the array)...
            if (child1N < length) {
                // Look it up and compute its score.
                var child1 = this.content[child1N],
                    child1Score = child1;
                // If the score is less than our element's, we need to swap.
                if (child1Score < elemScore)
                    swap = child1N;
            }
            // Do the same checks for the other child.
            if (child2N < length) {
                var child2 = this.content[child2N],
                    child2Score = child2;
                if (child2Score < (swap == null ? elemScore : child1Score))
                    swap = child2N;
            }

            // No need to swap further, we are done.
            if (swap == null) break;

            // Otherwise, swap and continue.
            this.content[n] = this.content[swap];
            this.content[swap] = element;
            n = swap;
        }
    }
};

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

function _random(min, max) {
    return Math.random() * (max - min) + min
}

const IGVColor = {

    rgbListFromHSV: () => {

        let s = 1;
        let accumulation = [];
        for (let v = 1; v >= 0.5; v -= .1) {
            for (let h = 0; h < 1; h += 1 / 28) {
                const r = "rgb(" + IGVColor.hsvToRgb(h, s, v).join(",") + ")";
                accumulation.push(r);
            }
        }

        // add black
        accumulation.pop();
        accumulation.push(IGVColor.rgbColor(16, 16, 16));

        return accumulation
    },

    rgbToHex: function (rgb) {
        rgb = rgb.match(/^rgba?[\s+]?\([\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?/i);
        return (rgb && rgb.length === 4) ? "#" +
            ("0" + parseInt(rgb[1], 10).toString(16)).slice(-2) +
            ("0" + parseInt(rgb[2], 10).toString(16)).slice(-2) +
            ("0" + parseInt(rgb[3], 10).toString(16)).slice(-2) : ''
    },

    hexToRgb: function (hex) {

        var cooked = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);

        if (null === cooked) {
            return undefined
        }

        return "rgb(" + parseInt(cooked[1], 16) + "," + parseInt(cooked[2], 16) + "," + parseInt(cooked[3], 16) + ")"
    },

    /**
     * Converts an HSV color value to RGB. Conversion formula
     * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
     * Assumes h, s, and v are contained in the set [0, 1] and
     * returns r, g, and b in the set [0, 255].
     *
     * Credit: https://gist.githubusercontent.com/mjackson/5311256
     *
     * @param   h       The hue
     * @param   s       The saturation
     * @param   v       The value
     * @return  Array   The RGB representation
     */
    hsvToRgb: function (h, s, v) {
        var r, g, b;

        var i = Math.floor(h * 6);
        var f = h * 6 - i;
        var p = v * (1 - s);
        var q = v * (1 - f * s);
        var t = v * (1 - (1 - f) * s);

        switch (i % 6) {
            case 0:
                r = v, g = t, b = p;
                break
            case 1:
                r = q, g = v, b = p;
                break
            case 2:
                r = p, g = v, b = t;
                break
            case 3:
                r = p, g = q, b = v;
                break
            case 4:
                r = t, g = p, b = v;
                break
            case 5:
                r = v, g = p, b = q;
                break
        }

        return [Math.floor(r * 255), Math.floor(g * 255), Math.floor(b * 255)]
    },

    /**
     * Converts an HSL color value to RGB. Conversion formula
     * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
     * Assumes h, s, and l are contained in the set [0, 1] and
     * returns r, g, and b in the set [0, 255].
     *
     * Credit: https://gist.githubusercontent.com/mjackson/5311256
     *
     * @param   h       The hue
     * @param   s       The saturation
     * @param   l       The lightness
     * @return  Array   The RGB representation
     */
    hslToRgb: function (h, s, l) {
        var r, g, b;

        if (s === 0) {
            r = g = b = l; // achromatic
        } else {


            var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
            var p = 2 * l - q;

            r = IGVColor.hue2rgb(p, q, h + 1 / 3);
            g = IGVColor.hue2rgb(p, q, h);
            b = IGVColor.hue2rgb(p, q, h - 1 / 3);
        }

        return [r * 255, g * 255, b * 255]
    },

    hue2rgb: (p, q, t) => {
        if (t < 0) t += 1;
        if (t > 1) t -= 1;
        if (t < 1 / 6) return p + (q - p) * 6 * t
        if (t < 1 / 2) return q
        if (t < 2 / 3) return p + (q - p) * (2 / 3 - t) * 6
        return p
    },

    rgbaColor: function (r, g, b, a) {

        r = IGVMath.clamp(r, 0, 255);
        g = IGVMath.clamp(g, 0, 255);
        b = IGVMath.clamp(b, 0, 255);
        a = IGVMath.clamp(a, 0.0, 1.0);

        return "rgba(" + r + "," + g + "," + b + "," + a + ")"
    },

    rgbColor: function (r, g, b) {

        r = IGVMath.clamp(r, 0, 255);
        g = IGVMath.clamp(g, 0, 255);
        b = IGVMath.clamp(b, 0, 255);

        return "rgb(" + r + "," + g + "," + b + ")"
    },

    greyScale: function (value) {

        var grey = IGVMath.clamp(value, 0, 255);

        return "rgb(" + grey + "," + grey + "," + grey + ")"
    },

    randomGrey: function (min, max) {

        min = IGVMath.clamp(min, 0, 255);
        max = IGVMath.clamp(max, 0, 255);

        var g = Math.round(_random(min, max)).toString(10);

        return "rgb(" + g + "," + g + "," + g + ")"
    },

    randomRGB: function (min, max) {

        min = IGVMath.clamp(min, 0, 255);
        max = IGVMath.clamp(max, 0, 255);

        var r = Math.round(_random(min, max)).toString(10);
        var g = Math.round(_random(min, max)).toString(10);
        var b = Math.round(_random(min, max)).toString(10);

        return "rgb(" + r + "," + g + "," + b + ")"
    },

    randomRGBConstantAlpha: function (min, max, alpha) {

        min = IGVMath.clamp(min, 0, 255);
        max = IGVMath.clamp(max, 0, 255);

        var r = Math.round(_random(min, max)).toString(10);
        var g = Math.round(_random(min, max)).toString(10);
        var b = Math.round(_random(min, max)).toString(10);

        return "rgba(" + r + "," + g + "," + b + "," + alpha + ")"
    },

    addAlpha: function (color, alpha) {

        if (color === "0" || color === ".") {
            color = "rgb(0,0,0)";
        } else {
            const c = this.colorNameToHex(color);
            if (c) {
                color = c;
            }
        }

        var isHex = /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(color);

        if (color.startsWith("rgba")) {
            const idx = color.lastIndexOf(",");
            return color.substring(0, idx + 1) + alpha.toString() + ")"
        }

        if (isHex) {
            color = IGVColor.hexToRgb(color);
        }

        if (color.startsWith("rgb")) {
            return color.replace("rgb", "rgba").replace(")", ", " + alpha + ")")
        } else {
            return color
        }
    },

    rgbComponents: function (color) {

        if (color === "0" || color === ".") {
            return [0, 0, 0]
        }
        const isHex = /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(color);
        if (isHex) {
            color = IGVColor.hexToRgb(color);
        } else {
            if (!color.startsWith("rgb")) {
                const hex = this.colorNameToHex(color);
                color = this.hexToRgb(hex);
            }
        }

        if (color.startsWith("rgb(")) {
            return color.substring(4, color.length - 1).split(",").map(s => Number.parseInt(s.trim()))
        } else if (color.startsWith("rgba(")) {
            return color.substring(5, color.length - 1).split(",").map((s, i) => {
                s = s.trim();
                return i === 3 ? Number.parseFloat(s) : Number.parseInt(s)
            })
        } else {
            throw Error("Unrecognized color string: color")
        }
    },

    /**
     *
     * @param dest  RGB components as an array
     * @param src  RGB components as an array
     * @param alpha   alpha transparancy in the range 0-1
     * @returns {}
     */
    getCompositeColor: function (dest, src, alpha) {

        var r = Math.floor(alpha * src[0] + (1 - alpha) * dest[0]),
            g = Math.floor(alpha * src[1] + (1 - alpha) * dest[1]),
            b = Math.floor(alpha * src[2] + (1 - alpha) * dest[2]);

        return "rgb(" + r + "," + g + "," + b + ")"

    },

    /**
     * Return a color represented by the string.  If the string is not a recognized color format return the
     * string itself.  This function exists for backward compatibility,  createColorStringSafe is preferredz
     * @param str
     */
    createColorString: function (str) {
        // Excel will quote color strings, strip all quotes
        const color = this.createColorStringSafe(str);
        return color ? color : str
    },

    /**
     * If str is a recognized color format return a string encoding the color
     * @param str
     */
    createColorStringSafe: function (str) {
        // Excel will quote color strings, strip all quotes
        str = stripQuotes$2(str);

        if (str.startsWith('rgb(') && str.endsWith(')')) {
            return str
        }
        if (str.startsWith('#') && str.length < 8) {
            return str
        }

        // See if string is a recognized web color
        const hex = this.colorNameToHex(str);
        if (hex) {
            return hex
        }

        // UCSC format (e.g.  0,0,255)
        const tokens = str.split(",");
        if (tokens.length === 3 && isColorComponent(tokens[0]) && isColorComponent(tokens[1]) && isColorComponent(tokens[2])) {
            return `rgb(${str})`
        }
    },

    darkenLighten: function (color, amt) {

        let src;
        let hexColor = this.colorNameToHex(color);
        if (hexColor) {
            src = IGVColor.hexToRgb(hexColor);
        } else {
            src = color.startsWith('rgb(') ? color : IGVColor.hexToRgb(color);
        }

        const components = src.replace(")", "").substring(4).split(",");

        const r = Math.max(0, Math.min(255, Number.parseInt(components[0].trim()) + amt));
        const g = Math.max(0, Math.min(255, Number.parseInt(components[1].trim()) + amt));
        const b = Math.max(0, Math.min(255, Number.parseInt(components[2].trim()) + amt));

        return 'rgb(' + r.toString() + ',' + g.toString() + ',' + b.toString() + ')'

    },

    /**
     * Convert html/css color name to hex value.  Adapted from https://gist.github.com/mxfh/4719348
     * @param colorName
     * @returns {*}
     */
    colorNameToHex: function (colorName) { // color list from http://stackoverflow.com/q/1573053/731179  with added gray/gray
        const definedColorNames = {
            "aliceblue": "#f0f8ff",
            "antiquewhite": "#faebd7",
            "aqua": "#00ffff",
            "aquamarine": "#7fffd4",
            "azure": "#f0ffff",
            "beige": "#f5f5dc",
            "bisque": "#ffe4c4",
            "black": "#000000",
            "blanchedalmond": "#ffebcd",
            "blue": "#0000ff",
            "blueviolet": "#8a2be2",
            "brown": "#a52a2a",
            "burlywood": "#deb887",
            "cadetblue": "#5f9ea0",
            "chartreuse": "#7fff00",
            "chocolate": "#d2691e",
            "coral": "#ff7f50",
            "cornflowerblue": "#6495ed",
            "cornsilk": "#fff8dc",
            "crimson": "#dc143c",
            "cyan": "#00ffff",
            "darkblue": "#00008b",
            "darkcyan": "#008b8b",
            "darkgoldenrod": "#b8860b",
            "darkgray": "#a9a9a9",
            "darkgreen": "#006400",
            "darkkhaki": "#bdb76b",
            "darkmagenta": "#8b008b",
            "darkolivegreen": "#556b2f",
            "darkorange": "#ff8c00",
            "darkorchid": "#9932cc",
            "darkred": "#8b0000",
            "darksalmon": "#e9967a",
            "darkseagreen": "#8fbc8f",
            "darkslateblue": "#483d8b",
            "darkslategray": "#2f4f4f",
            "darkturquoise": "#00ced1",
            "darkviolet": "#9400d3",
            "deeppink": "#ff1493",
            "deepskyblue": "#00bfff",
            "dimgray": "#696969",
            "dodgerblue": "#1e90ff",
            "firebrick": "#b22222",
            "floralwhite": "#fffaf0",
            "forestgreen": "#228b22",
            "fuchsia": "#ff00ff",
            "gainsboro": "#dcdcdc",
            "ghostwhite": "#f8f8ff",
            "gold": "#ffd700",
            "goldenrod": "#daa520",
            "gray": "#808080",
            "green": "#008000",
            "greenyellow": "#adff2f",
            "honeydew": "#f0fff0",
            "hotpink": "#ff69b4",
            "indianred ": "#cd5c5c",
            "indigo ": "#4b0082",
            "ivory": "#fffff0",
            "khaki": "#f0e68c",
            "lavender": "#e6e6fa",
            "lavenderblush": "#fff0f5",
            "lawngreen": "#7cfc00",
            "lemonchiffon": "#fffacd",
            "lightblue": "#add8e6",
            "lightcoral": "#f08080",
            "lightcyan": "#e0ffff",
            "lightgoldenrodyellow": "#fafad2",
            "lightgrey": "#d3d3d3",
            "lightgreen": "#90ee90",
            "lightpink": "#ffb6c1",
            "lightsalmon": "#ffa07a",
            "lightseagreen": "#20b2aa",
            "lightskyblue": "#87cefa",
            "lightslategray": "#778899",
            "lightsteelblue": "#b0c4de",
            "lightyellow": "#ffffe0",
            "lime": "#00ff00",
            "limegreen": "#32cd32",
            "linen": "#faf0e6",
            "magenta": "#ff00ff",
            "maroon": "#800000",
            "mediumaquamarine": "#66cdaa",
            "mediumblue": "#0000cd",
            "mediumorchid": "#ba55d3",
            "mediumpurple": "#9370d8",
            "mediumseagreen": "#3cb371",
            "mediumslateblue": "#7b68ee",
            "mediumspringgreen": "#00fa9a",
            "mediumturquoise": "#48d1cc",
            "mediumvioletred": "#c71585",
            "midnightblue": "#191970",
            "mintcream": "#f5fffa",
            "mistyrose": "#ffe4e1",
            "moccasin": "#ffe4b5",
            "navajowhite": "#ffdead",
            "navy": "#000080",
            "oldlace": "#fdf5e6",
            "olive": "#808000",
            "olivedrab": "#6b8e23",
            "orange": "#ffa500",
            "orangered": "#ff4500",
            "orchid": "#da70d6",
            "palegoldenrod": "#eee8aa",
            "palegreen": "#98fb98",
            "paleturquoise": "#afeeee",
            "palevioletred": "#d87093",
            "papayawhip": "#ffefd5",
            "peachpuff": "#ffdab9",
            "peru": "#cd853f",
            "pink": "#ffc0cb",
            "plum": "#dda0dd",
            "powderblue": "#b0e0e6",
            "purple": "#800080",
            "red": "#ff0000",
            "rosybrown": "#bc8f8f",
            "royalblue": "#4169e1",
            "saddlebrown": "#8b4513",
            "salmon": "#fa8072",
            "sandybrown": "#f4a460",
            "seagreen": "#2e8b57",
            "seashell": "#fff5ee",
            "sienna": "#a0522d",
            "silver": "#c0c0c0",
            "skyblue": "#87ceeb",
            "slateblue": "#6a5acd",
            "slategray": "#708090",
            "snow": "#fffafa",
            "springgreen": "#00ff7f",
            "steelblue": "#4682b4",
            "tan": "#d2b48c",
            "teal": "#008080",
            "thistle": "#d8bfd8",
            "tomato": "#ff6347",
            "turquoise": "#40e0d0",
            "violet": "#ee82ee",
            "wheat": "#f5deb3",
            "white": "#ffffff",
            "whitesmoke": "#f5f5f5",
            "yellow": "#ffff00",
            "yellowgreen": "#9acd32",
            "darkgrey": "#a9a9a9",
            "darkslategrey": "#2f4f4f",
            "dimgrey": "#696969",
            "grey": "#808080",
            "lightgray": "#d3d3d3",
            "lightslategrey": "#778899",
            "slategrey": "#708090"
        };
        return definedColorNames[colorName]
    }
};

function isColorComponent(str) {
    const num = Number.parseInt(str);
    return !Number.isNaN(num) && num >= 0 && num <= 255
}

// Support for oauth token based authorization
// This class supports explicit setting of an oauth token either globally or for specific hosts.
//
// The variable oauth.google.access_token, which becomes igv.oauth.google.access_token on ES5 conversion is
// supported for backward compatibility

const DEFAULT_HOST$1 = "googleapis";

class Oauth$1 {

    constructor() {
        this.oauthTokens = {};
    }


    setToken(token, host) {
        host = host || DEFAULT_HOST$1;
        this.oauthTokens[host] = token;
    }

    getToken(host) {
        host = host || DEFAULT_HOST$1;
        let token;
        for (let key of Object.keys(this.oauthTokens)) {
            const regex = wildcardToRegExp$1(key);
            if (regex.test(host)) {
                token = this.oauthTokens[key];
                break
            }
        }
        return token
    }

    removeToken(host) {
        host = host || DEFAULT_HOST$1;
        for (let key of Object.keys(this.oauthTokens)) {
            const regex = wildcardToRegExp$1(key);
            if (regex.test(host)) {
                this.oauthTokens[key] = undefined;
            }
        }
    }

    // Special object for google -- legacy support
    // google: {
    //     setToken: function (token) {
    //         oauth.setToken(token);
    //     }
    // }
}


/**
 * Creates a RegExp from the given string, converting asterisks to .* expressions,
 * and escaping all other characters.
 *
 * credit https://gist.github.com/donmccurdy/6d073ce2c6f3951312dfa45da14a420f
 */
function wildcardToRegExp$1(s) {
    return new RegExp('^' + s.split(/\*+/).map(regExpEscape$1).join('.*') + '$')
}

/**
 * RegExp-escapes all characters in the given string.
 *
 * credit https://gist.github.com/donmccurdy/6d073ce2c6f3951312dfa45da14a420f
 */
function regExpEscape$1(s) {
    return s.replace(/[|\\{}()[\]^$+*?.]/g, '\\$&')
}

function isGoogleURL$1(url) {
    return (url.includes("googleapis") && !url.includes("urlshortener")) ||
        isGoogleStorageURL$1(url) ||
        isGoogleDriveURL$1(url)
}

function isGoogleStorageURL$1(url) {
    return url.startsWith("gs://") ||
        url.startsWith("https://www.googleapis.com/storage") ||
        url.startsWith("https://storage.cloud.google.com") ||
        url.startsWith("https://storage.googleapis.com");
}

function isGoogleDriveURL$1(url) {
    return url.indexOf("drive.google.com") >= 0 || url.indexOf("www.googleapis.com/drive") > 0
}

/**
 * Translate gs:// urls to https
 * See https://cloud.google.com/storage/docs/json_api/v1
 * @param gsUrl
 * @returns {string|*}
 */
function translateGoogleCloudURL$1(gsUrl) {

    let {bucket, object} = parseBucketName$1(gsUrl);
    object = encode$1(object);

    const qIdx = gsUrl.indexOf('?');
    const paramString = (qIdx > 0) ? gsUrl.substring(qIdx) + "&alt=media" : "?alt=media";

    return `https://storage.googleapis.com/storage/v1/b/${bucket}/o/${object}${paramString}`
}

/**
 * Parse a google bucket and object name from a google storage URL.  Known forms include
 *
 * gs://BUCKET_NAME/OBJECT_NAME
 * https://storage.googleapis.com/BUCKET_NAME/OBJECT_NAME
 * https://storage.googleapis.com/storage/v1/b/BUCKET_NAME/o/OBJECT_NAME
 * https://www.googleapis.com/storage/v1/b/BUCKET_NAME/o/OBJECT_NAME"
 * https://storage.googleapis.com/download/storage/v1/b/BUCKET_NAME/o/OBJECT_NAME
 *
 * @param url
 */
function parseBucketName$1(url) {

    let bucket;
    let object;

    if (url.startsWith("gs://")) {
        const i = url.indexOf('/', 5);
        if (i >= 0) {
            bucket = url.substring(5, i);
            const qIdx = url.indexOf('?');
            object = (qIdx < 0) ? url.substring(i + 1) : url.substring(i + 1, qIdx);
        }

    } else if (url.startsWith("https://storage.googleapis.com") || url.startsWith("https://storage.cloud.google.com")) {
        const bucketIdx = url.indexOf("/v1/b/", 8);
        if (bucketIdx > 0) {
            const objIdx = url.indexOf("/o/", bucketIdx);
            if (objIdx > 0) {
                const queryIdx = url.indexOf("?", objIdx);
                bucket = url.substring(bucketIdx + 6, objIdx);
                object = queryIdx > 0 ? url.substring(objIdx + 3, queryIdx) : url.substring(objIdx + 3);
            }

        } else {
            const idx1 = url.indexOf("/", 8);
            const idx2 = url.indexOf("/", idx1+1);
            const idx3 = url.indexOf("?", idx2);
            if (idx2 > 0) {
                bucket = url.substring(idx1+1, idx2);
                object = idx3 < 0 ? url.substring(idx2+1) : url.substring(idx2+1, idx3);
            }
        }

    } else if (url.startsWith("https://www.googleapis.com/storage/v1/b")) {
        const bucketIdx = url.indexOf("/v1/b/", 8);
        const objIdx = url.indexOf("/o/", bucketIdx);
        if (objIdx > 0) {
            const queryIdx = url.indexOf("?", objIdx);
            bucket = url.substring(bucketIdx + 6, objIdx);
            object = queryIdx > 0 ? url.substring(objIdx + 3, queryIdx) : url.substring(objIdx + 3);
        }
    }

    if (bucket && object) {
        return {
            bucket, object
        }
    } else {
        throw Error(`Unrecognized Google Storage URI: ${url}`)
    }

}

/**
 * Percent a GCS object name.  See https://cloud.google.com/storage/docs/request-endpoints
 * Specific characters to encode:
 *   !, #, $, &, ', (, ), *, +, ,, /, :, ;, =, ?, @, [, ], and space characters.
 * @param obj
 */

function encode$1(objectName) {

    let result = '';
    objectName.split('').forEach(function(letter) {
        if(encodings$1$1.has(letter)) {
            result += encodings$1$1.get(letter);
        } else {
            result += letter;
        }
    });
    return result;
}

//	%23	%24	%25	%26	%27	%28	%29	%2A	%2B	%2C	%2F	%3A	%3B	%3D	%3F	%40	%5B	%5D
const encodings$1$1 = new Map();
encodings$1$1.set("!", "%21");
encodings$1$1.set("#", "%23");
encodings$1$1.set("$", "%24");
encodings$1$1.set("%", "%25");
encodings$1$1.set("&", "%26");
encodings$1$1.set("'", "%27");
encodings$1$1.set("(", "%28");
encodings$1$1.set(")", "%29");
encodings$1$1.set("*", "%2A");
encodings$1$1.set("+", "%2B");
encodings$1$1.set(",", "%2C");
encodings$1$1.set("/", "%2F");
encodings$1$1.set(":", "%3A");
encodings$1$1.set(";", "%3B");
encodings$1$1.set("=", "%3D");
encodings$1$1.set("?", "%3F");
encodings$1$1.set("@", "%40");
encodings$1$1.set("[", "%5B");
encodings$1$1.set("]", "%5D");
encodings$1$1.set(" ", "%20");

// Convenience functions for the gapi oAuth library.

async function init$2(config) {

    if (!(google.accounts.oauth2.initTokenClient)) {
        throw new Error("Google accounts token client not loaded (https://accounts.google.com/gsi/client)")
    }

    if (isInitialized$2()) {
        throw new Error("Google client is already initialized")
    }

    // Note: callback is added when accessToken is requested
    const codeClientConfig = {
        client_id: config.client_id || config.clientId,
        scope: config.scope || 'https://www.googleapis.com/auth/userinfo.profile',
        state: config.state || 'igv',
        error: (err) => {
            throw new Error(err.type)
        },
        hint: config.hint,     // Optional
        hosted_domain: config.hosted_domain  // Optional
    };

    const tokenClient = google.accounts.oauth2.initTokenClient(codeClientConfig);

    // Attach an object to keep igv state
    google.igv = {
        tokenClient: tokenClient,
        apiKey: config.apiKey
    };
}

function isInitialized$2() {
    return window.google && window.google.igv
}

/**
 * Return the current access token if the user is signed in, or undefined otherwise.  This function does not
 * attempt a signIn or request any specfic scopes.
 *
 * @returns access_token || undefined
 */
function getCurrentAccessToken$2() {
    return (isInitialized$2() && google.igv.tokenResponse && Date.now() < google.igv.tokenExpiresAt) ?
        google.igv.tokenResponse.access_token :
        undefined
}


let promise$1;
/**
 * Return a promise for an access token for the given scope.  If the user hasn't authorized the scope request it
 *
 * @param scope
 * @returns {Promise<unknown>}
 */
async function getAccessToken$2(scope) {

    if (!isInitialized$2()) {
        throw Error("Google oAuth has not been initialized")
    }

    if (google.igv.tokenResponse &&
        Date.now() < google.igv.tokenExpiresAt &&
        google.accounts.oauth2.hasGrantedAllScopes(google.igv.tokenResponse, scope)) {
        return google.igv.tokenResponse.access_token
    } else {
        const tokenClient = google.igv.tokenClient;
        if(!promise$1) {
            promise$1 = new Promise((resolve, reject) => {
                try {
                    // Settle this promise in the response callback for requestAccessToken()
                    tokenClient.callback = (tokenResponse) => {
                        if (tokenResponse.error !== undefined) {
                            reject(tokenResponse);
                        }
                        google.igv.tokenResponse = tokenResponse;
                        google.igv.tokenExpiresAt = Date.now() + tokenResponse.expires_in * 1000;
                        resolve(tokenResponse.access_token);
                    };
                    tokenClient.requestAccessToken({scope});
                } catch (err) {
                }
            });
        }
        return promise$1
    }
}

// gapi.auth2.getAuthInstance().isSignedIn.listen(status => {
//     const user = gapi.auth2.getAuthInstance().currentUser.get()
//     queryGoogleAuthenticationStatus(user, status)
// })

function getScopeForURL$1(url) {
    if (isGoogleDriveURL$1(url)) {
        return "https://www.googleapis.com/auth/drive.file"
    } else if (isGoogleStorageURL$1(url)) {
        return "https://www.googleapis.com/auth/devstorage.read_only"
    } else {
        return 'https://www.googleapis.com/auth/userinfo.profile'
    }
}

function getApiKey$1() {
    return google.igv.apiKey
}

/**
 * Return information about a specific google drive URL
 *
 * @param googleDriveURL
 * @returns {Promise<any>}
 */
async function getDriveFileInfo$1(googleDriveURL) {

    const id = getGoogleDriveFileID$1(googleDriveURL);
    let endPoint = "https://www.googleapis.com/drive/v3/files/" + id + "?supportsTeamDrives=true";
    const apiKey = getApiKey$1();
    if (apiKey) {
        endPoint += "&key=" + apiKey;
    }
    const response = await fetch(endPoint);
    let json = await response.json();
    if (json.error && json.error.code === 404) {
        let scope = "https://www.googleapis.com/auth/drive.readonly";
        const access_token = await getAccessToken$2(scope);
        if (access_token) {
            const response = await fetch(endPoint, {
                headers: {
                    'Authorization': `Bearer ${access_token}`
                }
            });
            json = await response.json();
            if (json.error) {
                throw Error(json.error);
            }
        } else {
            throw Error(json.error);
        }
    }
    return json;
}


function getDriveDownloadURL$1(link) {
    // Return a google drive download url for the sharable link
    //https://drive.google.com/open?id=0B-lleX9c2pZFbDJ4VVRxakJzVGM
    //https://drive.google.com/file/d/1_FC4kCeO8E3V4dJ1yIW7A0sn1yURKIX-/view?usp=sharing
    var id = getGoogleDriveFileID$1(link);
    return id ? "https://www.googleapis.com/drive/v3/files/" + id + "?alt=media&supportsTeamDrives=true" : link;
}

function getGoogleDriveFileID$1(link) {

    //https://drive.google.com/file/d/1_FC4kCeO8E3V4dJ1yIW7A0sn1yURKIX-/view?usp=sharing
    //https://www.googleapis.com/drive/v3/files/1w-tvo6p1SH4p1OaQSVxpkV_EJgGIstWF?alt=media&supportsTeamDrives=true"

    if (link.includes("/open?id=")) {
        const i1 = link.indexOf("/open?id=") + 9;
        const i2 = link.indexOf("&");
        if (i1 > 0 && i2 > i1) {
            return link.substring(i1, i2)
        } else if (i1 > 0) {
            return link.substring(i1);
        }

    } else if (link.includes("/file/d/")) {
        const i1 = link.indexOf("/file/d/") + 8;
        const i2 = link.lastIndexOf("/");
        return link.substring(i1, i2);

    } else if (link.startsWith("https://www.googleapis.com/drive")) {
        let i1 = link.indexOf("/files/");
        const i2 = link.indexOf("?");
        if (i1 > 0) {
            i1 += 7;
            return i2 > 0 ?
                link.substring(i1, i2) :
                link.substring(i1)
        }
    }

    throw Error("Unknown Google Drive url format: " + link);


}

// The MIT License (MIT)

/**
 * @constructor
 * @param {Object} options A set op options to pass to the throttle function
 *        @param {number} requestsPerSecond The amount of requests per second
 *                                          the library will limit to
 */
class Throttle$1 {
    constructor(options) {
        this.requestsPerSecond = options.requestsPerSecond || 10;
        this.lastStartTime = 0;
        this.queued = [];
    }

    /**
     * Adds a promise
     * @param {Function} async function to be executed
     * @param {Object} options A set of options.
     * @return {Promise} A promise
     */
    add(asyncFunction, options) {

        var self = this;
        return new Promise(function (resolve, reject) {
            self.queued.push({
                resolve: resolve,
                reject: reject,
                asyncFunction: asyncFunction,
            });
            self.dequeue();
        });
    }

    /**
     * Adds all the promises passed as parameters
     * @param {Function} promises An array of functions that return a promise
     * @param {Object} options A set of options.
     * @param {number} options.signal An AbortSignal object that can be used to abort the returned promise
     * @param {number} options.weight A "weight" of each operation resolving by array of promises
     * @return {Promise} A promise that succeeds when all the promises passed as options do
     */
    addAll(promises, options) {
        var addedPromises = promises.map(function (promise) {
            return this.add(promise, options);
        }.bind(this));

        return Promise.all(addedPromises);
    };

    /**
     * Dequeues a promise
     * @return {void}
     */
    dequeue() {
        if (this.queued.length > 0) {
            var now = new Date(),
                inc = (1000 / this.requestsPerSecond) + 1,
                elapsed = now - this.lastStartTime;

            if (elapsed >= inc) {
                this._execute();
            } else {
                // we have reached the limit, schedule a dequeue operation
                setTimeout(function () {
                    this.dequeue();
                }.bind(this), inc - elapsed);
            }
        }
    }

    /**
     * Executes the promise
     * @private
     * @return {void}
     */
    async _execute() {
        this.lastStartTime = new Date();
        var candidate = this.queued.shift();
        const f = candidate.asyncFunction;
        try {
            const r = await f();
            candidate.resolve(r);
        } catch (e) {
            candidate.reject(e);
        }

    }


}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class IGVXhr$1 {

    constructor() {
        this.apiKey = undefined;
        this.googleThrottle = new Throttle$1({
            requestsPerSecond: 8
        });
        this.RANGE_WARNING_GIVEN = false;
        this.oauth = new Oauth$1();
    }

    setApiKey(key) {
        this.apiKey = key;
    }

    async loadArrayBuffer(url, options) {
        options = options || {};
        if (!options.responseType) {
            options.responseType = "arraybuffer";
        }
        if (isFile$1(url)) {
            return this._loadFileSlice(url, options)
        } else {
            return this.load(url, options)
        }
    }

    /**
     * A wrapper around loadArrayBuffer that inflates gzipped data
     * @param url
     * @param options
     * @returns {Promise<Uint8Array>}
     */
     async loadByteArray(url, options) {
        const arraybuffer = await this.loadArrayBuffer(url, options);
        let plain;
        if (isgzipped$1(arraybuffer)) {
            plain = ungzip_blocks$1(arraybuffer);
        } else {
            plain = new Uint8Array(arraybuffer);
        }
        return plain
    }


    async loadJson(url, options) {
        options = options || {};
        const method = options.method || (options.sendData ? "POST" : "GET");
        if (method === "POST") {
            options.contentType = "application/json";
        }
        const result = await this.loadString(url, options);
        if (result) {
            return JSON.parse(result)
        } else {
            return result
        }
    }

    async loadString(path, options) {
        options = options || {};
        if (path instanceof File) {
            return this._loadStringFromFile(path, options)
        } else {
            return this._loadStringFromUrl(path, options)
        }
    }

    async load(url, options) {

        options = options || {};
        const urlType = typeof url;

        // Resolve functions, promises, and functions that return promises
        url = await (typeof url === 'function' ? url() : url);

        if (isFile$1(url)) {
            return this._loadFileSlice(url, options)
        } else if (typeof url.startsWith === 'function') {   // Test for string
            if (url.startsWith("data:")) {
                const buffer = decodeDataURI$1(url).buffer;
                if (options.range) {
                    const rangeEnd = options.range.size ? options.range.start + options.range.size : buffer.byteLength;
                    return buffer.slice(options.range.start, rangeEnd)
                } else {
                    return buffer
                }
            } else {
                if (url.startsWith("https://drive.google.com")) {
                    url = getDriveDownloadURL$1(url);
                }
                if (isGoogleDriveURL$1(url) || url.startsWith("https://www.dropbox.com")) {
                    return this.googleThrottle.add(async () => {
                        return this._loadURL(url, options)
                    })
                } else {
                    return this._loadURL(url, options)
                }
            }
        } else {
            throw Error(`url must be either a 'File', 'string', 'function', or 'Promise'.  Actual type: ${urlType}`)
        }
    }

    async _loadURL(url, options) {

        const self = this;
        const _url = url;   // The unmodified URL, needed in case of an oAuth retry

        url = mapUrl$1(url);

        options = options || {};

        let oauthToken = options.oauthToken || this.getOauthToken(url);
        if (oauthToken) {
            oauthToken = await (typeof oauthToken === 'function' ? oauthToken() : oauthToken);
        }

        return new Promise(function (resolve, reject) {

            // Various Google tansformations
            if (isGoogleURL$1(url) && !isGoogleStorageSigned$1(url)) {
                if (isGoogleStorageURL$1(url)) {
                    url = translateGoogleCloudURL$1(url);
                }
                url = addApiKey$1(url);

                if (isGoogleDriveURL$1(url)) {
                    addTeamDrive$1(url);
                }

                // If we have an access token try it, but don't force a signIn or request for scopes yet
                if (!oauthToken) {
                    oauthToken = getCurrentGoogleAccessToken$1();
                }
            }

            const headers = options.headers || {};
            if (oauthToken) {
                addOauthHeaders$1(headers, oauthToken);
            }
            const range = options.range;


            const xhr = new XMLHttpRequest();
            const sendData = options.sendData || options.body;
            const method = options.method || (sendData ? "POST" : "GET");
            const responseType = options.responseType;
            const contentType = options.contentType;
            const mimeType = options.mimeType;

            xhr.open(method, url);

            if (options.timeout) {
                xhr.timeout = options.timeout;
            }

            if (range) {
                let rangeEnd = "";
                if (range.size) {
                    rangeEnd = range.start + range.size - 1;
                }
                xhr.setRequestHeader("Range", "bytes=" + range.start + "-" + rangeEnd);
                //      xhr.setRequestHeader("Cache-Control", "no-cache");    <= This can cause CORS issues, disabled for now
            }
            if (contentType) {
                xhr.setRequestHeader("Content-Type", contentType);
            }
            if (mimeType) {
                xhr.overrideMimeType(mimeType);
            }
            if (responseType) {
                xhr.responseType = responseType;
            }
            if (headers) {
                for (let key of Object.keys(headers)) {
                    const value = headers[key];
                    xhr.setRequestHeader(key, value);
                }
            }

            // NOTE: using withCredentials with servers that return "*" for access-allowed-origin will fail
            if (options.withCredentials === true) {
                xhr.withCredentials = true;
            }

            xhr.onload = async function (event) {

                // when the url points to a local file, the status is 0
                if (xhr.status === 0 || (xhr.status >= 200 && xhr.status <= 300)) {
                    if ("HEAD" === options.method) {
                        // Support fetching specific headers.  Attempting to fetch all headers can be problematic with CORS
                        const headers = options.requestedHeaders || ['content-length'];
                        const headerMap = {};
                        for (let h of headers) {
                            headerMap[h] = xhr.getResponseHeader(h);
                        }
                        resolve(headerMap);
                    } else {
                        // Assume "GET" or "POST"
                        if (range && xhr.status !== 206 && range.start !== 0) {

                            // For small files a range starting at 0 can return the whole file => 200
                            // Provide just the slice we asked for, throw out the rest quietly
                            // If file is large warn user
                            if (xhr.response.length > 100000 && !self.RANGE_WARNING_GIVEN) {
                                alert(`Warning: Range header ignored for URL: ${url}.  This can have severe performance impacts.`);
                            }
                            resolve(xhr.response.slice(range.start, range.start + range.size));
                        } else {
                            resolve(xhr.response);
                        }
                    }
                } else if (xhr.status === 416) {
                    handleError(Error(`416 Unsatisfiable Range`));
                } else if (isInitialized$2() &&
                    ((xhr.status === 404 || xhr.status === 401 || xhr.status === 403) &&
                        isGoogleURL$1(url)) &&
                    !options.retries) {
                    tryGoogleAuth();

                } else {
                    if (xhr.status === 403) {
                        handleError("Access forbidden: " + url);
                    } else {
                        handleError(xhr.status);
                    }
                }
            };


            xhr.onerror = function (event) {
                if (isGoogleURL$1(url) && !options.retries) {
                    tryGoogleAuth();
                } else {
                    handleError("Error accessing resource: " + url + " Status: " + xhr.status);
                }
            };

            xhr.ontimeout = function (event) {
                handleError("Timed out");
            };

            xhr.onabort = function (event) {
                reject(event);
            };

            try {
                xhr.send(sendData);
            } catch (e) {
                if (isGoogleURL$1(url) && !options.retries) {
                    tryGoogleAuth();
                } else {
                    handleError(e);
                }
            }


            function handleError(error) {
                if (reject) {
                    reject(error);
                } else {
                    throw error
                }
            }

            async function tryGoogleAuth() {
                try {
                    const accessToken = await fetchGoogleAccessToken$1(_url);
                    options.retries = 1;
                    options.oauthToken = accessToken;
                    const response = await self.load(_url, options);
                    resolve(response);
                } catch (e) {
                    if (e.error) {
                        const msg = e.error.startsWith("popup_blocked") ?
                            "Google login popup blocked by browser." :
                            e.error;
                        alert(msg);
                    } else {
                        handleError(e);
                    }
                }
            }
        })

    }

    async _loadFileSlice(localfile, options) {

        let blob = (options && options.range) ?
            localfile.slice(options.range.start, options.range.start + options.range.size) :
            localfile;

        const arrayBuffer = await blob.arrayBuffer();

        if ("arraybuffer" === options.responseType) {
            return arrayBuffer
        } else {
            return arrayBufferToString$1(arrayBuffer)
        }
    }

    async _loadStringFromFile(localfile, options) {

        const blob = options.range ? localfile.slice(options.range.start, options.range.start + options.range.size) : localfile;
        const arrayBuffer = await blob.arrayBuffer();
        return arrayBufferToString$1(arrayBuffer)
    }

    async _loadStringFromUrl(url, options) {

        options = options || {};
        options.responseType = "arraybuffer";
        const data = await this.load(url, options);
        return arrayBufferToString$1(data)
    }

    /**
     * Explicity set an oAuth token for use with given host.  If host is undefined token is used for google api access*
     * @param token
     * @param host
     */
    setOauthToken(token, host) {
        this.oauth.setToken(token, host);
    }

    /**
     * Return an oauth token for the URL if we have one.  This method does not force sign-in, and the token may
     * or may not be valid.  Sign-in is triggered on request failure.
     * *
     * @param url
     * @returns {*}
     */
    getOauthToken(url) {

        // Google is the default provider, don't try to parse host for google URLs
        const host = isGoogleURL$1(url) ?
            undefined :
            parseUri$1(url).host;

        // First check the explicit settings (i.e. token set through the API)
        let token = this.oauth.getToken(host);
        if (token) {
            return token
        } else if (host === undefined) {
            // Now try Google oauth tokens previously obtained.  This will return undefined if google oauth is not
            // configured.
            const googleToken = getCurrentGoogleAccessToken$1();
            if (googleToken && googleToken.expires_at > Date.now()) {
                return googleToken.access_token
            }
        }
    }

    /**
     * This method should only be called when it is known the server supports HEAD requests.  It is used to recover
     * from 416 errors from out-of-spec WRT range request servers.  Notably Globus.
     * * *
     * @param url
     * @param options
     * @returns {Promise<unknown>}
     */
    async getContentLength(url, options) {
        options = options || {};
        options.method = 'HEAD';
        options.requestedHeaders = ['content-length'];
        const headerMap = await this._loadURL(url, options);
        const contentLengthString = headerMap['content-length'];
        return contentLengthString ? Number.parseInt(contentLengthString) : 0
    }

}

function isGoogleStorageSigned$1(url) {
    return url.indexOf("X-Goog-Signature") > -1
}


/**
 * Return a Google oAuth token, triggering a sign in if required.   This method should not be called until we know
 * a token is required, that is until we've tried the url and received a 401, 403, or 404.
 *
 * @param url
 * @returns the oauth token
 */
async function fetchGoogleAccessToken$1(url) {
    if (isInitialized$2()) {
        const scope = getScopeForURL$1(url);
        const access_token = await getAccessToken$2(scope);
        return access_token
    } else {
        throw Error(
            `Authorization is required, but Google oAuth has not been initalized. Contact your site administrator for assistance.`)
    }
}

/**
 * Return the current google access token, if one exists.  Do not triger signOn or request additional scopes.
 * @returns {undefined|access_token}
 */
function getCurrentGoogleAccessToken$1() {
    if (isInitialized$2()) {
        const access_token = getCurrentAccessToken$2();
        return access_token
    } else {
        return undefined
    }
}

function addOauthHeaders$1(headers, acToken) {
    if (acToken) {
        headers["Cache-Control"] = "no-cache";
        headers["Authorization"] = "Bearer " + acToken;
    }
    return headers
}


function addApiKey$1(url) {
    let apiKey = igvxhr$1.apiKey;
    if (!apiKey && typeof gapi !== "undefined") {
        apiKey = gapi.apiKey;
    }
    if (apiKey !== undefined && !url.includes("key=")) {
        const paramSeparator = url.includes("?") ? "&" : "?";
        url = url + paramSeparator + "key=" + apiKey;
    }
    return url
}

function addTeamDrive$1(url) {
    if (url.includes("supportsTeamDrive")) {
        return url
    } else {
        const paramSeparator = url.includes("?") ? "&" : "?";
        url = url + paramSeparator + "supportsTeamDrive=true";
    }
}

/**
 * Perform some well-known url mappings.
 * @param url
 */
function mapUrl$1(url) {

    if (url.startsWith("https://www.dropbox.com")) {
        return url.replace("//www.dropbox.com", "//dl.dropboxusercontent.com")
    } else if (url.startsWith("https://drive.google.com")) {
        return getDriveDownloadURL$1(url)
    } else if (url.includes("//www.broadinstitute.org/igvdata")) {
        return url.replace("//www.broadinstitute.org/igvdata", "//data.broadinstitute.org/igvdata")
    } else if (url.includes("//igvdata.broadinstitute.org")) {
        return url.replace("//igvdata.broadinstitute.org", "//s3.amazonaws.com/igv.broadinstitute.org")
    } else if (url.includes("//igv.genepattern.org")) {
        return url.replace("//igv.genepattern.org", "//igv-genepattern-org.s3.amazonaws.com")
    } else if (url.startsWith("ftp://ftp.ncbi.nlm.nih.gov/geo")) {
        return url.replace("ftp://", "https://")
    } else {
        return url
    }
}


function arrayBufferToString$1(arraybuffer) {

    let plain;
    if (isgzipped$1(arraybuffer)) {
        plain = ungzip_blocks$1(arraybuffer);
    } else {
        plain = new Uint8Array(arraybuffer);
    }

    if ('TextDecoder' in getGlobalObject$1()) {
        return new TextDecoder().decode(plain)
    } else {
        return decodeUTF8$1(plain)
    }
}

/**
 * Use when TextDecoder is not available (primarily IE).
 *
 * From: https://gist.github.com/Yaffle/5458286
 *
 * @param octets
 * @returns {string}
 */
function decodeUTF8$1(octets) {
    var string = "";
    var i = 0;
    while (i < octets.length) {
        var octet = octets[i];
        var bytesNeeded = 0;
        var codePoint = 0;
        if (octet <= 0x7F) {
            bytesNeeded = 0;
            codePoint = octet & 0xFF;
        } else if (octet <= 0xDF) {
            bytesNeeded = 1;
            codePoint = octet & 0x1F;
        } else if (octet <= 0xEF) {
            bytesNeeded = 2;
            codePoint = octet & 0x0F;
        } else if (octet <= 0xF4) {
            bytesNeeded = 3;
            codePoint = octet & 0x07;
        }
        if (octets.length - i - bytesNeeded > 0) {
            var k = 0;
            while (k < bytesNeeded) {
                octet = octets[i + k + 1];
                codePoint = (codePoint << 6) | (octet & 0x3F);
                k += 1;
            }
        } else {
            codePoint = 0xFFFD;
            bytesNeeded = octets.length - i;
        }
        string += String.fromCodePoint(codePoint);
        i += bytesNeeded + 1;
    }
    return string
}


function getGlobalObject$1() {
    if (typeof self !== 'undefined') {
        return self
    }
    if (typeof global !== 'undefined') {
        return global
    } else {
        return window
    }
}


const igvxhr$1 = new IGVXhr$1();

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/** An implementation of an interval tree, following the explanation.
 * from CLR.
 *
 * Public interface:
 *   Constructor  IntervalTree
 *   Insertion    insert
 *   Search       findOverlapping
 */

var BLACK$1 = 1;
var RED = 2;

var NIL$1 = {};
NIL$1.color = BLACK$1;
NIL$1.parent = NIL$1;
NIL$1.left = NIL$1;
NIL$1.right = NIL$1;


class IntervalTree {

    constructor() {
        this.root = NIL$1;
    }

    insert(start, end, value) {

        var interval = new Interval(start, end, value);
        var x = new Node$1(interval);
        this.treeInsert(x);
        x.color = RED;
        while (x !== this.root && x.parent.color === RED) {
            if (x.parent === x.parent.parent.left) {
                let y = x.parent.parent.right;
                if (y.color === RED) {
                    x.parent.color = BLACK$1;
                    y.color = BLACK$1;
                    x.parent.parent.color = RED;
                    x = x.parent.parent;
                } else {
                    if (x === x.parent.right) {
                        x = x.parent;
                        leftRotate.call(this, x);
                    }
                    x.parent.color = BLACK$1;
                    x.parent.parent.color = RED;
                    rightRotate.call(this, x.parent.parent);
                }
            } else {
                let y = x.parent.parent.left;
                if (y.color === RED) {
                    x.parent.color = BLACK$1;
                    y.color = BLACK$1;
                    x.parent.parent.color = RED;
                    x = x.parent.parent;
                } else {
                    if (x === x.parent.left) {
                        x = x.parent;
                        rightRotate.call(this, x);
                    }
                    x.parent.color = BLACK$1;
                    x.parent.parent.color = RED;
                    leftRotate.call(this, x.parent.parent);
                }
            }
        }
        this.root.color = BLACK$1;
    }

    /**
     *
     * @param start - query interval
     * @param end - query interval
     * @returns Array of all intervals overlapping the query region
     */
    findOverlapping(start, end) {


        var searchInterval = new Interval(start, end, 0);

        if (this.root === NIL$1) return [];

        var intervals = searchAll.call(this, searchInterval, this.root, []);

        if (intervals.length > 1) {
            intervals.sort(function (i1, i2) {
                return i1.low - i2.low;
            });
        }

        return intervals;
    }

    /**
     * Dump info on intervals to console.  For debugging.
     */
    logIntervals() {

        logNode(this.root);

        function logNode(node, indent) {

            if (node.left !== NIL$1) logNode(node.left);
            if (node.right !== NIL$1) logNode(node.right);
        }

    }

    mapIntervals(func) {

        applyInterval(this.root);

        function applyInterval(node) {

            func(node.interval);

            if (node.left !== NIL$1) applyInterval(node.left);
            if (node.right !== NIL$1) applyInterval(node.right);
        }
    }


    /**
     * Note:  Does not maintain RB constraints,  this is done post insert
     *
     * @param x  a Node
     */
    treeInsert(x) {
        var node = this.root;
        var y = NIL$1;
        while (node !== NIL$1) {
            y = node;
            if (x.interval.low <= node.interval.low) {
                node = node.left;
            } else {
                node = node.right;
            }
        }
        x.parent = y;

        if (y === NIL$1) {
            this.root = x;
            x.left = x.right = NIL$1;
        } else {
            if (x.interval.low <= y.interval.low) {
                y.left = x;
            } else {
                y.right = x;
            }
        }

        applyUpdate.call(this, x);
    }
}

function searchAll(interval, node, results) {

    if (node.interval.overlaps(interval)) {
        results.push(node.interval);
    }

    if (node.left !== NIL$1 && node.left.max >= interval.low) {
        searchAll.call(this, interval, node.left, results);
    }

    if (node.right !== NIL$1 && node.right.min <= interval.high) {
        searchAll.call(this, interval, node.right, results);
    }

    return results;
}

function leftRotate(x) {
    var y = x.right;
    x.right = y.left;
    if (y.left !== NIL$1) {
        y.left.parent = x;
    }
    y.parent = x.parent;
    if (x.parent === NIL$1) {
        this.root = y;
    } else {
        if (x.parent.left === x) {
            x.parent.left = y;
        } else {
            x.parent.right = y;
        }
    }
    y.left = x;
    x.parent = y;

    applyUpdate.call(this, x);
    // no need to apply update on y, since it'll y is an ancestor
    // of x, and will be touched by applyUpdate().
}


function rightRotate(x) {
    var y = x.left;
    x.left = y.right;
    if (y.right !== NIL$1) {
        y.right.parent = x;
    }
    y.parent = x.parent;
    if (x.parent === NIL$1) {
        this.root = y;
    } else {
        if (x.parent.right === x) {
            x.parent.right = y;
        } else {
            x.parent.left = y;
        }
    }
    y.right = x;
    x.parent = y;


    applyUpdate.call(this, x);
    // no need to apply update on y, since it'll y is an ancestor
    // of x, and will be touched by applyUpdate().
}


// Applies the statistic update on the node and its ancestors.
function applyUpdate(node) {
    while (node !== NIL$1) {
        var nodeMax = node.left.max > node.right.max ? node.left.max : node.right.max;
        var intervalHigh = node.interval.high;
        node.max = nodeMax > intervalHigh ? nodeMax : intervalHigh;

        var nodeMin = node.left.min < node.right.min ? node.left.min : node.right.min;
        var intervalLow = node.interval.low;
        node.min = nodeMin < intervalLow ? nodeMin : intervalLow;

        node = node.parent;
    }
}


class Interval {
    constructor(low, high, value) {
        this.low = low;
        this.high = high;
        this.value = value;
    }

    equals(other) {
        if (!other) {
            return false;
        }
        if (this === other) {
            return true;
        }
        return (this.low === other.low &&
            this.high === other.high);

    }

    compareTo(other) {
        if (this.low < other.low)
            return -1;
        if (this.low > other.low)
            return 1;

        if (this.high < other.high)
            return -1;
        if (this.high > other.high)
            return 1;

        return 0;
    }

    /**
     * Returns true if this interval overlaps the other.
     */
    overlaps(other) {
            return (this.low <= other.high && other.low <= this.high);
    }
}

function Node$1(interval) {
    this.parent = NIL$1;
    this.left = NIL$1;
    this.right = NIL$1;
    this.interval = interval;
    this.color = RED;
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * Object for caching lists of features.  Supports effecient queries for sub-range  (chr, start, end)
 *
 * @param featureList
 * @param The genomic range spanned by featureList (optional)
 * @constructor
 */

class FeatureCache$1 {

    constructor(featureList, genome, range) {

        featureList = featureList || [];
        this.treeMap = this.buildTreeMap(featureList, genome);
        this.range = range;
        this.count = featureList.length;
    }

    containsRange(genomicRange) {
        // No range means cache contains all features
        return (this.range === undefined || this.range.contains(genomicRange.chr, genomicRange.start, genomicRange.end));
    }

    /**
     * Search loaded features
     * @param fn
     */
    findFeatures(fn) {
        const found = [];
        for(let featureList of Object.values(this.allFeatures)) {
            for(let f of featureList) {
                if(fn(f)) {
                    found.push(f);
                }
            }
        }
        return found
    }

    queryFeatures(chr, start, end) {

        const tree = this.treeMap[chr];

        if (!tree) return [];

        const intervals = tree.findOverlapping(start, end);

        if (intervals.length === 0) {
            return [];
        } else {
            // Trim the list of features in the intervals to those
            // overlapping the requested range.
            // Assumption: features are sorted by start position

            const featureList = [];
            const all = this.allFeatures[chr];
            if (all) {
                for (let interval of intervals) {
                    const indexRange = interval.value;
                    for (let i = indexRange.start; i < indexRange.end; i++) {
                        let feature = all[i];
                        if (feature.start > end) break;
                        else if (feature.end >= start) {
                            featureList.push(feature);
                        }
                    }
                }
                featureList.sort(function (a, b) {
                    return a.start - b.start;
                });
            }
            return featureList;
        }
    };

    /**
     * Returns all features, unsorted.
     *
     * @returns {Array}
     */
    getAllFeatures() {
        return this.allFeatures;
    }

    buildTreeMap(featureList, genome) {

        const treeMap = {};
        const chromosomes = [];
        this.allFeatures = {};

        if (featureList) {
            for (let feature of featureList) {

                let chr = feature.chr;
                // Translate to "official" name
                if (genome) {
                    chr = genome.getChromosomeName(chr);
                }

                let geneList = this.allFeatures[chr];
                if (!geneList) {
                    chromosomes.push(chr);
                    geneList = [];
                    this.allFeatures[chr] = geneList;
                }
                geneList.push(feature);
            }


            // Now build interval tree for each chromosome
            for (let chr of chromosomes) {
                const chrFeatures = this.allFeatures[chr];
                chrFeatures.sort(function (f1, f2) {
                    return (f1.start === f2.start ? 0 : (f1.start > f2.start ? 1 : -1));
                });
                treeMap[chr] = buildIntervalTree$1(chrFeatures);
            }
        }

        return treeMap;
    }
}

/**
 * Build an interval tree from the feature list for fast interval based queries.   We lump features in groups
 * of 10, or total size / 100,   to reduce size of the tree.
 *
 * @param featureList
 */
function buildIntervalTree$1(featureList) {

    const tree = new IntervalTree();
    const len = featureList.length;
    const chunkSize = Math.max(10, Math.round(len / 10));

    for (let i = 0; i < len; i += chunkSize) {
        const e = Math.min(len, i + chunkSize);
        const subArray = new IndexRange(i, e); //featureList.slice(i, e);
        const iStart = featureList[i].start;
        //
        let iEnd = iStart;
        for (let j = i; j < e; j++) {
            iEnd = Math.max(iEnd, featureList[j].end);
        }
        tree.insert(iStart, iEnd, subArray);
    }

    return tree;
}


class IndexRange {
    constructor(start, end) {
        this.start = start;
        this.end = end;
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const FeatureUtils = {

    packFeatures: function (features, maxRows, sorted) {

        var start;
        var end;

        if (!features) return;

        maxRows = maxRows || 10000;

        if (!sorted) {
            features.sort(function (a, b) {
                return a.start - b.start;
            });
        }


        if (features.length === 0) {
            return [];

        } else {

            var bucketList = [],
                allocatedCount = 0,
                lastAllocatedCount = 0,
                nextStart,
                row,
                index,
                bucket,
                feature,
                gap = 2,
                bucketStart;

            start = features[0].start;
            end = features[features.length - 1].start;

            bucketStart = Math.max(start, features[0].start);
            nextStart = bucketStart;

            features.forEach(function (alignment) {

                var buckListIndex = Math.max(0, alignment.start - bucketStart);
                if (bucketList[buckListIndex] === undefined) {
                    bucketList[buckListIndex] = [];
                }
                bucketList[buckListIndex].push(alignment);
            });


            row = 0;


            while (allocatedCount < features.length && row <= maxRows) {


                while (nextStart <= end) {

                    bucket = undefined;

                    while (!bucket && nextStart <= end) {

                        index = nextStart - bucketStart;
                        if (bucketList[index] === undefined) {
                            ++nextStart;                     // No buckets at this index
                        } else {
                            bucket = bucketList[index];
                        }

                    } // while (bucket)

                    if (!bucket) {
                        break;
                    }
                    feature = bucket.pop();
                    if (0 === bucket.length) {
                        bucketList[index] = undefined;
                    }

                    feature.row = row;

                    nextStart = feature.end + gap;
                    ++allocatedCount;

                } // while (nextStart)

                row++;
                nextStart = bucketStart;

                if (allocatedCount === lastAllocatedCount) break;   // Protect from infinite loops

                lastAllocatedCount = allocatedCount;

            } // while (allocatedCount)

        }
    },


    /**
     * Find features overlapping the given interval.  It is assumed that all features share the same chromosome.
     *
     * TODO -- significant overlap with FeatureCache, refactor to combine
     *
     * @param featureList
     * @param start
     * @param end
     */
    findOverlapping: function (featureList, start, end) {

        if (!featureList || featureList.length === 0) {
            return [];
        } else {
            const tree = buildIntervalTree(featureList);
            const intervals = tree.findOverlapping(start, end);

            if (intervals.length === 0) {
                return [];
            } else {
                // Trim the list of features in the intervals to those
                // overlapping the requested range.
                // Assumption: features are sorted by start position

                featureList = [];

                intervals.forEach(function (interval) {
                    const intervalFeatures = interval.value;
                    const len = intervalFeatures.length;
                    for (let i = 0; i < len; i++) {
                        const feature = intervalFeatures[i];
                        if (feature.start > end) break;
                        else if (feature.end > start) {
                            featureList.push(feature);
                        }
                    }
                });

                featureList.sort(function (a, b) {
                    return a.start - b.start;
                });

                return featureList;
            }
        }

    }
};


/**
 * Build an interval tree from the feature list for fast interval based queries.   We lump features in groups
 * of 10, or total size / 100,   to reduce size of the tree.
 *
 * @param featureList
 */
function buildIntervalTree(featureList) {

    const tree = new IntervalTree();
    const len = featureList.length;
    const chunkSize = Math.max(10, Math.round(len / 100));

    featureList.sort(function (f1, f2) {
        return (f1.start === f2.start ? 0 : (f1.start > f2.start ? 1 : -1));
    });

    for (let i = 0; i < len; i += chunkSize) {
        const e = Math.min(len, i + chunkSize);
        const subArray = featureList.slice(i, e);
        const iStart = subArray[0].start;
        let iEnd = iStart;
        subArray.forEach(function (feature) {
            iEnd = Math.max(iEnd, feature.end);
        });
        tree.insert(iStart, iEnd, subArray);
    }

    return tree;
}

function hexToRGB(hex) {
    // Ensure the hex value is in the proper format
    hex = hex.replace(/^#/, '');

    // If it's a shorthand hex color (like #f06), double each character
    if (hex.length === 3) {
        hex = hex.split('').map(char => char + char).join('');
    }

    if (hex.length !== 6) {
        throw new Error('Invalid HEX color.');
    }

    // Parse the r, g, b values
    let bigint = parseInt(hex, 16);
    let r = (bigint >> 16) & 255;
    let g = (bigint >> 8) & 255;
    let b = bigint & 255;

    return `rgb(${r}, ${g}, ${b})`;
}

const genericColorPickerPalette =
    {
        licorice: "#000000",
        steel: "#6e6e6e",
        magnesium: "#b8b8b8",
        mercury: "#e8e8e8",
        cayenne: "#891100",
        mocha: "#894800",
        aspargus: "#888501",
        fern: "#458401",
        teal: "#008688",
        salmon: "#ff726e",
        tangerine: "#ff8802",
        cantaloupe: "#ffce6e",
        lemon: "#fffa03",
        lime: "#83f902",
        honeydew: "#cefa6e",
        ice: "#68fdff",
        aqua: "#008cff",
        blueberry: "#002eff",
        midnight: "#001888",
        grape: "#8931ff",
        lavender: "#d278ff",
        orchid: "#6e76ff",
        strawberry: "#ff2987",
        magenta: "#ff39ff",
        carnation: "#ff7fd3"
    };

const appleCrayonRGBPalette =
    {
        cantaloupe: {r: 255, g: 206, b: 110},
        honeydew: {r: 206, g: 250, b: 110},
        spindrift: {r: 104, g: 251, b: 208},
        sky: {r: 106, g: 207, b: 255},
        lavender: {r: 210, g: 120, b: 255},
        carnation: {r: 255, g: 127, b: 211},
        licorice: {r: 0, g: 0, b: 0},
        snow: {r: 255, g: 255, b: 255},
        salmon: {r: 255, g: 114, b: 110},
        banana: {r: 255, g: 251, b: 109},
        flora: {r: 104, g: 249, b: 110},
        ice: {r: 104, g: 253, b: 255},
        orchid: {r: 110, g: 118, b: 255},
        bubblegum: {r: 255, g: 122, b: 255},
        lead: {r: 30, g: 30, b: 30},
        mercury: {r: 232, g: 232, b: 232},
        tangerine: {r: 255, g: 136, b: 2},
        lime: {r: 131, g: 249, b: 2},
        sea_foam: {r: 3, g: 249, b: 135},
        aqua: {r: 0, g: 140, b: 255},
        grape: {r: 137, g: 49, b: 255},
        strawberry: {r: 255, g: 41, b: 135},
        tungsten: {r: 58, g: 58, b: 58},
        silver: {r: 208, g: 208, b: 208},
        maraschino: {r: 255, g: 33, b: 1},
        lemon: {r: 255, g: 250, b: 3},
        spring: {r: 5, g: 248, b: 2},
        turquoise: {r: 0, g: 253, b: 255},
        blueberry: {r: 0, g: 46, b: 255},
        magenta: {r: 255, g: 57, b: 255},
        iron: {r: 84, g: 84, b: 83},
        magnesium: {r: 184, g: 184, b: 184},
        mocha: {r: 137, g: 72, b: 0},
        fern: {r: 69, g: 132, b: 1},
        moss: {r: 1, g: 132, b: 72},
        ocean: {r: 0, g: 74, b: 136},
        eggplant: {r: 73, g: 26, b: 136},
        maroon: {r: 137, g: 22, b: 72},
        steel: {r: 110, g: 110, b: 110},
        aluminum: {r: 160, g: 159, b: 160},
        cayenne: {r: 137, g: 17, b: 0},
        aspargus: {r: 136, g: 133, b: 1},
        clover: {r: 2, g: 132, b: 1},
        teal: {r: 0, g: 134, b: 136},
        midnight: {r: 0, g: 24, b: 136},
        plum: {r: 137, g: 30, b: 136},
        tin: {r: 135, g: 134, b: 135},
        nickel: {r: 136, g: 135, b: 135}
    };

function appleCrayonRGB(name) {
    const {r, g, b} = appleCrayonRGBPalette[name];
    return `rgb(${r},${g},${b})`
}

function appleCrayonRGBA(name, alpha) {
    const {r, g, b} = appleCrayonRGBPalette[name];
    return `rgba(${r},${g},${b},${alpha})`
}

const colorPalettes = {

    Set1:
        [
            "rgb(228,26,28)",
            "rgb(55,126,184)",
            "rgb(77,175,74)",
            "rgb(166,86,40)",
            "rgb(152,78,163)",
            "rgb(255,127,0)",
            "rgb(247,129,191)",
            "rgb(153,153,153)",
            "rgb(255,255,51)"
        ],

    Dark2:
        [
            "rgb(27,158,119)",
            "rgb(217,95,2)",
            "rgb(117,112,179)",
            "rgb(231,41,138)",
            "rgb(102,166,30)",
            "rgb(230,171,2)",
            "rgb(166,118,29)",
            "rgb(102,102,102)"
        ],

    Set2:
        [
            "rgb(102, 194,165)",
            "rgb(252,141,98)",
            "rgb(141,160,203)",
            "rgb(231,138,195)",
            "rgb(166,216,84)",
            "rgb(255,217,47)",
            "rgb(229,196,148)",
            "rgb(179,179,179)"
        ],

    Set3:
        [
            "rgb(141,211,199)",
            "rgb(255,255,179)",
            "rgb(190,186,218)",
            "rgb(251,128,114)",
            "rgb(128,177,211)",
            "rgb(253,180,98)",
            "rgb(179,222,105)",
            "rgb(252,205,229)",
            "rgb(217,217,217)",
            "rgb(188,128,189)",
            "rgb(204,235,197)",
            "rgb(255,237,111)"
        ],

    Pastel1:
        [
            "rgb(251,180,174)",
            "rgb(179,205,227)",
            "rgb(204,235,197)",
            "rgb(222,203,228)",
            "rgb(254,217,166)",
            "rgb(255,255,204)",
            "rgb(229,216,189)",
            "rgb(253,218,236)"
        ],

    Pastel2:
        [
            "rgb(173,226,207)",
            "rgb(253,205,172)",
            "rgb(203,213,232)",
            "rgb(244,202,228)",
            "rgb(230,245,201)",
            "rgb(255,242,174)",
            "rgb(243,225,206)"
        ],

    Accent:
        [
            "rgb(127,201,127)",
            "rgb(190,174,212)",
            "rgb(253,192,134)",
            "rgb(255,255,153)",
            "rgb(56,108,176)",
            "rgb(240,2,127)",
            "rgb(191,91,23)"
        ]
};

class PaletteColorTable {

    constructor(palette) {
        this.colors = colorPalettes[palette];
        if (!Array.isArray(this.colors)) this.colors = [];
        this.colorTable = new Map();
        this.nextIdx = 0;
        this.colorGenerator = new RandomColorGenerator();
    }

    getColor(key) {
        if (!this.colorTable.has(key)) {
            if (this.nextIdx < this.colors.length) {
                this.colorTable.set(key, this.colors[this.nextIdx]);
            } else {
                this.colorTable.set(key, this.colorGenerator.get());
            }
            this.nextIdx++;
        }
        return this.colorTable.get(key)
    }
}

class ColorTable {
    constructor(colors) {
        this.colorTable = colors || {};
        this.nextIdx = 0;
        this.colorGenerator = new RandomColorGenerator();
    }

    getColor(key) {
        if (!this.colorTable.hasOwnProperty(key)) {
            if (this.colorTable.hasOwnProperty("*")) {
                return this.colorTable["*"]
            }
            this.colorTable[key] = this.colorGenerator.get();
        }
        return this.colorTable[key]
    }
}

// Random color generator from https://github.com/sterlingwes/RandomColor/blob/master/rcolor.js
// Free to use & distribute under the MIT license
// Wes Johnson (@SterlingWes)
//
// inspired by http://martin.ankerl.com/2009/12/09/how-to-create-random-colors-programmatically/
function RandomColorGenerator() {
    this.hue = Math.random();
    this.goldenRatio = 0.618033988749895;
    this.hexwidth = 2;
}

RandomColorGenerator.prototype.hsvToRgb = function (h, s, v) {
    var h_i = Math.floor(h * 6),
        f = h * 6 - h_i,
        p = v * (1 - s),
        q = v * (1 - f * s),
        t = v * (1 - (1 - f) * s),
        r = 255,
        g = 255,
        b = 255;
    switch (h_i) {
        case 0:
            r = v, g = t, b = p;
            break
        case 1:
            r = q, g = v, b = p;
            break
        case 2:
            r = p, g = v, b = t;
            break
        case 3:
            r = p, g = q, b = v;
            break
        case 4:
            r = t, g = p, b = v;
            break
        case 5:
            r = v, g = p, b = q;
            break
    }
    return [Math.floor(r * 256), Math.floor(g * 256), Math.floor(b * 256)]
};

RandomColorGenerator.prototype.padHex = function (str) {
    if (str.length > this.hexwidth) return str
    return new Array(this.hexwidth - str.length + 1).join('0') + str
};

RandomColorGenerator.prototype.get = function (saturation, value) {
    this.hue += this.goldenRatio;
    this.hue %= 1;
    if (typeof saturation !== "number") saturation = 0.5;
    if (typeof value !== "number") value = 0.95;
    var rgb = this.hsvToRgb(this.hue, saturation, value);

    return "#" + this.padHex(rgb[0].toString(16))
        + this.padHex(rgb[1].toString(16))
        + this.padHex(rgb[2].toString(16))

};

new RandomColorGenerator();

function rgbaColor(r, g, b, a) {
    r = IGVMath.clamp(r, 0, 255);
    g = IGVMath.clamp(g, 0, 255);
    b = IGVMath.clamp(b, 0, 255);
    a = IGVMath.clamp(a, 0.0, 1.0);
    return `rgba(${r}, ${g}, ${b}, ${a})`
}

function rgbColor(r, g, b) {
    r = IGVMath.clamp(r, 0, 255);
    g = IGVMath.clamp(g, 0, 255);
    b = IGVMath.clamp(b, 0, 255);
    return `rgb(${r}, ${g}, ${b})`
}

function randomRGB$1(min, max) {

    min = IGVMath.clamp(min, 0, 255);
    max = IGVMath.clamp(max, 0, 255);

    const r = Math.round(Math.random() * (max - min) + min).toString(10);
    const g = Math.round(Math.random() * (max - min) + min).toString(10);
    const b = Math.round(Math.random() * (max - min) + min).toString(10);
    return `rgb(${r},${g},${b})`

}

function rgbaStringTokens(rgbaString) {

    if (rgbaString.startsWith('rgba(')) {

        const [ignore, pass0 ] = rgbaString.split('(');

        const [ rgba ] = pass0.split(')');

        return rgba.split(',').map((string, index) => index < 3 ? parseInt(string) : parseFloat(string))

    } else {
        return undefined
    }
}

function rgbStringTokens(rgbString) {

    if (rgbString.startsWith('rgb(')) {

        const [ignore, pass0 ] = rgbString.split('(');

        const [ rgb ] = pass0.split(')');

        return rgb.split(',').map(string => parseInt(string))

    } else {
        return undefined
    }
}

const fudge = 0.005;
function rgbStringHeatMapLerp(_a, _b, interpolant) {

    if (interpolant < fudge) {
        return _a
    } else if (interpolant > 1.0 - fudge) {
        return _b
    } else {
        let rA, gA, bA;
        let rB, gB, bB;
        if (interpolant < 0.5) {

            interpolant /= .5;

            [ rA, gA, bA ] = rgbStringTokens(_a);
            [ rB, gB, bB ] = rgbStringTokens(appleCrayonRGB('snow'));
        } else {

            interpolant = (interpolant - .5) / .5;

            [ rA, gA, bA ] = rgbStringTokens(appleCrayonRGB('snow'));
            [ rB, gB, bB ] = rgbStringTokens(_b);
        }

        const [ r, g, b ] =
            [
                Math.floor(IGVMath.lerp(rA, rB, interpolant)),
                Math.floor(IGVMath.lerp(gA, gB, interpolant)),
                Math.floor(IGVMath.lerp(bA, bB, interpolant))
            ];

        return rgbColor(r, g, b)

    }


}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * Test if the given value is a string or number.  Not using typeof as it fails on boxed primitives.
 *
 * @param value
 * @returns boolean
 */

function isSimpleType(value) {
    const simpleTypes = new Set(["boolean", "number", "string", "symbol"]);
    const valueType = typeof value;
    return (value !== undefined && (simpleTypes.has(valueType) || value.substring || value.toFixed))
}

function buildOptions(config, options) {

    var defaultOptions = {
        oauthToken: config.oauthToken,
        headers: config.headers,
        withCredentials: config.withCredentials,
        filename: config.filename
    };

    return Object.assign(defaultOptions, options)
}

/**
 * isMobile test from http://detectmobilebrowsers.com
 * TODO -- improve UI design so this isn't neccessary
 * @returns {boolean}
 */

// igv.isMobile = function () {
//
//     const a = (navigator.userAgent || navigator.vendor || window.opera);
//     return (/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i.test(a) ||
//         /1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i.test(a.substr(0, 4)))
//
// }

const doAutoscale = function (features) {
    var min, max;

    if (features && features.length > 0) {
        min = Number.MAX_VALUE;
        max = -Number.MAX_VALUE;

        for(let f of features) {
            if (!Number.isNaN(f.value)) {
                min = Math.min(min, f.value);
                max = Math.max(max, f.value);
            }
        }

        // Insure we have a zero baseline
        if (max > 0) min = Math.min(0, min);
        if (max < 0) max = 0;
    } else {
        // No features -- default
        min = 0;
        max = 100;
    }

    return {min: min, max: max}
};

const validateGenomicExtent = function (chromosomeLengthBP, genomicExtent, minimumBP) {

    let ss = genomicExtent.start;
    let ee = genomicExtent.end;

    if (undefined === ee) {

        ss -= minimumBP / 2;
        ee = ss + minimumBP;

        if (ee > chromosomeLengthBP) {
            ee = chromosomeLengthBP;
            ss = ee - minimumBP;
        } else if (ss < 0) {
            ss = 0;
            ee = minimumBP;
        }

    } else if (ee - ss < minimumBP) {

        const center = (ee + ss) / 2;

        if (center - minimumBP / 2 < 0) {
            ss = 0;
            ee = ss + minimumBP;
        } else if (center + minimumBP / 2 > chromosomeLengthBP) {
            ee = chromosomeLengthBP;
            ss = ee - minimumBP;
        } else {
            ss = center - minimumBP / 2;
            ee = ss + minimumBP;
        }
    }

    genomicExtent.start = Math.ceil(ss);
    genomicExtent.end = Math.floor(ee);
};

/*!
 * is-number <https://github.com/jonschlinkert/is-number>
 *
 * Copyright (c) 2014-present, Jon Schlinkert.
 * Released under the MIT License.
 */

const isNumber = function (num) {
    if (typeof num === 'number') {
        return num - num === 0
    }
    if (typeof num === 'string' && num.trim() !== '') {
        return Number.isFinite(+num)
    }
    return false
};

async function getFilename$1(url) {
    if (isString$3(url) && url.startsWith("https://drive.google.com")) {
        // This will fail if Google API key is not defined
        if (getApiKey$1() === undefined) {
            throw Error("Google drive is referenced, but API key is not defined.  An API key is required for Google Drive access")
        }
        const json = await getDriveFileInfo$1(url);
        return json.originalFileName || json.name
    } else {
        return getFilename$2(url)
    }
}

function prettyBasePairNumber(raw) {

    var denom,
        units,
        value,
        floored;

    if (raw > 1e7) {
        denom = 1e6;
        units = " mb";
    } else if (raw > 1e4) {

        denom = 1e3;
        units = " kb";

        value = raw / denom;
        floored = Math.floor(value);
        return numberFormatter$1(floored) + units
    } else {
        return numberFormatter$1(raw) + " bp"
    }

    value = raw / denom;
    floored = Math.floor(value);

    return floored.toString() + units
}


function isDataURL(obj) {
    return (isString$3(obj) && obj.startsWith("data:"))
}

function createColumn(columnContainer, className) {
    const column = div$2({class: className});
    columnContainer.appendChild(column);
}


function insertElementBefore(element, referenceNode) {
    referenceNode.parentNode.insertBefore(element, referenceNode);
}

function insertElementAfter(element, referenceNode) {
    referenceNode.parentNode.insertBefore(element, referenceNode.nextSibling);
}

/**
 * Test to see if page is loaded in a secure context, that is by https or is localhost.
 */
function isSecureContext() {
    return window.location.protocol === "https:" || window.location.hostname === "localhost"
}

function getElementVerticalDimension(element) {

    const style = window.getComputedStyle(element);

    const marginTop = parseInt(style.marginTop);
    const marginBottom = parseInt(style.marginBottom);

    const { top, bottom, height } = element.getBoundingClientRect();
    return {
        top: Math.floor(top) - marginTop,
        bottom: Math.floor(bottom) + marginBottom,
        height: Math.floor(height) + marginTop + marginBottom
    };
}

class Popover {

    constructor(parent, isDraggable, title, closeHandler) {

        this.parent = parent;

        this.popover = div$2({ class: "igv-ui-popover" });
        parent.appendChild(this.popover);

        this.popoverHeader = div$2();
        this.popover.appendChild(this.popoverHeader);

        const titleElement = div$2();
        this.popoverHeader.appendChild(titleElement);
        if (title) {
            titleElement.textContent = title;
        }

        // attach close handler
        const el = div$2();
        this.popoverHeader.appendChild(el);
        el.appendChild(createIcon$1('times'));
        el.addEventListener('click', e => {
            e.stopPropagation();
            e.preventDefault();
            closeHandler ? closeHandler() : this.dismiss();
        });

        if (true === isDraggable) {
            makeDraggable$1(this.popover, this.popoverHeader, { minX:0, minY:0 });
        }

        this.popoverContent = div$2();
        this.popover.appendChild(this.popoverContent);

        this.popover.style.display = 'none';


    }

    configure(menuItems) {

        if (0 === menuItems.length) {
            return
        }

        const menuElements = createMenuElements$1(menuItems, this.popover);

        for (const { element } of menuElements) {
            this.popoverContent.appendChild(element);
        }

    }

    present(event) {

        this.popover.style.display = 'block';

        const parent = this.popover.parentNode;
        const { x, y, width } = translateMouseCoordinates(event, parent);
        this.popover.style.top  = `${ y }px`;

        const { width: w } = this.popover.getBoundingClientRect();

        const xmax = x + w;
        const delta = xmax - width;

        this.popover.style.left = `${ xmax > width ? (x - delta) : x }px`;
        this.popoverContent.style.maxWidth = `${ Math.min(w, width) }px`;
    }

    presentContentWithEvent(e, content) {

        this.popover.style.display = 'block';

        this.popoverContent.innerHTML = content;

        present$1(e, this.popover, this.popoverContent);

    }

    presentMenu(e, menuItems) {

        if (0 === menuItems.length) {
            return
        }

        this.popover.style.display = 'block';

        const menuElements = createMenuElements$1(menuItems, this.popover);
        for (let item of menuElements) {
            this.popoverContent.appendChild(item.element);
        }

        present$1(e, this.popover, this.popoverContent);
    }

    dismiss() {
        this.popover.style.display = 'none';
    }

    hide() {
        this.popover.style.display = 'none';
        this.dispose();
    }

    dispose() {

        if (this.popover) {
            this.popover.parentNode.removeChild(this.popover);
        }

        const keys = Object.keys(this);
        for (let key of keys) {
            this[ key ] = undefined;
        }
    }

}

function present$1(e, popover, popoverContent) {

    const { x, y, width } = translateMouseCoordinates(e, popover.parentNode);
    popover.style.top  = `${ y }px`;

    const { width: w } = popover.getBoundingClientRect();

    const xmax = x + w;
    const delta = xmax - width;

    popover.style.left = `${ xmax > width ? (x - delta) : x }px`;
    popoverContent.style.maxWidth = `${ Math.min(w, width) }px`;


}

function createMenuElements$1(itemList, popover) {

    const list  = itemList.map(function (item, i) {
        let element;

        if (typeof item === 'string') {
            element = div$2();
            element.innerHTML = item;
        } else if (typeof item === 'Node') {
            element = item;
        } else {
            if (typeof item.init === 'function') {
                item.init();
            }

            if ("checkbox" === item.type) {
                element = createCheckbox$1("Show all bases", item.value);
            }

            else {
                element = div$2();
                if (typeof item.label === 'string') {
                    element.innerHTML = item.label;
                }
            }

            if (item.click && "color" !== item.type) {
                element.addEventListener('click', handleClick);
                element.addEventListener('touchend', handleClick);
                element.addEventListener('mouseup', function (e) {
                    e.preventDefault();
                    e.stopPropagation();
                });

                // eslint-disable-next-line no-inner-declarations
                function handleClick(e) {
                    item.click();
                    hide$1(popover);
                    e.preventDefault();
                    e.stopPropagation();
                }
            }
        }


        return { element, init: item.init };
    });

    return list;
}

/*! @license DOMPurify 3.2.6 | (c) Cure53 and other contributors | Released under the Apache license 2.0 and Mozilla Public License 2.0 | github.com/cure53/DOMPurify/blob/3.2.6/LICENSE */

const {
  entries,
  setPrototypeOf,
  isFrozen,
  getPrototypeOf,
  getOwnPropertyDescriptor
} = Object;
let {
  freeze,
  seal,
  create: create$2
} = Object; // eslint-disable-line import/no-mutable-exports
let {
  apply,
  construct
} = typeof Reflect !== 'undefined' && Reflect;
if (!freeze) {
  freeze = function freeze(x) {
    return x;
  };
}
if (!seal) {
  seal = function seal(x) {
    return x;
  };
}
if (!apply) {
  apply = function apply(fun, thisValue, args) {
    return fun.apply(thisValue, args);
  };
}
if (!construct) {
  construct = function construct(Func, args) {
    return new Func(...args);
  };
}
const arrayForEach = unapply(Array.prototype.forEach);
const arrayLastIndexOf = unapply(Array.prototype.lastIndexOf);
const arrayPop = unapply(Array.prototype.pop);
const arrayPush = unapply(Array.prototype.push);
const arraySplice = unapply(Array.prototype.splice);
const stringToLowerCase = unapply(String.prototype.toLowerCase);
const stringToString = unapply(String.prototype.toString);
const stringMatch = unapply(String.prototype.match);
const stringReplace = unapply(String.prototype.replace);
const stringIndexOf = unapply(String.prototype.indexOf);
const stringTrim = unapply(String.prototype.trim);
const objectHasOwnProperty = unapply(Object.prototype.hasOwnProperty);
const regExpTest = unapply(RegExp.prototype.test);
const typeErrorCreate = unconstruct(TypeError);
/**
 * Creates a new function that calls the given function with a specified thisArg and arguments.
 *
 * @param func - The function to be wrapped and called.
 * @returns A new function that calls the given function with a specified thisArg and arguments.
 */
function unapply(func) {
  return function (thisArg) {
    if (thisArg instanceof RegExp) {
      thisArg.lastIndex = 0;
    }
    for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }
    return apply(func, thisArg, args);
  };
}
/**
 * Creates a new function that constructs an instance of the given constructor function with the provided arguments.
 *
 * @param func - The constructor function to be wrapped and called.
 * @returns A new function that constructs an instance of the given constructor function with the provided arguments.
 */
function unconstruct(func) {
  return function () {
    for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }
    return construct(func, args);
  };
}
/**
 * Add properties to a lookup table
 *
 * @param set - The set to which elements will be added.
 * @param array - The array containing elements to be added to the set.
 * @param transformCaseFunc - An optional function to transform the case of each element before adding to the set.
 * @returns The modified set with added elements.
 */
function addToSet(set, array) {
  let transformCaseFunc = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : stringToLowerCase;
  if (setPrototypeOf) {
    // Make 'in' and truthy checks like Boolean(set.constructor)
    // independent of any properties defined on Object.prototype.
    // Prevent prototype setters from intercepting set as a this value.
    setPrototypeOf(set, null);
  }
  let l = array.length;
  while (l--) {
    let element = array[l];
    if (typeof element === 'string') {
      const lcElement = transformCaseFunc(element);
      if (lcElement !== element) {
        // Config presets (e.g. tags.js, attrs.js) are immutable.
        if (!isFrozen(array)) {
          array[l] = lcElement;
        }
        element = lcElement;
      }
    }
    set[element] = true;
  }
  return set;
}
/**
 * Clean up an array to harden against CSPP
 *
 * @param array - The array to be cleaned.
 * @returns The cleaned version of the array
 */
function cleanArray(array) {
  for (let index = 0; index < array.length; index++) {
    const isPropertyExist = objectHasOwnProperty(array, index);
    if (!isPropertyExist) {
      array[index] = null;
    }
  }
  return array;
}
/**
 * Shallow clone an object
 *
 * @param object - The object to be cloned.
 * @returns A new object that copies the original.
 */
function clone(object) {
  const newObject = create$2(null);
  for (const [property, value] of entries(object)) {
    const isPropertyExist = objectHasOwnProperty(object, property);
    if (isPropertyExist) {
      if (Array.isArray(value)) {
        newObject[property] = cleanArray(value);
      } else if (value && typeof value === 'object' && value.constructor === Object) {
        newObject[property] = clone(value);
      } else {
        newObject[property] = value;
      }
    }
  }
  return newObject;
}
/**
 * This method automatically checks if the prop is function or getter and behaves accordingly.
 *
 * @param object - The object to look up the getter function in its prototype chain.
 * @param prop - The property name for which to find the getter function.
 * @returns The getter function found in the prototype chain or a fallback function.
 */
function lookupGetter(object, prop) {
  while (object !== null) {
    const desc = getOwnPropertyDescriptor(object, prop);
    if (desc) {
      if (desc.get) {
        return unapply(desc.get);
      }
      if (typeof desc.value === 'function') {
        return unapply(desc.value);
      }
    }
    object = getPrototypeOf(object);
  }
  function fallbackValue() {
    return null;
  }
  return fallbackValue;
}

const html$1 = freeze(['a', 'abbr', 'acronym', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'big', 'blink', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'content', 'data', 'datalist', 'dd', 'decorator', 'del', 'details', 'dfn', 'dialog', 'dir', 'div', 'dl', 'dt', 'element', 'em', 'fieldset', 'figcaption', 'figure', 'font', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'img', 'input', 'ins', 'kbd', 'label', 'legend', 'li', 'main', 'map', 'mark', 'marquee', 'menu', 'menuitem', 'meter', 'nav', 'nobr', 'ol', 'optgroup', 'option', 'output', 'p', 'picture', 'pre', 'progress', 'q', 'rp', 'rt', 'ruby', 's', 'samp', 'section', 'select', 'shadow', 'small', 'source', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td', 'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'tr', 'track', 'tt', 'u', 'ul', 'var', 'video', 'wbr']);
const svg$1 = freeze(['svg', 'a', 'altglyph', 'altglyphdef', 'altglyphitem', 'animatecolor', 'animatemotion', 'animatetransform', 'circle', 'clippath', 'defs', 'desc', 'ellipse', 'filter', 'font', 'g', 'glyph', 'glyphref', 'hkern', 'image', 'line', 'lineargradient', 'marker', 'mask', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialgradient', 'rect', 'stop', 'style', 'switch', 'symbol', 'text', 'textpath', 'title', 'tref', 'tspan', 'view', 'vkern']);
const svgFilters = freeze(['feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting', 'feDisplacementMap', 'feDistantLight', 'feDropShadow', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feImage', 'feMerge', 'feMergeNode', 'feMorphology', 'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence']);
// List of SVG elements that are disallowed by default.
// We still need to know them so that we can do namespace
// checks properly in case one wants to add them to
// allow-list.
const svgDisallowed = freeze(['animate', 'color-profile', 'cursor', 'discard', 'font-face', 'font-face-format', 'font-face-name', 'font-face-src', 'font-face-uri', 'foreignobject', 'hatch', 'hatchpath', 'mesh', 'meshgradient', 'meshpatch', 'meshrow', 'missing-glyph', 'script', 'set', 'solidcolor', 'unknown', 'use']);
const mathMl$1 = freeze(['math', 'menclose', 'merror', 'mfenced', 'mfrac', 'mglyph', 'mi', 'mlabeledtr', 'mmultiscripts', 'mn', 'mo', 'mover', 'mpadded', 'mphantom', 'mroot', 'mrow', 'ms', 'mspace', 'msqrt', 'mstyle', 'msub', 'msup', 'msubsup', 'mtable', 'mtd', 'mtext', 'mtr', 'munder', 'munderover', 'mprescripts']);
// Similarly to SVG, we want to know all MathML elements,
// even those that we disallow by default.
const mathMlDisallowed = freeze(['maction', 'maligngroup', 'malignmark', 'mlongdiv', 'mscarries', 'mscarry', 'msgroup', 'mstack', 'msline', 'msrow', 'semantics', 'annotation', 'annotation-xml', 'mprescripts', 'none']);
const text = freeze(['#text']);

const html = freeze(['accept', 'action', 'align', 'alt', 'autocapitalize', 'autocomplete', 'autopictureinpicture', 'autoplay', 'background', 'bgcolor', 'border', 'capture', 'cellpadding', 'cellspacing', 'checked', 'cite', 'class', 'clear', 'color', 'cols', 'colspan', 'controls', 'controlslist', 'coords', 'crossorigin', 'datetime', 'decoding', 'default', 'dir', 'disabled', 'disablepictureinpicture', 'disableremoteplayback', 'download', 'draggable', 'enctype', 'enterkeyhint', 'face', 'for', 'headers', 'height', 'hidden', 'high', 'href', 'hreflang', 'id', 'inputmode', 'integrity', 'ismap', 'kind', 'label', 'lang', 'list', 'loading', 'loop', 'low', 'max', 'maxlength', 'media', 'method', 'min', 'minlength', 'multiple', 'muted', 'name', 'nonce', 'noshade', 'novalidate', 'nowrap', 'open', 'optimum', 'pattern', 'placeholder', 'playsinline', 'popover', 'popovertarget', 'popovertargetaction', 'poster', 'preload', 'pubdate', 'radiogroup', 'readonly', 'rel', 'required', 'rev', 'reversed', 'role', 'rows', 'rowspan', 'spellcheck', 'scope', 'selected', 'shape', 'size', 'sizes', 'span', 'srclang', 'start', 'src', 'srcset', 'step', 'style', 'summary', 'tabindex', 'title', 'translate', 'type', 'usemap', 'valign', 'value', 'width', 'wrap', 'xmlns', 'slot']);
const svg = freeze(['accent-height', 'accumulate', 'additive', 'alignment-baseline', 'amplitude', 'ascent', 'attributename', 'attributetype', 'azimuth', 'basefrequency', 'baseline-shift', 'begin', 'bias', 'by', 'class', 'clip', 'clippathunits', 'clip-path', 'clip-rule', 'color', 'color-interpolation', 'color-interpolation-filters', 'color-profile', 'color-rendering', 'cx', 'cy', 'd', 'dx', 'dy', 'diffuseconstant', 'direction', 'display', 'divisor', 'dur', 'edgemode', 'elevation', 'end', 'exponent', 'fill', 'fill-opacity', 'fill-rule', 'filter', 'filterunits', 'flood-color', 'flood-opacity', 'font-family', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 'fx', 'fy', 'g1', 'g2', 'glyph-name', 'glyphref', 'gradientunits', 'gradienttransform', 'height', 'href', 'id', 'image-rendering', 'in', 'in2', 'intercept', 'k', 'k1', 'k2', 'k3', 'k4', 'kerning', 'keypoints', 'keysplines', 'keytimes', 'lang', 'lengthadjust', 'letter-spacing', 'kernelmatrix', 'kernelunitlength', 'lighting-color', 'local', 'marker-end', 'marker-mid', 'marker-start', 'markerheight', 'markerunits', 'markerwidth', 'maskcontentunits', 'maskunits', 'max', 'mask', 'media', 'method', 'mode', 'min', 'name', 'numoctaves', 'offset', 'operator', 'opacity', 'order', 'orient', 'orientation', 'origin', 'overflow', 'paint-order', 'path', 'pathlength', 'patterncontentunits', 'patterntransform', 'patternunits', 'points', 'preservealpha', 'preserveaspectratio', 'primitiveunits', 'r', 'rx', 'ry', 'radius', 'refx', 'refy', 'repeatcount', 'repeatdur', 'restart', 'result', 'rotate', 'scale', 'seed', 'shape-rendering', 'slope', 'specularconstant', 'specularexponent', 'spreadmethod', 'startoffset', 'stddeviation', 'stitchtiles', 'stop-color', 'stop-opacity', 'stroke-dasharray', 'stroke-dashoffset', 'stroke-linecap', 'stroke-linejoin', 'stroke-miterlimit', 'stroke-opacity', 'stroke', 'stroke-width', 'style', 'surfacescale', 'systemlanguage', 'tabindex', 'tablevalues', 'targetx', 'targety', 'transform', 'transform-origin', 'text-anchor', 'text-decoration', 'text-rendering', 'textlength', 'type', 'u1', 'u2', 'unicode', 'values', 'viewbox', 'visibility', 'version', 'vert-adv-y', 'vert-origin-x', 'vert-origin-y', 'width', 'word-spacing', 'wrap', 'writing-mode', 'xchannelselector', 'ychannelselector', 'x', 'x1', 'x2', 'xmlns', 'y', 'y1', 'y2', 'z', 'zoomandpan']);
const mathMl = freeze(['accent', 'accentunder', 'align', 'bevelled', 'close', 'columnsalign', 'columnlines', 'columnspan', 'denomalign', 'depth', 'dir', 'display', 'displaystyle', 'encoding', 'fence', 'frame', 'height', 'href', 'id', 'largeop', 'length', 'linethickness', 'lspace', 'lquote', 'mathbackground', 'mathcolor', 'mathsize', 'mathvariant', 'maxsize', 'minsize', 'movablelimits', 'notation', 'numalign', 'open', 'rowalign', 'rowlines', 'rowspacing', 'rowspan', 'rspace', 'rquote', 'scriptlevel', 'scriptminsize', 'scriptsizemultiplier', 'selection', 'separator', 'separators', 'stretchy', 'subscriptshift', 'supscriptshift', 'symmetric', 'voffset', 'width', 'xmlns']);
const xml = freeze(['xlink:href', 'xml:id', 'xlink:title', 'xml:space', 'xmlns:xlink']);

// eslint-disable-next-line unicorn/better-regex
const MUSTACHE_EXPR = seal(/\{\{[\w\W]*|[\w\W]*\}\}/gm); // Specify template detection regex for SAFE_FOR_TEMPLATES mode
const ERB_EXPR = seal(/<%[\w\W]*|[\w\W]*%>/gm);
const TMPLIT_EXPR = seal(/\$\{[\w\W]*/gm); // eslint-disable-line unicorn/better-regex
const DATA_ATTR = seal(/^data-[\-\w.\u00B7-\uFFFF]+$/); // eslint-disable-line no-useless-escape
const ARIA_ATTR = seal(/^aria-[\-\w]+$/); // eslint-disable-line no-useless-escape
const IS_ALLOWED_URI = seal(/^(?:(?:(?:f|ht)tps?|mailto|tel|callto|sms|cid|xmpp|matrix):|[^a-z]|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i // eslint-disable-line no-useless-escape
);
const IS_SCRIPT_OR_DATA = seal(/^(?:\w+script|data):/i);
const ATTR_WHITESPACE = seal(/[\u0000-\u0020\u00A0\u1680\u180E\u2000-\u2029\u205F\u3000]/g // eslint-disable-line no-control-regex
);
const DOCTYPE_NAME = seal(/^html$/i);
const CUSTOM_ELEMENT = seal(/^[a-z][.\w]*(-[.\w]+)+$/i);

var EXPRESSIONS = /*#__PURE__*/Object.freeze({
  __proto__: null,
  ARIA_ATTR: ARIA_ATTR,
  ATTR_WHITESPACE: ATTR_WHITESPACE,
  CUSTOM_ELEMENT: CUSTOM_ELEMENT,
  DATA_ATTR: DATA_ATTR,
  DOCTYPE_NAME: DOCTYPE_NAME,
  ERB_EXPR: ERB_EXPR,
  IS_ALLOWED_URI: IS_ALLOWED_URI,
  IS_SCRIPT_OR_DATA: IS_SCRIPT_OR_DATA,
  MUSTACHE_EXPR: MUSTACHE_EXPR,
  TMPLIT_EXPR: TMPLIT_EXPR
});

/* eslint-disable @typescript-eslint/indent */
// https://developer.mozilla.org/en-US/docs/Web/API/Node/nodeType
const NODE_TYPE = {
  element: 1,
  attribute: 2,
  text: 3,
  cdataSection: 4,
  entityReference: 5,
  // Deprecated
  entityNode: 6,
  // Deprecated
  progressingInstruction: 7,
  comment: 8,
  document: 9,
  documentType: 10,
  documentFragment: 11,
  notation: 12 // Deprecated
};
const getGlobal = function getGlobal() {
  return typeof window === 'undefined' ? null : window;
};
/**
 * Creates a no-op policy for internal use only.
 * Don't export this function outside this module!
 * @param trustedTypes The policy factory.
 * @param purifyHostElement The Script element used to load DOMPurify (to determine policy name suffix).
 * @return The policy created (or null, if Trusted Types
 * are not supported or creating the policy failed).
 */
const _createTrustedTypesPolicy = function _createTrustedTypesPolicy(trustedTypes, purifyHostElement) {
  if (typeof trustedTypes !== 'object' || typeof trustedTypes.createPolicy !== 'function') {
    return null;
  }
  // Allow the callers to control the unique policy name
  // by adding a data-tt-policy-suffix to the script element with the DOMPurify.
  // Policy creation with duplicate names throws in Trusted Types.
  let suffix = null;
  const ATTR_NAME = 'data-tt-policy-suffix';
  if (purifyHostElement && purifyHostElement.hasAttribute(ATTR_NAME)) {
    suffix = purifyHostElement.getAttribute(ATTR_NAME);
  }
  const policyName = 'dompurify' + (suffix ? '#' + suffix : '');
  try {
    return trustedTypes.createPolicy(policyName, {
      createHTML(html) {
        return html;
      },
      createScriptURL(scriptUrl) {
        return scriptUrl;
      }
    });
  } catch (_) {
    // Policy creation failed (most likely another DOMPurify script has
    // already run). Skip creating the policy, as this will only cause errors
    // if TT are enforced.
    console.warn('TrustedTypes policy ' + policyName + ' could not be created.');
    return null;
  }
};
const _createHooksMap = function _createHooksMap() {
  return {
    afterSanitizeAttributes: [],
    afterSanitizeElements: [],
    afterSanitizeShadowDOM: [],
    beforeSanitizeAttributes: [],
    beforeSanitizeElements: [],
    beforeSanitizeShadowDOM: [],
    uponSanitizeAttribute: [],
    uponSanitizeElement: [],
    uponSanitizeShadowNode: []
  };
};
function createDOMPurify() {
  let window = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : getGlobal();
  const DOMPurify = root => createDOMPurify(root);
  DOMPurify.version = '3.2.6';
  DOMPurify.removed = [];
  if (!window || !window.document || window.document.nodeType !== NODE_TYPE.document || !window.Element) {
    // Not running in a browser, provide a factory function
    // so that you can pass your own Window
    DOMPurify.isSupported = false;
    return DOMPurify;
  }
  let {
    document
  } = window;
  const originalDocument = document;
  const currentScript = originalDocument.currentScript;
  const {
    DocumentFragment,
    HTMLTemplateElement,
    Node,
    Element,
    NodeFilter,
    NamedNodeMap = window.NamedNodeMap || window.MozNamedAttrMap,
    HTMLFormElement,
    DOMParser,
    trustedTypes
  } = window;
  const ElementPrototype = Element.prototype;
  const cloneNode = lookupGetter(ElementPrototype, 'cloneNode');
  const remove = lookupGetter(ElementPrototype, 'remove');
  const getNextSibling = lookupGetter(ElementPrototype, 'nextSibling');
  const getChildNodes = lookupGetter(ElementPrototype, 'childNodes');
  const getParentNode = lookupGetter(ElementPrototype, 'parentNode');
  // As per issue #47, the web-components registry is inherited by a
  // new document created via createHTMLDocument. As per the spec
  // (http://w3c.github.io/webcomponents/spec/custom/#creating-and-passing-registries)
  // a new empty registry is used when creating a template contents owner
  // document, so we use that as our parent document to ensure nothing
  // is inherited.
  if (typeof HTMLTemplateElement === 'function') {
    const template = document.createElement('template');
    if (template.content && template.content.ownerDocument) {
      document = template.content.ownerDocument;
    }
  }
  let trustedTypesPolicy;
  let emptyHTML = '';
  const {
    implementation,
    createNodeIterator,
    createDocumentFragment,
    getElementsByTagName
  } = document;
  const {
    importNode
  } = originalDocument;
  let hooks = _createHooksMap();
  /**
   * Expose whether this browser supports running the full DOMPurify.
   */
  DOMPurify.isSupported = typeof entries === 'function' && typeof getParentNode === 'function' && implementation && implementation.createHTMLDocument !== undefined;
  const {
    MUSTACHE_EXPR,
    ERB_EXPR,
    TMPLIT_EXPR,
    DATA_ATTR,
    ARIA_ATTR,
    IS_SCRIPT_OR_DATA,
    ATTR_WHITESPACE,
    CUSTOM_ELEMENT
  } = EXPRESSIONS;
  let {
    IS_ALLOWED_URI: IS_ALLOWED_URI$1
  } = EXPRESSIONS;
  /**
   * We consider the elements and attributes below to be safe. Ideally
   * don't add any new ones but feel free to remove unwanted ones.
   */
  /* allowed element names */
  let ALLOWED_TAGS = null;
  const DEFAULT_ALLOWED_TAGS = addToSet({}, [...html$1, ...svg$1, ...svgFilters, ...mathMl$1, ...text]);
  /* Allowed attribute names */
  let ALLOWED_ATTR = null;
  const DEFAULT_ALLOWED_ATTR = addToSet({}, [...html, ...svg, ...mathMl, ...xml]);
  /*
   * Configure how DOMPurify should handle custom elements and their attributes as well as customized built-in elements.
   * @property {RegExp|Function|null} tagNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any custom elements)
   * @property {RegExp|Function|null} attributeNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any attributes not on the allow list)
   * @property {boolean} allowCustomizedBuiltInElements allow custom elements derived from built-ins if they pass CUSTOM_ELEMENT_HANDLING.tagNameCheck. Default: `false`.
   */
  let CUSTOM_ELEMENT_HANDLING = Object.seal(create$2(null, {
    tagNameCheck: {
      writable: true,
      configurable: false,
      enumerable: true,
      value: null
    },
    attributeNameCheck: {
      writable: true,
      configurable: false,
      enumerable: true,
      value: null
    },
    allowCustomizedBuiltInElements: {
      writable: true,
      configurable: false,
      enumerable: true,
      value: false
    }
  }));
  /* Explicitly forbidden tags (overrides ALLOWED_TAGS/ADD_TAGS) */
  let FORBID_TAGS = null;
  /* Explicitly forbidden attributes (overrides ALLOWED_ATTR/ADD_ATTR) */
  let FORBID_ATTR = null;
  /* Decide if ARIA attributes are okay */
  let ALLOW_ARIA_ATTR = true;
  /* Decide if custom data attributes are okay */
  let ALLOW_DATA_ATTR = true;
  /* Decide if unknown protocols are okay */
  let ALLOW_UNKNOWN_PROTOCOLS = false;
  /* Decide if self-closing tags in attributes are allowed.
   * Usually removed due to a mXSS issue in jQuery 3.0 */
  let ALLOW_SELF_CLOSE_IN_ATTR = true;
  /* Output should be safe for common template engines.
   * This means, DOMPurify removes data attributes, mustaches and ERB
   */
  let SAFE_FOR_TEMPLATES = false;
  /* Output should be safe even for XML used within HTML and alike.
   * This means, DOMPurify removes comments when containing risky content.
   */
  let SAFE_FOR_XML = true;
  /* Decide if document with <html>... should be returned */
  let WHOLE_DOCUMENT = false;
  /* Track whether config is already set on this instance of DOMPurify. */
  let SET_CONFIG = false;
  /* Decide if all elements (e.g. style, script) must be children of
   * document.body. By default, browsers might move them to document.head */
  let FORCE_BODY = false;
  /* Decide if a DOM `HTMLBodyElement` should be returned, instead of a html
   * string (or a TrustedHTML object if Trusted Types are supported).
   * If `WHOLE_DOCUMENT` is enabled a `HTMLHtmlElement` will be returned instead
   */
  let RETURN_DOM = false;
  /* Decide if a DOM `DocumentFragment` should be returned, instead of a html
   * string  (or a TrustedHTML object if Trusted Types are supported) */
  let RETURN_DOM_FRAGMENT = false;
  /* Try to return a Trusted Type object instead of a string, return a string in
   * case Trusted Types are not supported  */
  let RETURN_TRUSTED_TYPE = false;
  /* Output should be free from DOM clobbering attacks?
   * This sanitizes markups named with colliding, clobberable built-in DOM APIs.
   */
  let SANITIZE_DOM = true;
  /* Achieve full DOM Clobbering protection by isolating the namespace of named
   * properties and JS variables, mitigating attacks that abuse the HTML/DOM spec rules.
   *
   * HTML/DOM spec rules that enable DOM Clobbering:
   *   - Named Access on Window (§7.3.3)
   *   - DOM Tree Accessors (§3.1.5)
   *   - Form Element Parent-Child Relations (§4.10.3)
   *   - Iframe srcdoc / Nested WindowProxies (§4.8.5)
   *   - HTMLCollection (§4.2.10.2)
   *
   * Namespace isolation is implemented by prefixing `id` and `name` attributes
   * with a constant string, i.e., `user-content-`
   */
  let SANITIZE_NAMED_PROPS = false;
  const SANITIZE_NAMED_PROPS_PREFIX = 'user-content-';
  /* Keep element content when removing element? */
  let KEEP_CONTENT = true;
  /* If a `Node` is passed to sanitize(), then performs sanitization in-place instead
   * of importing it into a new Document and returning a sanitized copy */
  let IN_PLACE = false;
  /* Allow usage of profiles like html, svg and mathMl */
  let USE_PROFILES = {};
  /* Tags to ignore content of when KEEP_CONTENT is true */
  let FORBID_CONTENTS = null;
  const DEFAULT_FORBID_CONTENTS = addToSet({}, ['annotation-xml', 'audio', 'colgroup', 'desc', 'foreignobject', 'head', 'iframe', 'math', 'mi', 'mn', 'mo', 'ms', 'mtext', 'noembed', 'noframes', 'noscript', 'plaintext', 'script', 'style', 'svg', 'template', 'thead', 'title', 'video', 'xmp']);
  /* Tags that are safe for data: URIs */
  let DATA_URI_TAGS = null;
  const DEFAULT_DATA_URI_TAGS = addToSet({}, ['audio', 'video', 'img', 'source', 'image', 'track']);
  /* Attributes safe for values like "javascript:" */
  let URI_SAFE_ATTRIBUTES = null;
  const DEFAULT_URI_SAFE_ATTRIBUTES = addToSet({}, ['alt', 'class', 'for', 'id', 'label', 'name', 'pattern', 'placeholder', 'role', 'summary', 'title', 'value', 'style', 'xmlns']);
  const MATHML_NAMESPACE = 'http://www.w3.org/1998/Math/MathML';
  const SVG_NAMESPACE = 'http://www.w3.org/2000/svg';
  const HTML_NAMESPACE = 'http://www.w3.org/1999/xhtml';
  /* Document namespace */
  let NAMESPACE = HTML_NAMESPACE;
  let IS_EMPTY_INPUT = false;
  /* Allowed XHTML+XML namespaces */
  let ALLOWED_NAMESPACES = null;
  const DEFAULT_ALLOWED_NAMESPACES = addToSet({}, [MATHML_NAMESPACE, SVG_NAMESPACE, HTML_NAMESPACE], stringToString);
  let MATHML_TEXT_INTEGRATION_POINTS = addToSet({}, ['mi', 'mo', 'mn', 'ms', 'mtext']);
  let HTML_INTEGRATION_POINTS = addToSet({}, ['annotation-xml']);
  // Certain elements are allowed in both SVG and HTML
  // namespace. We need to specify them explicitly
  // so that they don't get erroneously deleted from
  // HTML namespace.
  const COMMON_SVG_AND_HTML_ELEMENTS = addToSet({}, ['title', 'style', 'font', 'a', 'script']);
  /* Parsing of strict XHTML documents */
  let PARSER_MEDIA_TYPE = null;
  const SUPPORTED_PARSER_MEDIA_TYPES = ['application/xhtml+xml', 'text/html'];
  const DEFAULT_PARSER_MEDIA_TYPE = 'text/html';
  let transformCaseFunc = null;
  /* Keep a reference to config to pass to hooks */
  let CONFIG = null;
  /* Ideally, do not touch anything below this line */
  /* ______________________________________________ */
  const formElement = document.createElement('form');
  const isRegexOrFunction = function isRegexOrFunction(testValue) {
    return testValue instanceof RegExp || testValue instanceof Function;
  };
  /**
   * _parseConfig
   *
   * @param cfg optional config literal
   */
  // eslint-disable-next-line complexity
  const _parseConfig = function _parseConfig() {
    let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    if (CONFIG && CONFIG === cfg) {
      return;
    }
    /* Shield configuration object from tampering */
    if (!cfg || typeof cfg !== 'object') {
      cfg = {};
    }
    /* Shield configuration object from prototype pollution */
    cfg = clone(cfg);
    PARSER_MEDIA_TYPE =
    // eslint-disable-next-line unicorn/prefer-includes
    SUPPORTED_PARSER_MEDIA_TYPES.indexOf(cfg.PARSER_MEDIA_TYPE) === -1 ? DEFAULT_PARSER_MEDIA_TYPE : cfg.PARSER_MEDIA_TYPE;
    // HTML tags and attributes are not case-sensitive, converting to lowercase. Keeping XHTML as is.
    transformCaseFunc = PARSER_MEDIA_TYPE === 'application/xhtml+xml' ? stringToString : stringToLowerCase;
    /* Set configuration parameters */
    ALLOWED_TAGS = objectHasOwnProperty(cfg, 'ALLOWED_TAGS') ? addToSet({}, cfg.ALLOWED_TAGS, transformCaseFunc) : DEFAULT_ALLOWED_TAGS;
    ALLOWED_ATTR = objectHasOwnProperty(cfg, 'ALLOWED_ATTR') ? addToSet({}, cfg.ALLOWED_ATTR, transformCaseFunc) : DEFAULT_ALLOWED_ATTR;
    ALLOWED_NAMESPACES = objectHasOwnProperty(cfg, 'ALLOWED_NAMESPACES') ? addToSet({}, cfg.ALLOWED_NAMESPACES, stringToString) : DEFAULT_ALLOWED_NAMESPACES;
    URI_SAFE_ATTRIBUTES = objectHasOwnProperty(cfg, 'ADD_URI_SAFE_ATTR') ? addToSet(clone(DEFAULT_URI_SAFE_ATTRIBUTES), cfg.ADD_URI_SAFE_ATTR, transformCaseFunc) : DEFAULT_URI_SAFE_ATTRIBUTES;
    DATA_URI_TAGS = objectHasOwnProperty(cfg, 'ADD_DATA_URI_TAGS') ? addToSet(clone(DEFAULT_DATA_URI_TAGS), cfg.ADD_DATA_URI_TAGS, transformCaseFunc) : DEFAULT_DATA_URI_TAGS;
    FORBID_CONTENTS = objectHasOwnProperty(cfg, 'FORBID_CONTENTS') ? addToSet({}, cfg.FORBID_CONTENTS, transformCaseFunc) : DEFAULT_FORBID_CONTENTS;
    FORBID_TAGS = objectHasOwnProperty(cfg, 'FORBID_TAGS') ? addToSet({}, cfg.FORBID_TAGS, transformCaseFunc) : clone({});
    FORBID_ATTR = objectHasOwnProperty(cfg, 'FORBID_ATTR') ? addToSet({}, cfg.FORBID_ATTR, transformCaseFunc) : clone({});
    USE_PROFILES = objectHasOwnProperty(cfg, 'USE_PROFILES') ? cfg.USE_PROFILES : false;
    ALLOW_ARIA_ATTR = cfg.ALLOW_ARIA_ATTR !== false; // Default true
    ALLOW_DATA_ATTR = cfg.ALLOW_DATA_ATTR !== false; // Default true
    ALLOW_UNKNOWN_PROTOCOLS = cfg.ALLOW_UNKNOWN_PROTOCOLS || false; // Default false
    ALLOW_SELF_CLOSE_IN_ATTR = cfg.ALLOW_SELF_CLOSE_IN_ATTR !== false; // Default true
    SAFE_FOR_TEMPLATES = cfg.SAFE_FOR_TEMPLATES || false; // Default false
    SAFE_FOR_XML = cfg.SAFE_FOR_XML !== false; // Default true
    WHOLE_DOCUMENT = cfg.WHOLE_DOCUMENT || false; // Default false
    RETURN_DOM = cfg.RETURN_DOM || false; // Default false
    RETURN_DOM_FRAGMENT = cfg.RETURN_DOM_FRAGMENT || false; // Default false
    RETURN_TRUSTED_TYPE = cfg.RETURN_TRUSTED_TYPE || false; // Default false
    FORCE_BODY = cfg.FORCE_BODY || false; // Default false
    SANITIZE_DOM = cfg.SANITIZE_DOM !== false; // Default true
    SANITIZE_NAMED_PROPS = cfg.SANITIZE_NAMED_PROPS || false; // Default false
    KEEP_CONTENT = cfg.KEEP_CONTENT !== false; // Default true
    IN_PLACE = cfg.IN_PLACE || false; // Default false
    IS_ALLOWED_URI$1 = cfg.ALLOWED_URI_REGEXP || IS_ALLOWED_URI;
    NAMESPACE = cfg.NAMESPACE || HTML_NAMESPACE;
    MATHML_TEXT_INTEGRATION_POINTS = cfg.MATHML_TEXT_INTEGRATION_POINTS || MATHML_TEXT_INTEGRATION_POINTS;
    HTML_INTEGRATION_POINTS = cfg.HTML_INTEGRATION_POINTS || HTML_INTEGRATION_POINTS;
    CUSTOM_ELEMENT_HANDLING = cfg.CUSTOM_ELEMENT_HANDLING || {};
    if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck)) {
      CUSTOM_ELEMENT_HANDLING.tagNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck;
    }
    if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck)) {
      CUSTOM_ELEMENT_HANDLING.attributeNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck;
    }
    if (cfg.CUSTOM_ELEMENT_HANDLING && typeof cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements === 'boolean') {
      CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements = cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements;
    }
    if (SAFE_FOR_TEMPLATES) {
      ALLOW_DATA_ATTR = false;
    }
    if (RETURN_DOM_FRAGMENT) {
      RETURN_DOM = true;
    }
    /* Parse profile info */
    if (USE_PROFILES) {
      ALLOWED_TAGS = addToSet({}, text);
      ALLOWED_ATTR = [];
      if (USE_PROFILES.html === true) {
        addToSet(ALLOWED_TAGS, html$1);
        addToSet(ALLOWED_ATTR, html);
      }
      if (USE_PROFILES.svg === true) {
        addToSet(ALLOWED_TAGS, svg$1);
        addToSet(ALLOWED_ATTR, svg);
        addToSet(ALLOWED_ATTR, xml);
      }
      if (USE_PROFILES.svgFilters === true) {
        addToSet(ALLOWED_TAGS, svgFilters);
        addToSet(ALLOWED_ATTR, svg);
        addToSet(ALLOWED_ATTR, xml);
      }
      if (USE_PROFILES.mathMl === true) {
        addToSet(ALLOWED_TAGS, mathMl$1);
        addToSet(ALLOWED_ATTR, mathMl);
        addToSet(ALLOWED_ATTR, xml);
      }
    }
    /* Merge configuration parameters */
    if (cfg.ADD_TAGS) {
      if (ALLOWED_TAGS === DEFAULT_ALLOWED_TAGS) {
        ALLOWED_TAGS = clone(ALLOWED_TAGS);
      }
      addToSet(ALLOWED_TAGS, cfg.ADD_TAGS, transformCaseFunc);
    }
    if (cfg.ADD_ATTR) {
      if (ALLOWED_ATTR === DEFAULT_ALLOWED_ATTR) {
        ALLOWED_ATTR = clone(ALLOWED_ATTR);
      }
      addToSet(ALLOWED_ATTR, cfg.ADD_ATTR, transformCaseFunc);
    }
    if (cfg.ADD_URI_SAFE_ATTR) {
      addToSet(URI_SAFE_ATTRIBUTES, cfg.ADD_URI_SAFE_ATTR, transformCaseFunc);
    }
    if (cfg.FORBID_CONTENTS) {
      if (FORBID_CONTENTS === DEFAULT_FORBID_CONTENTS) {
        FORBID_CONTENTS = clone(FORBID_CONTENTS);
      }
      addToSet(FORBID_CONTENTS, cfg.FORBID_CONTENTS, transformCaseFunc);
    }
    /* Add #text in case KEEP_CONTENT is set to true */
    if (KEEP_CONTENT) {
      ALLOWED_TAGS['#text'] = true;
    }
    /* Add html, head and body to ALLOWED_TAGS in case WHOLE_DOCUMENT is true */
    if (WHOLE_DOCUMENT) {
      addToSet(ALLOWED_TAGS, ['html', 'head', 'body']);
    }
    /* Add tbody to ALLOWED_TAGS in case tables are permitted, see #286, #365 */
    if (ALLOWED_TAGS.table) {
      addToSet(ALLOWED_TAGS, ['tbody']);
      delete FORBID_TAGS.tbody;
    }
    if (cfg.TRUSTED_TYPES_POLICY) {
      if (typeof cfg.TRUSTED_TYPES_POLICY.createHTML !== 'function') {
        throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createHTML" hook.');
      }
      if (typeof cfg.TRUSTED_TYPES_POLICY.createScriptURL !== 'function') {
        throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createScriptURL" hook.');
      }
      // Overwrite existing TrustedTypes policy.
      trustedTypesPolicy = cfg.TRUSTED_TYPES_POLICY;
      // Sign local variables required by `sanitize`.
      emptyHTML = trustedTypesPolicy.createHTML('');
    } else {
      // Uninitialized policy, attempt to initialize the internal dompurify policy.
      if (trustedTypesPolicy === undefined) {
        trustedTypesPolicy = _createTrustedTypesPolicy(trustedTypes, currentScript);
      }
      // If creating the internal policy succeeded sign internal variables.
      if (trustedTypesPolicy !== null && typeof emptyHTML === 'string') {
        emptyHTML = trustedTypesPolicy.createHTML('');
      }
    }
    // Prevent further manipulation of configuration.
    // Not available in IE8, Safari 5, etc.
    if (freeze) {
      freeze(cfg);
    }
    CONFIG = cfg;
  };
  /* Keep track of all possible SVG and MathML tags
   * so that we can perform the namespace checks
   * correctly. */
  const ALL_SVG_TAGS = addToSet({}, [...svg$1, ...svgFilters, ...svgDisallowed]);
  const ALL_MATHML_TAGS = addToSet({}, [...mathMl$1, ...mathMlDisallowed]);
  /**
   * @param element a DOM element whose namespace is being checked
   * @returns Return false if the element has a
   *  namespace that a spec-compliant parser would never
   *  return. Return true otherwise.
   */
  const _checkValidNamespace = function _checkValidNamespace(element) {
    let parent = getParentNode(element);
    // In JSDOM, if we're inside shadow DOM, then parentNode
    // can be null. We just simulate parent in this case.
    if (!parent || !parent.tagName) {
      parent = {
        namespaceURI: NAMESPACE,
        tagName: 'template'
      };
    }
    const tagName = stringToLowerCase(element.tagName);
    const parentTagName = stringToLowerCase(parent.tagName);
    if (!ALLOWED_NAMESPACES[element.namespaceURI]) {
      return false;
    }
    if (element.namespaceURI === SVG_NAMESPACE) {
      // The only way to switch from HTML namespace to SVG
      // is via <svg>. If it happens via any other tag, then
      // it should be killed.
      if (parent.namespaceURI === HTML_NAMESPACE) {
        return tagName === 'svg';
      }
      // The only way to switch from MathML to SVG is via`
      // svg if parent is either <annotation-xml> or MathML
      // text integration points.
      if (parent.namespaceURI === MATHML_NAMESPACE) {
        return tagName === 'svg' && (parentTagName === 'annotation-xml' || MATHML_TEXT_INTEGRATION_POINTS[parentTagName]);
      }
      // We only allow elements that are defined in SVG
      // spec. All others are disallowed in SVG namespace.
      return Boolean(ALL_SVG_TAGS[tagName]);
    }
    if (element.namespaceURI === MATHML_NAMESPACE) {
      // The only way to switch from HTML namespace to MathML
      // is via <math>. If it happens via any other tag, then
      // it should be killed.
      if (parent.namespaceURI === HTML_NAMESPACE) {
        return tagName === 'math';
      }
      // The only way to switch from SVG to MathML is via
      // <math> and HTML integration points
      if (parent.namespaceURI === SVG_NAMESPACE) {
        return tagName === 'math' && HTML_INTEGRATION_POINTS[parentTagName];
      }
      // We only allow elements that are defined in MathML
      // spec. All others are disallowed in MathML namespace.
      return Boolean(ALL_MATHML_TAGS[tagName]);
    }
    if (element.namespaceURI === HTML_NAMESPACE) {
      // The only way to switch from SVG to HTML is via
      // HTML integration points, and from MathML to HTML
      // is via MathML text integration points
      if (parent.namespaceURI === SVG_NAMESPACE && !HTML_INTEGRATION_POINTS[parentTagName]) {
        return false;
      }
      if (parent.namespaceURI === MATHML_NAMESPACE && !MATHML_TEXT_INTEGRATION_POINTS[parentTagName]) {
        return false;
      }
      // We disallow tags that are specific for MathML
      // or SVG and should never appear in HTML namespace
      return !ALL_MATHML_TAGS[tagName] && (COMMON_SVG_AND_HTML_ELEMENTS[tagName] || !ALL_SVG_TAGS[tagName]);
    }
    // For XHTML and XML documents that support custom namespaces
    if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && ALLOWED_NAMESPACES[element.namespaceURI]) {
      return true;
    }
    // The code should never reach this place (this means
    // that the element somehow got namespace that is not
    // HTML, SVG, MathML or allowed via ALLOWED_NAMESPACES).
    // Return false just in case.
    return false;
  };
  /**
   * _forceRemove
   *
   * @param node a DOM node
   */
  const _forceRemove = function _forceRemove(node) {
    arrayPush(DOMPurify.removed, {
      element: node
    });
    try {
      // eslint-disable-next-line unicorn/prefer-dom-node-remove
      getParentNode(node).removeChild(node);
    } catch (_) {
      remove(node);
    }
  };
  /**
   * _removeAttribute
   *
   * @param name an Attribute name
   * @param element a DOM node
   */
  const _removeAttribute = function _removeAttribute(name, element) {
    try {
      arrayPush(DOMPurify.removed, {
        attribute: element.getAttributeNode(name),
        from: element
      });
    } catch (_) {
      arrayPush(DOMPurify.removed, {
        attribute: null,
        from: element
      });
    }
    element.removeAttribute(name);
    // We void attribute values for unremovable "is" attributes
    if (name === 'is') {
      if (RETURN_DOM || RETURN_DOM_FRAGMENT) {
        try {
          _forceRemove(element);
        } catch (_) {}
      } else {
        try {
          element.setAttribute(name, '');
        } catch (_) {}
      }
    }
  };
  /**
   * _initDocument
   *
   * @param dirty - a string of dirty markup
   * @return a DOM, filled with the dirty markup
   */
  const _initDocument = function _initDocument(dirty) {
    /* Create a HTML document */
    let doc = null;
    let leadingWhitespace = null;
    if (FORCE_BODY) {
      dirty = '<remove></remove>' + dirty;
    } else {
      /* If FORCE_BODY isn't used, leading whitespace needs to be preserved manually */
      const matches = stringMatch(dirty, /^[\r\n\t ]+/);
      leadingWhitespace = matches && matches[0];
    }
    if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && NAMESPACE === HTML_NAMESPACE) {
      // Root of XHTML doc must contain xmlns declaration (see https://www.w3.org/TR/xhtml1/normative.html#strict)
      dirty = '<html xmlns="http://www.w3.org/1999/xhtml"><head></head><body>' + dirty + '</body></html>';
    }
    const dirtyPayload = trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
    /*
     * Use the DOMParser API by default, fallback later if needs be
     * DOMParser not work for svg when has multiple root element.
     */
    if (NAMESPACE === HTML_NAMESPACE) {
      try {
        doc = new DOMParser().parseFromString(dirtyPayload, PARSER_MEDIA_TYPE);
      } catch (_) {}
    }
    /* Use createHTMLDocument in case DOMParser is not available */
    if (!doc || !doc.documentElement) {
      doc = implementation.createDocument(NAMESPACE, 'template', null);
      try {
        doc.documentElement.innerHTML = IS_EMPTY_INPUT ? emptyHTML : dirtyPayload;
      } catch (_) {
        // Syntax error if dirtyPayload is invalid xml
      }
    }
    const body = doc.body || doc.documentElement;
    if (dirty && leadingWhitespace) {
      body.insertBefore(document.createTextNode(leadingWhitespace), body.childNodes[0] || null);
    }
    /* Work on whole document or just its body */
    if (NAMESPACE === HTML_NAMESPACE) {
      return getElementsByTagName.call(doc, WHOLE_DOCUMENT ? 'html' : 'body')[0];
    }
    return WHOLE_DOCUMENT ? doc.documentElement : body;
  };
  /**
   * Creates a NodeIterator object that you can use to traverse filtered lists of nodes or elements in a document.
   *
   * @param root The root element or node to start traversing on.
   * @return The created NodeIterator
   */
  const _createNodeIterator = function _createNodeIterator(root) {
    return createNodeIterator.call(root.ownerDocument || root, root,
    // eslint-disable-next-line no-bitwise
    NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT | NodeFilter.SHOW_PROCESSING_INSTRUCTION | NodeFilter.SHOW_CDATA_SECTION, null);
  };
  /**
   * _isClobbered
   *
   * @param element element to check for clobbering attacks
   * @return true if clobbered, false if safe
   */
  const _isClobbered = function _isClobbered(element) {
    return element instanceof HTMLFormElement && (typeof element.nodeName !== 'string' || typeof element.textContent !== 'string' || typeof element.removeChild !== 'function' || !(element.attributes instanceof NamedNodeMap) || typeof element.removeAttribute !== 'function' || typeof element.setAttribute !== 'function' || typeof element.namespaceURI !== 'string' || typeof element.insertBefore !== 'function' || typeof element.hasChildNodes !== 'function');
  };
  /**
   * Checks whether the given object is a DOM node.
   *
   * @param value object to check whether it's a DOM node
   * @return true is object is a DOM node
   */
  const _isNode = function _isNode(value) {
    return typeof Node === 'function' && value instanceof Node;
  };
  function _executeHooks(hooks, currentNode, data) {
    arrayForEach(hooks, hook => {
      hook.call(DOMPurify, currentNode, data, CONFIG);
    });
  }
  /**
   * _sanitizeElements
   *
   * @protect nodeName
   * @protect textContent
   * @protect removeChild
   * @param currentNode to check for permission to exist
   * @return true if node was killed, false if left alive
   */
  const _sanitizeElements = function _sanitizeElements(currentNode) {
    let content = null;
    /* Execute a hook if present */
    _executeHooks(hooks.beforeSanitizeElements, currentNode, null);
    /* Check if element is clobbered or can clobber */
    if (_isClobbered(currentNode)) {
      _forceRemove(currentNode);
      return true;
    }
    /* Now let's check the element's type and name */
    const tagName = transformCaseFunc(currentNode.nodeName);
    /* Execute a hook if present */
    _executeHooks(hooks.uponSanitizeElement, currentNode, {
      tagName,
      allowedTags: ALLOWED_TAGS
    });
    /* Detect mXSS attempts abusing namespace confusion */
    if (SAFE_FOR_XML && currentNode.hasChildNodes() && !_isNode(currentNode.firstElementChild) && regExpTest(/<[/\w!]/g, currentNode.innerHTML) && regExpTest(/<[/\w!]/g, currentNode.textContent)) {
      _forceRemove(currentNode);
      return true;
    }
    /* Remove any occurrence of processing instructions */
    if (currentNode.nodeType === NODE_TYPE.progressingInstruction) {
      _forceRemove(currentNode);
      return true;
    }
    /* Remove any kind of possibly harmful comments */
    if (SAFE_FOR_XML && currentNode.nodeType === NODE_TYPE.comment && regExpTest(/<[/\w]/g, currentNode.data)) {
      _forceRemove(currentNode);
      return true;
    }
    /* Remove element if anything forbids its presence */
    if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
      /* Check if we have a custom element to handle */
      if (!FORBID_TAGS[tagName] && _isBasicCustomElement(tagName)) {
        if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, tagName)) {
          return false;
        }
        if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(tagName)) {
          return false;
        }
      }
      /* Keep content except for bad-listed elements */
      if (KEEP_CONTENT && !FORBID_CONTENTS[tagName]) {
        const parentNode = getParentNode(currentNode) || currentNode.parentNode;
        const childNodes = getChildNodes(currentNode) || currentNode.childNodes;
        if (childNodes && parentNode) {
          const childCount = childNodes.length;
          for (let i = childCount - 1; i >= 0; --i) {
            const childClone = cloneNode(childNodes[i], true);
            childClone.__removalCount = (currentNode.__removalCount || 0) + 1;
            parentNode.insertBefore(childClone, getNextSibling(currentNode));
          }
        }
      }
      _forceRemove(currentNode);
      return true;
    }
    /* Check whether element has a valid namespace */
    if (currentNode instanceof Element && !_checkValidNamespace(currentNode)) {
      _forceRemove(currentNode);
      return true;
    }
    /* Make sure that older browsers don't get fallback-tag mXSS */
    if ((tagName === 'noscript' || tagName === 'noembed' || tagName === 'noframes') && regExpTest(/<\/no(script|embed|frames)/i, currentNode.innerHTML)) {
      _forceRemove(currentNode);
      return true;
    }
    /* Sanitize element content to be template-safe */
    if (SAFE_FOR_TEMPLATES && currentNode.nodeType === NODE_TYPE.text) {
      /* Get the element's text content */
      content = currentNode.textContent;
      arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
        content = stringReplace(content, expr, ' ');
      });
      if (currentNode.textContent !== content) {
        arrayPush(DOMPurify.removed, {
          element: currentNode.cloneNode()
        });
        currentNode.textContent = content;
      }
    }
    /* Execute a hook if present */
    _executeHooks(hooks.afterSanitizeElements, currentNode, null);
    return false;
  };
  /**
   * _isValidAttribute
   *
   * @param lcTag Lowercase tag name of containing element.
   * @param lcName Lowercase attribute name.
   * @param value Attribute value.
   * @return Returns true if `value` is valid, otherwise false.
   */
  // eslint-disable-next-line complexity
  const _isValidAttribute = function _isValidAttribute(lcTag, lcName, value) {
    /* Make sure attribute cannot clobber */
    if (SANITIZE_DOM && (lcName === 'id' || lcName === 'name') && (value in document || value in formElement)) {
      return false;
    }
    /* Allow valid data-* attributes: At least one character after "-"
        (https://html.spec.whatwg.org/multipage/dom.html#embedding-custom-non-visible-data-with-the-data-*-attributes)
        XML-compatible (https://html.spec.whatwg.org/multipage/infrastructure.html#xml-compatible and http://www.w3.org/TR/xml/#d0e804)
        We don't need to check the value; it's always URI safe. */
    if (ALLOW_DATA_ATTR && !FORBID_ATTR[lcName] && regExpTest(DATA_ATTR, lcName)) ; else if (ALLOW_ARIA_ATTR && regExpTest(ARIA_ATTR, lcName)) ; else if (!ALLOWED_ATTR[lcName] || FORBID_ATTR[lcName]) {
      if (
      // First condition does a very basic check if a) it's basically a valid custom element tagname AND
      // b) if the tagName passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
      // and c) if the attribute name passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.attributeNameCheck
      _isBasicCustomElement(lcTag) && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, lcTag) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(lcTag)) && (CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.attributeNameCheck, lcName) || CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.attributeNameCheck(lcName)) ||
      // Alternative, second condition checks if it's an `is`-attribute, AND
      // the value passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
      lcName === 'is' && CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, value) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(value))) ; else {
        return false;
      }
      /* Check value is safe. First, is attr inert? If so, is safe */
    } else if (URI_SAFE_ATTRIBUTES[lcName]) ; else if (regExpTest(IS_ALLOWED_URI$1, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if ((lcName === 'src' || lcName === 'xlink:href' || lcName === 'href') && lcTag !== 'script' && stringIndexOf(value, 'data:') === 0 && DATA_URI_TAGS[lcTag]) ; else if (ALLOW_UNKNOWN_PROTOCOLS && !regExpTest(IS_SCRIPT_OR_DATA, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if (value) {
      return false;
    } else ;
    return true;
  };
  /**
   * _isBasicCustomElement
   * checks if at least one dash is included in tagName, and it's not the first char
   * for more sophisticated checking see https://github.com/sindresorhus/validate-element-name
   *
   * @param tagName name of the tag of the node to sanitize
   * @returns Returns true if the tag name meets the basic criteria for a custom element, otherwise false.
   */
  const _isBasicCustomElement = function _isBasicCustomElement(tagName) {
    return tagName !== 'annotation-xml' && stringMatch(tagName, CUSTOM_ELEMENT);
  };
  /**
   * _sanitizeAttributes
   *
   * @protect attributes
   * @protect nodeName
   * @protect removeAttribute
   * @protect setAttribute
   *
   * @param currentNode to sanitize
   */
  const _sanitizeAttributes = function _sanitizeAttributes(currentNode) {
    /* Execute a hook if present */
    _executeHooks(hooks.beforeSanitizeAttributes, currentNode, null);
    const {
      attributes
    } = currentNode;
    /* Check if we have attributes; if not we might have a text node */
    if (!attributes || _isClobbered(currentNode)) {
      return;
    }
    const hookEvent = {
      attrName: '',
      attrValue: '',
      keepAttr: true,
      allowedAttributes: ALLOWED_ATTR,
      forceKeepAttr: undefined
    };
    let l = attributes.length;
    /* Go backwards over all attributes; safely remove bad ones */
    while (l--) {
      const attr = attributes[l];
      const {
        name,
        namespaceURI,
        value: attrValue
      } = attr;
      const lcName = transformCaseFunc(name);
      const initValue = attrValue;
      let value = name === 'value' ? initValue : stringTrim(initValue);
      /* Execute a hook if present */
      hookEvent.attrName = lcName;
      hookEvent.attrValue = value;
      hookEvent.keepAttr = true;
      hookEvent.forceKeepAttr = undefined; // Allows developers to see this is a property they can set
      _executeHooks(hooks.uponSanitizeAttribute, currentNode, hookEvent);
      value = hookEvent.attrValue;
      /* Full DOM Clobbering protection via namespace isolation,
       * Prefix id and name attributes with `user-content-`
       */
      if (SANITIZE_NAMED_PROPS && (lcName === 'id' || lcName === 'name')) {
        // Remove the attribute with this value
        _removeAttribute(name, currentNode);
        // Prefix the value and later re-create the attribute with the sanitized value
        value = SANITIZE_NAMED_PROPS_PREFIX + value;
      }
      /* Work around a security issue with comments inside attributes */
      if (SAFE_FOR_XML && regExpTest(/((--!?|])>)|<\/(style|title)/i, value)) {
        _removeAttribute(name, currentNode);
        continue;
      }
      /* Did the hooks approve of the attribute? */
      if (hookEvent.forceKeepAttr) {
        continue;
      }
      /* Did the hooks approve of the attribute? */
      if (!hookEvent.keepAttr) {
        _removeAttribute(name, currentNode);
        continue;
      }
      /* Work around a security issue in jQuery 3.0 */
      if (!ALLOW_SELF_CLOSE_IN_ATTR && regExpTest(/\/>/i, value)) {
        _removeAttribute(name, currentNode);
        continue;
      }
      /* Sanitize attribute content to be template-safe */
      if (SAFE_FOR_TEMPLATES) {
        arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
          value = stringReplace(value, expr, ' ');
        });
      }
      /* Is `value` valid for this attribute? */
      const lcTag = transformCaseFunc(currentNode.nodeName);
      if (!_isValidAttribute(lcTag, lcName, value)) {
        _removeAttribute(name, currentNode);
        continue;
      }
      /* Handle attributes that require Trusted Types */
      if (trustedTypesPolicy && typeof trustedTypes === 'object' && typeof trustedTypes.getAttributeType === 'function') {
        if (namespaceURI) ; else {
          switch (trustedTypes.getAttributeType(lcTag, lcName)) {
            case 'TrustedHTML':
              {
                value = trustedTypesPolicy.createHTML(value);
                break;
              }
            case 'TrustedScriptURL':
              {
                value = trustedTypesPolicy.createScriptURL(value);
                break;
              }
          }
        }
      }
      /* Handle invalid data-* attribute set by try-catching it */
      if (value !== initValue) {
        try {
          if (namespaceURI) {
            currentNode.setAttributeNS(namespaceURI, name, value);
          } else {
            /* Fallback to setAttribute() for browser-unrecognized namespaces e.g. "x-schema". */
            currentNode.setAttribute(name, value);
          }
          if (_isClobbered(currentNode)) {
            _forceRemove(currentNode);
          } else {
            arrayPop(DOMPurify.removed);
          }
        } catch (_) {
          _removeAttribute(name, currentNode);
        }
      }
    }
    /* Execute a hook if present */
    _executeHooks(hooks.afterSanitizeAttributes, currentNode, null);
  };
  /**
   * _sanitizeShadowDOM
   *
   * @param fragment to iterate over recursively
   */
  const _sanitizeShadowDOM = function _sanitizeShadowDOM(fragment) {
    let shadowNode = null;
    const shadowIterator = _createNodeIterator(fragment);
    /* Execute a hook if present */
    _executeHooks(hooks.beforeSanitizeShadowDOM, fragment, null);
    while (shadowNode = shadowIterator.nextNode()) {
      /* Execute a hook if present */
      _executeHooks(hooks.uponSanitizeShadowNode, shadowNode, null);
      /* Sanitize tags and elements */
      _sanitizeElements(shadowNode);
      /* Check attributes next */
      _sanitizeAttributes(shadowNode);
      /* Deep shadow DOM detected */
      if (shadowNode.content instanceof DocumentFragment) {
        _sanitizeShadowDOM(shadowNode.content);
      }
    }
    /* Execute a hook if present */
    _executeHooks(hooks.afterSanitizeShadowDOM, fragment, null);
  };
  // eslint-disable-next-line complexity
  DOMPurify.sanitize = function (dirty) {
    let cfg = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let body = null;
    let importedNode = null;
    let currentNode = null;
    let returnNode = null;
    /* Make sure we have a string to sanitize.
      DO NOT return early, as this will return the wrong type if
      the user has requested a DOM object rather than a string */
    IS_EMPTY_INPUT = !dirty;
    if (IS_EMPTY_INPUT) {
      dirty = '<!-->';
    }
    /* Stringify, in case dirty is an object */
    if (typeof dirty !== 'string' && !_isNode(dirty)) {
      if (typeof dirty.toString === 'function') {
        dirty = dirty.toString();
        if (typeof dirty !== 'string') {
          throw typeErrorCreate('dirty is not a string, aborting');
        }
      } else {
        throw typeErrorCreate('toString is not a function');
      }
    }
    /* Return dirty HTML if DOMPurify cannot run */
    if (!DOMPurify.isSupported) {
      return dirty;
    }
    /* Assign config vars */
    if (!SET_CONFIG) {
      _parseConfig(cfg);
    }
    /* Clean up removed elements */
    DOMPurify.removed = [];
    /* Check if dirty is correctly typed for IN_PLACE */
    if (typeof dirty === 'string') {
      IN_PLACE = false;
    }
    if (IN_PLACE) {
      /* Do some early pre-sanitization to avoid unsafe root nodes */
      if (dirty.nodeName) {
        const tagName = transformCaseFunc(dirty.nodeName);
        if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
          throw typeErrorCreate('root node is forbidden and cannot be sanitized in-place');
        }
      }
    } else if (dirty instanceof Node) {
      /* If dirty is a DOM element, append to an empty document to avoid
         elements being stripped by the parser */
      body = _initDocument('<!---->');
      importedNode = body.ownerDocument.importNode(dirty, true);
      if (importedNode.nodeType === NODE_TYPE.element && importedNode.nodeName === 'BODY') {
        /* Node is already a body, use as is */
        body = importedNode;
      } else if (importedNode.nodeName === 'HTML') {
        body = importedNode;
      } else {
        // eslint-disable-next-line unicorn/prefer-dom-node-append
        body.appendChild(importedNode);
      }
    } else {
      /* Exit directly if we have nothing to do */
      if (!RETURN_DOM && !SAFE_FOR_TEMPLATES && !WHOLE_DOCUMENT &&
      // eslint-disable-next-line unicorn/prefer-includes
      dirty.indexOf('<') === -1) {
        return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(dirty) : dirty;
      }
      /* Initialize the document to work on */
      body = _initDocument(dirty);
      /* Check we have a DOM node from the data */
      if (!body) {
        return RETURN_DOM ? null : RETURN_TRUSTED_TYPE ? emptyHTML : '';
      }
    }
    /* Remove first element node (ours) if FORCE_BODY is set */
    if (body && FORCE_BODY) {
      _forceRemove(body.firstChild);
    }
    /* Get node iterator */
    const nodeIterator = _createNodeIterator(IN_PLACE ? dirty : body);
    /* Now start iterating over the created document */
    while (currentNode = nodeIterator.nextNode()) {
      /* Sanitize tags and elements */
      _sanitizeElements(currentNode);
      /* Check attributes next */
      _sanitizeAttributes(currentNode);
      /* Shadow DOM detected, sanitize it */
      if (currentNode.content instanceof DocumentFragment) {
        _sanitizeShadowDOM(currentNode.content);
      }
    }
    /* If we sanitized `dirty` in-place, return it. */
    if (IN_PLACE) {
      return dirty;
    }
    /* Return sanitized string or DOM */
    if (RETURN_DOM) {
      if (RETURN_DOM_FRAGMENT) {
        returnNode = createDocumentFragment.call(body.ownerDocument);
        while (body.firstChild) {
          // eslint-disable-next-line unicorn/prefer-dom-node-append
          returnNode.appendChild(body.firstChild);
        }
      } else {
        returnNode = body;
      }
      if (ALLOWED_ATTR.shadowroot || ALLOWED_ATTR.shadowrootmode) {
        /*
          AdoptNode() is not used because internal state is not reset
          (e.g. the past names map of a HTMLFormElement), this is safe
          in theory but we would rather not risk another attack vector.
          The state that is cloned by importNode() is explicitly defined
          by the specs.
        */
        returnNode = importNode.call(originalDocument, returnNode, true);
      }
      return returnNode;
    }
    let serializedHTML = WHOLE_DOCUMENT ? body.outerHTML : body.innerHTML;
    /* Serialize doctype if allowed */
    if (WHOLE_DOCUMENT && ALLOWED_TAGS['!doctype'] && body.ownerDocument && body.ownerDocument.doctype && body.ownerDocument.doctype.name && regExpTest(DOCTYPE_NAME, body.ownerDocument.doctype.name)) {
      serializedHTML = '<!DOCTYPE ' + body.ownerDocument.doctype.name + '>\n' + serializedHTML;
    }
    /* Sanitize final string template-safe */
    if (SAFE_FOR_TEMPLATES) {
      arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
        serializedHTML = stringReplace(serializedHTML, expr, ' ');
      });
    }
    return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(serializedHTML) : serializedHTML;
  };
  DOMPurify.setConfig = function () {
    let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    _parseConfig(cfg);
    SET_CONFIG = true;
  };
  DOMPurify.clearConfig = function () {
    CONFIG = null;
    SET_CONFIG = false;
  };
  DOMPurify.isValidAttribute = function (tag, attr, value) {
    /* Initialize shared config vars if necessary. */
    if (!CONFIG) {
      _parseConfig({});
    }
    const lcTag = transformCaseFunc(tag);
    const lcName = transformCaseFunc(attr);
    return _isValidAttribute(lcTag, lcName, value);
  };
  DOMPurify.addHook = function (entryPoint, hookFunction) {
    if (typeof hookFunction !== 'function') {
      return;
    }
    arrayPush(hooks[entryPoint], hookFunction);
  };
  DOMPurify.removeHook = function (entryPoint, hookFunction) {
    if (hookFunction !== undefined) {
      const index = arrayLastIndexOf(hooks[entryPoint], hookFunction);
      return index === -1 ? undefined : arraySplice(hooks[entryPoint], index, 1)[0];
    }
    return arrayPop(hooks[entryPoint]);
  };
  DOMPurify.removeHooks = function (entryPoint) {
    hooks[entryPoint] = [];
  };
  DOMPurify.removeAllHooks = function () {
    hooks = _createHooksMap();
  };
  return DOMPurify;
}
var purify = createDOMPurify();

const httpMessages$1 =
    {
        "401": "Access unauthorized",
        "403": "Access forbidden",
        "404": "Not found"
    };

class AlertDialog$1 {
    /**
     * Initialize a new alert dialog
     * @param parent
     * @param alertProps - Optional - properties such as scroll to error
     */
    constructor(parent, alertProps) {
        this.alertProps = Object.assign({
            /** When an alert is presented - focus occur */
            shouldFocus: true,
            /** When focus occur - scroll into that element in the view */
            preventScroll: false
        }, alertProps);

        // container
        this.container = document.createElement('div');
        this.container.className = "igv-ui-alert-dialog-container";
        parent.appendChild(this.container);
        this.container.setAttribute('tabIndex', '-1');

        // header
        const header = document.createElement('div');
        this.container.appendChild(header);

        this.errorHeadline = document.createElement('div');
        header.appendChild(this.errorHeadline);
        this.errorHeadline.textContent = '';

        // body container
        let bodyContainer = document.createElement('div');
        bodyContainer.className = 'igv-ui-alert-dialog-body';
        this.container.appendChild(bodyContainer);

        // body copy
        this.body = document.createElement('div');
        this.body.className = 'igv-ui-alert-dialog-body-copy';
        bodyContainer.appendChild(this.body);

        // ok container
        let ok_container = document.createElement('div');
        this.container.appendChild(ok_container);

        // ok
        this.ok = document.createElement('div');
        ok_container.appendChild(this.ok);
        this.ok.textContent = 'OK';

        const okHandler = () => {

            if (typeof this.callback === 'function') {
                this.callback("OK");
                this.callback = undefined;
            }
            this.body.innerHTML = '';
            this.container.style.display = 'none';
        };

        this.ok.addEventListener('click', event => {
            event.stopPropagation();
            okHandler();
        });

        this.container.addEventListener('keypress', event => {
            event.stopPropagation();
            if ('Enter' === event.key) {
                okHandler();
            }
        });

        makeDraggable$1(this.container, header);

        this.container.style.display = 'none';
    }

    present(alert, callback) {

        this.errorHeadline.textContent = alert.message ? 'ERROR' : '';
        let string = alert.message || alert;

        if (httpMessages$1.hasOwnProperty(string)) {
            string = httpMessages$1[string];
        }

        this.body.innerHTML = purify.sanitize(string);

        this.callback = callback;
        this.container.style.display = 'flex';
        if (this.alertProps.shouldFocus) {
            this.container.focus({ preventScroll: this.alertProps.preventScroll });
        }
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of ctx software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and ctx permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


const IGVGraphics = {

    configureHighDPICanvas: function (ctx, w, h) {

        const scaleFactor = window.devicePixelRatio;
        // const scaleFactor = 1

        ctx.canvas.style.width = (`${w}px`);
        ctx.canvas.width = Math.floor(scaleFactor * w);

        ctx.canvas.style.height = (`${h}px`);
        ctx.canvas.height = Math.floor(scaleFactor * h);

        ctx.scale(scaleFactor, scaleFactor);

    },

    setProperties: function (ctx, properties) {

        for (var key in properties) {
            if (properties.hasOwnProperty(key)) {
                var value = properties[key];
                ctx[key] = value;
            }
        }
    },

    strokeLine: function (ctx, x1, y1, x2, y2, properties) {

        x1 = Math.floor(x1) + 0.5;
        y1 = Math.floor(y1) + 0.5;
        x2 = Math.floor(x2) + 0.5;
        y2 = Math.floor(y2) + 0.5;

        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }

        ctx.beginPath();
        ctx.moveTo(x1, y1);
        ctx.lineTo(x2, y2);
        ctx.stroke();

        if (properties) ctx.restore();
    },

    fillRect: function (ctx, x, y, w, h, properties) {

        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }

        ctx.fillRect(x, y, w, h);

        if (properties) ctx.restore();
    },

    fillPolygon: function (ctx, x, y, properties) {
        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }
        doPath(ctx, x, y);
        ctx.fill();
        if (properties) ctx.restore();
    },

    strokePolygon: function (ctx, x, y, properties) {
        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }
        doPath(ctx, x, y);
        ctx.stroke();
        if (properties) ctx.restore();
    },

    fillText: function (ctx, text, x, y, properties, transforms) {

        if (properties || transforms) {
            ctx.save();
        }

        if (properties) {
            IGVGraphics.setProperties(ctx, properties);
        }

        if (transforms) {
            // Slow path with context saving and extra translate
            ctx.translate(x, y);

            for (var transform in transforms) {
                var value = transforms[transform];

                // TODO: Add error checking for robustness
                if (transform === 'translate') {
                    ctx.translate(value['x'], value['y']);
                }
                if (transform === 'rotate') {
                    ctx.rotate(value['angle'] * Math.PI / 180);
                }
            }

            ctx.fillText(text, 0, 0);
        } else {
            ctx.fillText(text, x, y);
        }

        if (properties || transforms) ctx.restore();
    },

    strokeText: function (ctx, text, x, y, properties, transforms) {


        if (properties || transforms) {
            ctx.save();
        }

        if (properties) {
            IGVGraphics.setProperties(ctx, properties);
        }

        if (transforms) {
            ctx.translate(x, y);

            for (var transform in transforms) {
                var value = transforms[transform];

                // TODO: Add error checking for robustness
                if (transform === 'translate') {
                    ctx.translate(value['x'], value['y']);
                }
                if (transform === 'rotate') {
                    ctx.rotate(value['angle'] * Math.PI / 180);
                }
            }

            ctx.strokeText(text, 0, 0);
        } else {
            ctx.strokeText(text, x, y);
        }

        if (properties || transforms) ctx.restore();
    },

    strokeCircle: function (ctx, x, y, radius, properties) {
        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }
        ctx.beginPath();
        ctx.arc(x, y, radius, 0, 2 * Math.PI);
        ctx.stroke();
        if (properties) ctx.restore();
    },

    fillCircle: function (ctx, x, y, radius, properties) {
        if (properties) {
            ctx.save();
            IGVGraphics.setProperties(ctx, properties);
        }
        ctx.beginPath();
        ctx.arc(x, y, radius, 0, 2 * Math.PI);
        ctx.fill();
        if (properties) ctx.restore();
    },

    drawArrowhead: function (ctx, x, y, size, lineWidth) {

        ctx.save();
        if (!size) {
            size = 5;
        }
        if (lineWidth) {
            ctx.lineWidth = lineWidth;
        }
        ctx.beginPath();
        ctx.moveTo(x, y - size / 2);
        ctx.lineTo(x, y + size / 2);
        ctx.lineTo(x + size, y);
        ctx.lineTo(x, y - size / 2);
        ctx.closePath();
        ctx.fill();
        ctx.restore();
    },

    dashedLine: function (ctx, x1, y1, x2, y2, dashLen, properties = {}) {
        if (dashLen === undefined) dashLen = 2;
        ctx.setLineDash([dashLen, dashLen]);
        IGVGraphics.strokeLine(ctx, x1, y1, x2, y2, properties);
        ctx.setLineDash([]);
    },

    roundRect: function (ctx, x, y, width, height, radius, fill, stroke) {

        if (typeof stroke == "undefined") {
            stroke = true;
        }
        if (typeof radius === "undefined") {
            radius = 5;
        }
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
        ctx.lineTo(x + radius, y + height);
        ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
        ctx.lineTo(x, y + radius);
        ctx.quadraticCurveTo(x, y, x + radius, y);
        ctx.closePath();
        if (stroke) {
            ctx.stroke();
        }
        if (fill) {
            ctx.fill();
        }
    },
    polygon: function (ctx, x, y, fill, stroke) {

        if (typeof stroke == "undefined") {
            stroke = true;
        }

        ctx.beginPath();
        var len = x.length;
        ctx.moveTo(x[0], y[0]);
        for (var i = 1; i < len; i++) {
            ctx.lineTo(x[i], y[i]);
            // this.moveTo(x[i], y[i]);
        }

        ctx.closePath();
        if (stroke) {
            ctx.stroke();
        }
        if (fill) {
            ctx.fill();
        }
    },

    drawRandomColorVerticalLines: ctx => {
        for (let x = 0; x < ctx.canvas.width; x++) {
            IGVGraphics.fillRect(ctx, x, 0, 1, ctx.canvas.height, { fillStyle: randomRGB$1(100, 250) });
        }
    },

    labelTransformWithContext: (ctx, exe) => {
        ctx.translate(exe, 0);
        ctx.scale(-1, 1);
        ctx.translate(-exe, 0);
    }

};

function doPath(ctx, x, y) {
    var i, len = x.length;
    ctx.beginPath();
    ctx.moveTo(x[0], y[0]);
    for (i = 1; i < len; i++) {
        ctx.lineTo(x[i], y[i]);
    }
    ctx.closePath();
}

const pairs =
    [
        ['A', 'T'],
        ['G', 'C'],
        ['Y', 'R'],
        ['W', 'S'],
        ['K', 'M'],
        ['D', 'H'],
        ['B', 'V']
    ];

const complements = new Map();
for (let p of pairs) {
    const p1 = p[0];
    const p2 = p[1];
    complements.set(p1, p2);
    complements.set(p2, p1);
    complements.set(p1.toLowerCase(), p2.toLowerCase());
    complements.set(p2.toLowerCase(), p1.toLowerCase());
}

function complementBase(base) {
    return complements.has(base) ? complements.get(base) : base
}

function complementSequence(sequence) {
    let comp = '';
    for (let base of sequence) {
        comp += complements.has(base) ? complements.get(base) : base;
    }
    return comp
}

function reverseComplementSequence(sequence) {

    let comp = '';
    let idx = sequence.length;
    while (idx-- > 0) {
        const base = sequence[idx];
        comp += complements.has(base) ? complements.get(base) : base;
    }
    return comp
}

/**
 * Object for chromosome meta-data
 */


class Chromosome {
    constructor(name, order, bpLength, altNames) {
        this.name = name;
        this.order = order;
        this.bpLength = bpLength;
        this.altNames = altNames;
    }

    getAltName(key) {
        return this.altNames && this.altNames.has(key) ? this.altNames.get(key) : this.name
    }
}

class GenomicInterval {

    constructor(chr, start, end, features) {
        this.chr = chr;
        this.start = start;
        this.end = end;
        this.features = features;
    }

    contains(chr, start, end) {
        return this.chr === chr &&
            this.start <= start &&
            this.end >= end
    }

    containsRange(range) {
        return this.chr === range.chr &&
            this.start <= range.start &&
            this.end >= range.end
    }

    get locusString() {
        return `${this.chr}:${this.start + 1}-${this.end}`
    }
}

class SequenceInterval extends GenomicInterval {

    constructor(chr, start, end, features) {
        super(chr, start, end, features);
    }

    getSequence(start, end) {
        if (start < this.start || end > this.end) {
            return null
        }
        const offset = start - this.start;
        const n = end - start;
        const seq = this.features ? this.features.substring(offset, offset + n) : null;
        return seq
    }

    hasSequence(start, end) {
        return start >= this.start && end <= this.end
    }

}

const splitLines$2 = splitLines$3;

const reservedProperties$1 = new Set(['fastaURL', 'indexURL', 'cytobandURL', 'indexed']);

class NonIndexedFasta {

    #chromosomeNames
    chromosomes = new Map()
    sequences = new Map()

    constructor(reference) {

        this.fastaURL = reference.fastaURL;
        this.withCredentials = reference.withCredentials;


        // Build a track-like config object from the referenceObject
        const config = {};
        for (let key in reference) {
            if (reference.hasOwnProperty(key) && !reservedProperties$1.has(key)) {
                config[key] = reference[key];
            }
        }
        this.config = config;
    }


    async init() {
        return this.loadAll()
    }

    getSequenceRecord(chr) {
        return this.chromosomes.get(chr)
    }

    get chromosomeNames() {
        if (!this.#chromosomeNames) {
            this.#chromosomeNames = Array.from(this.chromosomes.keys());
        }
        return this.#chromosomeNames
    }

    getFirstChromosomeName() {
        return this.chromosomeNames[0]
    }

    async getSequence(chr, start, end) {

        if (this.sequences.size === 0) {
            await this.loadAll();
        }

        if (!this.sequences.has(chr)) {
            return undefined
        }

        let seqSlice = this.sequences.get(chr).find(ss => ss.contains(start, end));
        if (!seqSlice) {
            seqSlice = this.sequences.get(chr).find(ss => ss.overlaps(start, end));
            if (!seqSlice) {
                return undefined
            }
        }

        start -= seqSlice.offset;
        end -= seqSlice.offset;

        let prefix = "";
        if (start < 0) {
            for (let i = start; i < Math.min(end, 0); i++) {
                prefix += "*";
            }
        }

        if (end <= 0) {
            return prefix
        }

        const seq = seqSlice.sequence;
        const seqEnd = Math.min(end, seq.length);
        return prefix + seq.substring(start, seqEnd)
    }

    async loadAll() {


        const pushChromosome = (current, order) => {
            const length = current.length || (current.offset + current.seq.length);
            if (!chrNameSet.has(current.chr)) {
                this.sequences.set(current.chr, []);
                this.chromosomes.set(current.chr, new Chromosome(current.chr, order, length));
                chrNameSet.add(current.chr);
            } else {
                const c = this.chromosomes.get(current.chr);
                c.bpLength = Math.max(c.bpLength, length);
            }
            this.sequences.get(current.chr).push(new SequenceSlice(current.offset, current.seq));
        };

        let data;
        if (isDataURL(this.fastaURL)) {
            let bytes = decodeDataURI$1(this.fastaURL);
            data = "";
            for (let b of bytes) {
                data += String.fromCharCode(b);
            }
        } else {
            data = await igvxhr$1.load(this.fastaURL, buildOptions(this.config));
        }

        const chrNameSet = new Set();
        const lines = splitLines$2(data);
        let order = 0;
        let current = {};
        for (let nextLine of lines) {
            if (nextLine.startsWith("#") || nextLine.length === 0) ; else if (nextLine.startsWith(">")) {
                // Start the next sequence
                if (current.seq && current.seq.length > 0) {
                    pushChromosome(current, order++);
                }
                current.seq = "";

                const parts = nextLine.substr(1).split(/\s+/);


                // Check for @len= token, which is a non-standard extension supporting igv-reports.
                if (nextLine.includes("@len=")) {
                    const nameParts = parts[0].split(':');
                    current.chr = nameParts[0];
                    if (nameParts.length > 1 && nameParts[1].indexOf('-') > 0) {

                        const locusParts = nameParts[1].split('-');
                        if (locusParts.length === 2 &&
                            /^[0-9]+$/.test(locusParts[0]) &&
                            /^[0-9]+$/.test(locusParts[1])) ;
                        const from = Number.parseInt(locusParts[0]);
                        Number.parseInt(locusParts[1]);
                        current.offset = from - 1;

                        // Check for chromosome length token
                        if (parts.length > 1 && parts[1].startsWith("@len=")) {
                            try {
                                current.length = parseInt(parts[1].trim().substring(5));
                            } catch (e) {
                                current.length = undefined;
                                console.error(`Error parsing sequence length for ${nextLine}`);
                            }
                        } else {
                            current.length = undefined;
                        }
                    }
                } else {
                    // No special tokens, a standard FASTA header
                    current.chr = parts[0];
                    current.offset = 0;
                }

            } else {
                // Not a header or comment, so it must be sequence data
                current.seq += nextLine;
            }
        }

        // Handle the last sequence
        if (current.seq && current.seq.length > 0) {
            pushChromosome(current, order);
        }

    }

    /**
     * Return the first cached interval containing the specified region, or undefined if no interval is found.
     *
     * @param chr
     * @param start
     * @param end
     * @returns a SequenceInterval or undefined
     */
    getSequenceInterval(chr, start, end) {

        const slices = this.sequences.get(chr);
        if (!slices) return undefined

        for (let sequenceSlice of slices) {
            const seq = sequenceSlice.sequence;
            const seqStart = sequenceSlice.offset;
            const seqEnd = seqStart + seq.length;

            if (seqStart <= start && seqEnd >= end) {
                return new SequenceInterval(chr, seqStart, seqEnd, seq)
            }
        }
        return undefined

    }
}


class SequenceSlice {

    constructor(offset, sequence) {
        this.offset = offset;
        this.sequence = sequence;
    }

    contains(start, end) {
        return this.offset <= start && this.end >= end
    }

    overlaps(start, end) {
        return this.offset < end && this.end > start
    }

    get end() {
        return this.offset + this.sequence.length
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const splitLines$1 = splitLines$3;

const reservedProperties = new Set(['fastaURL', 'indexURL', 'compressedIndexURL', 'cytobandURL', 'indexed']);

class FastaSequence {

    #chromosomeNames
    chromosomes = new Map()

    constructor(reference) {

        this.file = reference.fastaURL;
        this.indexFile = reference.indexURL;
        this.compressedIndexFile = reference.compressedIndexURL;
        this.withCredentials = reference.withCredentials;


        // Build a track-like config object from the referenceObject
        const config = {};
        for (let key in reference) {
            if (reference.hasOwnProperty(key) && !reservedProperties.has(key)) {
                config[key] = reference[key];
            }
        }
        this.config = config;
    }

    getSequenceRecord(chr) {
        return this.chromosomes.get(chr)
    }

    async init() {
        return this.getIndex()
    }

    get chromosomeNames() {
        if(!this.#chromosomeNames) {
            this.#chromosomeNames = Array.from(this.chromosomes.keys());
        }
        return this.#chromosomeNames
    }

    getFirstChromosomeName() {
        return this.chromosomeNames[0]
    }

    async getIndex() {

        if (this.index) {
            return this.index
        } else {
            const data = await igvxhr$1.load(this.indexFile, buildOptions(this.config));
            const lines = splitLines$1(data);
            const len = lines.length;
            let lineNo = 0;
            let order = 0;
            this.index = {};

            while (lineNo < len) {
                const tokens = lines[lineNo++].split("\t");
                const nTokens = tokens.length;

                if (nTokens === 5) {
                    // Parse the index line.
                    const chr = tokens[0];
                    const size = parseInt(tokens[1]);
                    const position = parseInt(tokens[2]);
                    const basesPerLine = parseInt(tokens[3]);
                    const bytesPerLine = parseInt(tokens[4]);

                    const indexEntry = {
                        size: size,
                        position: position,
                        basesPerLine: basesPerLine,
                        bytesPerLine: bytesPerLine
                    };

                    this.index[chr] = indexEntry;
                    this.chromosomes.set(chr, new Chromosome(chr, order++, size));
                }
            }
            return this.index
        }
    }


    //Code is losely based on https://github.com/GMOD/bgzf-filehandle
    //Reworked however in orde to work with the igvxhr interface for loading files
    //Additionally, replaced calls to the Long.js interface with standard JS calls for ArrayBuffers and the associated views
    //
    //The compressed index is an array of blocks, with each block being a pair: compressed-position & uncompressed-position (both in bytes)
    async getCompressedIndex() {
        const GZI_NUM_BYTES_OFFSET = 8;
        const GZI_NUM_BYTES_BLOCK = 8;
        if (this.compressedIndex) {
            return this.compressedIndex
        }
        if (!this.compressedIndexFile) {
            this.compressedIndex = [];
            return this.compressedIndex
        }
        //In contrast to the 'normal' reference (for which the index is chromosome based), this index is block-based
        //As such there is not need to make it a hash. An array is sufficient.
        this.compressedIndex = [];
        const gziData = await igvxhr$1.loadArrayBuffer(this.compressedIndexFile, buildOptions(this.config));
        const givenFileSize = gziData.byteLength;
        if (givenFileSize < GZI_NUM_BYTES_OFFSET) {
            return this.compressedIndex
        }
        //First 8 bytes are a little endian unsigned bigint (64bit), indicating the number of blocks in the index.
        const numBlocksBuffer = gziData.slice(0, GZI_NUM_BYTES_OFFSET);
        const numBlocks = Number((new DataView(numBlocksBuffer)).getBigUint64(0, true));
        //The remainder of the gzi content are pairs of little endian unsigned bigint (64bit) numbers.
        //The first of the pair is the compressed position of a block
        //The second of the pair is the uncompressed position of a block

        //Sanity check:
        //Is the size of the array-buffer (of the entire file) correct with regards to the number of blocks detailled by the first 8 bytes of the file?
        //Total file-size should be:
        // 8 + 2*(num_entries*8) bytes, with the first 8 bytes indicating the number of entries
        const expectedFileSize = GZI_NUM_BYTES_OFFSET + numBlocks * 2 * GZI_NUM_BYTES_BLOCK;
        if (givenFileSize != expectedFileSize) {
            return this.compressedIndex
        }

        //Push the first block to the index: the first block always has positions 0 for both the compressed and uncompressed file
        this.compressedIndex.push([0, 0]);

        //Further process all the blocks of the GZI index, and keep them in memory
        for (let blockNumber = 0; blockNumber < numBlocks; blockNumber++) {
            const bufferBlockStart = GZI_NUM_BYTES_OFFSET + blockNumber * 2 * GZI_NUM_BYTES_BLOCK;
            const bufferBlockEnd = GZI_NUM_BYTES_OFFSET + blockNumber * 2 * GZI_NUM_BYTES_BLOCK + 2 * GZI_NUM_BYTES_BLOCK;
            const bufferBlock = gziData.slice(bufferBlockStart, bufferBlockEnd);
            const viewBlock = new DataView(bufferBlock);
            const compressedPosition = Number(viewBlock.getBigUint64(0, true));  //First 8 bytes
            const uncompressedPosition = Number(viewBlock.getBigUint64(GZI_NUM_BYTES_BLOCK, true)); //Last 8 bytes
            this.compressedIndex.push([compressedPosition, uncompressedPosition]);
        }
        return this.compressedIndex
    }

    //The Fasta-index gives a byte-position of the chromosomal sequences within the FASTA file.
    //These locations need to be remapped to the locations within the zipped reference genome, using the GZI index
    //This function provides this functionality by 
    //1) taking the indicated start/stop byte locations within the UNCOMPRESSED FASTA file
    //2) remapping these byte locations to the correct blocks (and associated positions) within the COMPRESSED FASTA file
    //Subsequently, the calling method can then extract the correct blocks from the compressed FASTA files and uncompressed the data
    async getRelevantCompressedBlockNumbers(queryPositionStart, queryPositionEnd) {
        const UNCOMPRESSED_POSITION = 1;
        //Fallback for impossible values
        if (queryPositionStart < 0 || queryPositionEnd < 0 || queryPositionEnd < queryPositionStart) {
            return []
        }
        //Ensure compressed index is loaded
        await this.getCompressedIndex();
        let result = [];
        //Now search for the correct block-numbers (going from 0 to length(compressed-index)) which overlap with the provided byte-positions
        const lowestBlockNumber = 0;
        const highestBlockNumber = this.compressedIndex.length - 1;
        //Failsafe if for some reason the compressed index wasn't loaded or doesn't contain any data
        if (this.compressedIndex.length == 0) {
            return []
        }
        //Failsafe: if the queryPositionStart is greater than the uncompressed-position of the final block,
        //then this final block is the only possible result
        if (queryPositionStart > (this.compressedIndex)[highestBlockNumber][UNCOMPRESSED_POSITION]) {
            return [highestBlockNumber]
        }

        //Rather than doing a linear search over all blocks, a binary search is done for speed considerations
        //We are searching for the highest block number for which its position is smaller than the query start position
        //Afterwards we will simply expand the blocks until the entire query range is covered
        let searchLow = lowestBlockNumber;
        let searchHigh = highestBlockNumber;
        let searchPosition = Math.floor(this.compressedIndex.length / 2);
        let maxIterations = this.compressedIndex.length + 1;
        let solutionFound = false;
        //instead of doing a while(true), this for-loop prevents eternal loops in case of issues
        for (let iteration = 0; iteration < maxIterations; iteration++) {
            const searchUncompressedPosition = (this.compressedIndex)[searchPosition][UNCOMPRESSED_POSITION];
            const nextSearchUncompressedPosition = (searchPosition < (this.compressedIndex.length - 1)) ? (this.compressedIndex)[searchPosition + 1][UNCOMPRESSED_POSITION] : Infinity;
            //The query position lies within the current search block
            if (searchUncompressedPosition <= queryPositionStart && nextSearchUncompressedPosition > queryPositionStart) {
                solutionFound = true;
                break //searchPosition is the correct block number index
            }
            //Current block lies before the query position
            else if (searchUncompressedPosition < queryPositionStart) {
                searchLow = searchPosition + 1;
            }
            //Current block lies after the query position
            else {
                searchHigh = searchPosition - 1;
            }
            searchPosition = Math.ceil((searchHigh - searchLow) / 2) + searchLow;
        }
        //If for some reason the binary search did not reveal a correct block index, then we return the empty result
        if (!solutionFound) {
            return []
        }

        //Now extend the result by adding additional blocks until the entire query range is covered
        result.push(searchPosition);
        for (let blockIndex = searchPosition + 1; blockIndex < this.compressedIndex.length; blockIndex++) {
            result.push(blockIndex);
            const blockUncompressedPosition = (this.compressedIndex)[blockIndex][UNCOMPRESSED_POSITION];
            if (blockUncompressedPosition >= queryPositionEnd) {
                break
            }
        }

        //It is possible that the query end position lies AFTER the start of the final block
        //If this is the case, we add a 'fake' negative index which will be interpreted by the loadAndUncompressBlocks method as an indicator
        //to read until the end of the file 
        const finalRelevantBlock = result[result.length - 1];
        const finalIndexBlock = this.compressedIndex.length - 1;
        if (finalRelevantBlock === finalIndexBlock && (this.compressedIndex)[finalRelevantBlock][UNCOMPRESSED_POSITION] < queryPositionEnd) {
            result.push(-1);
        }

        return result
    }


    //Load the content from the blockIndices.
    //This is done on a per-block basis
    //Content of the first block will be trimmed in order to match the expected offset
    async loadAndUncompressBlocks(blockIndices, startByte) {
        const COMPRESSED_POSITION = 0;
        const UNCOMPRESSED_POSITION = 1;
        //Normally the compressed index should already exist, we're just makeing sure here
        await this.getCompressedIndex();

        if (blockIndices.length == 0) {
            return ""
        }

        //Storing data in seperate array with indices in order to assert order due to async behaviour of loops
        let resultCache = Array(blockIndices.length - 1);
        for (let i = 0; i < blockIndices.length - 1; i++) {
            const currentBlockNumber = blockIndices[i];
            const currentBlockInfo = (this.compressedIndex)[currentBlockNumber];
            const currentBlockCompressedPosition = currentBlockInfo[COMPRESSED_POSITION];

            const nextBlockNumber = blockIndices[i + 1];
            let compressedBytes = [];
            if (nextBlockNumber != -1) {  //default : read current entire block only
                const nextBlockInfo = (this.compressedIndex)[nextBlockNumber];
                const nextBlockCompressedPosition = nextBlockInfo[COMPRESSED_POSITION];
                const compressedLength = nextBlockCompressedPosition - currentBlockCompressedPosition;
                compressedBytes = await igvxhr$1.loadArrayBuffer(this.file, buildOptions(this.config, {
                    range: {
                        start: currentBlockCompressedPosition,
                        size: compressedLength
                    }
                }));
            } else {   // special case for query within final block: read until the end of the file
                compressedBytes = await igvxhr$1.loadArrayBuffer(this.file, buildOptions(this.config, {
                    range: {
                        start: currentBlockCompressedPosition
                    }
                }));
            }
            //now unzip the compressed bytes, and store them in the resultCache
            const uncompressedBytes = await unbgzf$1(compressedBytes);
            resultCache[i] = uncompressedBytes;
        }

        //Iterate over the result cache, create sequences from the data, and create a full sequence string from the data
        let result = "";
        for (let i = 0; i < resultCache.length; i++) {
            for (let j = 0; j < resultCache[i].length; j++) {
                const c = String.fromCharCode(resultCache[i][j]);
                result = result + c;
            }
        }

        //postprocess this data: because entire blocks are read we need to remove the first N bases of the first used block, 
        //which are not included in the original query positions
        const firstBlockInfo = (this.compressedIndex)[blockIndices[0]];
        const offset = startByte - firstBlockInfo[UNCOMPRESSED_POSITION];
        result = result.substring(offset);

        return result
    }


    async readSequence(chr, qstart, qend) {

        await this.getIndex();
        await this.getCompressedIndex(); //This will work even if no compressed index file is set

        const idxEntry = this.index[chr];
        if (!idxEntry) {
            // Tag interval with null so we don't try again
            // this.interval = new GenomicInterval(chr, qstart, qend, null)
            this.index[chr] = null;
            return null
        }

        const start = Math.max(0, qstart);    // qstart should never be < 0
        const end = Math.min(idxEntry.size, qend);
        const bytesPerLine = idxEntry.bytesPerLine;
        const basesPerLine = idxEntry.basesPerLine;
        const position = idxEntry.position;
        const nEndBytes = bytesPerLine - basesPerLine;
        const startLine = Math.floor(start / basesPerLine);
        const endLine = Math.floor(end / basesPerLine);
        const base0 = startLine * basesPerLine;   // Base at beginning of start line
        const offset = start - base0;
        const startByte = position + startLine * bytesPerLine + offset;
        const base1 = endLine * basesPerLine;
        const offset1 = end - base1;
        const endByte = position + endLine * bytesPerLine + offset1 - 1;
        const byteCount = endByte - startByte + 1;

        if (byteCount <= 0) {
            console.error("No sequence for " + chr + ":" + qstart + "-" + qend);
            return null
        }

        //If the compressed index file is set, then we are dealing with a compressed genome sequence
        //The selection of startByte/endByte is done for the non-compressed genome sequence.
        //These need to be 'converted' to the correct byte positions in the compressed genome sequence,
        //by making use of the compressed index (GZI file)
        let allBytes;
        if (!this.compressedIndexFile) {
            allBytes = await igvxhr$1.load(this.file, buildOptions(this.config, {
                range: {
                    start: startByte,
                    size: byteCount
                }
            }));
        } else {
            let relevantBlockIndices = await this.getRelevantCompressedBlockNumbers(startByte, endByte);
            if (relevantBlockIndices.length === 0) {
                return null
            }
            allBytes = await this.loadAndUncompressBlocks(relevantBlockIndices, startByte);
        }

        if (!allBytes) {
            return null
        }

        let nBases,
            seqBytes = "",
            srcPos = 0,
            allBytesLength = allBytes.length;

        if (offset > 0) {
            nBases = Math.min(end - start, basesPerLine - offset);
            seqBytes += allBytes.substr(srcPos, nBases);
            srcPos += (nBases + nEndBytes);
        }

        while (srcPos < allBytesLength) {
            nBases = Math.min(basesPerLine, allBytesLength - srcPos);
            seqBytes += allBytes.substr(srcPos, nBases);
            srcPos += (nBases + nEndBytes);
        }

        return seqBytes

    }
}

const splitLines = splitLines$3;

/**
 * Represents a reference object created from a ChromSizes file.  This is unusual, primarily for testing.
 */
class ChromSizes {

    #chromosomeNames
    chromosomes = new Map()

    constructor(url) {
        this.url = url;
    }

    async init() {
        return this.loadAll()
    }

    getSequenceRecord(chr) {
        return this.chromosomes.get(chr)
    }

    get chromosomeNames() {
        if(!this.#chromosomeNames) {
            this.#chromosomeNames = Array.from(this.chromosomes.keys());
        }
    }

    async getSequence(chr, start, end) {
        return null // TODO -- return array of "N"s?
    }

    async loadAll() {

        let data;
        if (isDataURL(this.url)) {
            let bytes = decodeDataURI$1(this.fastaURL);
            data = "";
            for (let b of bytes) {
                data += String.fromCharCode(b);
            }
        } else {
            data = await igvxhr$1.load(this.url, {});
        }

        const lines = splitLines(data);
        let order = 0;
        for (let nextLine of lines) {
            const tokens = nextLine.split('\t');
            if(tokens.length > 1) {
                const chrLength = Number.parseInt(tokens[1]);
                const chromosome = new Chromosome(tokens[0], order++, chrLength);
                this.chromosomes.set(tokens[0], chromosome);
            }
        }
    }

}

async function loadChromSizes(url) {

    const chromosomeSizes = new Map();

    let data;
    if (isDataURL(url)) {
        let bytes = decodeDataURI$1(url);
        data = "";
        for (let b of bytes) {
            data += String.fromCharCode(b);
        }
    } else {
        data = await igvxhr$1.load(url, {});
    }

    const lines = splitLines(data);
    let order = 0;
    for (let nextLine of lines) {
        const tokens = nextLine.split('\t');
        if(tokens.length > 1) {
            const chrLength = Number.parseInt(tokens[1]);
            chromosomeSizes.set(tokens[0], new Chromosome(tokens[0], order++, chrLength));
        }
    }
    return chromosomeSizes
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */



class BinaryParser$1 {
    constructor(dataView, littleEndian = true) {

        this.littleEndian = littleEndian;
        this.position = 0;
        this.view = dataView;
        this.length = dataView.byteLength;
    }

    /**
     * Print the first "n" bytes to the console.  Used for debugging.
     * @param n
     */
    dumpBytes (n = 100) {
        const pos = this.position;
        const bytes = [];
        for(let i=0; i<= n; i++) {
            bytes.push(this.getByte());
        }
        this.setPosition(pos);
    }

    setPosition(position) {
        this.position = position;
    }

    available() {
        return this.length - this.position
    }

    remLength() {
        return this.length - this.position
    }

    hasNext() {
        return this.position < this.length - 1
    }

    getByte() {
        const retValue = this.view.getUint8(this.position, this.littleEndian);
        this.position++;
        return retValue
    }

    getShort() {
        const retValue = this.view.getInt16(this.position, this.littleEndian);
        this.position += 2;
        return retValue
    }

    getUShort() {
        const retValue = this.view.getUint16(this.position, this.littleEndian);
        this.position += 2;
        return retValue
    }


    getInt() {
        const retValue = this.view.getInt32(this.position, this.littleEndian);
        this.position += 4;
        return retValue
    }


    getUInt() {
        const retValue = this.view.getUint32(this.position, this.littleEndian);
        this.position += 4;
        return retValue
    }

    getLong() {

        // DataView doesn't support long. So we'll try manually
        var b = [];
        b[0] = this.view.getUint8(this.position);
        b[1] = this.view.getUint8(this.position + 1);
        b[2] = this.view.getUint8(this.position + 2);
        b[3] = this.view.getUint8(this.position + 3);
        b[4] = this.view.getUint8(this.position + 4);
        b[5] = this.view.getUint8(this.position + 5);
        b[6] = this.view.getUint8(this.position + 6);
        b[7] = this.view.getUint8(this.position + 7);

        let value = 0;
        if (this.littleEndian) {
            for (let i = b.length - 1; i >= 0; i--) {
                value = (value * 256) + b[i];
            }
        } else {
            for (let i = 0; i < b.length; i++) {
                value = (value * 256) + b[i];
            }
        }
        this.position += 8;
        return value
    }

    getString(len) {

        let s = "";
        let c;
        while ((c = this.view.getUint8(this.position++)) !== 0) {
            s += String.fromCharCode(c);
            if (len && s.length === len) break
        }
        return s
    }

    getFixedLengthString(len) {

        let s = "";
        for (let i = 0; i < len; i++) {
            const c = this.view.getUint8(this.position++);
            if (c > 0) {
                s += String.fromCharCode(c);
            }
        }
        return s
    }

    getFloat() {

        var retValue = this.view.getFloat32(this.position, this.littleEndian);
        this.position += 4;
        return retValue


    }

    getDouble() {

        var retValue = this.view.getFloat64(this.position, this.littleEndian);
        this.position += 8;
        return retValue
    }

    skip(n) {
        this.position += n;
        return this.position
    }


    /**
     * Return a BGZip (bam and tabix) virtual pointer
     * TODO -- why isn't 8th byte used ?
     * TODO -- does endian matter here ?
     * @returns {*}
     */
    getVPointer() {

        var position = this.position,
            offset = (this.view.getUint8(position + 1) << 8) | (this.view.getUint8(position)),
            byte6 = ((this.view.getUint8(position + 6) & 0xff) * 0x100000000),
            byte5 = ((this.view.getUint8(position + 5) & 0xff) * 0x1000000),
            byte4 = ((this.view.getUint8(position + 4) & 0xff) * 0x10000),
            byte3 = ((this.view.getUint8(position + 3) & 0xff) * 0x100),
            byte2 = ((this.view.getUint8(position + 2) & 0xff)),
            block = byte6 + byte5 + byte4 + byte3 + byte2;
        this.position += 8;

        return new VPointer$1(block, offset)
    }
}

class VPointer$1 {
    constructor(block, offset) {
        this.block = block;
        this.offset = offset;
    }

    isLessThan(vp) {
        return this.block < vp.block ||
            (this.block === vp.block && this.offset < vp.offset)
    }

    isGreaterThan(vp) {
        return this.block > vp.block ||
            (this.block === vp.block && this.offset > vp.offset)
    }

    isEqualTo(vp) {
        return this.block === vp.block && this.offset === vp.offset
    }

    print() {
        return "" + this.block + ":" + this.offset
    }
}

/**
 * A UCSC BigBed B+ tree, used to support searching the "extra indexes".
 *
 * Nodes are loaded on demand during search, avoiding the need to read the entire tree into
 * memory.  Tree nodes can be scattered across the file, making loading the entire tree unfeasible in reasonable time.
 */
class BPTree {

    static magic = 2026540177
    littleEndian = true
    type = 'BPTree'          // Either BPTree or BPChromTree
    nodeCache = new Map()

    static async loadBpTree(path, config, startOffset, type, loader) {
        const bpTree = new BPTree(path, config, startOffset, type, loader);
        return bpTree.init()
    }

    constructor(path, config, startOffset, type, loader) {
        this.path = path;
        this.config = config;
        this.startOffset = startOffset;
        if(type) {
            this.type = type;
        }
        this.loader = loader || igvxhr$1;
    }

    async init() {
        const binaryParser = await this.#getParserFor(this.startOffset, 32);
        let magic = binaryParser.getInt();
        if(magic !== BPTree.magic) {
            binaryParser.setPosition(0);
            this.littleEndian = !this.littleEndian;
            binaryParser.littleEndian = this.littleEndian;
            magic = binaryParser.getInt();
            if(magic !== BPTree.magic) {
                throw Error(`Bad magic number ${magic}`)
            }
        }

        const blockSize = binaryParser.getInt();
        const keySize = binaryParser.getInt();
        const valSize = binaryParser.getInt();
        const itemCount = binaryParser.getLong();
        const reserved = binaryParser.getLong();
        const nodeOffset = this.startOffset + 32;
        this.header = {magic, blockSize, keySize, valSize, itemCount, reserved, nodeOffset};
        return this
    }

    getItemCount() {
        if(!this.header) {
            throw Error("Header not initialized")
        }
        return this.header.itemCount
    }

    async search(term) {

        if(!this.header) {
            await this.init();
        }

        const walkTreeNode = async (offset) => {

            const node = await this.readTreeNode(offset);

            if (node.type === 1) {
                // Leaf node
                for (let item of node.items) {
                    if (term === item.key) {
                        return item.value
                    }
                }
            } else {
                // Non leaf node

                // Read and discard the first key.
                let childOffset = node.items[0].offset;

                for (let i = 1; i < node.items.length; i++) {
                    const key = node.items[i].key;
                    if (term.localeCompare(key) < 0) {
                        break
                    }
                    childOffset = node.items[i].offset;
                }

                return walkTreeNode(childOffset)
            }
        };

        // Kick things off
        return walkTreeNode(this.header.nodeOffset)
    }

    async readTreeNode (offset)  {

        if (this.nodeCache.has(offset)) {
            return this.nodeCache.get(offset)
        } else {
            let binaryParser = await this.#getParserFor(offset, 4);
            const type = binaryParser.getByte();
            binaryParser.getByte();
            const count = binaryParser.getUShort();
            const items = [];

            const {keySize, valSize} = this.header;

            if (type === 1) {
                // Leaf node
                const size = count * (keySize + valSize);
                binaryParser = await this.#getParserFor(offset + 4, size);
                for (let i = 0; i < count; i++) {
                    const key = binaryParser.getFixedLengthString(keySize);
                    let value;
                    if(this.type === 'BPChromTree') {
                        const id = binaryParser.getInt();
                        const size = binaryParser.getInt();
                        value = {id, size};
                    } else {
                        const offset = binaryParser.getLong();
                        if (valSize === 16) {
                            const length = binaryParser.getLong();
                            value = {offset, length};
                        } else {
                            value = {offset};
                        }
                    }
                    items.push({key, value});
                }
            } else {
                // Non leaf node
                const size = count * (keySize + 8);
                binaryParser = await this.#getParserFor(offset + 4, size);

                for (let i = 0; i < count; i++) {
                    const key = binaryParser.getFixedLengthString(keySize);
                    const offset = binaryParser.getLong();
                    items.push({key, offset});
                }
            }

            const node = {type, count, items};
            this.nodeCache.set(offset, node);
            return node
        }
    }

    async #getParserFor(start, size) {
        try {
            const data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {range: {start, size}}));
            return new BinaryParser$1(new DataView(data), this.littleEndian)
        } catch (e) {
            console.error(e);
        }
    }

}

/**
 * Note: Some portions of this code adapated from the GMOD two-bit.js project, @Copyright (c) 2017 Robert Buels
 * * https://github.com/GMOD/twobit-js/blob/master/src/twoBitFile.ts*
 */

const twoBit = ['T', 'C', 'A', 'G'];
const byteTo4Bases = [];
for (let i = 0; i < 256; i++) {
    byteTo4Bases.push(
        twoBit[(i >> 6) & 3] +
        twoBit[(i >> 4) & 3] +
        twoBit[(i >> 2) & 3] +
        twoBit[i & 3],
    );
}
const maskedByteTo4Bases = byteTo4Bases.map(bases => bases.toLowerCase());

class TwobitSequence {

    littleEndian
    metaIndex = new Map()
    chromosomeNames

    constructor(config) {
        this.url = config.twoBitURL || config.fastaURL;
        this.config = config;
        if(config.twoBitBptURL) {
            this.bptURL = config.twoBitBptURL;
        }
     }

    async init() {
        if(this.bptURL) {
            this.index = await BPTree.loadBpTree(this.bptURL, this.config, 0);
        } else {
            const idx = await this._readIndex();
            this.index = {
                search: async (name) =>  {
                    return idx.has(name) ? {offset: idx.get(name)} : undefined
                }
            };
        }
    }

    async readSequence(seqName, regionStart, regionEnd) {

        if (!this.index) {
            await this.init();
        }

        const record = await this.getSequenceRecord(seqName);
        if (!record) {
            return null
        }

        if (regionStart < 0) {
            throw new TypeError('regionStart cannot be less than 0')
        }
        // end defaults to the end of the sequence
        if (regionEnd === undefined || regionEnd > record.dnaSize) {
            regionEnd = record.dnaSize;
        }

        const nBlocks = this._getOverlappingBlocks(
            regionStart,
            regionEnd,
            record.nBlocks
        );
        const maskBlocks = this._getOverlappingBlocks(
            regionStart,
            regionEnd,
            record.maskBlocks
        );

        const baseBytesOffset = Math.floor(regionStart / 4);
        const start = record.packedPos + baseBytesOffset;
        const size = Math.floor(regionEnd / 4) - baseBytesOffset + 1;

        const baseBytesArrayBuffer = await igvxhr$1.loadArrayBuffer(this.url, {range: {start, size}});
        const baseBytes = new Uint8Array(baseBytesArrayBuffer);

        let sequenceBases = '';
        for (let genomicPosition = regionStart; genomicPosition < regionEnd; genomicPosition += 1) {

            // function checks if  we are currently masked

            while (maskBlocks.length && maskBlocks[0].end <= genomicPosition) {
                maskBlocks.shift();
            }
            const baseIsMasked = maskBlocks[0] && maskBlocks[0].start <= genomicPosition && maskBlocks[0].end > genomicPosition;


            // process the N block if we have one.  Masked "N" ("n")  is not supported
            if (nBlocks[0] && genomicPosition >= nBlocks[0].start && genomicPosition < nBlocks[0].end) {
                const currentNBlock = nBlocks.shift();
                while (genomicPosition < currentNBlock.end && genomicPosition < regionEnd) {
                    sequenceBases += 'N';
                    genomicPosition++;
                }
                genomicPosition--;
            } else {
                const bytePosition = Math.floor(genomicPosition / 4) - baseBytesOffset;
                const subPosition = genomicPosition % 4;
                const byte = baseBytes[bytePosition];

                sequenceBases += baseIsMasked
                    ? maskedByteTo4Bases[byte][subPosition]
                    : byteTo4Bases[byte][subPosition];

            }
        }
        return sequenceBases
    }

    /**
     * Read the internal index of the 2bit file.  This is a list of sequence names and their offsets in the file.
     *
     * @returns {Promise<Map<any, any>>}
     * @private
     */
    async _readIndex() {

        const index = new Map();
        this.chromosomeNames = [];

        const loadRange = {start: 0, size: 64};
        let arrayBuffer = await igvxhr$1.loadArrayBuffer(this.url, {range: loadRange});
        let dataView = new DataView(arrayBuffer);

        let ptr = 0;
        const magicLE = dataView.getUint32(ptr, true);
        const magicBE = dataView.getUint32(ptr, false);
        ptr += 4;

        const magic = 0x1A412743;
        if (magicLE === magic) {
            this.littleEndian = true;
        } else if (magicBE === magic) {
            this.littleEndian = false;
        } else {
            throw Error(`Bad magic number ${magic}`)
        }

        this.version = dataView.getUint32(ptr, this.littleEndian);
        ptr += 4;

        this.sequenceCount = dataView.getUint32(ptr, this.littleEndian);
        ptr += 4;

        this.reserved = dataView.getUint32(ptr, this.littleEndian);
        ptr += 4;

        // Loop through sequences loading name and file offset.  We don't know the precise size in bytes in advance.
        let estSize;
        let binaryBuffer;

        let estNameLength = 20;
        for (let i = 0; i < this.sequenceCount; i++) {

            if (!binaryBuffer || binaryBuffer.available() < 1) {
                estSize = (this.sequenceCount - i) * estNameLength;
                binaryBuffer = await this._loadBinaryBuffer(ptr, estSize);
            }
            const len = binaryBuffer.getByte();
            ptr += 1;

            if (binaryBuffer.available() < len + 5) {
                estSize = (this.sequenceCount - i) * estNameLength + 100;
                binaryBuffer = await this._loadBinaryBuffer(ptr, estSize);
            }
            const name = binaryBuffer.getString(len);
            const offset = binaryBuffer.getUInt();
            ptr += len + 4;
            index.set(name, offset);

            estNameLength = Math.floor(estNameLength * (i / (i + 1)) + name.length / (i + 1));

            this.chromosomeNames.push(name);
        }
        return index
    }

    /**
     * Fetch the sequence metadata for the given seq name *
     *
     * @param seqName
     * @returns {Promise<void>}
     */
    async getSequenceRecord(seqName) {

        if (!this.metaIndex.has(seqName)) {

            if (!this.index) {
                throw Error("TwobitSequence object must be initialized before accessing sequence")
            }

            let result = await this.index.search(seqName);
            if (!result) {
                return
            }
            let offset = result.offset;

            // Read size of dna data & # of "N" blocks
            let size = 8;
            let binaryBuffer = await this._loadBinaryBuffer(offset, size);
            const dnaSize = binaryBuffer.getUInt();
            const nBlockCount = binaryBuffer.getUInt();
            offset += size;

            // Read "N" blocks and # of mask blocks
            size = nBlockCount * (4 + 4) + 4;
            binaryBuffer = await this._loadBinaryBuffer(offset, size);
            const nBlockStarts = [];
            for (let i = 0; i < nBlockCount; i++) {
                nBlockStarts.push(binaryBuffer.getUInt());
            }
            const nBlockSizes = [];
            for (let i = 0; i < nBlockCount; i++) {
                nBlockSizes.push(binaryBuffer.getUInt());
            }
            const maskBlockCount = binaryBuffer.getUInt();
            offset += size;

            // Read "mask" blocks
            size = maskBlockCount * (4 + 4) + 4;
            binaryBuffer = await this._loadBinaryBuffer(offset, size);
            const maskBlockStarts = [];
            for (let i = 0; i < maskBlockCount; i++) {
                maskBlockStarts.push(binaryBuffer.getUInt());
            }
            const maskBlockSizes = [];
            for (let i = 0; i < maskBlockCount; i++) {
                maskBlockSizes.push(binaryBuffer.getUInt());
            }

            //Transform "N" and "mask" block data into something more useful
            const nBlocks = [];
            for (let i = 0; i < nBlockCount; i++) {
                nBlocks.push(new Block$1(nBlockStarts[i], nBlockSizes[i]));
            }
            const maskBlocks = [];
            for (let i = 0; i < maskBlockCount; i++) {
                maskBlocks.push(new Block$1(maskBlockStarts[i], maskBlockSizes[i]));
            }

            const reserved = binaryBuffer.getUInt();
            if (reserved != 0) {
                throw Error("Bad 2-bit file")
            }
            offset += size;
            const packedPos = offset;

            const meta = {
                dnaSize,
                nBlocks,
                maskBlocks,
                packedPos,
                bpLength: dnaSize
            };
            this.metaIndex.set(seqName, meta);


        }
        return this.metaIndex.get(seqName)
    }

    /**
     * Return blocks overlapping the genome region [start, end]
     *
     * TODO -- optimize this, currently it uses linear search
     * * *
     * @param start
     * @param end
     * @param blocks
     * @returns {*[]}
     * @private
     */
    _getOverlappingBlocks(start, end, blocks) {

        const overlappingBlocks = [];
        for (let block of blocks) {
            if (block.start > end) {
                break
            } else if (block.end < start) {
                continue
            } else {
                overlappingBlocks.push(block);
            }
        }
        return overlappingBlocks
    }

    async _loadBinaryBuffer(start, size) {
        const arrayBuffer = await igvxhr$1.loadArrayBuffer(this.url, {range: {start, size}});
        return new BinaryParser$1(new DataView(arrayBuffer), this.littleEndian)
    }
}

class Block$1 {

    constructor(start, size) {
        this.start = start;
        this.size = size;
    }

    get end() {
        return this.start + this.size

    }
}

/**
 * Wrapper for a sequence loader that provides caching
 */

class CachedSequence {

    static #minQuerySize = 1e5
    #currentQuery
    #cachedIntervals = []
    #maxIntervals = 10   // TODO - this should be >= the number of viewports for multi-locus view

    constructor(sequenceReader, browser) {
        this.sequenceReader = sequenceReader;
        this.browser = browser;
    }

    get chromosomes() {
        return this.sequenceReader.chromosomes
    }

    async getSequenceRecord(chr) {
        return this.sequenceReader.getSequenceRecord(chr)
    }

    async getSequence(chr, start, end) {

        let interval = this.#cachedIntervals.find(i => i.contains(chr, start, end));
        if (!interval) {
            interval =  await this.#queryForSequence(chr, start, end);
            this.#trimCache(interval);
            this.#cachedIntervals.push(interval);
        }

        if (interval) {
            const offset = start - interval.start;
            const n = end - start;
            const seq = interval.features ? interval.features.substring(offset, offset + n) : null;
            return seq
        } else {
            return undefined
        }
    }

    #trimCache(interval) {
        // Filter out redundant (subsumed) cached intervals
        this.#cachedIntervals = this.#cachedIntervals.filter(i => !interval.contains(i));
        if (this.#cachedIntervals.length === this.#maxIntervals) {
            this.#cachedIntervals.shift();
        }

        // Filter out out-of-view cached intervals.  Don't try this if there are too many frames, inefficient
        if (this.browser && this.browser.referenceFrameList.length < 100) {
            this.#cachedIntervals = this.#cachedIntervals.filter(i => {
                const b = undefined !== this.browser.referenceFrameList.find(frame => frame.overlaps(i));
                return b;
            });
        }
    }

    /**
     * Return the first cached interval containing the specified region, or undefined if no interval is found.
     *
     * @param chr
     * @param start
     * @param end
     * @returns a SequenceInterval or undefined
     */
    getSequenceInterval(chr, start, end) {
        return this.#cachedIntervals.find(i => i.contains(chr, start, end))
    }

    /**
     * Query for a sequence.  Returns a promise that is resolved when the asynchronous call to read sequence returns.
     *
     * @param chr
     * @param start
     * @param end
     * @returns {Promise<sequence>}
     */
    async #queryForSequence(chr, start, end) {
        // Expand query, to minimum of 100kb
        let qstart = start;
        let qend = end;
        if ((end - start) < CachedSequence.#minQuerySize) {
            const w = (end - start);
            const center = Math.round(start + w / 2);
            qstart = Math.max(0, center - CachedSequence.#minQuerySize/2);
            qend = qstart + CachedSequence.#minQuerySize;
        }
        const interval = new SequenceInterval(chr, qstart, qend);

        if (this.#currentQuery && this.#currentQuery[0].contains(chr, start, end)) {
            return this.#currentQuery[1]
        } else {
            const queryPromise = new Promise(async (resolve, reject) => {
                interval.features = await this.sequenceReader.readSequence(chr, qstart, qend);
                resolve(interval);
            });
            this.#currentQuery = [interval, queryPromise];
            return queryPromise
        }
    }


    async init() {
        return this.sequenceReader.init()
    }

    get chromosomeNames() {
        return this.sequenceReader.chromosomeNames
    }

    getFirstChromosomeName() {
        return typeof this.sequenceReader.getFirstChromosomeName === 'function' ? this.sequenceReader.getFirstChromosomeName() : undefined
    }

    #isIntervalInView(interval) {
        this.browser.referenceFrameList;
    }
}

/**
 * Create a sequence object.  The referenced object can include multiple sequence references, in particular
 * fasta and 2bit URLs.  This is for backward compatibility, the 2bit URL has preference.
 *
 * @param reference
 * @returns {Promise<CachedSequence|ChromSizes|NonIndexedFasta>}
 */
async function loadSequence(reference, browser) {

    let fasta;
    if ("chromsizes" === reference.format) {
        fasta = new ChromSizes(reference.fastaURL || reference.url);
    } else if ("2bit" === reference.format || reference.twoBitURL) {
        fasta = new CachedSequence(new TwobitSequence(reference), browser);
    } else if (isDataURL(reference.fastaURL) || !reference.indexURL) {
        fasta = new NonIndexedFasta(reference);
    } else if("gbk" === reference.format || reference.gbkURL) ;

    else {
        fasta = new CachedSequence(new FastaSequence(reference), browser);
    }
    await fasta.init();
    return fasta
}

const defaultNucleotideColors = {
    "A": "rgb(  0, 200,   0)",
    "C": "rgb(  0,0,200)",
    "T": "rgb(255,0,0)",
    "G": "rgb(209,113,  5)",
    "N": "rgb(80,80,80)"
};

/**
 * Decoder for bedpe records.
 *
 * Bedpe format was created by Aaron Quinlan et al as part of the bedtools project.
 * The spec is here:  https://bedtools.readthedocs.io/en/latest/content/general-usage.html,
 *
 * 1      2      3    4      5      6    7    8     9       10      11-
 * chrom1 start1 end1 chrom2 start2 end2 name score strand1 strand2 <Any number of additional, user-defined fields>
 *
 * However there are off spec variants, an important one being a 7 column format with score in place of the standard
 * name column.
 *
 * A common variant is a "hiccups" output file, which is standard bedpe with the exception of a header line
 * of the form
 *
 * chr1    x1    x2    chr2    y1    y2    name    score    strand1    strand2    color    observed    expectedBL    expectedDonut    expectedH    expectedV    fdrBL    fdrDonut    fdrH    fdrV
 *
 * The "hiccups" output is apparently not standardized as this is found at ENCODE, with a non-helpful "tsv" extension
 *
 * chr1    x1    x2    chr2    y1    y2    color    observed    expectedBL    expectedDonut    expectedH    expectedV    fdrBL    fdrDonut    fdrH    fdrV    numCollapsed    centroid1    centroid2    radius
 * chr9    136790000    136795000    chr9    136990000    136995000    0,255,255    101.0    31.100368    38.40316    56.948116    34.040756    1.1876738E-13    1.05936405E-13    2.5148233E-4    1.7220993E-13    1    136792500    136992500    25590
 *
 * The "hiccups" documentation specfies yet another set of column headers
 * chromosome1    x1    x2    chromosome2    y1    y2    color    observed expected_bottom_left    expected_donut    expected_horizontal    expected_vertical fdr_bottom_left    fdr_donut    fdr_horizontal    fdr_vertical number_collapsed    centroid1    centroid2    radius
 *
 * @param tokens
 * @param ignore
 * @returns {{start1: number, end2: number, end1: number, chr1: *, chr2: *, start2: number}|undefined}
 */

function decodeBedpe(tokens, header) {

    if (tokens.length < 6) {
        return undefined
    }

    var feature = {
        chr1: tokens[0],
        start1: Number.parseInt(tokens[1]),
        end1: Number.parseInt(tokens[2]),
        chr2: tokens[3],
        start2: Number.parseInt(tokens[4]),
        end2: Number.parseInt(tokens[5])
    };

    if (isNaN(feature.start1) || isNaN(feature.end1) || isNaN(feature.start2) || isNaN(feature.end2)) {
        //throw Error(`Error parsing line: ${tokens.join('\t')}`);
        return undefined
    }

    // Determine if this is a "hiccups" file.  Store result on "header" so it doesn't need repeated for every feature
    if(header && header.hiccups === undefined) {
        header.hiccups = header.columnNames ? isHiccups(header.columnNames) : false;
    }
    const hiccups = header && header.hiccups;
    const stdColumns = hiccups ? 6 : 10;

    if(!hiccups) {
        if (tokens.length > 6 && tokens[6] !== ".") {
            feature.name = tokens[6];
        }

        if (tokens.length > 7 && tokens[7] !== ".") {
            feature.score = Number(tokens[7]);
        }

        if (tokens.length > 8 && tokens[8] !== ".") {
            feature.strand1 = tokens[8];
        }

        if (tokens.length > 9 && tokens[9] !== ".") {
            feature.strand2 = tokens[9];
        }
    }

    // Optional extra columns
    if (header) {
        const colorColumn = header.colorColumn;
        if (colorColumn && colorColumn < tokens.length) {
            feature.color = IGVColor.createColorString(tokens[colorColumn]);
        }
        const thicknessColumn = header.thicknessColumn;
        if (thicknessColumn && thicknessColumn < tokens.length) {
            feature.thickness = tokens[thicknessColumn];
        }

        if (tokens.length > stdColumns && header.columnNames && header.columnNames.length === tokens.length) {
            feature.extras = tokens.slice(stdColumns);
        }
    }


    // Set total extent of feature
    if (feature.chr1 === feature.chr2) {
        feature.chr = feature.chr1;
        feature.start = Math.min(feature.start1, feature.start2);
        feature.end = Math.max(feature.end1, feature.end2);

    }
    return feature
}

/**
 * Hack for non-standard bedPE formats, where numeric score can be in column 7 (name field from spec)
 * @param features
 */
function fixBedPE(features) {

    if (features.length == 0) return

    // Assume all features have same properties
    const firstFeature = features[0];
    if (firstFeature.score === undefined && firstFeature.name !== undefined) {
        // Name field (col 7) is sometimes used for score.
        for (let f of features) {
            if (!(isNumber(f.name) || f.name === '.')) return
        }
        for (let f of features) {
            f.score = Number(f.name);
            delete f.name;
        }
    }

    // Make copies of inter-chr features, one for each chromosome
    const interChrFeatures = features.filter(f => f.chr1 !== f.chr2);
    for (let f1 of interChrFeatures) {
        const f2 = Object.assign({}, f1);
        f2.dup = true;
        features.push(f2);

        f1.chr = f1.chr1;
        f1.start = f1.start1;
        f1.end = f1.end1;

        f2.chr = f2.chr2;
        f2.start = f2.start2;
        f2.end = f2.end2;
    }
}


/**
 * Special decoder for Hic Domain files.   In these files feature1 == feature2, they are really bed records.
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeBedpeDomain(tokens, header) {

    if (tokens.length < 8) return undefined

    return {
        chr: tokens[0],
        start: Number.parseInt(tokens[1]),
        end: Number.parseInt(tokens[2]),
        color: IGVColor.createColorString(tokens[6]),
        value: Number(tokens[7])
    }
}

function isHiccups(columns) {
    return columns && (columns.includes("fdrDonut") || columns.includes("fdr_donut"))
}

/**
 * Decode UCSC "interact" files.  See https://genome.ucsc.edu/goldenpath/help/interact.html
 *
 0  string chrom;        "Chromosome (or contig, scaffold, etc.). For interchromosomal, use 2 records"
 1  uint chromStart;     "Start position of lower region. For interchromosomal, set to chromStart of this region"
 2  uint chromEnd;       "End position of upper region. For interchromosomal, set to chromEnd of this region"
 3  string name;         "Name of item, for display.  Usually 'sourceName/targetName/exp' or empty"
 4  uint score;          "Score (0-1000)"
 5  double value;        "Strength of interaction or other data value. Typically basis for score"
 6  string exp;          "Experiment name (metadata for filtering). Use . if not applicable"
 7  string color;        "Item color.  Specified as r,g,b or hexadecimal #RRGGBB or html color name, as in //www.w3.org/TR/css3-color/#html4. Use 0 and spectrum setting to shade by score"
 8  string sourceChrom;  "Chromosome of source region (directional) or lower region. For non-directional interchromosomal, chrom of this region."
 9  uint sourceStart;    "Start position in chromosome of source/lower/this region"
 10 uint sourceEnd;      "End position in chromosome of source/lower/this region"
 11 string sourceName;   "Identifier of source/lower/this region"
 12 string sourceStrand; "Orientation of source/lower/this region: + or -.  Use . if not applicable"
 13 string targetChrom;  "Chromosome of target region (directional) or upper region. For non-directional interchromosomal, chrom of other region"
 14 uint targetStart;    "Start position in chromosome of target/upper/this region"
 15 uint targetEnd;      "End position in chromosome of target/upper/this region"
 16 string targetName;   "Identifier of target/upper/this region"
 17 string targetStrand; "Orientation of target/upper/this region: + or -.  Use . if not applicable"
 *
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeInteract(tokens, header) {

    if (tokens.length < 6) {
        return undefined
    }

    var feature = {
        chr: tokens[0],
        start: Number.parseInt(tokens[1]),
        end: Number.parseInt(tokens[2]),

        chr1: tokens[8],
        start1: Number.parseInt(tokens[9]),
        end1: Number.parseInt(tokens[10]),

        chr2: tokens[13],
        start2: Number.parseInt(tokens[14]),
        end2: Number.parseInt(tokens[15]),

        name: tokens[3],
        score: Number(tokens[4]),
        value: Number(tokens[5]),
        color: tokens[7] === '.' ? undefined : tokens[7] === "0" ? "rgb(0,0,0)" : tokens[7],

    };

    return feature
}

/**
 * Wrapper class to record a decoding error.
 */

class DecodeError {
    constructor(message) {
        this.message = message;
    }
}

const encodings$2 = new Map([
    ["%09", "\t"],
    ["%0A", "\n"],
    ["%0D", "\r"],
    ["%25", "%"],
    ["%3B", ";"],
    ["%3D", "="],
    ["%26", "&"],
    ["%2C", ","]
]);


// GFF3 attributes have specific percent encoding rules, the list below are required, all others are forbidden
/*
tab (%09)
newline (%0A)
carriage return (%0D)
% percent (%25)
control characters (%00 through %1F, %7F)
In addition, the following characters have reserved meanings in column 9 and must be escaped when used in other contexts:
; semicolon (%3B)
= equals (%3D)
& ampersand (%26)
, comma (%2C)
 */

function decodeGFFAttribute(str) {

    if (!str.includes("%")) {
        return str
    }
    let decoded = "";
    for (let i = 0; i < str.length; i++) {

        if (str.charCodeAt(i) === 37 && i < str.length - 2) {
            const key = str.substring(i, i + 3);
            if (encodings$2.has(key)) {
                decoded += encodings$2.get(key);
            } else {
                decoded += key;
            }
            i += 2;
        } else {
            decoded += str.charAt(i);
        }
    }
    return decoded

}

/**
 * Parse the attribute string, returning an array of key-value pairs.  An array is used rather than a map as attribute
 * keys are not required to be unique.
 *
 * @param attributeString
 * @param keyValueDelim
 * @returns {[]}
 */
function parseAttributeString(attributeString, keyValueDelim = "=") {
    // parse 'attributes' string (see column 9 docs in https://github.com/The-Sequence-Ontology/Specifications/blob/master/gff3.md)
    const isGff3 = ('=' === keyValueDelim);
    var attributes = [];
    for (let kv of attributeString.split(';')) {
        kv = kv.trim();
        const idx = kv.indexOf(keyValueDelim);
        if (idx > 0 && idx < kv.length - 1) {
            let key = decodeGFFAttribute(kv.substring(0, idx).trim());
            let value = decodeGFFAttribute(kv.substring(idx + 1).trim());
            if (!isGff3) {
                key = stripQuotes$1(key);
                value = stripQuotes$1(value);
            }
            attributes.push([key, value]);
        }
    }
    return attributes
}

function stripQuotes$1(value) {
    if (value.startsWith('"') && value.endsWith('"')) {
        value = value.substr(1, value.length - 2);
    }
    return value
}

/**
 * Some interpretations of the sequence ontology needed to assemble GFF transcripts.
 *
 */

const transcriptTypes = new Set(['transcript', 'primary_transcript', 'processed_transcript', 'mRNA', 'mrna',
    'lnc_RNA', 'miRNA', 'ncRNA', 'rRNA', 'scRNA', 'snRNA', 'snoRNA', 'tRNA']);
const cdsTypes = new Set(['CDS', 'cds']);
const codonTypes = new Set(['start_codon', 'stop_codon']);
const utrTypes = new Set(['5UTR', '3UTR', 'UTR', 'five_prime_UTR', 'three_prime_UTR', "3'-UTR", "5'-UTR"]);
const exonTypes = new Set(['exon', 'coding-exon']);

const transcriptPartTypes = new Set();
for (let cltn of [cdsTypes, codonTypes, utrTypes, exonTypes]) {
    for (let t of cltn) {
        transcriptPartTypes.add(t);
    }
}

function isExon(type) {
    return exonTypes.has(type)
}

function isIntron(type) {
    return type.includes("intron")
}

function isCoding(type) {
    return cdsTypes.has(type) || codonTypes.has(type)
}

function isUTR(type) {
    return utrTypes.has(type)
}

function isTranscript(type) {
    return transcriptTypes.has(type) || type.endsWith("RNA") || type.endsWith("transcript")
}

function isTranscriptPart(type) {
    return transcriptPartTypes.has(type) || type.endsWith("RNA") || isIntron(type)
}

const filterPopupProperties = new Set(["id", "parent", "name"]);

class GFFFeature {

    constructor(properties) {

        Object.assign(this, properties);

        if (properties.phase !== undefined && "." !== properties.phase) {
            this.readingFrame = (3 - parseInt(properties.phase)) % 3;
        }

    }


    popupData(genomicLocation) {

        const pd = this.geneObject ? this.geneObject.popupData() : [];

        if (this.geneObject) {
            pd.push('<hr/>');
        }

        if (this.name) {
            pd.push({name: 'Name', value: this.name});
        }

        pd.push({name: 'Type', value: this.type});
        pd.push({name: 'Source', value: this.source});
        if (this.score !== undefined) {
            pd.push({name: 'Score', value: this.score});
        }
        pd.push({name: 'Phase', value: this.phase});

        if (this.attributeString) {
            const atts = parseAttributeString(this.attributeString, this.delim);
            for (let [key, value] of atts) {
                if (value !== undefined && value.length > 0 && !filterPopupProperties.has(key.toLowerCase())) {
                    pd.push({name: key + ":", value: value});
                }
            }
        }
        pd.push({
            name: 'Location',
            value: `${this.chr}:${numberFormatter$1(this.start + 1)}-${numberFormatter$1(this.end)}`
        });
        return pd
    }

    getAttributeValue(attributeName) {
        if (this.hasOwnProperty(attributeName)) {
            return this[attributeName]
        } else {
            // TODO -- fetch from attribute string and cache
            if (!this._attributeCache) {
                this._attributeCache = new Map();
            }
            if (this._attributeCache.has(attributeName)) {
                return this._attributeCache.get(attributeName)
            } else {
                const atts = parseAttributeString(this.attributeString, this.delim);
                let v;
                for (let [key, value] of atts) {
                    if (key === attributeName) {
                        v = value;
                        break
                    }
                }
                this._attributeCache.set(attributeName, v);
                return v
            }
        }
    }
}

class GFFTranscript extends GFFFeature {

    constructor(feature) {
        super(feature);
        this.exons = [];
        this.parts = [];
    }

    addExon(feature) {

        this.exons.push(feature);

        // Expand feature --  for transcripts not explicitly represented in the file (gtf)
        this.start = Math.min(this.start, feature.start);
        this.end = Math.max(this.end, feature.end);
    }

    addPart(feature) {
        this.parts.push(feature);
    }

    assembleParts() {

        if (this.parts.length === 0) return

        this.parts.sort(function (a, b) {
            return a.start - b.start
        });

        // Create exons, if neccessary
        let lastStart = this.parts[0].start;
        let lastEnd = this.parts[0].end;
        for (let i = 1; i < this.parts.length; i++) {
            const part = this.parts[i];
            if (isIntron(part.type)) {
                continue
            }
            if (part.start <= lastEnd) {
                lastEnd = Math.max(lastEnd, part.end);
            } else {
                let exon = this.findExonContaining({start: lastStart, end: lastEnd});
                if (!exon) {
                    this.exons.push({start: lastStart, end: lastEnd, psuedo: true});
                }
                lastStart = part.start;
                lastEnd = part.end;
            }
        }
        let exon = this.findExonContaining({start: lastStart, end: lastEnd});
        if (!exon) {
            this.exons.push({start: lastStart, end: lastEnd, psuedo: true});
            this.start = Math.min(this.start, lastStart);
            this.end = Math.max(this.end, lastEnd);
        }


        for (let part of this.parts) {
            const type = part.type;
            if (isCoding(type)) {
                this.addCDS(part);
            } else if (isUTR(type)) {
                this.addUTR(part);
            }
        }
    }

    findExonContaining({start, end}) {
        for (let exon of this.exons) {
            if (exon.end >= end && exon.start <= start) {
                return exon
            }
        }
        return undefined
    }

    addCDS(cds) {

        let exon;
        const exons = this.exons;

        for (let e of exons) {
            if (e.start <= cds.start && e.end >= cds.end) {
                exon = e;
                break
            }
        }

        if (exon) {
            exon.cdStart = exon.cdStart ? Math.min(cds.start, exon.cdStart) : cds.start;
            exon.cdEnd = exon.cdEnd ? Math.max(cds.end, exon.cdEnd) : cds.end;
            exon.readingFrame = cds.readingFrame;
            // TODO -- merge attributes?
        } else {
            // cds.cdStart = cds.start
            // cds.cdEnd = cds.end
            // exons.push(cds)
            console.error("No exon found spanning " + cds.start + "-" + cds.end);
        }

        // Expand feature --  for transcripts not explicitly represented in the file (gtf files)
        // this.start = Math.min(this.start, cds.start);
        // this.end = Math.max(this.end, cds.end);

        this.cdStart = this.cdStart ? Math.min(cds.start, this.cdStart) : cds.start;
        this.cdEnd = this.cdEnd ? Math.max(cds.end, this.cdEnd) : cds.end;
    }

    addUTR(utr) {

        let exon;
        const exons = this.exons;

        // Find exon containing CDS
        for (let i = 0; i < exons.length; i++) {
            if (exons[i].start <= utr.start && exons[i].end >= utr.end) {
                exon = exons[i];
                break
            }
        }

        if (exon) {
            if (utr.start === exon.start && utr.end === exon.end) {
                exon.utr = true;
            } else {
                if (utr.end < exon.end) {
                    exon.cdStart = utr.end;
                }
                if (utr.start > exon.start) {
                    exon.cdEnd = utr.start;
                }
            }

        } else {
            // utr.utr = true
            // exons.push(utr)
            console.error("No exon found spanning " + cds.start + "-" + cds.end);
        }

        // Expand feature --  for transcripts not explicitly represented in the file
        // this.start = Math.min(this.start, utr.start);
        // this.end = Math.max(this.end, utr.end);

    }

    finish() {

        this.assembleParts();

        var cdStart = this.cdStart;
        var cdEnd = this.cdEnd;

        this.exons.sort(function (a, b) {
            return a.start - b.start
        });

        // Search for UTR exons that were not explicitly tagged
        if (cdStart) {
            this.exons.forEach(function (exon) {
                if (exon.end < cdStart || exon.start > cdEnd) exon.utr = true;
            });
        }
    }

    popupData(genomicLocation) {

        const pd = super.popupData(genomicLocation);

        // If clicked over an exon add its attributes
        for (let exon of this.exons) {
            if (exon.pseudo) continue  // An implicit exon
            if (genomicLocation >= exon.start && genomicLocation < exon.end && typeof exon.popupData === 'function') {
                pd.push('<hr/>');
                const exonData = exon.popupData(genomicLocation);
                for (let att of exonData) {
                    pd.push(att);
                }
            }
        }

        for (let part of this.parts) {
            if (genomicLocation >= part.start && genomicLocation < part.end && typeof part.popupData === 'function') {
                pd.push('<hr/>');
                const partData = part.popupData(genomicLocation);
                for (let att of partData) {
                    pd.push(att);
                }
            }
        }


        return pd
    }
}

class GFFHelper {

    static gffNameFields = new Set(["Name", "transcript_id", "gene_name", "gene", "gene_id", "alias", "locus", "name" ])
    constructor(options) {
        this.format = options.format;
        this.nameField = options.nameField;
        this.filterTypes = options.filterTypes === undefined ?
            new Set(['chromosome']) :
            new Set(options.filterTypes);
    }

    combineFeatures(features, genomicInterval) {

        let combinedFeatures;

        const filterTypes = this.filterTypes;
        features = features.filter(f => filterTypes === undefined || !filterTypes.has(f.type));

        if ("gff3" === this.format) {
            const tmp = this.combineFeaturesById(features);
            combinedFeatures = this.combineFeaturesByType(tmp);
        } else {
            combinedFeatures = this.combineFeaturesByType(features);
        }

        this.numberExons(combinedFeatures, genomicInterval);
        this.nameFeatures(combinedFeatures);
        return combinedFeatures
    }

    /**
     * Combine multiple non-transcript model features with the same ID on the same chromosome into a single feature.
     * Features that are part of the transcript model (e.g. exon, mRNA, etc) are combined later.
     *
     * @param features
     * @returns {[]}
     */
    combineFeaturesById(features) {

        const chrIdMap = new Map();
        const combinedFeatures = [];

        for (let f of features) {
            if (isTranscriptPart(f.type) || isTranscript(f.type) || !f.id) {
                combinedFeatures.push(f);
            } else {
                let idMap = chrIdMap.get(f.chr);
                if (!idMap) {
                    idMap = new Map();
                    chrIdMap.set(f.chr, idMap);
                }

                let featureArray = idMap.get(f.id);
                if (featureArray) {
                    featureArray.push(f);
                } else {
                    idMap.set(f.id, [f]);
                }
            }
        }

        for (let idMap of chrIdMap.values()) {
            for (let featureArray of idMap.values()) {
                if (featureArray.length > 1) {
                    // Use the first feature as prototypical (for column 9 attributes), and adjust start/end
                    // Parts are represented as "exons", as that is how they are presented visually
                    const cf = featureArray[0];
                    cf.exons = [];
                    for (let f of featureArray) {
                        cf.start = Math.min(cf.start, f.start);
                        cf.end = Math.max(cf.end, f.end);
                        cf.exons.push({
                            start: f.start,
                            end: f.end
                        });
                    }
                    combinedFeatures.push(cf);
                } else {
                    combinedFeatures.push(featureArray[0]);
                }
            }
        }

        return combinedFeatures
    }

    combineFeaturesByType(features) {

        // Build dictionary of genes
        const genes = features.filter(f => "gene" === f.type || f.type.endsWith("_gene"));
        const geneMap = Object.create(null);
        for (let g of genes) {
            geneMap[g.id] = g;
        }

        // 1. Build dictionary of transcripts
        const transcripts = Object.create(null);
        const combinedFeatures = [];
        const consumedFeatures = new Set();
        const filterTypes = this.filterTypes;

        features = features.filter(f => filterTypes === undefined || !filterTypes.has(f.type));

        for (let f of features) {
            if (isTranscript(f.type)) {
                const transcriptId = f.id; // getAttribute(f.attributeString, "transcript_id", /\s+/);
                if (undefined !== transcriptId) {
                    const gffTranscript = new GFFTranscript(f);
                    transcripts[transcriptId] = gffTranscript;
                    combinedFeatures.push(gffTranscript);
                    consumedFeatures.add(f);
                    const g = geneMap[f.parent];
                    if (g) {
                        gffTranscript.geneObject = g;
                        consumedFeatures.add(g);
                    }
                }
            }
        }

        // Add exons and transcript parts
        for (let f of features) {
            if (isTranscriptPart(f.type)) {
                const parents = getParents(f);
                if (parents) {
                    for (let id of parents) {

                        let transcript = transcripts[id];
                        if (!transcript && this.format === "gtf") {
                            // GTF does not require explicit a transcript or mRNA record, start one with this feature.
                            const psuedoTranscript = Object.assign({}, f);
                            psuedoTranscript.type = "transcript";
                            transcript = new GFFTranscript(psuedoTranscript);
                            transcripts[id] = transcript;
                            combinedFeatures.push(transcript);
                        }
                        if (transcript !== undefined) {

                            if (isExon(f.type)) {
                                if (parents.length > 1) {
                                    // Multiple parents, this is unusual.  Make a copy as exon can be modified
                                    // differently by CDS, etc, for each parent
                                    const e2 = new GFFFeature(f);
                                    transcript.addExon(e2);
                                } else {
                                    transcript.addExon(f);
                                }
                            } else {
                                transcript.addPart(f);
                            }
                            consumedFeatures.add(f);
                        }
                    }
                }
            }
        }

        // Finish transcripts
        combinedFeatures.forEach(function (f) {
            if (typeof f.finish === "function") {
                f.finish();
            }
        });

        // Add other features
        const others = features.filter(f => !consumedFeatures.has(f));
        for (let f of others) {
            combinedFeatures.push(f);
        }

        return combinedFeatures

        function getParents(f) {
            if (f.parent && f.parent.trim() !== "") {
                return f.parent.trim().split(",")
            } else {
                return null
            }
        }
    }

    numberExons(features, genomicInterval) {

        for (let f of features) {
            if (f.exons &&
                (!genomicInterval ||
                    (f.end <= genomicInterval.end && f.start > genomicInterval.start))) {
                for (let i = 0; i < f.exons.length; i++) {
                    const exon = f.exons[i];
                    exon.number = f.strand === "-" ? f.exons.length - i : i + 1;
                }
            }
        }
    }

    nameFeatures(features) {
        // Find name (label) property
        for (let f of features) {
            if(typeof f.getAttributeValue === 'function') {
                if (this.nameField) {
                    f.name = f.getAttributeValue(this.nameField);
                } else {
                    for (let nameField of GFFHelper.gffNameFields) {
                        const v = f.getAttributeValue(nameField);
                        if (v) {
                            f.name = v;
                            break
                        }
                    }
                }
            }
        }
    }
}

/**
 * Decode the UCSC bed format.  Only the first 3 columns (chr, start, end) are required.   The remaining columns
 * must follow standard bed order, but we will tolerate deviations after column 3.
 *
 * @param tokens
 * @param ignore
 * @returns decoded feature, or null if this is not a valid record
 */
function decodeBed(tokens, header, maxColumnCount = Number.MAX_SAFE_INTEGER) {

    if (tokens.length < 3) return undefined

    const gffTags = header && header.gffTags;

    const chr = tokens[0];
    const start = parseInt(tokens[1]);
    const end = tokens.length > 2 ? parseInt(tokens[2]) : start + 1;
    if (isNaN(start) || isNaN(end)) {
        return new DecodeError(`Unparsable bed record.`)
    }
    const feature = new UCSCBedFeature({chr: chr, start: start, end: end, score: 1000});

    let columnCount = 3;
    try {
        if (tokens.length > 3 && columnCount++ < maxColumnCount) {

            // Potentially parse name field as GFF column 9 style streng.
            if (tokens[3].indexOf(';') > 0 && tokens[3].indexOf('=') > 0) {
                const attributeKVs = parseAttributeString(tokens[3], '=');
                feature.attributes = {};
                for (let kv of attributeKVs) {
                    feature.attributes[kv[0]] = kv[1];
                    if(gffTags) {
                        if (header.nameField != undefined && kv[0] === header.nameField) {
                            feature.name = kv[1];
                        } else if (!feature.name && GFFHelper.gffNameFields.has(kv[0])) {
                            feature.name = kv[1];
                        }
                    }
                }
            }
            if (!feature.name && !gffTags) {
                feature.name = tokens[3] === '.' ? '' : tokens[3];
            }
        }

        if (tokens.length > 4 && columnCount++ < maxColumnCount) {
            feature.score = tokens[4] === '.' ? 0 : Number(tokens[4]);
            if (isNaN(feature.score)) {
                return feature
            }
        }

        if (tokens.length > 5 && columnCount++ < maxColumnCount) {
            feature.strand = tokens[5];
            if (!(feature.strand === '.' || feature.strand === '+' || feature.strand === '-')) {
                return feature
            }
        }

        if (tokens.length > 6 && columnCount++ < maxColumnCount) {
            feature.cdStart = parseInt(tokens[6]);
            if (isNaN(feature.cdStart)) {
                return feature
            }
        }

        if (tokens.length > 7 && columnCount++ < maxColumnCount) {
            feature.cdEnd = parseInt(tokens[7]);
            if (isNaN(feature.cdEnd)) {
                return feature
            }
        }

        if (tokens.length > 8 && columnCount++ < maxColumnCount) {
            if (tokens[8] !== "." && tokens[8] !== "0")
                feature.color = IGVColor.createColorString(tokens[8]);
        }

        if (tokens.length > 11 && columnCount++ < maxColumnCount) {
            const exonCount = parseInt(tokens[9]);
            // Some basic validation
            if (exonCount > 1000) {
                // unlikely
                return feature
            }

            const exonSizes = tokens[10].replace(/,$/, '').split(',');
            const exonStarts = tokens[11].replace(/,$/, '').split(',');
            if (!(exonSizes.length === exonStarts.length && exonCount === exonSizes.length)) {
                return feature
            }

            const exons = [];
            for (let i = 0; i < exonCount; i++) {
                const eStart = start + parseInt(exonStarts[i]);
                const eEnd = eStart + parseInt(exonSizes[i]);
                exons.push({start: eStart, end: eEnd});
            }
            if (exons.length > 0) {
                findUTRs$1(exons, feature.cdStart, feature.cdEnd);
                feature.exons = exons;
            }
        }

        // Optional extra columns
        if (header) {
            let thicknessColumn = header.thicknessColumn;
            let colorColumn = header.colorColumn;
            if (colorColumn && colorColumn < tokens.length) {
                feature.color = IGVColor.createColorString(tokens[colorColumn]);
            }
            if (thicknessColumn && thicknessColumn < tokens.length) {
                feature.thickness = tokens[thicknessColumn];
            }
        }
    } catch
        (e) {

    }

    return feature
}


function decodeGappedPeak(tokens, header) {
    const feature = decodeBed(tokens, header);
    if (feature && tokens.length > 14) {
        feature.signal = Number(tokens[12]);
        feature.pValue = Number(tokens[13]);
        feature.qValue  = Number(tokens[14]);
    }
    return feature
}

/**
 * Decode a bedMethyl file.
 * Reference: https://www.encodeproject.org/data-standards/wgbs/
 * @param tokens
 * @param header
 */
function decodeBedmethyl(tokens, header) {

    // Bedmethyl is a 9+9 format
    const feature = decodeBed(tokens, header, 9);
    if (feature) {
        const extraColumnHeadings = ["Coverage", "% Showing Methylation", "N-mod", "N-canonical", "N-other mod",
            "N-delete", "N-fail", "N-dff", "N-nocall"];
        for (let i = 9; i < tokens.length; i++) {
            const heading = extraColumnHeadings[i - 9];
            feature[heading] = tokens[i];
        }
    }


    return feature
}


/**
 * Decode a UCSC repeat masker record.
 */
function decodeRepeatMasker(tokens, header) {

    /**
     * Columns, from UCSC documentation
     *
     * 0  bin    585    smallint(5) unsigned    Indexing field to speed chromosome range queries.
     * 1  swScore    1504    int(10) unsigned    Smith Waterman alignment score
     * 2  milliDiv    13    int(10) unsigned    Base mismatches in parts per thousand
     * 3  milliDel    4    int(10) unsigned    Bases deleted in parts per thousand
     * 4  milliIns    13    int(10) unsigned    Bases inserted in parts per thousand
     * 5  genoName    chr1    varchar(255)    Genomic sequence name
     * 6  genoStart    10000    int(10) unsigned    Start in genomic sequence
     * 7  genoEnd    10468    int(10) unsigned    End in genomic sequence
     * 8  genoLeft    -249240153    int(11)    -#bases after match in genomic sequence
     * 9  strand    +    char(1)    Relative orientation + or -
     * 10 repName    (CCCTAA)n    varchar(255)    Name of repeat
     * 11 repClass    Simple_repeat    varchar(255)    Class of repeat
     * 12 repFamily    Simple_repeat    varchar(255)    Family of repeat
     * 13 repStart    1    int(11)    Start (if strand is +) or -#bases after match (if strand is -) in repeat sequence
     * 14 repEnd    463    int(11)    End in repeat sequence
     * 15 repLeft    0    int(11)    -#bases after match (if strand is +) or start (if strand is -) in repeat sequence
     * 16 id    1    char(1)    First digit of id field in RepeatMasker .out file. Best ignored.
     */
    if (tokens.length <= 15) return undefined

    const feature = {
        swScore: Number.parseInt(tokens[1]),
        milliDiv: Number.parseInt(tokens[2]),
        milliDel: Number.parseInt(tokens[3]),
        milliIns: Number.parseInt(tokens[4]),
        chr: tokens[5],
        start: Number.parseInt(tokens[6]),
        end: Number.parseInt(tokens[7]),
        //genoLeft: tokens[8],
        strand: tokens[9],
        repName: tokens[10],
        repClass: tokens[11],
        repFamily: tokens[12],
        repStart: Number.parseInt(tokens[13]),
        repEnd: Number.parseInt(tokens[14]),
        repLeft: Number.parseInt(tokens[15])
    };

    return feature

}

/**
 * Decode a UCSC "genePred" record.
 *
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeGenePred(tokens, header) {

    var shift = header.shift === undefined ? 0 : 1;

    if (tokens.length <= 9 + shift) return undefined

    const cdStart = parseInt(tokens[5 + shift]);
    const cdEnd = parseInt(tokens[6 + shift]);
    var feature = {
        name: tokens[0 + shift],
        chr: tokens[1 + shift],
        strand: tokens[2 + shift],
        start: parseInt(tokens[3 + shift]),
        end: parseInt(tokens[4 + shift]),
        cdStart: cdStart,
        cdEnd: cdEnd,
        id: tokens[0 + shift]
    };
    const exons = decodeExons(parseInt(tokens[7 + shift]), tokens[8 + shift], tokens[9 + shift]);
    findUTRs$1(exons, cdStart, cdEnd);

    feature.exons = exons;

    return feature

}

/**
 * Decode a UCSC "genePredExt" record.  refGene files are in this format.
 *
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeGenePredExt(tokens, header) {

    var shift = header.shift === undefined ? 0 : 1;

    if (tokens.length <= 11 + shift) return undefined

    const cdStart = parseInt(tokens[5 + shift]);
    const cdEnd = parseInt(tokens[6 + shift]);
    const feature = {
        name: tokens[11 + shift],
        chr: tokens[1 + shift],
        strand: tokens[2 + shift],
        start: parseInt(tokens[3 + shift]),
        end: parseInt(tokens[4 + shift]),
        cdStart: cdStart,
        cdEnd: cdEnd,
        id: tokens[0 + shift]
    };

    const exons = decodeExons(
        parseInt(tokens[7 + shift]),
        tokens[8 + shift],
        tokens[9 + shift],
        tokens[14 + shift]);
    findUTRs$1(exons, cdStart, cdEnd);

    feature.exons = exons;

    return feature
}

/**
 * Decode a UCSC "refFlat" record
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeReflat(tokens, header) {

    var shift = header.shift === undefined ? 0 : 1;

    if (tokens.length <= 10 + shift) return undefined

    const cdStart = parseInt(tokens[6 + shift]);
    const cdEnd = parseInt(tokens[7 + shift]);
    var feature = {
        name: tokens[0 + shift],
        id: tokens[1 + shift],
        chr: tokens[2 + shift],
        strand: tokens[3 + shift],
        start: parseInt(tokens[4 + shift]),
        end: parseInt(tokens[5 + shift]),
        cdStart: cdStart,
        cdEnd: cdEnd
    };

    const exons = decodeExons(parseInt(tokens[8 + shift]), tokens[9 + shift], tokens[10 + shift]);
    findUTRs$1(exons, cdStart, cdEnd);
    feature.exons = exons;

    return feature
}

/**
 * Decode a UCS PSL record *
 * @param tokens
 * @param header
 * @returns {DecodeError|UCSCBedFeature|undefined}
 */
function decodePSL(tokens, header) {

    /*
    * 0 matches - Number of bases that match that aren't repeats
    * 1 misMatches - Number of bases that don't match
    * 2 repMatches - Number of bases that match but are part of repeats
    * 3 nCount - Number of "N" bases
    * 4 qNumInsert - Number of inserts in query
    * 5 qBaseInsert - Number of bases inserted in query
    * 6 tNumInsert - Number of inserts in target
    * 7 tBaseInsert - Number of bases inserted in target
    * 8 strand - "+" or "-" for query strand. For translated alignments, second "+"or "-" is for target genomic strand.
    * 9 qName - Query sequence name
    * 10 qSize - Query sequence size.
    * 11 qStart - Alignment start position in query
    * 12 qEnd - Alignment end position in query
    * 13 tName - Target sequence name
    * 14 tSize - Target sequence size
    * 15 tStart - Alignment start position in target
    * 16 tEnd - Alignment end position in target
    * 17 blockCount - Number of blocks in the alignment (a block contains no gaps)
    * 18 blockSizes - Comma-separated list of sizes of each block. If the query is a protein and the target the genome, blockSizes are in amino acids. See below for more information on protein query PSLs.
    * 19 qStarts - Comma-separated list of starting positions of each block in query
    * 20 tStarts - Comma-separated list of starting positions of each block in target
     */


    if (tokens.length < 21) return undefined

    const chr = tokens[13];
    const start = parseInt(tokens[15]);
    const end = parseInt(tokens[16]);
    const strand = tokens[8].charAt(0);
    const exonCount = parseInt(tokens[17]);
    const exons = [];
    const exonStarts = tokens[20].replace(/,$/, '').split(',');
    const exonSizes = tokens[18].replace(/,$/, '').split(',');

    for (let i = 0; i < exonCount; i++) {
        const start = parseInt(exonStarts[i]);
        const end = start + parseInt(exonSizes[i]);
        exons.push({start: start, end: end});
    }

    return new PSLFeature({chr, start, end, strand, exons, tokens})
}


function decodeExons(exonCount, startsString, endsString, frameOffsetsString) {

    const exonStarts = startsString.replace(/,$/, '').split(',');
    const exonEnds = endsString.replace(/,$/, '').split(',');
    const frameOffsets = frameOffsetsString ? frameOffsetsString.replace(/,$/, '').split(',') : undefined;
    const exons = [];
    for (let i = 0; i < exonCount; i++) {
        const start = parseInt(exonStarts[i]);
        const end = parseInt(exonEnds[i]);
        const exon = {start, end};
        if (frameOffsets) {
            const fo = parseInt(frameOffsets[i]);
            if (fo != -1) exon.readingFrame = fo;
        }
        exons.push(exon);
    }
    return exons

}

function findUTRs$1(exons, cdStart, cdEnd) {

    for (let exon of exons) {
        const end = exon.end;
        const start = exon.start;
        if (end < cdStart || start > cdEnd) {
            exon.utr = true;
        } else {
            if (cdStart >= start && cdStart <= end) {
                exon.cdStart = cdStart;
            }
            if (cdEnd >= start && cdEnd <= end) {
                exon.cdEnd = cdEnd;
            }
        }
    }

}

function decodePeak(tokens, header) {

    var tokenCount, chr, start, end, strand, name, score, qValue, signal, pValue;

    tokenCount = tokens.length;
    if (tokenCount < 9) {
        return undefined
    }

    chr = tokens[0];
    start = parseInt(tokens[1]);
    end = parseInt(tokens[2]);
    name = tokens[3];
    score = Number(tokens[4]);
    strand = tokens[5].trim();
    signal = Number(tokens[6]);
    pValue = Number(tokens[7]);
    qValue = Number(tokens[8]);

    if (score === 0) score = signal;

    return {
        chr: chr, start: start, end: end, name: name, score: score, strand: strand, signal: signal,
        pValue: pValue, qValue: qValue
    }
}

function decodeNarrowPeak(tokens, header) {

    const feature = decodePeak(tokens);
    if(tokens.length > 9) {
        feature.peak = Number(tokens[9]);
    }
    return feature
}

function decodeBedGraph(tokens, header) {

    var chr, start, end, value;

    if (tokens.length <= 3) return undefined

    chr = tokens[0];
    start = parseInt(tokens[1]);
    end = parseInt(tokens[2]);
    value = Number(tokens[3]);
    const feature = {chr: chr, start: start, end: end, value: value};

    // Optional extra columns
    if (header) {
        let colorColumn = header.colorColumn;
        if (colorColumn && colorColumn < tokens.length) {
            feature.color = IGVColor.createColorString(tokens[colorColumn]);
        }
    }

    return feature
}

function decodeWig(tokens, header) {

    const wig = header.wig;

    if (wig && wig.format === "fixedStep") {
        const ss = (wig.index * wig.step) + wig.start;
        const ee = ss + wig.span;
        const value = Number(tokens[0]);
        ++(wig.index);
        return isNaN(value) ? null : {chr: wig.chrom, start: ss, end: ee, value: value}
    } else if (wig && wig.format === "variableStep") {

        if (tokens.length < 2) return null
        const ss = parseInt(tokens[0], 10) - 1;
        const ee = ss + wig.span;
        const value = Number(tokens[1]);
        return isNaN(value) ? null : {chr: wig.chrom, start: ss, end: ee, value: value}

    } else {
        return decodeBedGraph(tokens)
    }
}

function decodeSNP(tokens, header) {

    if (tokens.length < 6) return undefined

    const autoSql = [
        'bin',
        'chr',
        'start',
        'end',
        'name',
        'score',
        'strand',
        'refNCBI',
        'refUCSC',
        'observed',
        'molType',
        'class',
        'valid',
        'avHet',
        'avHetSE',
        'func',
        'locType',
        'weight',
        'exceptions',
        'submitterCount',
        'submitters',
        'alleleFreqCount',
        'alleles',
        'alleleNs',
        'alleleFreqs',
        'bitfields'
    ];


    const feature = {
        chr: tokens[1],
        start: Number.parseInt(tokens[2]),
        end: Number.parseInt(tokens[3]),
        name: tokens[4],
        score: Number.parseInt(tokens[5])
    };

    const n = Math.min(tokens.length, autoSql.length);
    for (let i = 6; i < n; i++) {
        feature[autoSql[i]] = tokens[i];
    }
    return feature

}


class UCSCBedFeature {

    constructor(properties) {
        Object.assign(this, properties);
    }

    getAttributeValue(attributeName) {
        if (this.hasOwnProperty(attributeName)) {
            return this[attributeName]
        } else if (this.attributes) {
            return this.attributes[attributeName]
        }
    }
}

/*
* 0 matches - Number of bases that match that aren't repeats
* 1 misMatches - Number of bases that don't match
* 2 repMatches - Number of bases that match but are part of repeats
* 3 nCount - Number of "N" bases
* 4 qNumInsert - Number of inserts in query
* 5 qBaseInsert - Number of bases inserted in query
* 6 tNumInsert - Number of inserts in target
* 7 tBaseInsert - Number of bases inserted in target
* 8 strand - "+" or "-" for query strand. For translated alignments, second "+"or "-" is for target genomic strand.
* 9 qName - Query sequence name
* 10 qSize - Query sequence size.
* 11 qStart - Alignment start position in query
* 12 qEnd - Alignment end position in query
* 13 tName - Target sequence name
* 14 tSize - Target sequence size
* 15 tStart - Alignment start position in target
* 16 tEnd - Alignment end position in target
* 17 blockCount - Number of blocks in the alignment (a block contains no gaps)
* 18 blockSizes - Comma-separated list of sizes of each block. If the query is a protein and the target the genome, blockSizes are in amino acids. See below for more information on protein query PSLs.
* 19 qStarts - Comma-separated list of starting positions of each block in query
* 20 tStarts - Comma-separated list of starting positions of each block in target
 */

class PSLFeature {


    constructor(properties) {
        Object.assign(this, properties);
    }

    get score() {
        const tokens = this.tokens;
        const match = parseInt(tokens[0]);
        const repMatch = parseInt(tokens[2]);
        const misMatch = parseInt(tokens[1]);
        const qGapCount = parseInt(tokens[4]);
        const tGapCount = parseInt(tokens[6]);
        const qSize = parseInt(tokens[10]);
        return Math.floor((1000 * (match + repMatch - misMatch - qGapCount - tGapCount)) / qSize)
    }

    get matches() {
        return this.tokens[0]
    }

    get misMatches() {
        return this.tokens[1]
    }

    get repMatches() {
        return this.tokens[2]
    }

    get nCount() {
        return this.tokens[3]
    }

    get qNumInsert() {
        return this.tokens[4]
    }

    get qBaseInsert() {
        return this.tokens[5]
    }

    get tNumInsert() {
        return this.tokens[6]
    }

    get tBaseInsert() {
        return this.tokens[7]

    }

    popupData() {
        return [
            {name: 'chr', value: this.chr},
            {name: 'start', value: this.start + 1},
            {name: 'end', value: this.end},
            {name: 'strand', value: this.strand},
            {name: 'score', value: this.score},
            {name: 'match', value: this.matches},
            {name: "mis-match", value: this.misMatches},
            {name: "rep. match", value: this.repMatches},
            {name: "N's", value: this.nCount},
            {name: 'Q gap count', value: this.qNumInsert},
            {name: 'Q gap bases', value: this.qBaseInsert},
            {name: 'T gap count', value: this.tNumInsert},
            {name: 'T gap bases', value: this.tBaseInsert},
        ]
    }

}

function decode(tokens, header) {

    const format = header.format;
    if (tokens.length < 9) {
        return undefined      // Not a valid gff record
    }

    const delim = ('gff3' === format) ? '=' : ' ';
    return new GFFFeature({
        source: decodeGFFAttribute(tokens[1]),
        type: tokens[2],
        chr: tokens[0],
        start: parseInt(tokens[3]) - 1,
        end: parseInt(tokens[4]),
        score: "." === tokens[5] ? undefined : Number(tokens[5]),
        strand: tokens[6],
        phase: "." === tokens[7] ? "." : parseInt(tokens[7]),
        attributeString: tokens[8],
        delim: delim
    })
}


/**
 * Decode a single gff record (1 line in file).  Aggregations such as gene models are constructed at a higher level.
 *      ctg123 . mRNA            1050  9000  .  +  .  ID=mRNA00001;Parent=gene00001
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeGFF3(tokens, header) {

    const feature = decode(tokens, header);

    if (!feature) {
        return
    }

    const attributes =  parseAttributeString(feature.attributeString, feature.delim);

    // Search for color value as case insenstivie key
    for (let [key, value] of attributes) {
        const keyLower = key.toLowerCase();
        if ("color" === keyLower || "colour" === keyLower) {
            feature.color = IGVColor.createColorString(value);
        } else if (key === "ID") {
            feature.id = value;
        } else if (key === "Parent") {
            feature.parent = value;
        }
    }
    return feature
}

/**
 * GTF format example:
 NC_000008.11    BestRefSeq    gene    127735434    127742951    .    +    .    gene_id "MYC"; transcript_id ""; db_xref "GeneID:4609"; db_xref "HGNC:HGNC:7553"; db_xref "MIM:190080"; description "MYC proto-oncogene, bHLH transcription factor"; gbkey "Gene"; gene "MYC"; gene_biotype "protein_coding"; gene_synonym "bHLHe39"; gene_synonym "c-Myc"; gene_synonym "MRTL"; gene_synonym "MYCC";
 NC_000008.11    BestRefSeq    transcript    127735434    127742951    .    +    .    gene_id "MYC"; transcript_id "NM_001354870.1"; db_xref "GeneID:4609"; gbkey "mRNA"; gene "MYC"; product "MYC proto-oncogene, bHLH transcription factor, transcript variant 2"; transcript_biotype "mRNA";
 NC_000008.11    BestRefSeq    exon    127735434    127736623    .    +    .    gene_id "MYC"; transcript_id "NM_001354870.1"; db_xref "GeneID:4609"; gene "MYC"; product "MYC proto-oncogene, bHLH transcription factor, transcript variant 2"; transcript_biotype "mRNA"; exon_number "1";
 *
 * @param tokens
 * @param ignore
 * @returns {*}
 */
function decodeGTF(tokens, header) {

    const feature = decode(tokens, header);

    if (!feature) {
        return
    }

    const attributes = parseAttributeString(feature.attributeString, feature.delim);

    // GTF files specify neither ID nor parent fields, but they can be inferred from common conventions
    let idField;
    let parentField;
    switch (feature.type) {
        case "gene":
            idField = "gene_id";
            break
        case "transcript":
            idField = "transcript_id";
            parentField = "gene_id";
            break
        default:
            parentField = "transcript_id";
    }

    for (let [key, value] of attributes) {
        const keyLower = key.toLowerCase();
        if ("color" === keyLower || "colour" === keyLower) {
            feature.color = IGVColor.createColorString(value);
        } else if (key === idField) {
            feature.id = value;
        } else if (key === parentField) {
            feature.parent = value;
        }
    }
    return feature

}

function decodeFusionJuncSpan(tokens, header) {

    /*
     Format:

     0       #scaffold
     1       fusion_break_name
     2       break_left
     3       break_right
     4       num_junction_reads
     5       num_spanning_frags
     6       spanning_frag_coords

     0       B3GNT1--NPSR1
     1       B3GNT1--NPSR1|2203-10182
     2       2203
     3       10182
     4       189
     5       1138
     6       1860-13757,1798-13819,1391-18127,1443-17174,...

     */


    if (tokens.length < 7) return undefined

    var chr = tokens[0];
    var fusion_name = tokens[1];
    var junction_left = parseInt(tokens[2]);
    var junction_right = parseInt(tokens[3]);
    var num_junction_reads = parseInt(tokens[4]);
    var num_spanning_frags = parseInt(tokens[5]);

    var spanning_frag_coords_text = tokens[6];

    var feature = {
        chr: chr,
        name: fusion_name,
        junction_left: junction_left,
        junction_right: junction_right,
        num_junction_reads: num_junction_reads,
        num_spanning_frags: num_spanning_frags,
        spanning_frag_coords: [],

        start: -1,
        end: -1
    }; // set start and end later based on min/max of span coords

    var min_coord = junction_left;
    var max_coord = junction_right;

    if (num_spanning_frags > 0) {

        var coord_pairs = spanning_frag_coords_text.split(',');

        for (var i = 0; i < coord_pairs.length; i++) {
            var split_coords = coord_pairs[i].split('-');

            var span_left = split_coords[0];
            var span_right = split_coords[1];

            if (span_left < min_coord) {
                min_coord = span_left;
            }
            if (span_right > max_coord) {
                max_coord = span_right;
            }
            feature.spanning_frag_coords.push({left: span_left, right: span_right});

        }
    }

    feature.start = min_coord;
    feature.end = max_coord;


    return feature

}

function decodeGtexGWAS(tokens, header) {
    //chrom	chromStart	chromEnd	Strongest SNP-risk allele	Disease/Phenotype	P-value	Odds ratio or beta	PUBMEDID
    //1	1247493	1247494	rs12103-A	Inflammatory bowel disease	8.00E-13	1.1	23128233

    const tokenCount = tokens.length;
    if (tokenCount < 7) {
        return null
    }
    const feature = {
        chr: tokens[0],
        start: parseInt(tokens[1]) - 1,
        end: parseInt(tokens[2]),
        'Strongest SNP-risk allele': tokens[3],
        'Disease/Phenotype': tokens[4],
        'P-value': tokens[5],
        'Odds ratio or beta': tokens[6],
    };
    if (tokens.length > 6) {
        feature['PUBMEDID'] = `<a target = "blank" href = "https://www.ncbi.nlm.nih.gov/pubmed/${tokens[7]}">${tokens[7]}</a>`;
    }
    return feature
}

/**
 * Decode a custom columnar format.  Required columns are 'chr' and 'start'
 *
 * @param tokens
 * @param ignore
 * @returns decoded feature, or null if this is not a valid record
 */
function decodeCustom(tokens, header) {

    const format = header.customFormat;

    if (tokens.length < format.fields.length) return undefined

    const coords = format.coords || 0;

    const chr = tokens[format.chr];
    const start = parseInt(tokens[format.start]) - coords;
    const end = format.end !== undefined ? parseInt(tokens[format.end]) : start + 1;

    const feature = {chr: chr, start: start, end: end};

    if (format.fields) {
        format.fields.forEach(function (field, index) {

            if (index !== format.chr &&
                index !== format.start &&
                index !== format.end) {

                feature[field] = tokens[index];
            }
        });
    }

    return feature

}

/**
 * Decode a gcnv record, a bed style format encoding copy number variation
 *
 * @param tokens
 * @param header
 */

function decodeGcnv(tokens, header) {

    const columnNames = header.columnNames;
    if (!columnNames) {
        throw Error("Sample names are not defined.   Missing column headers?")
    }
    const sampleCount = columnNames.length - 3;

    const chr = tokens[0];
    const start = parseInt(tokens[1]);
    const end = parseInt(tokens[2]);
    const values = tokens.slice(3).map(Number);

    if (values.length == sampleCount) {
        return {
            chr: chr,
            start: start,
            end: end,
            values: values,
        }
    } else {
        // TODO Throw error?
        console.warn(`${chr}:${start}-${end} row contains ${values.length} sample columns instead of the expected ${sampleCount} columns. Skipping...`);
        return undefined
    }
}

function decodeShoebox(tokens, header, maxColumnCount = Number.MAX_SAFE_INTEGER) {

    if (tokens.length < 4) return undefined

    const chr = tokens[0];
    const start = parseInt(tokens[1]);
    const end = parseInt(tokens[2]);
    if (isNaN(start) || isNaN(end)) {
        return new DecodeError(`Unparsable bed record.`)
    }
    const feature = {chr, start, end};

    const values = [];
    for(let i = 3; i< tokens.length; i++) {
        values.push(Number.parseFloat(tokens[i]));
    }
    feature.values = values;


    return feature
}

/**
 * Utilities for creating and registering custom file formats for generic delimited files.  A format definition consists
 * of an ordered list of fields, and optional delimiter specified.
 *
 *
 */


/**
 * Register a new custom file format.
 * @param name
 * @param fields
 */
function registerFileFormats(name, fields) {
    FileFormats[name] = {fields: fields};
}

/**
 * Return a custom format object with the given name.
 * @param name
 * @returns {*}
 */
function getFormat(name) {

    if (FileFormats && FileFormats[name]) {
        return expandFormat(FileFormats[name])
    } else {
        return undefined
    }

    function expandFormat(format) {

        const fields = format.fields;
        const keys = ['chr', 'start', 'end'];

        for (let i = 0; i < fields.length; i++) {
            for (let key of keys) {
                if (key === fields[i]) {
                    format[key] = i;
                }
            }
        }

        return format
    }
}



/**
 * Table of custom formats, with several pre-defined.
 *
 * @type {{wgrna: {fields: string[]}, cpgislandext: {fields: string[]}, clinVarMain: {fields: string[]}, gwascatalog: {fields: string[]}}}
 */
const FileFormats = {

    gwascatalog: {
        fields: [
            'bin',
            'chr',
            'start',
            'end',
            'name',
            'pubMedID',
            'author',
            'pubDate',
            'journal',
            'title',
            'trait',
            'initSample',
            'replSample',
            'region',
            'genes',
            'riskAllele',
            'riskAlFreq',
            'pValue',
            'pValueDesc',
            'orOrBeta',
            'ci95',
            'platform',
            'cnv'
        ]
    },

    wgrna: {
        fields:
            [
                'bin',
                'chr',
                'start',
                'end',
                'name',
                'score',
                'strand',
                'thickStart',
                'thickEnd',
                'type'
            ]
    },

    cpgislandext: {
        fields:
            [
                'bin',
                'chr',
                'start',
                'end',
                'name',
                'length',
                'cpgNum',
                'gcNum',
                'perCpg',
                'perGc',
                'obsExp'
            ]
    },

    clinVarMain: {
        fields: [
            'chr1',
            'start',
            'end',
            'name',
            'score',
            'strand',
            'thickStart',
            'thickEnd',
            'reserved',
            'blockCount',  // Number of blocks
            'blockSizes',  // Comma separated list of block sizes
            'chromStarts', // Start positions relative to chromStart
            'origName',    // NM_198053.2(CD247):c.462C>T (p.Asp154=)	ClinVar Variation Report
            'clinSign',    // Likely benign	Clinical significance
            'reviewStatus', // 	based on: criteria provided,single submitter	Review Status
            'type',         // single nucleotide variant	Type of Variant
            'geneId', 	    // CD247	Gene Symbol
            'snpId',       //	181656780	dbSNP ID
            'nsvId',       //		dbVar ID
            'rcvAcc',      //	RCV000642347	ClinVar Allele Submission
            'testedInGtr', //	N	Genetic Testing Registry
            'phenotypeList', //	Immunodeficiency due to defect in cd3-zeta	Phenotypes
            'phenotype', //	MedGen:C1857798, OMIM:610163	Phenotype identifiers
            'origin', //	germline	Data origin
            'assembly', //	GRCh37	Genome assembly
            'cytogenetic', //	1q24.2	Cytogenetic status
            'hgvsCod', //	NM_198053.2:c.462C>T	Nucleotide HGVS
            'hgvsProt', //	NP_932170.1:p.Asp154=	Protein HGVS
            'numSubmit', //	1	Number of submitters
            'lastEval', //	Dec 19,2017	Last evaluation
            'guidelines', //		Guidelines
            'otherIds'
        ]
    }
};

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 *  Parser for column style (tab delimited, etc) text file formats (bed, gff, vcf, etc).
 *
 *
 */


/**
 * Return a parser for the given file format.
 */
class FeatureParser {

    constructor(config) {

        this.config = config;
        this.header = {};
        if (config.nameField) {
            this.header.nameField = config.nameField;
        }

        this.skipRows = 0;   // The number of fixed header rows to skip.  Override for specific types as needed

        if (config.decode) {
            this.decode = config.decode;
            this.delimiter = config.delimiter || "\t";
        } else if (config.format) {
            this.header.format = config.format.toLowerCase();
            this.setDecoder(this.header.format);
        }

        if (!this.delimiter) {
            this.delimiter = "\t";
        }
    }

    /**
     * Parse metadata from the file.   A variety of conventions are in use to supply metadata about file contents
     * through header lines (e.g. 'track') and # directives. This method unifies metadata as properties of a
     * 'header' object.
     *
     * @param data
     * @returns {{}}
     */
    async parseHeader(dataWrapper) {

        let header = this.header;
        let columnNames;
        let line;
        while ((line = await dataWrapper.nextLine()) !== undefined) {
            if (line.startsWith("track") || line.startsWith("#track")) {
                let h = parseTrackLine$1(line);
                Object.assign(header, h);
            } else if (line.startsWith("browser")) ; else if (line.startsWith("#columns")) {
                let h = parseColumnsDirective$1(line);
                Object.assign(header, h);
            } else if (line.startsWith("##gff-version 3")) {
                header.format = "gff3";
            } else if (line.startsWith("#gffTags")) {
                header.gffTags = true;
            } else if (line.startsWith("fixedStep") || line.startsWith("variableStep")) {
                // Wig directives -- we are in the data section
                break
            } else if (line.startsWith("#")) {
                const tokens = line.split(this.delimiter || "\t");
                if (tokens.length > 1) {
                    columnNames = tokens;   // Possible column names
                }
            } else {
                // All directives that could change the format, and thus decoder, should have been read by now.
                // Set the decoder, unless it is explicitly set in the track configuration (not common)
                if(!this.config.decode) {
                    this.setDecoder(header.format);
                }

                // If the line can be parsed as a feature assume we are beyond the header, if any
                const tokens = line.split(this.delimiter || "\t");
                try {
                    const tmpHeader = Object.assign({columnNames}, header);
                    let firstFeature;
                    if (firstFeature = this.decode(tokens, tmpHeader)) {
                        header.firstFeature = firstFeature;
                        break
                    } else {
                        if (tokens.length > 1) {
                            columnNames = tokens; // possible column names
                        }
                    }
                } catch (e) {
                    // Not a feature
                    if (tokens.length > 1) {
                        columnNames = tokens; // possible column names
                    }
                }
            }
        }

        if (columnNames) {
            header.columnNames = columnNames;
            for (let n = 0; n < columnNames.length; n++) {
                if (columnNames[n] === "color" || columnNames[n] === "colour") {
                    header.colorColumn = n;
                } else if (columnNames[n] === "thickness") {
                    header.thicknessColumn = n;
                }
            }
        }

        this.header = header;    // Directives might be needed for parsing lines
        return header
    }

    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        const decode = this.decode;
        const format = this.header.format;
        const delimiter = this.delimiter || "\t";
        let i = 0;
        let errorCount = 0;
        let line;
        while ((line = await dataWrapper.nextLine()) !== undefined) {
            i++;
            if (i <= this.skipRows) continue

            if (!line || line.startsWith("track") || line.startsWith("#") || line.startsWith("browser")) {
                continue
            } else if (format === "wig" && line.startsWith("fixedStep")) {
                this.header.wig = parseFixedStep(line);
                continue
            } else if (format === "wig" && line.startsWith("variableStep")) {
                this.header.wig = parseVariableStep(line);
                continue
            }

            const tokens = line.split(delimiter);
            if (tokens.length < 1) {
                continue
            }

            const feature = decode(tokens, this.header);

            if (feature instanceof DecodeError) {
                errorCount++;
                if (errorCount > 0) {
                    console.error(`Error parsing line '${line}': ${feature.message}`);
                }
                continue
            }

            if (feature) {
                allFeatures.push(feature);
            }
        }

        // Special hack for bedPE
        if (decode === decodeBedpe) {
            fixBedPE(allFeatures);
        }

        if (("gtf" === this.config.format || "gff3" === this.config.format || "gff" === this.config.format) &&
            this.config.assembleGFF !== false) {
            return (new GFFHelper(this.config)).combineFeatures(allFeatures)
        } else {
            return allFeatures
        }

    }

    setDecoder(format) {

        switch (format) {
            case "broadpeak":
            case "regionpeak":
            case "peaks":
                this.decode = decodePeak;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "narrowpeak":
                this.decode = decodeNarrowPeak;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "bedgraph":
                this.decode = decodeBedGraph;
                this.delimiter = /\s+/;
                break
            case "wig":
                this.decode = decodeWig;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "gff3" :
            case "gff":
                this.decode = decodeGFF3;
                this.delimiter = "\t";
                break
            case "gtf" :
                this.decode = decodeGTF;
                this.delimiter = "\t";
                break
            case "fusionjuncspan":
                // bhaas, needed for FusionInspector view
                this.decode = decodeFusionJuncSpan;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "gtexgwas":
                this.skipRows = 1;
                this.decode = decodeGtexGWAS;
                this.delimiter = "\t";
                break
            case "refflat":
                this.decode = decodeReflat;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "genepred":
                this.decode = decodeGenePred;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "genepredext":
                this.decode = decodeGenePredExt;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "ensgene":
                this.decode = decodeGenePred;
                this.header.shift = 1;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "refgene":
                this.decode = decodeGenePredExt;
                this.delimiter = this.config.delimiter || /\s+/;
                this.header.shift = 1;
                break
            case "bed":
                this.decode = decodeBed;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "gappedpeak":
                this.decode = decodeGappedPeak;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "bedmethyl":
                this.decode = decodeBedmethyl;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "bedpe":
            case "hiccups":
                this.decode = decodeBedpe;
                this.delimiter = this.config.delimiter || "\t";
                break
            case "bedpe-domain":
                this.decode = decodeBedpeDomain;
                this.headerLine = true;
                this.delimiter = this.config.delimiter || "\t";
                break
            case "bedpe-loop":
                this.decode = decodeBedpe;
                this.delimiter = this.config.delimiter || "\t";
                this.header = {colorColumn: 7};
                break
            case "interact":
                this.decode = decodeInteract;
                this.delimiter = this.config.delimiter || /\s+/;
                break
            case "snp":
                this.decode = decodeSNP;
                this.delimiter = "\t";
                break
            case "rmsk":
                this.decode = decodeRepeatMasker;
                this.delimiter = "\t";
                break
            case "gcnv":
                this.decode = decodeGcnv;
                this.delimiter = "\t";
                break
            case "shoebox":
                this.decode = decodeShoebox;
                this.delimiter = "\t";
                break
            default:
                const customFormat = getFormat(format);
                if (customFormat !== undefined) {
                    this.decode = decodeCustom;
                    this.header.customFormat = customFormat;
                    this.delimiter = customFormat.delimiter || "\t";
                } else {
                    this.decode = decodeBed;
                    this.delimiter = this.config.delimiter || /\s+/;
                }
        }

    }
}

function parseTrackLine$1(line) {

    const properties = {};
    const tokens = line.split(/(?:")([^"]+)(?:")|([^\s"]+)(?=\s+|$)/g);

    // Clean up tokens array
    let curr;
    const tmp = [];
    for (let tk of tokens) {
        if (!tk || tk.trim().length === 0) continue
        if (tk.endsWith("=")) {
            curr = tk;
        } else if (curr) {
            tmp.push(curr + tk);
            curr = undefined;
        } else {
            tmp.push(tk);
        }
    }

    for (let str of tmp) {
        if (!str) return
        var kv = str.split('=', 2);
        if (kv.length === 2) {
            const key = kv[0].trim();
            const value = kv[1].trim();
            if (properties.hasOwnProperty(key)) {
                let currentValue = properties[key];
                if (Array.isArray(currentValue)) {
                    currentValue.push(value);
                } else {
                    properties[key] = [currentValue, value];
                }
            } else {
                properties[key] = value;
            }
        }
    }
    if ("interact" == properties["type"]) {
        properties["format"] = "interact";
    } else if ("gcnv" === properties["type"]) {
        properties["format"] = "gcnv";
    }
    return properties
}

function parseColumnsDirective$1(line) {

    let properties = {};
    let t1 = line.split(/\s+/);

    if (t1.length === 2) {
        let t2 = t1[1].split(";");
        t2.forEach(function (keyValue) {
            let t = keyValue.split("=");
            if (t[0] === "color") {
                properties.colorColumn = Number.parseInt(t[1]) - 1;
            } else if (t[0] === "thickness") {
                properties.thicknessColumn = Number.parseInt(t[1]) - 1;
            }
        });
    }

    return properties
}

function parseFixedStep(line) {
    const tokens = line.split(/\s+/);
    const chrom = tokens[1].split("=")[1];
    const start = parseInt(tokens[2].split("=")[1], 10) - 1;
    const step = parseInt(tokens[3].split("=")[1], 10);
    const span = (tokens.length > 4) ? parseInt(tokens[4].split("=")[1], 10) : 1;
    return {format: "fixedStep", chrom, start, step, span, index: 0}
}

function parseVariableStep(line) {
    const tokens = line.split(/\s+/);
    const chrom = tokens[1].split("=")[1];
    const span = tokens.length > 2 ? parseInt(tokens[2].split("=")[1], 10) : 1;
    return {format: "variableStep", chrom, span}
}

function createCheckbox(name, initialState) {

    const container = div$2({class: 'igv-menu-popup-check-container'});

    const div$1 = div$2();
    container.appendChild(div$1);

    const svg = createIcon$1('check', (true === initialState ? '#444' : 'transparent'));
    div$1.appendChild(svg);

    const label = div$2();
    label.innerText = name;
    container.appendChild(label);

    return container
}

// Assigns a row # to each feature.  If the feature does not fit in any row and #rows == maxRows no
// row number is assigned.
function pack(featureList, maxRows) {

    maxRows = maxRows || Number.MAX_SAFE_INTEGER;
    const rows = [];
    featureList.sort(function (a, b) {
        return a.start - b.start
    });
    rows.push(-1000);

    for (let feature of featureList) {
        let r = 0;
        const len = Math.min(rows.length, maxRows);
        for (r = 0; r < len; r++) {
            if (feature.start >= rows[r]) {
                feature.row = r;
                rows[r] = feature.end;
                break
            }
        }
        feature.row = r;
        rows[r] = feature.end;
    }
}

const DEFAULT_MAX_WG_COUNT$1 = 10000;

/**
 * Return a collection of "whole genome" features wrapping the supplied features, possibly downsampled.  The purpose
 * is to support painting features in "whole genome view".
 *
 * @param allFeatures - dictionary (object), keys are chromosome names, values are lists of features
 * @param genome
 * @param maxWGCount - optional, maximum # of whole genome features to computer
 * @returns {*[]}
 */
async function computeWGFeatures(allFeatures, genome, maxWGCount) {

    const makeWGFeature = (f) => {
        const wg = Object.assign({}, f);
        wg.chr = "all";
        wg.start = genome.getGenomeCoordinate(f.chr, f.start);
        wg.end = genome.getGenomeCoordinate(f.chr, f.end);
        wg._f = f;
        // Don't draw exons in whole genome view
        if (wg["exons"]) delete wg["exons"];
        return wg
    };

    const wgChromosomeNames = new Set(genome.wgChromosomeNames);
    const wgFeatures = [];
    let count = 0;
    for (let c of genome.wgChromosomeNames) {

        if (Array.isArray(allFeatures)) {
            const featureDict = {};
            for (let f of allFeatures) {
                const chr = genome.getChromosomeName(f.chr);
                if (!featureDict.hasOwnProperty(chr)) {
                    featureDict[chr] = [];
                }
                featureDict[chr].push(f);
            }
            allFeatures = featureDict;
        }

        const features = allFeatures[c];

        if (features) {
            const max = maxWGCount || DEFAULT_MAX_WG_COUNT$1;
            for (let f of features) {
                let queryChr = genome.getChromosomeName(f.chr);
                if (wgChromosomeNames.has(queryChr)) {
                    if (wgFeatures.length < max) {
                        wgFeatures.push(makeWGFeature(f));
                    } else {
                        //Reservoir sampling
                        const samplingProb = max / (count + 1);
                        if (Math.random() < samplingProb) {
                            const idx = Math.floor(Math.random() * (max - 1));
                            wgFeatures[idx] = makeWGFeature(f);
                        }
                    }
                }
                count++;
            }
        }
    }

    wgFeatures.sort(function (a, b) {
        return a.start - b.start
    });

    return wgFeatures
}

/**
 * Assigns a row to each feature such that features do not overlap.
 *
 * @param features
 * @param maxRows
 * @param filter Function thta takes a feature and returns a boolean indicating visibility
 */
function packFeatures(features, maxRows, filter) {

    maxRows = maxRows || 1000;
    if (features == null || features.length === 0) {
        return
    }
    // Segregate by chromosome
    const chrFeatureMap = {};
    const chrs = [];
    for (let feature of features) {
        if(filter && !filter(feature)) {
            feature.row = undefined;
        } else {
            const chr = feature.chr;
            let flist = chrFeatureMap[chr];
            if (!flist) {
                flist = [];
                chrFeatureMap[chr] = flist;
                chrs.push(chr);
            }
            flist.push(feature);
        }
    }

    // Loop through chrosomosomes and pack features;
    for (let chr of chrs) {
        pack(chrFeatureMap[chr], maxRows);
    }
}


/**
 * Return the index at which a new feature should be inserted in the sorted featureList.  It is assumed
 * that featureList is sorted by the compare function.  If featureList has 1 or more features with compare === 0
 * the new feature should be inserted at the end.
 *
 * @param featureList
 * @param center
 * @param direction -- forward === true, reverse === false
 * @returns {feature}
 */

function findFeatureAfterCenter(featureList, center, direction = true) {

    const featureCenter = (feature) => (feature.start + feature.end) / 2;

    const compare = direction ?
        (o1, o2) => o1.start - o2.start + o1.end - o2.end :
        (o2, o1) => o1.start - o2.start + o1.end - o2.end;
    const sortedList = Array.from(featureList);
    sortedList.sort(compare);

    let low = 0;
    let high = sortedList.length;
    while (low < high) {
        let mid = Math.floor((low + high) / 2);
        if(direction) {
            if (featureCenter(sortedList[mid]) <= center) {
                low = mid + 1;
            } else {
                high = mid;
            }
        } else {
            if (featureCenter(sortedList[mid]) >= center) {
                low = mid + 1;
            } else {
                high = mid;
            }

        }
    }
    return sortedList[low]
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2018 Regents of the University of California
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const fixColor = (colorString) => {
    if (isString$3(colorString)) {
        return (colorString.indexOf(",") > 0 && !(colorString.startsWith("rgb(") || colorString.startsWith("rgba("))) ?
            `rgb(${colorString})` : colorString
    } else {
        return colorString
    }
};

/**
 * A collection of properties and methods shared by all (or most) track types.
 *
 * @param config
 * @param browser
 * @constructor
 */
class TrackBase {

    static defaultColor = 'rgb(150,150,150)'

    static defaults = {
        height: 50,
        autoHeight: false,
        visibilityWindow: undefined,   // Identifies property that should be copied from config
        color: undefined,  // Identifies property that should be copied from config
        altColor: undefined,  // Identifies property that should be copied from config
        supportHiDPI: true,
        selected: false
    }

    constructor(config, browser) {
        this.browser = browser;
        this.init(config);
    }

    /**
     * Initialize track properties from the config object.  This method is typically overriden in subclasses, which
     * will call this implementation as super.init(config).
     *
     * @param config
     */
    init(config) {

        this.config = config;

        if (config.displayMode) {
            config.displayMode = config.displayMode.toUpperCase();
        }

        // Base default settings
        const defaults = Object.assign({}, TrackBase.defaults);

        // Overide with class specific default settings
        if (this.constructor.defaults) {
            for (let key of Object.keys(this.constructor.defaults)) {
                defaults[key] = this.constructor.defaults[key];
            }
        }

        for (let key of Object.keys(defaults)) {
            this[key] = config.hasOwnProperty(key) ? config[key] : defaults[key];
            if ((key === 'color' || key === 'altColor') && this[key]) {
                this[key] = fixColor(this[key]);
            }
        }

        // this._initialColor = this.color || this.constructor.defaultColor
        // this._initialAltColor = this.altColor || this.constructor.defaultColor

        if (config.name || config.label) {
            this.name = config.name || config.label;
        } else if (isFile$1(config.url)) {
            this.name = config.url.name;
        } else if (isString$3(config.url) && !config.url.startsWith("data:")) {
            this.name = getFilename$2(config.url);
        }

        this.url = config.url;
        if (this.config.type) this.type = this.config.type;
        this.id = this.config.id === undefined ? this.name : this.config.id;
        this.order = config.order;
        this.autoscaleGroup = config.autoscaleGroup;
        this.removable = config.removable === undefined ? true : config.removable;      // Defaults to true
        this.minHeight = config.minHeight || Math.min(25, this.height);
        this.maxHeight = config.maxHeight || Math.max(1000, this.height);

        if (config.onclick) {
            this.onclick = config.onclick;
            config.onclick = undefined;   // functions cannot be saved in sessions, clear it here.
        }

        if (config.description) {
            // Override description -- displayed when clicking on track label.  Convert to function if neccessary
            if (typeof config.description === 'function') {
                this.description = config.description;
            } else {
                this.description = () => config.description;
            }
        }

        // Support for mouse hover text.  This can be expensive, off by default.
        // this.hoverText = function(clickState) => return tool tip text
        if (config.hoverTextFields) {
            this.hoverText = hoverText.bind(this);
        } else if (typeof this.config.hoverText === 'function') {
            this.hoverText = this.config.hoverText;
        }
    }

    async postInit() {

        this._initialColor = this.color || this.constructor.defaultColor;
        this._initialAltColor = this.altColor || this.constructor.defaultColor;
        return this
    }

    get name() {
        return this._name
    }

    set name(name) {
        this._name = name;
        if (this.trackView) {
            this.trackView.setTrackLabelName(name);
        }
    }

    clearCachedFeatures() {
        if (this.trackView) {
            this.trackView.clearCachedFeatures();
        }
    }

    updateViews() {
        if (this.trackView) {
            this.trackView.updateViews();
        }
    }

    repaintViews() {
        if (this.trackView) {
            this.trackView.repaintViews();
        }
    }

    /**
     * Used to create session object for bookmarking, sharing.  Only simple property values (string, number, boolean)
     * are saved.
     */
    getState() {

        const isJSONable = item => !(item === undefined || typeof item === 'function' || item instanceof Promise);

        // Create copy of config, minus transient properties (convention is name starts with '_').  Also, all
        // function properties are transient as they cannot be saved in json
        const state = {};

        const jsonableConfigKeys = Object.keys(this.config).filter(key => isJSONable(this.config[key]));

        for (const key of jsonableConfigKeys) {
            if (!key.startsWith("_")) {
                state[key] = this.config[key];
            }
        }

        // Update original config values with any changes
        for (let key of Object.keys(state)) {
            if (key.startsWith("_")) continue   // transient property
            const value = this[key];
            if (value !== undefined && (isSimpleType(value) || typeof value === "boolean" || key === "metadata")) {
                state[key] = value;
            }
        }

        // If user has changed other properties from defaults update state.
        const defs = Object.assign({}, TrackBase.defaults);
        if (this.constructor.defaults) {
            for (let key of Object.keys(this.constructor.defaults)) {
                defs[key] = this.constructor.defaults[key];
            }
        }
        for (let key of Object.keys(defs)) {
            if (undefined !== this[key] && defs[key] !== this[key]) {
                state[key] = this[key];
            }
        }

        // Flatten dataRange if present
        if (!this.autoscale && this.dataRange) {
            state.min = this.dataRange.min;
            state.max = this.dataRange.max;
        }

        if (this.autoscaleGroup) {
            state.autoscaleGroup = this.autoscaleGroup;
        }

        return state
    }

    get supportsWholeGenome() {
        return this.config.supportsWholeGenome === true
    }

    /**
     * Does the track support sample names.  Current sample aware tracks include VCF (with genotypes), MUT, MAF, and SEG
     * @returns {boolean}
     */
    hasSamples() {
        return false
    }

    getGenomeId() {
        return this.browser.genome ? this.browser.genome.id : undefined
    }

    /**
     * Set certain track properties, usually from a "track" line.  Not all UCSC properties are supported.
     *
     * Track configuration settings have precendence over track line properties, so if both are present ignore the
     * track line.
     *
     * @param properties
     */
    setTrackProperties(properties) {

        if (this.disposed) return   // This track was removed during async load

        const tracklineConfg = {};
        let tokens;
        for (let key of Object.keys(properties)) {
            switch (key.toLowerCase()) {
                case "usescore":
                    tracklineConfg.useScore = (
                        properties[key] === 1 || properties[key] === "1" || properties[key] === "on" || properties[key] === true);
                    break
                case "visibility":
                    //0 - hide, 1 - dense, 2 - full, 3 - pack, and 4 - squish
                    switch (properties[key]) {
                        case "2":
                        case "3":
                        case "pack":
                        case "full":
                            tracklineConfg.displayMode = "EXPANDED";
                            break
                        case "4":
                        case "squish":
                            tracklineConfg.displayMode = "SQUISHED";
                            break
                        case "1":
                        case "dense":
                            tracklineConfg.displayMode = "COLLAPSED";
                    }
                    break
                case "color":
                case "altcolor":
                    tracklineConfg[key] = properties[key].startsWith("rgb(") ? properties[key] : "rgb(" + properties[key] + ")";
                    break
                case "featurevisiblitywindow":
                case "visibilitywindow":
                    tracklineConfg.visibilityWindow = Number.parseInt(properties[key]);
                    break
                case "maxheightpixels":
                    tokens = properties[key].split(":");
                    if (tokens.length === 3) {
                        tracklineConfg.minHeight = Number.parseInt(tokens[2]);
                        tracklineConfg.height = Number.parseInt(tokens[1]);
                        tracklineConfg.maxHeight = Number.parseInt(tokens[0]);
                    }
                    break
                case "viewlimits":
                    if (!this.config.autoscale && !this.config.max) {   //config has precedence
                        tokens = properties[key].split(":");
                        let min = 0;
                        let max;
                        if (tokens.length == 1) {
                            max = Number(tokens[0]);
                        } else if (tokens.length == 2) {
                            min = Number(tokens[0]);
                            max = Number(tokens[1]);
                        }
                        if (Number.isNaN(max) || Number.isNaN(min)) {
                            console.warn(`Unexpected viewLimits value in track line: ${properties["viewLimits"]}`);
                        } else {
                            tracklineConfg.autoscale = false;
                            tracklineConfg.dataRange = {min, max};
                            this.viewLimitMin = min;
                            this.viewLimitMax = max;
                        }
                    }
                case "name":
                    tracklineConfg[key] = properties[key];
                    break
                case "url":
                    tracklineConfg["infoURL"] = properties[key];
                    break
                case "type":
                    const v = properties[key];
                    if (UCSCTypeMappings.has(v)) {
                        tracklineConfg[key] = UCSCTypeMappings.get(v);
                    } else {
                        tracklineConfg[key] = v;
                    }
                    break
                case "graphtype":
                    tracklineConfg["graphType"] = properties[key];
                    break
                default:
                    tracklineConfg[key] = properties[key];
            }
        }

        // Track configuration objects have precedence over track line properties in general.  The "name" property
        // is an exception if it was derived and not explicitly entered (that is derived from the web app from filename).
        for (let key of Object.keys(tracklineConfg)) {

            if (!this.config.hasOwnProperty(key) || (key === "name" && this.config._derivedName)) {
                let value = tracklineConfg[key];
                if ("true" === value) value = true;
                if ("false" === value) value = false;

                this[key] = value;
                if (key === "height" && this.trackView) {
                    try {
                        const h = Number.parseInt(value);
                        this.trackView.setTrackHeight(h);
                    } catch (e) {
                        console.error(e);
                    }
                }
            }
        }
    }

    /**
     * Return the features clicked over.  Default implementation assumes an array of features and only considers
     * the genomic location.   Overriden by most subclasses.
     *
     * @param clickState
     * @returns {[]|*[]}
     */
    clickedFeatures(clickState) {

        // We use the cached features rather than method to avoid async load.  If the
        // feature is not already loaded this won't work,  but the user wouldn't be mousing over it either.
        const features = clickState.viewport.cachedFeatures;

        if (!features || !Array.isArray(features) || features.length === 0) {
            return []
        }

        const genomicLocation = clickState.genomicLocation;

        // When zoomed out we need some tolerance around genomicLocation
        const tolerance = (clickState.referenceFrame.bpPerPixel > 0.2) ? 3 * clickState.referenceFrame.bpPerPixel : 0.2;
        const ss = genomicLocation - tolerance;
        const ee = genomicLocation + tolerance;
        return (FeatureUtils.findOverlapping(features, ss, ee))
    }

    /**
     * Default popup text function -- just extracts string and number properties in random order.
     * @param feature     * @returns {Array}
     */
    extractPopupData(feature, genomeId) {

        const filteredProperties = new Set(['row', 'color', 'chr', 'start', 'end', 'cdStart', 'cdEnd', 'strand', 'alpha']);
        const data = [];

        let alleles, alleleFreqs;
        for (let property in feature) {

            if (feature.hasOwnProperty(property) &&
                !filteredProperties.has(property) &&
                isSimpleType(feature[property])) {

                let value = feature[property];
                data.push({name: capitalize(property), value: value});

                if (property === "alleles") {
                    alleles = feature[property];
                } else if (property === "alleleFreqs") {
                    alleleFreqs = feature[property];
                }
            }
        }

        if (alleles && alleleFreqs) {

            if (alleles.endsWith(",")) {
                alleles = alleles.substr(0, alleles.length - 1);
            }
            if (alleleFreqs.endsWith(",")) {
                alleleFreqs = alleleFreqs.substr(0, alleleFreqs.length - 1);
            }

            let a = alleles.split(",");
            let af = alleleFreqs.split(",");
            if (af.length > 1) {
                let b = [];
                for (let i = 0; i < af.length; i++) {
                    b.push({a: a[i], af: Number(af[i])});
                }
                b.sort(function (x, y) {
                    return x.af - y.af
                });

                let ref = b[b.length - 1].a;
                if (ref.length === 1) {
                    for (let i = b.length - 2; i >= 0; i--) {
                        let alt = b[i].a;
                        if (alt.length === 1) {
                            if (!genomeId) genomeId = this.getGenomeId();
                            const cravatLink = TrackBase.getCravatLink(feature.chr, feature.start + 1, ref, alt, genomeId);
                            if (cravatLink) {
                                data.push('<hr/>');
                                data.push({html: cravatLink});
                                data.push('<hr/>');
                            }
                        }
                    }
                }
            }
        }

        if (feature.attributes) {
            for (let key of Object.keys(feature.attributes)) {
                data.push({name: key, value: feature.attributes[key]});
            }
        }

        // final chr position
        let posString = `${feature.chr}:${numberFormatter$1(feature.start + 1)}-${numberFormatter$1(feature.end)}`;
        if (feature.strand) {
            posString += ` (${feature.strand})`;
        }

        data.push({name: 'Location', value: posString});

        return data

    }


    /**
     * Default track description -- displayed on click of track label.  This can be overriden in the track
     * configuration, or in subclasses.
     */
    description() {

        const wrapKeyValue = (k, v) => `<div class="igv-track-label-popup-shim"><b>${k}: </b>${v}</div>`;

        let str = '<div class="igv-track-label-popup">';
        if (this.url) {
            if (isFile$1(this.url)) {
                str += wrapKeyValue('Filename', this.url.name);
            } else {
                str += wrapKeyValue('URL', this.url);
            }
        } else {
            str = this.name;
        }
        if (this.config) {
            if (this.config.metadata) {
                for (let key of Object.keys(this.config.metadata)) {
                    const value = this.config.metadata[key];
                    str += wrapKeyValue(key, value);
                }
            }

            // Add any config properties that are capitalized
            for (let key of Object.keys(this.config)) {
                if (key.startsWith("_")) continue   // transient property
                let first = key.substr(0, 1);
                if (first !== first.toLowerCase()) {
                    const value = this.config[key];
                    if (value && isSimpleType(value)) {
                        str += wrapKeyValue(key, value);
                    }
                }
            }

        }
        str += '</div>';
        return str
    }

    /**
     * Return color for a specific feature of this track.  This default implementation is overriden by subclasses*
     * @param f
     * @returns {*|string|string}
     */
    getColorForFeature(f) {
        return (typeof this.color === "function") ? this.color(feature) : this.color
    }

    numericDataMenuItems() {

        const menuItems = [];

        // Data range or color scale

        if ("heatmap" !== this.graphType) {

            menuItems.push('<hr/>');

            function dialogPresentationHandler(e) {

                if (this.trackView.track.selected) {
                    this.browser.dataRangeDialog.configure(this.trackView.browser.getSelectedTrackViews());
                } else {
                    this.browser.dataRangeDialog.configure(this.trackView);
                }
                this.browser.dataRangeDialog.present(e);
            }

            menuItems.push({label: 'Set data range', dialog: dialogPresentationHandler});

            if (this.logScale !== undefined) {

                function logScaleHandler() {
                    this.logScale = !this.logScale;
                    this.trackView.repaintViews();
                }

                menuItems.push({element: createCheckbox("Log scale", this.logScale), click: logScaleHandler});
            }

            function autoScaleHandler() {
                this.autoscaleGroup = undefined;
                this.autoscale = !this.autoscale;
                this.browser.updateViews();
            }

            menuItems.push({element: createCheckbox("Autoscale", this.autoscale), click: autoScaleHandler});
        }

        return menuItems
    }

    setDataRange({min, max}) {

        this.dataRange = {min, max};
        this.autoscale = false;
        this.autoscaleGroup = undefined;
        this.trackView.repaintViews();
    }

    /**
     * Return the first feature in this track whose start position is > position
     * @param chr
     * @param position
     * @param direction
     * @returns {Promise<void>}
     */
    async nextFeatureAfter(chr, position, direction) {
        const viewport = this.trackView.viewports[0];
        let features = viewport.cachedFeatures;
        if (features && Array.isArray(features) && features.length > 0) {
            // Check chromosome, all cached features will share a chromosome
            const chrName = this.browser.genome.getChromosomeName(features[0].chr);
            if (chrName === chr) {
                const next = findFeatureAfterCenter(features, position, direction);
                if (next) {
                    return next
                }
            }
        }

        if (typeof this.featureSource.nextFeature === 'function') {
            return this.featureSource.nextFeature(chr, position, direction, this.visibilityWindow)
        }
    }

    /**
     * Track has been permanently removed.  Release resources and other cleanup
     */
    dispose() {

        this.disposed = true;

        // This should not be neccessary, but in case there is some unknown reference holding onto this track object,
        // for example in client code, release any resources here.
        for (let key of Object.keys(this)) {
            this[key] = undefined;
        }
    }

    static getCravatLink(chr, position, ref, alt, genomeID) {

        if ("hg38" === genomeID || "GRCh38" === genomeID) {

            const cravatChr = chr.startsWith("chr") ? chr : "chr" + chr;
            return `<a target="_blank" href="https://run.opencravat.org/result/nocache/variant.html` +
                `?chrom=${cravatChr}&pos=${position}&ref_base=${ref}&alt_base=${alt}"><b>Cravat ${ref}->${alt}</b></a>`

        } else {
            return undefined
        }
    }

    static localFileInspection(config) {

        const cooked = Object.assign({}, config);
        const lut =
            {
                url: 'file',
                indexURL: 'indexFile'
            };

        for (const key of ['url', 'indexURL']) {
            if (cooked[key] && cooked[key] instanceof File) {
                cooked[lut[key]] = cooked[key].name;
                delete cooked[key];
            }
        }

        return cooked
    }

    // Methods to support filtering api


    set filter(f) {
        this._filter = f;
        this.trackView.repaintViews();
    }

    /**
     * Return features visible in current viewports.
     *
     * @returns {*[]}
     */
    getInViewFeatures() {
        const inViewFeatures = [];
        for (let viewport of this.trackView.viewports) {
            if (viewport.isVisible()) {
                const referenceFrame = viewport.referenceFrame;
                referenceFrame.chr;
                const start = referenceFrame.start;
                const end = start + referenceFrame.toBP(viewport.getWidth());

                // We use the cached features  to avoid async load.  If the
                // feature is not already loaded it is by definition not in view.
                if (viewport.cachedFeatures) {
                    const viewFeatures = FeatureUtils.findOverlapping(viewport.cachedFeatures, start, end);
                    for (let f of viewFeatures) {
                        if (!this._filter || this._filter(f)) {
                            inViewFeatures.push(f);
                        }
                    }
                }
            }
        }
        return inViewFeatures
    }

    /**
     * Return an object enumerating filterable attributes, that is attributes that can be used to filter features.
     * The base implementation returns an empty object, overriden in subclasses.
     *
     * @returns {{}}
     */
    getFilterableAttributes() {
        return {}
    }
}

function hoverText(clickState) {

    if (!this.hoverTextFields) return

    const features = this.clickedFeatures(clickState);

    if (features && features.length > 0) {
        let str = "";
        for (let i = 0; i < features.length; i++) {
            if (i === 10) {
                str += "; ...";
                break
            }
            if (!features[i]) continue

            const f = features[i]._f || features[i];
            if (str.length > 0) str += "\n";

            str = "";
            for (let field of this.hoverTextFields) {
                if (str.length > 0) str += "\n";
                if (f.hasOwnProperty(field)) {
                    str += f[field];
                } else if (typeof f.getAttribute === "function") {
                    str += f.getAttribute(field);
                }
            }

        }
        return str
    }
}

/**
 * Map UCSC track line "type" setting to file format.  In igv.js "type" refers to the track type, not the input file format
 * @type {Map<string, string>}
 */
const UCSCTypeMappings = new Map([
    ["wiggle_0", "wig"],
    ["bed", "bed"],
    ["bigBed", "bigBed"],
    ["bigWig", "bigWig"]
]);

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


/**
 *  Define parser for seg files  (.bed, .gff, .vcf, etc).  A parser should implement 2 methods
 *
 *     parseHeader(data) - return an object representing a header.  Details are format specific
 *
 *     parseFeatures(data) - return a list of features
 *
 */


class SegParser {

    constructor(type) {
        this.type = type || 'seg';   // One of seg, mut, or maf

        switch (this.type) {
            case 'mut':
                this.sampleColumn = 3;
                this.chrColumn = 0;
                this.startColumn = 1;
                this.endColumn = 2;
                this.dataColumn = 4;
                break
            case 'maf':
                this.sampleColumn = 15;
                this.chrColumn = 4;
                this.startColumn = 5;
                this.endColumn = 6;
                this.dataColumn = 8;
                break
            default:
                this.sampleColumn = 0;
                this.chrColumn = 1;
                this.startColumn = 2;
                this.endColumn = 3;
            // Data column determined after reading header
        }
    }

    async parseHeader(dataWrapper) {
        let line;
        while ((line = await dataWrapper.nextLine()) !== undefined) {
            if (line.startsWith("#")) ; else {
                const tokens = line.split("\t");
                this.header = {headings: tokens};
                break
            }
        }
        return this.header
    }

    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        let extraHeaders;
        if (!this.header) {
            this.header = await this.parseHeader(dataWrapper);  // This will only work for non-indexed files
        }
        if ('seg' === this.type) {
            this.dataColumn = this.header.headings.length - 1;
        }
        if (this.header.headings.length > 5) {
            extraHeaders = this.extractExtraColumns(this.header.headings);
        }
        const valueColumnName = this.header.headings[this.dataColumn];

        let line;
        while ((line = await dataWrapper.nextLine()) !== undefined) {
            const tokens = line.split("\t");
            const value = ('seg' === this.type) ? Number(tokens[this.dataColumn]) : tokens[this.dataColumn];
            if (tokens.length > this.dataColumn) {
                const feature = new SegFeature({
                    sample: tokens[this.sampleColumn],
                    chr: tokens[this.chrColumn],
                    start: parseInt(tokens[this.startColumn]) - 1,
                    end: parseInt(tokens[this.endColumn]),
                    value,
                    valueColumnName
                });
                if (extraHeaders) {
                    const extraValues = this.extractExtraColumns(tokens);
                    feature.setAttributes({names: extraHeaders, values: extraValues});
                }
                allFeatures.push(feature);
            }
        }
        return allFeatures
    }

    extractExtraColumns(tokens) {
        const extras = [];
        for (let i = 0; i < tokens.length; i++) {
            if (i !== this.chrColumn && i !== this.startColumn && i !== this.endColumn && i !== this.sampleColumn) {
                extras.push(tokens[i]);
            }
        }
        return extras
    }

}

class SegFeature {

    constructor({sample, chr, start, end, value, valueColumnName}) {
        this.sample = sample;
        this.chr = chr;
        this.start = start;
        this.end = end;
        this.value = value;
        this.valueColumnName = valueColumnName;
    }

    setAttributes({names, values}) {
        this.attributeNames = names;
        this.attributeValues = values;
    }

    getAttribute(name) {
        if (this.attributeNames) {
            const idx = this.attributeNames.indexOf(name);
            if (idx >= 0) {
                return this.attributeValues[idx]
            }
        }
        return undefined
    }


    popupData(type, genomeID) {
        const filteredProperties = new Set(['chr', 'start', 'end', 'sample', 'value', 'row', 'color', 'sampleKey',
            'uniqueSampleKey', 'sampleId', 'chromosome', 'uniquePatientKey']);
        const locationString = (this.chr + ":" +
            numberFormatter$1(this.start + 1) + "-" +
            numberFormatter$1(this.end));
        const pd = [
            {name: "Sample", value: this.sample},
            {name: "Location", value: locationString},
            {name: this.valueColumnName ? capitalize(this.valueColumnName) : "Value", value: this.value}
        ];

        // TODO -- the Cravat stuff should probably be in the track (SegTrack)
        if ("mut" === type && "hg38" === genomeID) {
            const l = this.extractCravatLink(genomeID);
            if (l) {
                pd.push('<hr/>');
                pd.push({html: l});
                pd.push('<hr/>');
            }
        }

        if (this.attributeNames && this.attributeNames.length > 0) {
            for (let i = 0; i < this.attributeNames.length; i++) {
                if (!filteredProperties.has(this.attributeNames[i]) & this.valueColumnName !== this.attributeNames[i]) {
                    pd.push({name: capitalize(this.attributeNames[i]), value: this.attributeValues[i]});
                }
            }
        }
        return pd
    }

    extractCravatLink(genomeId) {

        let ref, alt;
        if (this.attributeNames && this.attributeNames.length > 0) {
            for (let i = 0; i < this.attributeNames.length; i++) {
                if (!ref && "Reference_Allele" === this.attributeNames[i]) {
                    ref = this.attributeValues[i];
                }
                if (!alt && this.attributeNames[i].startsWith("Tumor_Seq_Allele") && this.attributeValues[i] !== ref) {
                    alt = this.attributeValues[i];
                }
                if (ref && alt) {
                    return TrackBase.getCravatLink(this.chr, this.start + 1, ref, alt, genomeId)
                }
            }
        }

    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * Create a variant from an array of tokens representing a line in a "VCF" file
 * @param tokens
 */

const STANDARD_FIELDS = new Map([["REF", "referenceBases"], ["ALT", "alternateBases"], ["QUAL", "quality"], ["FILTER", "filter"]]);


class Variant {

    constructor(tokens) {
        this.chr = tokens[0]; // TODO -- use genome aliases
        this.pos = parseInt(tokens[1]);
        this.names = tokens[2];    // id in VCF
        this.referenceBases = tokens[3];
        this.alternateBases = tokens[4];
        this.quality = tokens[5];
        this.filter = tokens[6];
        this.info = {};
        const infoStr = tokens[7];
        if (infoStr && infoStr !== '.') {
            for (let elem of infoStr.split(';')) {
                var element = elem.split('=');
                this.info[element[0]] = element[1];
            }
        }
        this.init();
    }


    getAttributeValue(key) {
        if (STANDARD_FIELDS.has(key)) {
            key = STANDARD_FIELDS.get(key);
        }
        return this.hasOwnProperty(key) ? this[key] : this.info[key]
    }


    init() {

        const ref = this.referenceBases;
        const altBases = this.alternateBases;

        if (this.info) {
            if (this.info["VT"]) {
                this.type = this.info["VT"];
            } else if (this.info["SVTYPE"]) {
                this.type = "SV";
            } else if (this.info["PERIOD"]) {
                this.type = "STR";
            }
        }
        if (this.type === undefined) {
            this.type = determineType(ref, altBases);
        }

        // Determine start/end coordinates -- these are the coordinates representing the actual variant,
        // not the leading or trailing reference
        if (this.info["END"]) {
            this.start = this.pos - 1;
            if (this.info["CHR2"] && this.info["CHR2"] !== this.chr) {
                this.end = this.start + 1;
            } else {
                this.end = Number.parseInt(this.info["END"]);
            }
        } else {
            if (this.type === "NONVARIANT") {
                this.start = this.pos - 1;      // convert to 0-based coordinate convention
                this.end = this.start + ref.length;
            } else {

                const altTokens = altBases.split(",").filter(token => token.length > 0);
                this.alleles = [];
                this.start = undefined;
                this.end = undefined;

                for (let alt of altTokens) {

                    this.alleles.push(alt);

                    // We don't yet handle  SV and other special alt representations
                    if ("SV" !== this.type && isKnownAlt(alt)) {

                        let altLength = alt.length;
                        let lengthOnRef = ref.length;
                        const lmin = Math.min(altLength, lengthOnRef);

                        // Trim off matching bases.  Try first match, then right -> left,  then any remaining left -> right
                        let s = 0;

                        while (s < lmin && (ref.charCodeAt(s) === alt.charCodeAt(s))) {
                            s++;
                            altLength--;
                            lengthOnRef--;
                        }

                        // right -> left from end
                        while (altLength > 0 && lengthOnRef > 0) {
                            const altIdx = s + altLength - 1;
                            const refIdx = s + lengthOnRef - 1;
                            if (alt.charCodeAt(altIdx) === ref.charCodeAt(refIdx)) {
                                altLength--;
                                lengthOnRef--;
                            } else {
                                break
                            }
                        }

                        // if any remaining, left -> right
                        while (altLength > 0 && lengthOnRef > 0) {
                            const altIdx = s;
                            const refIdx = s;
                            if (alt.charCodeAt(altIdx) === ref.charCodeAt(refIdx)) {
                                s++;
                                altLength--;
                                lengthOnRef--;
                            } else {
                                break
                            }
                        }

                        const alleleStart = this.pos + s - 1;      // -1 for zero based coordinates
                        const alleleEnd = alleleStart + lengthOnRef;
                        this.start = this.start === undefined ? alleleStart : Math.min(this.start, alleleStart);
                        this.end = this.end === undefined ? alleleEnd : Math.max(this.end, alleleEnd);
                    }
                }

                // Default to single base representation @ position for variant types not otherwise handled
                if (this.start === undefined) {
                    this.start = this.pos - 1;
                    this.end = this.pos;
                }

                // Infer an insertion from start === end
                if (this.start === this.end) {
                    this.start -= 0.5;
                    this.end += 0.5;
                }
            }
        }
    }

    popupData(genomicLocation, genomeId) {

        const posString = `${numberFormatter$1(this.pos)}`;
        const locString = this.start === this.end ?
            `${numberFormatter$1(this.start)} | ${numberFormatter$1(this.start + 1)}` :
            `${numberFormatter$1(this.start + 1)}-${numberFormatter$1(this.end)}`;
        const fields = [
            {name: "Chr", value: this.chr},
            {name: "Pos", value: posString},
            {name: "Loc", value: locString},
            {name: "ID", value: this.names ? this.names : ""},
            {name: "Ref", value: this.referenceBases},
            {name: "Alt", value: this.alternateBases.replace("<", "&lt;")},
            {name: "Qual", value: this.quality},
            {name: "Filter", value: this.filter}
        ];

        if (this.type) {
            fields.push({name: "Type", value: this.type});
        }

        if ("SNP" === this.type) {
            let ref = this.referenceBases;
            if (ref.length === 1) {
                let altArray = this.alternateBases.split(",");
                for (let alt of altArray) {
                    if (alt.length === 1) {
                        let l = TrackBase.getCravatLink(this.chr, this.pos, ref, alt, genomeId);
                        if (l) {
                            fields.push('<hr/>');
                            fields.push({html: l});
                        }
                    }
                }
            }
        }

        const infoKeys = Object.keys(this.info);
        if (this.info && infoKeys.length > 0) {
            fields.push({html: '<hr style="border-top: dotted 1px;border-color: #c9c3ba" />'});
            for (let key of infoKeys) {
                fields.push({name: key, value: arrayToString(decodeURIComponent(this.info[key]))});
            }
        }

        return fields
    }

    getInfo(tag) {
        return this.info ? this.info[tag] : undefined
    }

    isRefBlock() {
        return "NONVARIANT" === this.type
    }

    isFiltered() {
        return !("." === this.filter || "PASS" === this.filter)
    }

    alleleFreq() {
        return this.info ? this.info["AF"] : undefined
    }
}

/**
 * Represents the "other end" of an SV which specifies the breakpoint as CHR2 and END info fields.
 */
class SVComplement {

    constructor(v) {
        this.mate = v;
        this.chr = v.info.CHR2;
        this.pos = Number.parseInt(v.info.END);
        this.start = this.pos - 1;
        this.end = this.pos;
    }

    get info() {
        return this.mate.info
    }

    get names() {
        return this.mate.names
    }

    get referenceBases() {
        return this.mate.referenceBases
    }

    get alternateBases() {
        return this.mate.alternateBases
    }

    get quality() {
        return this.mate.quality
    }

    get filter() {
        return this.mate.filter
    }

    get calls() {
        return this.mate.calls
    }

    getAttributeValue(key) {
        return this.mate.getAttributeValue(key)
    }

    getInfo(tag) {
        this.mate.getInfo(tag);
    }

    isFiltered() {
        return this.mate.isFiltered()
    }

    alleleFreq() {
        return this.mate.alleleFreq()
    }

    popupData(genomicLocation, genomeId) {
        const popupData = [];

        popupData.push("SV Breakpoint");
        popupData.push({name: 'Chr', value: this.chr});
        popupData.push({name: 'Pos', value: `${numberFormatter$1(this.pos)}`});
        popupData.push({html: '<hr style="border-top: dotted 1px;border-color: #c9c3ba" />'});
        popupData.push("SV");
        popupData.push(...this.mate.popupData(genomicLocation, genomeId));

        return popupData
    }
}


class Call {

    constructor({formatFields, sample, token}) {

        this.info = {};
        this.sample = sample;
        const ct = token.split(":");
        for (let idx = 0; idx < ct.length; idx++) {
            const callToken = ct[idx];
            if (idx == formatFields.genotypeIndex) {
                this.genotype = [];
                for (let s of callToken.split(/[\|\/]/)) {
                    this.genotype.push('.' === s ? s : parseInt(s));
                }
            } else {
                this.info[formatFields.fields[idx]] = callToken;
            }
        }

    }


    get zygosity() {
        if (!this._zygosity) {
            if (!this.genotype) {
                this._zygosity = 'unknown';
            } else {
                let allVar = true;  // until proven otherwise
                let allRef = true;
                let noCall = false;

                for (let g of this.genotype) {
                    if ('.' === g) {
                        noCall = true;
                        break
                    } else {
                        if (g !== 0) allRef = false;
                        if (g === 0) allVar = false;
                    }
                }
                if (noCall) {
                    this._zygosity = 'nocall';
                } else if (allRef) {
                    this._zygosity = 'homref';
                } else if (allVar) {
                    this._zygosity = 'homvar';
                } else {
                    this._zygosity = 'hetvar';
                }
            }
        }
        return this._zygosity
    }

    /**
     * Used in sorting
     */
    zygosityScore() {
        const zygosity = this.zygosity;
        switch (zygosity) {
            case 'homvar':
                return 4
            case 'hetvar':
                return 3
            case 'homref':
                return 2
            case 'nocall':
                return 1
            default:
                return 0
        }
    }

    #zygosityLabel() {
        const zygosity = this.zygosity;
        switch (zygosity) {
            case 'homref':
                return 'Homozygous reference'
            case 'homvar':
                return 'Homozygous variant'
            case 'hetvar':
                return 'Heterozygous'
            default:
                return ''
        }
    }


    popupData(genomicLocation, genomeID) {

        const popupData = [];

        if (this.sample !== undefined) {
            popupData.push({name: 'Sample', value: this.sample});
        }

        // Genotype string is set in VariantTrack when call is clicked -- this is for memory efficienty, very few
        // calls will get clicked
        if (this.genotypeString) {
            popupData.push({name: 'Genotype', value: this.genotypeString});
        }

        const zygosity = this.#zygosityLabel();
        if (zygosity) {
            popupData.push({name: 'Zygosity', value: zygosity});
        }


        var infoKeys = Object.keys(this.info);
        if (infoKeys.length) {
            popupData.push('<hr/>');
        }
        for(let key of infoKeys) {
            popupData.push({name: key, value: decodeURIComponent(this.info[key])});
        }

        return popupData
    }
}

const knownAltBases = new Set(["A", "C", "T", "G"].map(c => c.charCodeAt(0)));

function isKnownAlt(alt) {
    for (let i = 0; i < alt.length; i++) {
        if (!knownAltBases.has(alt.charCodeAt(i))) {
            return false
        }
    }
    return true
}


function determineType(ref, altAlleles) {
    const refLength = ref.length;
    if (altAlleles === undefined) {
        return "UNKNOWN"
    } else if (altAlleles.trim().length === 0 ||
        altAlleles === "<NON_REF>" ||
        altAlleles === "<*>" ||
        altAlleles === ".") {
        return "NONVARIANT"
    } else {
        const alleles = altAlleles.split(",");
        const types = alleles.map(function (a) {
            if (refLength === 1 && a.length === 1) {
                return "SNP"
            } else if ("<NON_REF>" === a) {
                return "NONVARIANT"
            } else if (a.length > refLength && isKnownAlt(a)) {
                return "INSERTION"
            } else if (a.length < refLength && isKnownAlt(a)) {
                return "DELETION"
            } else {
                return "OTHER"
            }
        });
        let type = types[0];
        for (let t of types) {
            if (t !== type) {
                return "MIXED"
            }
        }
        return type
    }
}

function arrayToString(value, delim) {

    if (delim === undefined) delim = ",";

    if (!(Array.isArray(value))) {
        return value
    }
    return value.join(delim)
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * Parser for VCF files.
 */

class VcfParser {

    construtor() {
    }

    async parseHeader(dataWrapper, genome) {

        const header = {};

        header.chrAliasTable = new Map();

        // First line must be file format
        let line = await dataWrapper.nextLine();
        if (line.startsWith("##fileformat")) {
            header.version = line.substr(13);
        } else {
            throw new Error("Invalid VCF file: missing fileformat line")
        }

        while ((line = await dataWrapper.nextLine()) !== undefined) {

            if (line.startsWith("#")) {

                let id;
                const values = {};

                if (line.startsWith("##")) {

                    if (line.startsWith("##INFO") || line.startsWith("##FILTER") || line.startsWith("##FORMAT")) {

                        const ltIdx = line.indexOf("<");
                        const gtIdx = line.lastIndexOf(">");

                        if (!(ltIdx > 2 && gtIdx > 0)) {
                            continue
                        }

                        const type = line.substring(2, ltIdx - 1);
                        if (!header[type]) header[type] = {};

                        //##INFO=<ID=AF,Number=A,Type=Float,Description="Allele frequency based on Flow Evaluator observation counts">
                        // ##FILTER=<ID=NOCALL,Description="Generic filter. Filtering details stored in FR info tag.">
                        // ##FORMAT=<ID=AF,Number=A,Type=Float,Description="Allele frequency based on Flow Evaluator observation counts">

                        const tokens = splitStringRespectingQuotes(line.substring(ltIdx + 1, gtIdx - 1), ",");

                        for (let token of tokens) {
                            var kv = token.split("=");
                            if (kv.length > 1) {
                                if (kv[0] === "ID") {
                                    id = kv[1];
                                } else {
                                    values[kv[0]] = kv[1];
                                }
                            }
                        }

                        if (id) {
                            header[type][id] = values;
                        }
                    } else if (line.startsWith("##contig") && genome) {
                        const idx1 = line.indexOf("<ID=");
                        let idx2 = line.indexOf(",", idx1);
                        if (idx2 == -1) {
                            idx2 = line.indexOf(">", idx1);
                        }
                        const chr = line.substring(idx1 + 4, idx2);
                        const canonicalChromosome = genome.getChromosomeName(chr);
                        header.chrAliasTable.set(canonicalChromosome, chr);
                    } else ;
                } else if (line.startsWith("#CHROM")) {
                    const tokens = line.split("\t");
                    if (tokens.length > 8) {
                        // Map of sample name -> index
                        header.sampleNameMap = new Map();
                        for (let j = 9; j < tokens.length; j++) {
                            header.sampleNameMap.set(tokens[j], j - 9);
                        }
                    }
                }

            } else {
                break
            }
        }

        this.header = header;  // Will need to intrepret genotypes and info field

        return header
    }


    /**
     * Parse data as a collection of Variant objects.
     *
     * @param data
     * @returns {Array}
     */
    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        const sampleNames = this.header.sampleNameMap ? Array.from(this.header.sampleNameMap.keys()) : undefined;
        const nExpectedColumns = 8 + (sampleNames ? sampleNames.length + 1 : 0);
        let line;
        while ((line = await dataWrapper.nextLine()) !== undefined) {
            if (line && !line.startsWith("#")) {

                const tokens = line.trim().split("\t");
                if (tokens.length === nExpectedColumns) {
                    const variant = new Variant(tokens);
                    variant.header = this.header;       // Keep a pointer to the header to interpret fields for popup text
                    //variant.line = line              // Uncomment for debugging
                    allFeatures.push(variant);

                    if (tokens.length > 9) {
                        //example...	GT	0|0	0|0	0|0	0|0	0|0	0|0	0|0	0|0	0|0	0|0	0|0
                        //example...    GT:DR:DV	./.:.:11

                        const formatFields = extractFormatFields(tokens[8].split(":"));

                        variant.calls = [];
                        for (let index = 9; index < tokens.length; index++) {
                            const sample = sampleNames[index-9];
                            const token = tokens[index];
                            const call = new Call({formatFields, sample, token});
                            variant.calls.push(call);

                        }

                        // If this is a structural variant create a complement of this variant for the other end
                        // The test for "SV" is not comprehensive, there is not yet a standard for this
                        if (variant.info && variant.info.CHR2 && variant.info.END) {
                            allFeatures.push(svComplement(variant));
                        }
                    }
                }
            }
        }
        return allFeatures

    }
}

function extractFormatFields(tokens) {

    const callFields = {
        genotypeIndex: -1,
        fields: tokens
    };
    for (let i = 0; i < tokens.length; i++) {
        if ("GT" === tokens[i]) {
            callFields.genotypeIndex = i;
        }
    }
    return callFields
}

function svComplement(v) {

    return new SVComplement(v)

}

const MIN_EXPONENT$1 = Math.log10(Number.MIN_VALUE);

/**
 * Parser for IGV desktop GWAS files.  See http://software.broadinstitute.org/software/igv/GWAS
 */
class GWASParser {

    constructor(config) {
        // Defaults - can be overriden by header
        this.config = config;
        if (config.columns) {
            if (config.columns.chromosome === undefined ||
                config.columns.position === undefined ||
                config.columns.value === undefined) {
                throw Error("columns property must define chrCol, posCol, and valueCol")
            }
            this.posCol = config.columns.position - 1;
            this.chrCol = config.columns.chromosome - 1;
            this.valueCol = config.columns.value - 1;
        } else {
            // Defaults -- can be overriden in header
            this.posCol = 2;
            this.chrCol = 1;
            this.valueCol = 3;
        }
    }

    async parseHeader(dataWrapper) {
        const headerLine = await dataWrapper.nextLine();
        return this.parseHeaderLine(headerLine)
    }

    parseHeaderLine(headerLine) {
        this.columns = headerLine.split(/\t/);
        if (!this.config.columns) {
            for (let i = 0; i < this.columns.length; i++) {
                const c = this.columns[i].toLowerCase();
                switch (c) {
                    case 'chr':
                    case 'chromosome':
                    case 'chr_id':
                        this.chrCol = i;
                        break
                    case 'bp':
                    case 'pos':
                    case 'position':
                    case 'chr_pos':
                        this.posCol = i;
                        break
                    case 'p':
                    case 'pval':
                    case 'pvalue':
                    case 'p-value':
                    case 'p.value':
                        this.valueCol = i;
                        break
                }
            }
        }
        return this.columns
    }

    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        const headerLine = dataWrapper.nextLine();
        if (!this.columns) {
            this.parseHeaderLine(headerLine);
        }

        let line;

        const parseValue = (valueString) => {
            // Don't try to parse extremely small values
            const idx = valueString.indexOf("E");
            if(idx > 0) {
                const exp = Number.parseInt(valueString.substring(idx + 1));
                if (exp < MIN_EXPONENT$1) {
                   return Number.MIN_VALUE;
                }
            }
            return Number(valueString)
        };

        while ((line = dataWrapper.nextLine()) !== undefined) {
            const tokens = line.split(/\t/);
            if (tokens.length === this.columns.length) {
                const posString = tokens[this.posCol];
                if(posString.indexOf(";") > 0 || posString.length == 0 || posString.indexOf('x') > 0) {
                    continue
                }
                const chr = tokens[this.chrCol];
                const value = parseValue(tokens[this.valueCol]);
                const start = parseInt(posString) - 1;
                const end = start + 1;
                allFeatures.push(new GWASFeature({chr, start, end, value, line, columns: this.columns}));

            }
        }
        return allFeatures
    }

    /**
     * Test first line to see if this is a GWAS file.  Used to determine file format for the case of generic
     * extensions such as "tsv"
     * @param firstLine
     */
    static isGWAS(firstLine) {
        const tokens = firstLine.split('\t');
        if (tokens.length < 5) {
            return false
        }
        const requiredHeaders =
            [
                ['chr', 'chromosome', 'chr_id', 'chrom'],
                ['bp', 'pos', 'position', 'chr_pos', 'chromEnd'],
                ['p', 'pval', 'p-value', 'p.value']
            ];
        for (let h of requiredHeaders) {
            if (!tokens.some(t => h.includes(t.toLowerCase()))) return false
        }
        return true
    }

}

class GWASFeature {

    constructor({chr, start, end, value, line, columns}) {
        this.chr = chr;
        this.start = start;
        this.end = end;
        this.value = value;
        this.line = line;
        this.columns = columns;
    }

    popupData() {
        const tokens = this.line.split(/\t/);
        return this.columns.map(function (c, index) {
            return {name: c, value: tokens[index]}
        })
    }

    getAttribute(attrName) {
        const tokens = this.line.split(/\t/);
        for (let i = 0; i < this.columns.length; i++) {
            if (this.columns[i] === attrName) {
                return tokens[i]
            }
        }
        return undefined
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 *  Define parsers for bed-like files  (.bed, .gff, .vcf, etc).  A parser should implement 2 methods
 *
 *     parseHeader(data) - return an object representing a header or metadata.  Details are format specific
 *
 *     parseFeatures(data) - return an array of features
 *
 */

var aedRegexpNoNamespace = new RegExp("([^:]*)\\(([^)]*)\\)"); // name(type) for AED parsing (namespace undefined)
var aedRegexpNamespace = new RegExp("([^:]*):([^(]*)\\(([^)]*)\\)"); // namespace:name(type) for AED parsing


class AEDParser {

    constructor(config) {
        const decode = config ? config.decode : undefined;
        this.nameField = config ? config.nameField : undefined;
        this.skipRows = 0;   // The number of fixed header rows to skip.  Override for specific types as needed
        if (decode) {
            this.decode = decode;
        } else {
            this.decode = decodeAed;
        }
        this.delimiter = "\t";
    }

    async parseHeader(dataWrapper) {
        let line;
        let header;
        while (line = await dataWrapper.nextLine()) {
            if (line.startsWith("track") || line.startsWith("#") || line.startsWith("browser")) {
                if (line.startsWith("track") || line.startsWith("#track")) {
                    let h = parseTrackLine(line);
                    if (header) {
                        Object.assign(header, h);
                    } else {
                        header = h;
                    }
                } else if (line.startsWith("#columns")) {
                    let h = parseColumnsDirective(line);
                    if (header) {
                        Object.assign(header, h);
                    } else {
                        header = h;
                    }
                } else if (line.startsWith("##gff-version 3")) {
                    this.format = "gff3";
                    if (!header) header = {};
                    header["format"] = "gff3";
                }
            } else {
                break
            }
        }
        this.header = header;    // Directives might be needed for parsing lines
        return header
    }

    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        const decode = this.decode;
        const delimiter = this.delimiter || "\t";
        let i = 0;
        let line;
        let wig;

        while ((line = dataWrapper.nextLine()) !== undefined) {
            i++;
            if (i <= this.skipRows || line.startsWith("track") || line.startsWith("#") || line.startsWith("browser")) {
                continue
            }

            let tokens = readTokensAed();
            if (tokens.length < 1) {
                continue
            }

            if (!this.aed) {
                // Store information about the aed header in the parser itself
                // This is done only once - on the first row
                this.aed = parseAedHeaderRow(tokens);
                continue
            }

            const feature = decode.call(this, tokens, wig);
            if (feature) {
                allFeatures.push(feature);
            }
        }

        return allFeatures

        // Double quoted strings can contain newlines in AED
        // "" is an escape for a ".
        // Parse all this, clean it up, split into tokens in a custom way
        function readTokensAed() {
            var tokens = [],
                token = "",
                quotedString = false,
                n,
                c;

            while (line || line === '') {
                for (n = 0; n < line.length; n++) {
                    c = line.charAt(n);
                    if (c === delimiter) {
                        if (!quotedString) {
                            tokens.push(token);
                            token = "";
                        } else {
                            token += c;
                        }
                    } else if (c === "\"") {
                        // Look ahead to the next character
                        if (n + 1 < line.length && line.charAt(n + 1) === "\"") {
                            if (quotedString) {
                                // Turn "" into a single " in the output string
                                token += "\"";
                            }
                            // Skip the next double quote
                            n++;
                        } else {
                            // We know the next character is NOT a double quote, flip our state
                            quotedString = !quotedString;
                        }
                    } else {
                        token += c;
                    }
                }
                // We are at the end of the line
                if (quotedString) {
                    token += '\n'; // Add newline to the token
                    line = nextLine(); // Keep going
                } else {
                    // We can end the loop
                    break
                }
            }
            // Push the last token
            tokens.push(token);
            return tokens
        }
    }
}


function parseAedToken(value) {
    // Example: refseq:accessionNumber(aed:String)
    // refseq - namespace, will be declared later
    // accessionNumber - name of the field
    // aed:String - type of the field
    // The namespace part may be missing
    var match = aedRegexpNamespace.exec(value);
    if (match) {
        return {
            namespace: match[1],
            name: match[2],
            type: match[3]
        }
    }

    match = aedRegexpNoNamespace.exec(value);
    if (match) {
        return {
            namespace: '?',
            name: match[1],
            type: match[2]
        }
    } else {
        throw new Error("Error parsing the header row of AED file - column not in ns:name(ns:type) format")
    }
}

function parseAedHeaderRow(tokens) {
    // First row of AED file defines column names
    // Each header item is an aed token - see parseAedToken
    var aed,
        k,
        token,
        aedToken;

    // Initialize aed section to be filled in
    aed = {
        columns: [ // Information about the namespace, name and type of each column
            // Example entry:
            // { namespace: 'bio', name: 'start', type: 'aed:Integer' }
        ],
        metadata: { // Metadata about the entire AED file
            // Example:
            // {
            //    aed: {
            //       application: { value: "CHaS Browser 3.3.0.139 (r10838)", type: "aed:String" },
            //       created: { value: "2018-01-02T10:20:30.123+01:00", type: "aed:DateTime" },
            //       modified: { value: "2018-03-04T11:22:33.456+01:00", type: "aed:DateTime" },
            //    }
            //    affx: {
            //       ucscGenomeVersion: { value: "hg19", type: "aed:String" }
            //    },
            //    namespace: {
            //       omim: { value: "http://affymetrix.com/ontology/www.ncbi.nlm.nih.gov/omim/", type: "aed:URI" },
            //       affx: { value: "http://affymetrix.com/ontology/", type: "aed:URI" },
            //       refseq: { value: "http://affymetrix.com/ontology/www.ncbi.nlm.nih.gov/RefSeq/", type: "aed:URI" }
            //    }
            // }
        }
    };
    for (k = 0; k < tokens.length; k++) {
        token = tokens[k];
        aedToken = parseAedToken(token);
        aed.columns.push(aedToken);
    }

    return aed
}

function parseTrackLine(line) {

    const properties = {};
    const tokens = line.split(/(?:")([^"]+)(?:")|([^\s"]+)(?=\s+|$)/g);


    // Clean up tokens array
    let curr;
    const tmp = [];
    for (let tk of tokens) {
        if (!tk || tk.trim().length === 0) continue
        if (tk.endsWith("=") > 0) {
            curr = tk;
        } else if (curr) {
            tmp.push(curr + tk);
            curr = undefined;
        } else {
            tmp.push(tk);
        }
    }
    for (let str of tmp) {
        if (!str) return
        var kv = str.split('=', 2);
        if (kv.length === 2) {
            const key = kv[0].trim();
            const value = kv[1].trim();
            properties[key] = value;
        }

    }

    return properties
}

function parseColumnsDirective(line) {

    let properties = {};
    let t1 = line.split(/\s+/);

    if (t1.length === 2) {

        let t2 = t1[1].split(";");

        t2.forEach(function (keyValue) {

            let t = keyValue.split("=");

            if (t[0] === "color") {
                properties.colorColumn = Number.parseInt(t[1]) - 1;
            } else if (t[0] === "thickness") {
                properties.thicknessColumn = Number.parseInt(t[1]) - 1;
            }
        });
    }

    return properties
}

/**
 * AED file feature.
 *
 * @param aed link to the AED file object containing file-level metadata and column descriptors
 * @param allColumns All columns as parsed from the AED
 *
 * Other values are parsed one by one
 */
function AedFeature(aed, allColumns) {
    var token, aedColumn, aedColumns = aed.columns;

    // Link to AED file (for metadata)
    this.aed = aed;

    // Unparsed columns from AED file
    this.allColumns = allColumns;

    // Prepare space for the parsed values
    this.chr = null;
    this.start = null;
    this.end = null;
    this.score = 1000;
    this.strand = '.';
    this.cdStart = null;
    this.cdEnd = null;
    this.name = null;
    this.color = null;

    for (let i = 0; i < allColumns.length; i++) {
        token = allColumns[i];
        if (!token) {
            // Skip empty fields
            continue
        }
        aedColumn = aedColumns[i];
        if (aedColumn.type === 'aed:Integer') {
            token = parseInt(token);
        }
        var arr = [];
        if (aedColumn.namespace.length > 0) {
            for (let j = 0; j < aedColumn.namespace.length; j++) {
                arr.push(aedColumn.namespace.charCodeAt(j));
            }
        }
        if (aedColumn.namespace.trim() === 'bio') {
            if (aedColumn.name === 'sequence') {
                this.chr = token;
            } else if (aedColumn.name === 'start') {
                this.start = token;
            } else if (aedColumn.name === 'end') {
                this.end = token;
            } else if (aedColumn.name === 'cdsMin') {
                this.cdStart = token;
            } else if (aedColumn.name === 'cdsMax') {
                this.cdEnd = token;
            } else if (aedColumn.name === 'strand') {
                this.strand = token;
            }
        } else if (aedColumn.namespace === 'aed') {
            if (aedColumn.name === 'name') {
                this.name = token;
            }
        } else if (aedColumn.namespace === 'style') {
            if (aedColumn.name === 'color') {
                this.color = IGVColor.createColorString(token);
            }
        }
    }
}

AedFeature.prototype.popupData = function () {
    var data = [],
        aed = this.aed;
    // Just dump everything we have for now
    for (var i = 0; i < this.allColumns.length; i++) {
        var featureValue = this.allColumns[i];
        var name = aed.columns[i].name;
        // Skip columns that are not interesting - you know the sequence, and you can see color
        if (name !== 'sequence' && name !== 'color') {
            if (featureValue) {
                data.push({name: name, value: featureValue});
            }
        }
    }
    return data
};

/**
 * Decode the AED file format
 * @param tokens
 * @param ignore
 * @returns decoded feature, or null if this is not a valid record
 */
function decodeAed(tokens, ignore) {
    var name, value, token,
        nonEmptyTokens = 0,
        aedColumns = this.aed.columns,
        aedColumn,
        aedKey,
        i;

    // Each aed row must match the exact number of columns or we skip it
    if (tokens.length !== aedColumns.length) {
        return undefined
    }

    for (i = 0; i < tokens.length; i++) {
        aedColumn = aedColumns[i];
        token = tokens[i];
        if (token !== '') {
            nonEmptyTokens++;
        }
        if (aedColumn.name === 'name' && aedColumn.namespace === 'aed') {
            name = token;
        } else if (aedColumn.name === 'value' && aedColumn.namespace === 'aed') {
            value = token;
        }
    }

    if (nonEmptyTokens === 2 && name && value) {
        // Special row that defines metadata for the entire file
        aedKey = parseAedToken(name);
        // Store in the metadata section
        if (!this.aed.metadata[aedKey.namespace]) {
            this.aed.metadata[aedKey.namespace] = {};
        }
        if (!this.aed.metadata[aedKey.namespace][aedKey.name]) {
            this.aed.metadata[aedKey.namespace][aedKey.name] = {
                type: aedKey.type,
                value: value
            };
        }
        // Ignore this value
        return undefined
    }

    var feature = new AedFeature(this.aed, tokens);

    if (!feature.chr || (!feature.start && feature.start !== 0) || !feature.end) {
        return undefined
    }

    return feature
}

function optimizeChunks(chunks, lowest) {

    if (chunks.length === 0) return chunks

    chunks.sort(function (c0, c1) {

        const dif = c0.minv.block - c1.minv.block;
        if (dif !== 0) {
            return dif
        } else {
            return c0.minv.offset - c1.minv.offset
        }
    });

    if (chunks.length <= 1) {
        return chunks
    }

    // console.log("Before trimming " + chunks.length)
    // for (let c of chunks) {
    //     console.log(`${c.minv.block} ${c.minv.offset}  -  ${c.maxv.block} ${c.maxv.offset}`)
    // }

    if (lowest) {
        chunks = chunks.filter(c => c.maxv.isGreaterThan(lowest));
    }

    // console.log("Before merging " + chunks.length)
    // for (let c of chunks) {
    //     console.log(`${c.minv.block} ${c.minv.offset}  -  ${c.maxv.block} ${c.maxv.offset}`)
    // }

    const mergedChunks = [];
    let lastChunk;
    for (let chunk of chunks) {

        if (!lastChunk) {
            mergedChunks.push(chunk);
            lastChunk = chunk;
        } else {
            if (canMerge(lastChunk, chunk)) {
                if (chunk.maxv.isGreaterThan(lastChunk.maxv)) {
                    lastChunk.maxv = chunk.maxv;
                }
            } else {
                mergedChunks.push(chunk);
                lastChunk = chunk;
            }
        }
    }

    // console.log("After merging " + mergedChunks.length)
    // for (let c of mergedChunks) {
    //     console.log(`${c.minv.block} ${c.minv.offset}  -  ${c.maxv.block} ${c.maxv.offset}`)
    // }

    return mergedChunks
}


/**
 * Merge 2 blocks if the file position gap between them is < 16 kb, and the total size is < ~5 mb
 * @param chunk1
 * @param chunk2
 * @returns {boolean|boolean}
 */
function canMerge(chunk1, chunk2) {
    const gap = chunk2.minv.block - chunk1.maxv.block;
    const sizeEstimate = chunk2.maxv.block - chunk1.minv.block;
    return sizeEstimate < 5000000 && gap < 65000
    
}

// Represents a CSI Bam or Tabix index

const CSI1_MAGIC$1 = 21582659; // CSI\1
const CSI2_MAGIC$1 = 38359875; // CSI\2

async function parseCsiIndex(arrayBuffer) {

    const idx = new CSIIndex();
    idx.parse(arrayBuffer);
    return idx
}

class CSIIndex {

    constructor() {
        this.tabix = true;  // => i.e. not a tribble index.   This is important, if obtuse
    }

    parse(arrayBuffer) {
        const parser = new BinaryParser$1(new DataView(arrayBuffer));

        const magic = parser.getInt();

        if (magic !== CSI1_MAGIC$1) {
            if (magic === CSI2_MAGIC$1) {
                throw Error("CSI version 2 is not supported.  Please enter an issue at https://github.com/igvteam/igv.js")
            } else {
                throw Error("Not a CSI index")
            }
        }

        this.indices = [];
        this.blockMin = Number.MAX_SAFE_INTEGER;
        this.lastBlockPosition = [];
        this.sequenceIndexMap = {};

        this.minShift = parser.getInt();
        this.depth = parser.getInt();
        const lAux = parser.getInt();
        const seqNames = [];
        let bmax = 0;

        if (lAux >= 28) {
            // Tabix header parameters aren't used, but they must be read to advance the pointer
            parser.getInt();
            parser.getInt();
            parser.getInt();
            parser.getInt();
            parser.getInt();
            parser.getInt();
            const l_nm = parser.getInt();
            const nameEndPos = parser.position + l_nm;
            let i = 0;
            while (parser.position < nameEndPos) {
                let seq_name = parser.getString();
                this.sequenceIndexMap[seq_name] = i;
                seqNames[i] = seq_name;
                i++;
            }
        }

        const MAX_BIN = this.bin_limit() + 1;
        const nref = parser.getInt();
        for (let ref = 0; ref < nref; ref++) {
            const binIndex = [];
            const loffset = [];
            const nbin = parser.getInt();
            for (let b = 0; b < nbin; b++) {

                const binNumber = parser.getInt();
                loffset[binNumber] = parser.getVPointer();

                if (binNumber > MAX_BIN) {
                    // This is a psuedo bin, not used but we have to consume the bytes
                    parser.getInt(); // # of chunks for this bin
                    parser.getVPointer();   // unmapped beg
                    parser.getVPointer();   // unmapped end
                    parser.getLong();
                    parser.getLong();

                } else {

                    binIndex[binNumber] = [];
                    const nchnk = parser.getInt(); // # of chunks for this bin

                    for (let i = 0; i < nchnk; i++) {
                        const cs = parser.getVPointer();    //chunk_beg
                        const ce = parser.getVPointer();    //chunk_end
                        if (cs && ce) {
                            if (cs.block < this.blockMin) {
                                this.blockMin = cs.block;    // Block containing first alignment
                            }
                            if (ce.block > bmax) {
                                bmax = ce.block;
                            }
                            binIndex[binNumber].push([cs, ce]);
                        }
                    }
                }
            }

            if (nbin > 0) {
                this.indices[ref] = {
                    binIndex: binIndex,
                    loffset: loffset
                };
            }
        }
        this.lastBlockPosition = bmax;
    }

    get sequenceNames() {
        return Object.keys(this.sequenceIndexMap)
    }

    /**
     * Fetch blocks for a particular genomic range.  This method is public so it can be unit-tested.
     *
     * @param refId  the sequence dictionary id of the chromosome
     * @param min  genomic start position
     * @param max  genomic end position
     * @param return an array of {minv: {filePointer, offset}, {maxv: {filePointer, offset}}
     */
    chunksForRange(refId, min, max) {

        const ba = this.indices[refId];
        if (!ba) {
            return []
        } else {
            const overlappingBins = this.reg2bins(min, max);        // List of bin #s that overlap min, max
            if (overlappingBins.length == 0) return []

            const chunks = [];
            // Find chunks in overlapping bins.  Leaf bins (< 4681) are not pruned
            for (let binRange of overlappingBins) {
                for (let bin = binRange[0]; bin <= binRange[1]; bin++) {
                    if (ba.binIndex[bin]) {
                        const binChunks = ba.binIndex[bin];
                        for (let c of binChunks) {
                            const cs = c[0];
                            const ce = c[1];
                            chunks.push({minv: cs, maxv: ce, bin: bin});
                        }
                    }
                }
            }

            // Find from the lowest bin level
            let bin = overlappingBins[this.depth][0];
            do {
                const target = ba.binIndex[bin];
                if (target) {
                    break
                }
                const firstBin = (this.getParentBin(bin) << 3) + 1;
                if (bin > firstBin) {
                    bin--;
                } else {
                    bin = this.getParentBin(bin);
                }
            } while (bin != 0)
            
            const lowestOffset = ba.loffset[bin];
            
            return optimizeChunks(chunks, lowestOffset)
        }
    }

    getParentBin(bin) {
        if (bin == 0) {
            return 0;
        }
        return (bin - 1) >> 3;
    }

    // reg2bins implementation adapted from GMOD/tabix-js  https://github.com/GMOD/tabix-js/blob/master/src/csi.ts
    reg2bins(beg, end) {
        beg -= 1; // < convert to 1-based closed
        if (beg < 1) beg = 1;
        if (end > 2 ** 34) end = 2 ** 34; // 17 GiB ought to be enough for anybody
        end -= 1;
        let l = 0;
        let t = 0;
        let s = this.minShift + this.depth * 3;
        const bins = [];
        for (; l <= this.depth; s -= 3, t += (1 << l * 3), l += 1) {
            const b = t + (beg >> s);
            const e = t + (end >> s);
            //
            // ITS NOT CLEAR WHERE THIS TEST CAME FROM,  but maxBinNumber is never set, and its not clear what it represents.
            // if (e - b + bins.length > this.maxBinNumber)
            //     throw new Error(
            //         `query ${beg}-${end} is too large for current binning scheme (shift ${this.minShift}, depth ${this.depth}), try a smaller query or a coarser index binning scheme`,
            //     )
            //
            bins.push([b, e]);
        }
        return bins
    }


    bin_limit() {
        return ((1 << (this.depth + 1) * 3) - 1) / 7
    }

}

// Represents a BAM or Tabix index.

const BAI_MAGIC$1 = 21578050;
const TABIX_MAGIC$1 = 21578324;


async function parseBamIndex(arrayBuffer) {
    const index = new BamIndex();
    await index.parse(arrayBuffer, false);
    return index
}

async function parseTabixIndex(arrayBuffer) {
    const index = new BamIndex();
    await index.parse(arrayBuffer, true);
    return index
}

class BamIndex {

    constructor() {

    }

    async parse(arrayBuffer, tabix) {

        const indices = [];
        let blockMin = Number.MAX_SAFE_INTEGER;
        let blockMax = 0;

        const parser = new BinaryParser$1(new DataView(arrayBuffer));
        const magic = parser.getInt();
        const sequenceIndexMap = {};
        if (magic === BAI_MAGIC$1 || (tabix && magic === TABIX_MAGIC$1)) {

            const nref = parser.getInt();
            if (tabix) {
                // Tabix header parameters aren't used, but they must be read to advance the pointer
                parser.getInt();
                parser.getInt();
                parser.getInt();
                parser.getInt();
                parser.getInt();
                parser.getInt();
                parser.getInt();

                for (let i = 0; i < nref; i++) {
                    let seq_name = parser.getString();
                    sequenceIndexMap[seq_name] = i;
                }
            }

            // Loop through sequences
            for (let ref = 0; ref < nref; ref++) {

                const binIndex = {};
                const linearIndex = [];
                const nbin = parser.getInt();

                for (let b = 0; b < nbin; b++) {
                    const binNumber = parser.getInt();
                    if (binNumber === 37450) {
                        // This is a psuedo bin, not used but we have to consume the bytes
                        parser.getInt(); // # of chunks for this bin
                        parser.getVPointer();   // unmapped beg
                        parser.getVPointer();   // unmapped end
                        parser.getLong();
                        parser.getLong();

                    } else {

                        binIndex[binNumber] = [];
                        const nchnk = parser.getInt(); // # of chunks for this bin

                        for (let i = 0; i < nchnk; i++) {
                            const cs = parser.getVPointer();    //chunk_beg
                            const ce = parser.getVPointer();    //chunk_end
                            if (cs && ce) {
                                if (cs.block < blockMin) {
                                    blockMin = cs.block;    // Block containing first alignment
                                }
                                if (ce.block > blockMax) {
                                    blockMax = ce.block;
                                }
                                binIndex[binNumber].push([cs, ce]);
                            }
                        }
                    }
                }

                const nintv = parser.getInt();
                for (let i = 0; i < nintv; i++) {
                    const cs = parser.getVPointer();
                    linearIndex.push(cs);   // Might be null
                }

                if (nbin > 0) {
                    indices[ref] = {
                        binIndex: binIndex,
                        linearIndex: linearIndex
                    };
                }
            }

            this.firstBlockPosition = blockMin;
            this.lastBlockPosition = blockMax;
            this.indices = indices;
            this.sequenceIndexMap = sequenceIndexMap;
            this.tabix = tabix;

        } else {
            throw new Error(indexURL + " is not a " + (tabix ? "tabix" : "bai") + " file")
        }


    }

    get sequenceNames() {
        return Object.keys(this.sequenceIndexMap)
    }

    /**
     * Fetch chunks for a particular genomic range.  This method is public so it can be unit-tested.
     *
     * @param refId  the sequence dictionary index of the chromosome
     * @param min  genomic start position
     * @param max  genomic end position
     * @param return an array of objects representing chunks (file spans) {minv: {block, offset}, {maxv: {block, offset}}
     */
    chunksForRange(refId, min, max) {

        const bam = this;
        const ba = bam.indices[refId];

        if (!ba) {
            return []
        } else {
            const overlappingBins = reg2bins(min, max);        // List of bin #s that overlap min, max

            //console.log("bin ranges")
            //for(let b of overlappingBins) {
            //    console.log(`${b[0]} - ${b[1]}`)
            //}

            const chunks = [];
            // Find chunks in overlapping bins.  Leaf bins (< 4681) are not pruned
            for (let binRange of overlappingBins) {
                for (let bin = binRange[0]; bin <= binRange[1]; bin++) {
                    if (ba.binIndex[bin]) {
                        const binChunks = ba.binIndex[bin];
                        for (let c of binChunks) {
                            const cs = c[0];
                            const ce = c[1];
                            chunks.push({minv: cs, maxv: ce});
                        }
                    }
                }
            }

            // Use the linear index to find minimum file position of chunks that could contain alignments in the region
            const nintv = ba.linearIndex.length;

            let lowest;
            const minLin = Math.min(min >> 14, nintv - 1);    // i.e. min / 16384
            const maxLin = Math.min(max >> 14, nintv - 1);
            for (let i = minLin; i <= maxLin; i++) {
                const vp = ba.linearIndex[i];
                if (vp) {
                    lowest = vp;       // lowest file offset that contains alignments overlapping (min, max)
                    break
                }
            }

            return optimizeChunks(chunks, lowest)
        }
    }
}



/**
 * Calculate the list of bins that overlap with region [beg, end]
 *
 */
function reg2bins(beg, end) {
    const list = [];
    if (end >= 1 << 29) end = 1 << 29;
    --end;
    list.push([0, 0]);
    list.push([1 + (beg >> 26), 1 + (end >> 26)]);
    list.push([9 + (beg >> 23), 9 + (end >> 23)]);
    list.push([73 + (beg >> 20), 73 + (end >> 20)]);
    list.push([585 + (beg >> 17), 585 + (end >> 17)]);
    list.push([4681 + (beg >> 14), 4681 + (end >> 14)]);

    // for (k = 1 + (beg >> 26); k <= 1 + (end >> 26); ++k) list.push(k);
    // for (k = 9 + (beg >> 23); k <= 9 + (end >> 23); ++k) list.push(k);
    // for (k = 73 + (beg >> 20); k <= 73 + (end >> 20); ++k) list.push(k);
    // for (k = 585 + (beg >> 17); k <= 585 + (end >> 17); ++k) list.push(k);
    // for (k = 4681 + (beg >> 14); k <= 4681 + (end >> 14); ++k) list.push(k);
    return list
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

async function parseTribbleIndex(arrayBuffer) {

    const index = new TribbleIndex();
    index.parse(arrayBuffer);
    return index
}

class TribbleIndex {

    constructor() {
        this.tribble = true;
    }

    async parse(arrayBuffer) {

        let blockMax = 0;
        this.chrIndex = {};
        this.lastBlockPosition = [];
        const parser = new BinaryParser$1(new DataView(arrayBuffer));
        readHeader(parser);

        let nChrs = parser.getInt();
        while (nChrs-- > 0) {

            // todo -- support interval tree index, we're assuming its a linear index

            let chr = parser.getString();
            const binWidth = parser.getInt();
            const nBins = parser.getInt();
            const longestFeature = parser.getInt();
            parser.getInt() > 0;
            parser.getInt();

            // note the code below accounts for > 60% of the total time to read an index
            let pos = parser.getLong();
            const blocks = [];
            for (let binNumber = 0; binNumber < nBins; binNumber++) {
                const nextPos = parser.getLong();
                blocks.push({min: pos, max: nextPos}); //        {position: pos, size: size});
                pos = nextPos;
                if (nextPos > blockMax) {
                    blockMax = nextPos;
                }
            }
            this.chrIndex[chr] = {chr: chr, blocks: blocks, longestFeature: longestFeature, binWidth: binWidth};
        }

        this.lastBlockPosition = blockMax;


        /**
         * Read the header .   Data here is not used in igv.js but we need to read it to advance the pointer.
         * @param parser
         */
        function readHeader(parser) {

            parser.getInt();     //   view._getInt32(offset += 32, true);
            parser.getInt();
            const version = parser.getInt();
            parser.getString();
            parser.getLong();
            parser.getLong();
            parser.getString();
            parser.getInt();
            if (version >= 3) {
                let nProperties = parser.getInt();
                while (nProperties-- > 0) {
                    parser.getString();
                    parser.getString();
                }
            }
        }
    }

    get sequenceNames() {
        return Object.keys(this.chrIndex)
    }


    /**
     * Fetch blocks for a particular genomic range.
     *
     * @param queryChr
     * @param min  genomic start position
     * @param max  genomic end position
     */
    chunksForRange(queryChr, min, max) { //function (refId, min, max) {

        const chrIdx = this.chrIndex[queryChr];

        if (chrIdx) {
            const blocks = chrIdx.blocks;
            const longestFeature = chrIdx.longestFeature;
            const binWidth = chrIdx.binWidth;
            const adjustedPosition = Math.max(min - longestFeature, 0);
            const startBinNumber = Math.floor(adjustedPosition / binWidth);

            if (startBinNumber >= blocks.length) // are we off the end of the bin list, so return nothing
                return []
            else {
                const endBinNumber = Math.min(Math.floor((max - 1) / binWidth), blocks.length - 1);

                // By definition blocks are adjacent in the file for the liner index.  Combine them into one merged block
                const startPos = blocks[startBinNumber].min;
                const endPos = blocks[endBinNumber].max;
                const size = endPos - startPos;
                if (size === 0) {
                    return []
                } else {
                    const mergedBlock = {minv: {block: startPos, offset: 0}, maxv: {block: endPos, offset: 0}};
                    return [mergedBlock]
                }
            }
        } else {
            return undefined
        }
    }
}

const CSI1_MAGIC = 21582659; // CSI\1
const CSI2_MAGIC = 38359875; // CSI\2
const BAI_MAGIC = 21578050;
const TABIX_MAGIC = 21578324;
const TRIBBLE_MAGIC = 1480870228;   //  byte[]{'T', 'I', 'D', 'X'};

/**
 * @param indexURL
 * @param config
 * @param tabix
 *
 */
async function loadIndex(indexURL, config) {

    let arrayBuffer = await igvxhr$1.loadArrayBuffer(indexURL, buildOptions(config));
    let dv = new DataView(arrayBuffer);

    // Some indexs are bgzipped, specifically tabix, and csi.  Bam (bai) are not.  Tribble is usually not.
    // Check first 2 bytes of file for gzip magic number, and inflate if neccessary
    if (dv.getUint8(0) === 0x1f && dv.getUint8(1) === 0x8b) {    // gzipped
        const inflate = unbgzf$1(arrayBuffer);
        arrayBuffer = inflate.buffer;
        dv = new DataView(arrayBuffer);
    }

    const magic = dv.getInt32(0, true);
    switch (magic) {
        case BAI_MAGIC:
            return parseBamIndex(arrayBuffer)
        case TABIX_MAGIC:
            return parseTabixIndex(arrayBuffer)
        case CSI1_MAGIC:
            return parseCsiIndex(arrayBuffer)
        case TRIBBLE_MAGIC:
            return parseTribbleIndex(arrayBuffer)
        case CSI2_MAGIC:
            throw Error("CSI version 2 is not supported.")
        default:
            throw Error(`Unrecognized index type: ${indexURL}`)
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

function getDataWrapper$1(data) {

    if (typeof (data) == 'string' || data instanceof String) {
        return new StringDataWrapper$1(data)
    } else {
        return new ByteArrayDataWrapper$1(data)
    }
}


// Data might be a string, or an UInt8Array
class StringDataWrapper$1 {

    constructor(string) {
        this.data = string;
        this.ptr = 0;
    }

    nextLine() {
        var start = this.ptr,
            idx = this.data.indexOf('\n', start),
            data = this.data;

        if (idx > 0) {
            this.ptr = idx + 1;   // Advance pointer for next line
            if (idx > start && data.charAt(idx - 1) === '\r') {
                // Trim CR manually in CR/LF sequence
                return data.substring(start, idx - 1)
            }
            return data.substring(start, idx)
        } else {
            var length = data.length;
            this.ptr = length;
            // Return undefined only at the very end of the data
            return (start >= length) ? undefined : data.substring(start)
        }
    }
}

class ByteArrayDataWrapper$1 {

    /**
     *
     * @param {Uint8Array} array
     */
    constructor(array) {
        this.data = array;
        this.length = this.data.length;
        this.ptr = 0;
    }

    /**
     * Decode the next line as a UTF-8 string.  From: https://gist.github.com/Yaffle/5458286
     * @returns {undefined|string}
     */
    nextLine() {
        if(this.ptr >= this.data.length) {
            return undefined
        }
        let i = this.ptr;
        const octets = this.data;
        let string = "";
        let maybeCR = false;
        while (i < octets.length) {
            var octet = octets[i];
            var bytesNeeded = 0;
            var codePoint = 0;
            if (octet <= 0x7F) {
                bytesNeeded = 0;
                codePoint = octet & 0xFF;
            } else if (octet <= 0xDF) {
                bytesNeeded = 1;
                codePoint = octet & 0x1F;
            } else if (octet <= 0xEF) {
                bytesNeeded = 2;
                codePoint = octet & 0x0F;
            } else if (octet <= 0xF4) {
                bytesNeeded = 3;
                codePoint = octet & 0x07;
            }
            if (octets.length - i - bytesNeeded > 0) {
                var k = 0;
                while (k < bytesNeeded) {
                    octet = octets[i + k + 1];
                    codePoint = (codePoint << 6) | (octet & 0x3F);
                    k += 1;
                }
            } else {
                codePoint = 0xFFFD;
                bytesNeeded = octets.length - i;
            }
            i += bytesNeeded + 1;

            const c = String.fromCodePoint(codePoint);
            if(c === '\r') {
                maybeCR = true;
            } else if( c === '\n') {
                break
            } else {
                if(maybeCR) {
                    // Add cr's unless immediately followed by a line feed.
                    string += '\r';
                    maybeCR = false;
                }
                string += c;
            }
        }
        this.ptr = i;
        return string
    }


}

/**
 * Class to iterate line-by-line over a BGZipped text file.  This class is useful for iterating from the start of
 * the file.  Not useful for indexed queries.
 */

class BGZLineReader {

    constructor(config) {
        this.config = config;
        this.filePtr = 0;
        this.bufferPtr = 0;
        this.buffer;
    }

    async nextLine() {

        let result = undefined;

        try {
            while (true) {
                const length = this.buffer ? this.buffer.length : 0;
                while (this.bufferPtr < length) {
                    const c = String.fromCharCode(this.buffer[this.bufferPtr++]);
                    if (c === '\r') continue
                    if (c === '\n') {
                        return result
                    }
                    result = result ? result + c : c;
                }
                if (this.eof) {
                    return result
                } else {
                    await this.readNextBlock();
                }
            }
        } catch (e) {
            console.warn(e);
            this.eof = true;
            return result
        }
    }

    async readNextBlock() {

        const bsizeOptions = buildOptions(this.config, {
            range: {
                start: this.filePtr,
                size: 26
            }
        });
        const abuffer = await igvxhr$1.loadArrayBuffer(this.config.url, bsizeOptions);
        const bufferSize = bgzBlockSize$1(abuffer);

        if (bufferSize === 0) {
            this.eof = true;
            this.buffer = undefined;
        } else {

            const options = buildOptions(this.config, {range: {start: this.filePtr, size: bufferSize}});
            const data = await igvxhr$1.loadArrayBuffer(this.config.url, options);
            if (data.byteLength < bufferSize) {
                this.eof = true; // Assumption
            }
            this.buffer = unbgzf$1(data);
            if(this.buffer.byteLength == 0) {
                this.eof = true;
            }
            this.bufferPtr = 0;
            this.filePtr += data.byteLength; //data.byteLength;
        }
    }

}

function concatenateArrayBuffers(arrayBuffers) {

    if (arrayBuffers.length === 1) {
        return arrayBuffers[0]
    }

    let len = 0;
    for (const b of arrayBuffers) {
        len += b.byteLength;
    }
    const c = new Uint8Array(len);
    let offset = 0;
    for (const b of arrayBuffers) {
        c.set(new Uint8Array(b), offset);
        offset += b.byteLength;
    }
    return c.buffer
}

/**
 * Return the block size for the data buffer.
 * @param data
 * @returns {number}
 */
const bgzBlockSize = (data) => {
    const ba = ArrayBuffer.isView(data) ? data : new Uint8Array(data);
    const bsize = (ba[17] << 8 | ba[16]) + 1;
    return bsize
};

class BGZBlockLoader {

    constructor(config) {
        this.config = config;
        this.cacheBlocks = false != config.cacheBlocks;   // Default to true
        this.cache = undefined;
    }

    /**
     * Return inflated data from startBlock through endBlock as an UInt8Array
     *
     * @param minv minimum virtual pointer  {block, offset}
     * @param maxv maximum virtual pointer  {block, offset}
     * @returns {Promise<Uint8Array>}
     */
    async getData(minv, maxv) {

        const startBlock = minv.block;
        const endBlock = maxv.block;
        const skipEnd = maxv.offset === 0;

        const blocks = await this.getInflatedBlocks(startBlock, endBlock, skipEnd);
        if (blocks.length === 1) {
            return blocks[0]
        }

        let len = 0;
        for (const b of blocks) {
            len += b.byteLength;
        }
        const c = new Uint8Array(len);
        let offset = 0;
        for (const b of blocks) {
            c.set(b, offset);
            offset += b.byteLength;
        }
        return c
    }

    /**
     * Return the inflated data for the specified blocks as an array of Uint8Arrays.  This method is public so
     * it can be unit tested. *
     * @param startBlock
     * @param endBlock
     * @returns {Promise<*[Uint8Array]>}
     */
    async getInflatedBlocks(startBlock, endBlock, skipEnd) {

        if (!this.cacheBlocks) {
            const buffer = await this.loadBLockData(startBlock, endBlock, {skipEnd});
            return inflateBlocks(buffer)
        } else {

            const c = this.cache;
            if (c &&
                c.startBlock <= startBlock &&
                (c.endBlock >= endBlock || skipEnd && c.nextEndBlock === endBlock)) {
                //console.log("Complete overlap")
                const startOffset = startBlock - c.startBlock;
                const endOffset = endBlock - c.startBlock;
                return inflateBlocks(c.buffer, startOffset, endOffset)
                // Don't update cache, still valid
            } else {

                let buffer;
                if (!c || (c.startBlock > endBlock || c.endBlock < startBlock)) {
                    // no overlap with cache
                    buffer = await this.loadBLockData(startBlock, endBlock, {skipEnd});
                } else {

                    //console.log("Some overlap")
                    const arrayBuffers = [];

                    // Load blocks preceding cache start, if any
                    if (startBlock < c.startBlock) {
                        // load first blocks
                        const startBuffer = await this.loadBLockData(startBlock, c.startBlock, {skipEnd: true});
                        arrayBuffers.push(startBuffer);
                    }

                    // Slice cached buffer as needed
                    let cachedBuffer;
                    if (startBlock <= c.startBlock && endBlock >= c.endBlock) {
                        cachedBuffer = c.buffer;
                    } else {
                        const start = Math.max(0, startBlock - c.startBlock);
                        let end;
                        if (endBlock >= c.endBlock) {
                            end = c.buffer.byteLength;
                        } else {
                            // We need to find the byte position of the end of "endBlock"
                            const boundaries = findBlockBoundaries(c.buffer);
                            for (let i = 0; i < boundaries.length - 1; i++) {
                                if (c.startBlock + boundaries[i] === endBlock) {
                                    end = boundaries[i + 1];
                                    break
                                }
                            }
                            // Do something if end not found
                        }
                        cachedBuffer = c.buffer.slice(start, end);
                    }
                    arrayBuffers.push(cachedBuffer);

                    // Load end blocks, if any
                    if (endBlock > c.endBlock) {
                        const endBuffer = await this.loadBLockData(c.endBlock, endBlock, {skipStart: true, skipEnd});
                        arrayBuffers.push(endBuffer);
                    }

                    buffer = concatenateArrayBuffers(arrayBuffers);
                }

                // If skipEnd === true we need to find boundary of last block in cache
                let nextEndBlock = endBlock;
                if(skipEnd) {
                    const boundaries = findBlockBoundaries(buffer);
                    endBlock = boundaries[boundaries.length - 1];
                }

                this.cache = {startBlock, endBlock, nextEndBlock, buffer};
                return inflateBlocks(buffer)
            }
        }
    }

    async loadBLockData(startBlock, endBlock, options) {

        const config = this.config;
        const skipStart = options && options.skipStart;
        const skipEnd = options && options.skipEnd;

        // Get size of last block if not skipped
        let lastBlockSize = 0;
        if (!skipEnd) {
            const bsizeOptions = buildOptions(config, {range: {start: endBlock, size: 26}});
            const abuffer = await igvxhr$1.loadArrayBuffer(config.url, bsizeOptions);
            lastBlockSize = bgzBlockSize(abuffer);
        }

        if (skipStart) {
            const bsizeOptions = buildOptions(config, {range: {start: startBlock, size: 26}});
            const abuffer = await igvxhr$1.loadArrayBuffer(config.url, bsizeOptions);
            startBlock += bgzBlockSize(abuffer);
        }

        // Load data for all blocks
        const loadOptions = buildOptions(config, {
            range: {
                start: startBlock,
                size: endBlock + lastBlockSize - startBlock
            }
        });

        //console.log(`${this.config.name}  Loaded ${startBlock} - ${endBlock + lastBlockSize}   (${(endBlock + lastBlockSize - startBlock) / 1000} kb)`)

        return igvxhr$1.loadArrayBuffer(config.url, loadOptions)
    }
}

function findBlockBoundaries(arrayBuffer) {

    const byteLengh = arrayBuffer.byteLength;
    let offset = 0;
    const blockBoundaries = [0];
    while (offset < byteLengh) {
        //console.log("Cache block "  + offset)
        const ba = new Uint8Array(arrayBuffer, offset);
        const bsize = (ba[17] << 8 | ba[16]) + 1;
        offset += bsize;
        if (offset < byteLengh) {
            blockBoundaries.push(offset);
        }
    }
    return blockBoundaries
}


/**
 * Inflate compressed blocks within the data buffer*
 * @param data
 * @param startBlock - optional file location for start block.  Default == 0
 * @param endBlock - optional file location for last block to decompress.
 * @returns {*[]}
 */
function inflateBlocks(data, startBlock, endBlock) {

    startBlock = startBlock || 0;

    const oBlockList = [];
    let ptr = startBlock;

    const lim = data.byteLength - 18;
    while (ptr < lim) {
        try {
            //console.log(113873 + ptr)
            const header = new Uint8Array(data, ptr, 18);
            const xlen = (header[11] << 8) | (header[10]);
            const bsize = ((header[17] << 8) | (header[16]));  // Total block size, including header, minus 1
            const start = 12 + xlen + ptr;    // Start of CDATA
            const bytesLeft = data.byteLength - start;
            const cDataSize = bsize - xlen - 18;

            if (bytesLeft < cDataSize || cDataSize <= 0) {
                // This is unexpected.  Throw error?
                break
            }

            const cdata = new Uint8Array(data, start, cDataSize);
            const unc = inflateRaw_1$2(cdata);
            oBlockList.push(unc);

            if (endBlock === ptr) {
                break
            } else {
                // Advance to next block
                ptr += bsize + 1;
            }

        } catch (e) {
            console.error(e);
            break
        }
    }
    return oBlockList
}

const MIN_EXPONENT = Math.log10(Number.MIN_VALUE);

/**
 * Parser for xQTL files.
 *
 * Quantitative trait loci (QTL) are genomic variants that are significantly associated to a measurable phenotype.
 *
 * Currently there is no standard file format for this data. For igv.js the essential data is genomic position of
 * the variant and the phenotype (e.g. a gene for eQtls)
 *
 * From https://github.com/igvteam/igv.js/issues/1833
 * CHR	SNP	BP	P	Phenotype
 * 10	rs146165798	50023438	0.000106	A1CF
 *
 * UCSC (based on GTEX)
 * https://genome.ucsc.edu/cgi-bin/hgTables?db=hg19&hgta_group=regulation&hgta_track=gtexEqtlTissue&hgta_table=gtexEqtlTissueAdiposeSubcut&hgta_doSchema=describe+table+schema
 *
 * bin	590	int(10) unsigned	range	Indexing field to speed chromosome range queries.
 * chrom	chr1	varchar(255)	values	Reference sequence chromosome or scaffold
 * chromStart	701834	int(10) unsigned	range	Start position in chromosome
 * chromEnd	701835	int(10) unsigned	range	End position in chromosome
 * name	rs189800799/RP11-206L10.9	varchar(255)	values	Variant/gene pair
 * score	22	int(10) unsigned	range	Score from 0-1000 (highest probabiliity in cluster * 1000)
 * strand	.	char(1)	values	.
 * thickStart	701834	int(10) unsigned	range	Start position
 * thickEnd	701835	int(10) unsigned	range	End position
 * itemRgb	16752800	int(10) unsigned	range	R,G,B color: red +effect, blue -effect. Bright for high, pale for lower (cutoff effectSize 2.0 RPKM).
 * variant	rs189800799	varchar(255)	values	Variant (rsID or GTEx identifier if none)
 * geneId	ENSG00000237491.4	varchar(255)	values	Target gene identifier
 * gene	RP11-206L10.9	varchar(255)	values	Target gene symbol
 * distance	-12315	int(11)	range	Distance from TSS
 * effectSize	0.229	float	range	Effect size (FPKM)
 * pValue	13.862	float	range	Nominal p-value
 * causalProb	0.022	float	range	Probability variant is in high confidence causal set
 *
 * EMBL Eqtl catalog
 * https://github.com/eQTL-Catalogue/eQTL-Catalogue-resources/blob/master/tabix/Columns.md
 *
 * variant - The variant ID (chromosome_position_ref_alt) e.g. chr19_226776_C_T. Based on GRCh38 coordinates and reference genome. The chromosome, position, ref and alt values should exactly match same fields in the summary statistics file, with 'chr' prefix added to the chromosome number.
 * r2 - Optional imputation quality score from the imputation software, can be replaced with NA if not available.
 * pvalue - Nominal p-value of association between the variant and the molecular trait.
 * molecular_trait_object_id - For phenotypes with multiple correlated alternatives (multiple alternative transcripts or exons within a gene, multple alternative promoters in txrevise, multiple alternative intons in Leafcutter), this defines the level at which the phenotypes were aggregated. Permutation p-values are calculated accross this set of alternatives.
 * molecular_trait_id - ID of the molecular trait used for QTL mapping. Depending on the quantification method used, this can be either a gene id, exon id, transcript id or a txrevise promoter, splicing or 3'end event id. Examples: ENST00000356937, ENSG00000008128.
 * maf - Minor allele frequency within a QTL mapping context (e.g. cell type or tissues within a study).
 * gene_id - Ensembl gene ID of the molecular trait.
 * median_tpm - Median transcripts per million (TPM) expression value of the gene. Can be replaced with NA if not availble (e.g. in microarray studies).
 * beta - Regression coefficient from the linear model.
 * se - Standard error of the beta.
 * an - Total number of alleles. For autosomal variants, this is usually two times the sample size. Conversly, for autosomal variants, sample size is equal to an/2.
 * ac - Count of the alternative allele.
 * ma_samples - Number of samples carrying at least one copy of the minor allele.
 * chromosome - GRCh38 chromosome name of the variant (e.g. 1,2,3 ...,X).
 * position - GRCh38 position of the variant.
 * ref - GRCh38 reference allele.
 * alt - GRCh38 alternative allele (also the effect allele).
 * type - Type of the genetic variant; SNP, INDEL or OTHER.
 * rsid - The dbSNP v151 rsid of the variant. If the same variant has multiple rsids then these should be split over multiple rows so that all of the other values are duplicated.
 *
 */
class QTLParser {

    chrCol = -1
    snpCol = -1
    posCol = -1
    pValueCol = -1
    phenotypeColumn = -1
    delimiter = '\t'

    constructor(config) {
        this.config = config;

        //TODO -- allow specifying column
        // this.pValueField = config.pValueField || "pValue"
        // this.phenotypeField = config.phenotypeField || config.geneField ||  "geneSymbol"
        // this.snpField = config.snpField || "snp"
    }

    async parseHeader(dataWrapper) {

        const config = this.config;
        if (config.delimiter) this.delimiter = config.delimiter;

        const headerLine = await dataWrapper.nextLine();
        const columns = this.parseHeaderLine(headerLine);

        // Config overrides

        if (config.chrColumn) this.chrColumn = config.chrColumn - 1;
        if (config.snpColumn) this.snpColumn = config.snpColumn - 1;
        if (config.posColumn) this.posColumn = config.posColumn - 1;
        if (config.pValueColumn) this.pValueColumn = config.pValueColumn - 1;
        if (config.phenotypeColumn) this.phenotypeColumn = config.phenotypeColumn - 1;

        return columns
    }

    parseHeaderLine(headerLine) {
        this.columns = headerLine.split(this.delimiter);
        for (let i = 0; i < this.columns.length; i++) {
            const c = this.columns[i].toLowerCase();
            switch (c) {
                case 'chr':
                case 'chromosome':
                case 'chr_id':
                case 'chrom':
                    this.chrColumn = i;
                    break
                case 'bp':
                case 'pos':
                case 'position':
                case 'chr_pos':
                case 'chromEnd':
                    this.posColumn = i;
                    break
                case 'p':
                case 'pval':
                case 'pvalue':
                case 'p-value':
                case 'p.value':
                    this.pValueColumn = i;
                    break
                case 'rsid':
                case 'variant':
                case 'snp':
                    this.snpColumn = i;
                    break
                case 'phenotype':
                case 'gene':
                case 'gene_id':
                case 'molecular_trait_id':
                    this.phenotypeColumn = i;
                    break
            }
        }

        // TODO validate
        return this.columns
    }

    async parseFeatures(dataWrapper) {

        const allFeatures = [];
        const headerLine = dataWrapper.nextLine();
        if (!this.columns) {
            this.parseHeaderLine(headerLine);
        }

        let line;

        const parseValue = (valueString) => {
            // Don't try to parse extremely small values
            const idx = valueString.indexOf("E");
            if (idx > 0) {
                const exp = Number.parseInt(valueString.substring(idx + 1));
                if (exp < MIN_EXPONENT) {
                    return Number.MIN_VALUE
                }
            }
            return Number(valueString)
        };

        while ((line = dataWrapper.nextLine()) !== undefined) {
            const tokens = line.split(this.delimiter);
            if (tokens.length === this.columns.length) {
                const posString = tokens[this.posColumn];
                if (posString.indexOf(";") > 0 || posString.length == 0 || posString.indexOf('x') > 0) {
                    continue
                }
                const chr = tokens[this.chrColumn];
                const pValue = parseValue(tokens[this.pValueColumn]);
                const start = parseInt(posString) - 1;
                const end = start + 1;
                const snp = tokens[this.snpColumn];
                const phenotype = tokens[this.phenotypeColumn];
                const qtl = new QTL({chr, start, end, pValue, snp, phenotype}, this.columns, tokens);

                allFeatures.push(qtl);

            }
        }
        return allFeatures
    }

    /**
     * Test first line to see if this is a QTL file.  Used to determine file format for the case of generic
     * extensions such as "tsv"
     * @param firstLine
     */
    static isQTL(firstLine) {
        const tokens = firstLine.split('\t');
        if (tokens.length < 5) {
            return false
        }
        const requiredHeaders =
            [
                ['chr', 'chromosome', 'chr_id', 'chrom'],
                ['bp', 'pos', 'position', 'chr_pos', 'chromEnd'],
                ['p', 'pval', 'p-value', 'p.value'],
                ['rsid', 'variant', 'snp'],
                ['phenotype', 'gene', 'gene_id', 'molecular_trait_id']
            ];
        for (let h of requiredHeaders) {
            if (!tokens.some(t => h.includes(t.toLowerCase()))) return false
        }
        return true
    }
}


class QTL {

    constructor({chr, start, end, pValue, snp, phenotype}, headers, tokens) {
        this.chr = chr;
        this.start = start;
        this.end = end;
        this.pValue = pValue;
        this.snp = snp;
        this.phenotype = phenotype;
        this.headers = headers;
        this.tokens = tokens;
    }

    popupData() {
        const data = [];
        for (let i = 0; i < this.headers.length; i++) {
            data.push({name: this.headers[i], value: this.tokens[i]});
        }
        return data
        // return [
        //     {name: 'chromosome', value: this.chr},
        //     {name: 'position', value: this.start + 1},
        //     {name: 'pValue', value: this.pValue},
        //     {name: 'snp', value: this.snp},
        //     {name: 'phenotype', value: this.phenotype}
        // ]
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// Conservative estimate of the maximum allowed string length
const MAX_STRING_LENGTH = 500000000;

/**
 * Reader for "bed like" files (tab delimited files with 1 feature per line: bed, gff, vcf, etc)
 *
 * @param config
 * @constructor
 */
class FeatureFileReader {

    sequenceNames

    constructor(config, genome) {

        this.config = config || {};
        this.genome = genome;
        this.indexURL = config.indexURL;
        this.indexed = config.indexed || this.indexURL !== undefined;
        this.queryable = this.indexed;

        if (isFile$1(this.config.url)) {
            this.filename = this.config.url.name;
        } else if (isDataURL(this.config.url)) {
            this.indexed = false;  // by definition
            this.dataURI = config.url;
        } else {
            const uriParts = parseUri$1(this.config.url);
            this.filename = config.filename || uriParts.file;
        }

        this.parser = this.getParser(this.config);

        if (this.config.format === "vcf" && !this.config.indexURL) {
            console.warn("Warning: index file not specified.  The entire vcf file will be loaded.");
        }

    }

    async defaultVisibilityWindow() {
        if (this.config.indexURL) {
            const index = await this.getIndex();
            if (index && index.lastBlockPosition) {
                let gl = 0;
                const s = 10000;
                for (let c of index.sequenceNames) {
                    const chromosome = this.genome.getChromosome(c);
                    if (chromosome) {
                        gl += chromosome.bpLength;
                    }
                }
                return Math.round((gl / index.lastBlockPosition) * s)
            }
        }
    }

    /**
     * Return a promise to load features for the genomic interval
     * @param chr
     * @param start
     * @param end
     */
    async readFeatures(chr, start, end) {

        // insure that header has been loaded
        if (!this.dataURI && !this.header) {
            await this.readHeader();
        }

        let allFeatures;
        const index = await this.getIndex();
        if (index) {
            this.indexed = true;
            allFeatures = await this.loadFeaturesWithIndex(chr, start, end);
        } else if (this.dataURI) {
            this.indexed = false;
            allFeatures = await this.loadFeaturesFromDataURI();
        } else if ("service" === this.config.sourceType) {
            allFeatures = await this.loadFeaturesFromService(chr, start, end);
        } else {
            this.indexed = false;
            allFeatures = await this.loadFeaturesNoIndex();
        }

        allFeatures.sort(function (a, b) {
            if (a.chr === b.chr) {
                return a.start - b.start
            } else {
                return a.chr.localeCompare(b.chr)
            }
        });

        return allFeatures
    }

    async readHeader() {

        if (this.dataURI) {
            await this.loadFeaturesFromDataURI(this.dataURI);
            return this.header
        } else if (this.config.indexURL) {
            const index = await this.getIndex();
            if (!index) {
                // Note - it should be impossible to get here
                throw new Error("Unable to load index: " + this.config.indexURL)
            }
            this.sequenceNames = new Set(index.sequenceNames);

            let dataWrapper;
            if (index.tabix) {
                this._blockLoader = new BGZBlockLoader(this.config);
                dataWrapper = new BGZLineReader(this.config);
            } else {
                // Tribble
                const maxSize = Object.values(index.chrIndex)
                    .flatMap(chr => chr.blocks)
                    .map(block => block.max)
                    .reduce((previous, current) =>
                        Math.min(previous, current), Number.MAX_SAFE_INTEGER);

                const options = buildOptions(this.config, {bgz: index.tabix, range: {start: 0, size: maxSize}});
                const data = await igvxhr$1.loadString(this.config.url, options);
                dataWrapper = getDataWrapper$1(data);
            }

            this.header = await this.parser.parseHeader(dataWrapper);
            return this.header

        } else if ("service" === this.config.sourceType) {
            if (this.config.seqnamesURL) {
                // Side effect, a bit ugly
                const options = buildOptions(this.config, {});
                const seqnameString = await igvxhr$1.loadString(this.config.seqnamesURL, options);
                if (seqnameString) {
                    this.sequenceNames = new Set(seqnameString.split(",").map(sn => sn.trim()).filter(sn => sn));
                }
            }
            if (this.config.headerURL) {
                const options = buildOptions(this.config, {});
                const data = await igvxhr$1.loadString(this.config.headerURL, options);
                const dataWrapper = getDataWrapper$1(data);
                this.header = await this.parser.parseHeader(dataWrapper);  // Cache header, might be needed to parse features
                return this.header
            }

        } else {

            let data;

            if (this.config._filecontents) {
                // In rare instances the entire file must be read and decoded to determine the file format.
                // When this occurs the file contents are temporarily stashed to prevent needing to read the file twice
                data = this.config._filecontents;
                delete this.config._filecontents;
            } else {
                // If this is a non-indexed file we will load all features in advance
                const options = buildOptions(this.config);
                data = await igvxhr$1.loadByteArray(this.config.url, options);
            }

            // If the data size is < max string length decode entire string with TextDecoder.  This is much faster
            // than decoding by line
            if (data.length < MAX_STRING_LENGTH) {
                data = new TextDecoder().decode(data);
            }


            let dataWrapper = getDataWrapper$1(data);
            this.header = await this.parser.parseHeader(dataWrapper);

            // Reset data wrapper and parse features
            dataWrapper = getDataWrapper$1(data);
            this.features = await this.parser.parseFeatures(dataWrapper);   // cache features

            // Extract chromosome names
            this.sequenceNames = new Set();
            for (let f of this.features) this.sequenceNames.add(f.chr);

            return this.header
        }

    }


    getParser(config) {

        switch (config.format) {
            case "vcf":
                return new VcfParser(config)
            case "seg" :
                return new SegParser("seg")
            case "mut":
                return new SegParser("mut")
            case "maf":
                return new SegParser("maf")
            case "gwas" :
                return new GWASParser(config)
            case "qtl":
                return new QTLParser(config)
            case "aed" :
                return new AEDParser(config)
            default:
                return new FeatureParser(config)
        }
    }

    async loadFeaturesNoIndex() {

        if (this.features) {
            // An optimization hack for non-indexed files, features are temporarily cached when header is read.
            const tmp = this.features;
            delete this.features;
            return tmp
        } else {
            const options = buildOptions(this.config);    // Add oauth token, if any
            const data = await igvxhr$1.loadByteArray(this.config.url, options);
            if (!this.header) {
                const dataWrapper = getDataWrapper$1(data);
                this.header = await this.parser.parseHeader(dataWrapper);
            }
            const dataWrapper = getDataWrapper$1(data);
            const features = [];
            await this._parse(features, dataWrapper);   // <= PARSING DONE HERE
            return features
        }
    }

    async loadFeaturesWithIndex(chr, start, end) {

        //console.log("Using index"
        const config = this.config;
        this.parser;
        const tabix = this.index.tabix;

        const refId = tabix ? this.index.sequenceIndexMap[chr] : chr;
        if (refId === undefined) {
            return []
        }

        const chunks = this.index.chunksForRange(refId, start, end);
        if (!chunks || chunks.length === 0) {
            return []
        } else {
            const allFeatures = [];
            for (let chunk of chunks) {

                let inflated;
                if (tabix) {
                    inflated = await this._blockLoader.getData(chunk.minv, chunk.maxv);
                } else {
                    const options = buildOptions(config, {
                        range: {
                            start: chunk.minv.block,
                            size: chunk.maxv.block - chunk.minv.block + 1
                        }
                    });
                    inflated = await igvxhr$1.loadString(config.url, options);
                }

                const slicedData = chunk.minv.offset ? inflated.slice(chunk.minv.offset) : inflated;
                const dataWrapper = getDataWrapper$1(slicedData);
                await this._parse(allFeatures, dataWrapper, chr, end, start);

            }

            return allFeatures
        }
    }

    async loadFeaturesFromService(chr, start, end) {

        let url;
        if (typeof this.config.url === 'function') {
            url = this.config.url({chr, start, end});
        } else {
            url = this.config.url
                .replace("$CHR", chr)
                .replace("$START", start)
                .replace("$END", end);
        }
        const options = buildOptions(this.config);    // Adds oauth token, if any
        const data = await igvxhr$1.loadString(url, options);
        const dataWrapper = getDataWrapper$1(data);
        const features = [];
        await this._parse(features, dataWrapper);   // <= PARSING DONE HERE
        return features

    }

    async _parse(allFeatures, dataWrapper, chr, end, start) {

        let features = await this.parser.parseFeatures(dataWrapper);

        features.sort(function (a, b) {
            if (a.chr === b.chr) {
                return a.start - b.start
            } else {
                return a.chr.localeCompare(b.chr)
            }
        });

        // Filter features not in requested range.
        if (undefined === chr) {
            for (let f of features) allFeatures.push(f);   // Don't use spread operator !!!  slicedFeatures might be very large
        } else {
            let inInterval = false;
            for (let i = 0; i < features.length; i++) {
                const f = features[i];
                if (f.chr === chr) {
                    if (f.start > end) {
                        allFeatures.push(f);  // First feature beyond interval
                        break
                    }
                    if (f.end >= start && f.start <= end) {
                        // All this to grab first feature before start of interval.  Needed for some track renderers, like line plot
                        if (!inInterval) {
                            inInterval = true;
                            if (i > 0) {
                                allFeatures.push(features[i - 1]);
                            }
                        }
                        allFeatures.push(f);
                    }
                }
            }
        }
    }

    async getIndex() {
        if (this.index) {
            return this.index
        } else if (this.config.indexURL) {
            this.index = await this.loadIndex();
            return this.index
        }
    }

    /**
     * Return a Promise for the async loaded index
     */
    async loadIndex() {
        const indexURL = this.config.indexURL;
        return loadIndex(indexURL, this.config)
    }

    async loadFeaturesFromDataURI() {

        if (this.features) {
            // An optimization hack for non-indexed files, features are temporarily cached when header is read.
            const tmp = this.features;
            delete this.features;
            return tmp
        } else {
            const plain = decodeDataURI$1(this.dataURI);
            let dataWrapper = getDataWrapper$1(plain);
            this.header = await this.parser.parseHeader(dataWrapper);
            if (this.header instanceof String && this.header.startsWith("##gff-version 3")) {
                this.format = 'gff3';
            }

            dataWrapper = getDataWrapper$1(plain);
            const features = [];
            await this._parse(features, dataWrapper);
            return features
        }
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016 University of California San Diego
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const isString$2 = isString$3;


class CustomServiceReader {
    constructor(config) {
        this.config = config;
    }

    async readFeatures(chr, start, end) {

        let url;
        if (typeof this.config.url === 'function') {
            url = this.config.url({chr, start, end});
        } else {
            url = this.config.url
                .replace("$CHR", chr)
                .replace("$START", start)
                .replace("$END", end);
        }

        let config = Object.assign({}, this.config);
        if (this.config.body !== undefined) {
            if (typeof this.config.body === 'function') {
                config.body = this.config.body({chr, start, end});
            } else {
                config.body =
                    this.config.body
                        .replace("$CHR", chr)
                        .replace("$START", start)
                        .replace("$END", end);
            }
        }


        let features;
        const data = await igvxhr$1.load(url, config);
        if (data) {
            if (typeof this.config.parser === "function") {
                features = this.config.parser(data);
            } else if (isString$2(data)) {
                features = JSON.parse(data);
            } else {
                features = data;
            }
        }
        if (this.config.mappings) {
            let mappingKeys = Object.keys(this.config.mappings);
            for (let f of features) {
                for (let key of mappingKeys) {
                    f[key] = f[this.config.mappings[key]];
                }
            }
        }
        return features
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const UCSCServiceReader = function (config, genome) {
    this.config = config;
    this.genome = genome;
    this.expandQueryInterval = false;
};

UCSCServiceReader.prototype.readFeatures = function (chr, start, end) {

    const s = Math.max(0, Math.floor(start));
    let e = Math.ceil(end);

    if (this.genome) {
        const c = this.genome.getChromosome(chr);
        if (c && e > c.bpLength) {
            e = c.bpLength;
        }
    }


    const url = this.config.url + '?db=' + this.config.db + '&table=' + this.config.tableName + '&chr=' + chr + '&start=' + s + '&end=' + e;

    return igvxhr$1.loadJson(url, this.config)
        .then(function (data) {
            if (data) {
                data.forEach(function (sample) {
                    if (sample.hasOwnProperty('exonStarts') &&
                        sample.hasOwnProperty('exonEnds') &&
                        sample.hasOwnProperty('exonCount') &&
                        sample.hasOwnProperty('cdsStart') &&
                        sample.hasOwnProperty('cdsEnd')) {
                        addExons(sample);
                    }
                });
                return data
            } else {
                return null
            }
        })
};

function addExons(sample) {
    var exonCount, exonStarts, exonEnds, exons, eStart, eEnd;
    exonCount = sample['exonCount'];
    exonStarts = sample['exonStarts'].split(',');
    exonEnds = sample['exonEnds'].split(',');
    exons = [];

    for (var i = 0; i < exonCount; i++) {
        eStart = parseInt(exonStarts[i]);
        eEnd = parseInt(exonEnds[i]);
        var exon = {start: eStart, end: eEnd};

        if (sample.cdsStart > eEnd || sample.cdsEnd < sample.cdsStart) exon.utr = true;   // Entire exon is UTR
        if (sample.cdsStart >= eStart && sample.cdsStart <= eEnd) exon.cdStart = sample.cdsStart;
        if (sample.cdsEnd >= eStart && sample.cdsEnd <= eEnd) exon.cdEnd = sample.cdsEnd;

        exons.push(exon);
    }

    sample.exons = exons;
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2015 UC San Diego
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * EQTL reader for GTEX webservice
 */
class GtexReader {

    featureCaches = []

    gtexChrs = new Set(["chr1", "chr10", "chr11", "chr12", "chr13", "chr14", "chr15", "chr16", "chr17", "chr18",
        "chr19", "chr2", "chr20", "chr21", "chr22", "chr3", "chr4", "chr5", "chr6", "chr7", "chr8", "chr9", "chrM",
        "chrX", "chrY"])

    constructor(config) {

        this.config = config;
        this.url = config.url;
        this.tissueId = config.tissueSiteDetailId;
        this.datasetId = config.datasetId || "gtex_v8";
    }

    async readFeatures(chr, bpStart, bpEnd) {

        // GTEX uses UCSC chromosome naming conventions.
        const queryChr = chr.startsWith("chr") ? chr : chr === "MT" ? "chrM" : "chr" + chr;

        if (!this.gtexChrs.has(queryChr)) {
            return []
        }

        const queryStart = Math.floor(bpStart);
        const queryEnd = Math.ceil(bpEnd);
        const datasetId = this.datasetId;
        const queryURL = this.url + "?chromosome=" + queryChr + "&start=" + queryStart + "&end=" + queryEnd +
            "&tissueSiteDetailId=" + this.tissueId + "&datasetId=" + datasetId;

        const json = await igvxhr$1.loadJson(queryURL, {
            withCredentials: this.config.withCredentials
        });

        if (json && json.singleTissueEqtl) {
            return json.singleTissueEqtl.map(json => new EQTL(json))
        } else {
            return []
        }
    }
}

// Example GTEX eqtl
// {
//     "chromosome": "chr16",
//     "datasetId": "gtex_v8",
//     "gencodeId": "ENSG00000275445.1",
//     "geneSymbol": "CTD-2649C14.3",
//     "geneSymbolUpper": "CTD-2649C14.3",
//     "nes": 0.51295,
//     "pValue": 5.57674e-14,
//     "pos": 21999621,
//     "snpId": "rs35368623",
//     "tissueSiteDetailId": "Muscle_Skeletal",
//     "variantId": "chr16_21999621_G_GA_b38"
// }


class EQTL {

    constructor(eqtl) {
        this.chr = eqtl.chromosome;
        this.start = eqtl.pos - 1;
        this.end = this.start + 1;
        this.snp = eqtl.snpId;
        this.phenotype = eqtl.geneSymbol;
        this.pValue = eqtl.pValue;
        this.json = eqtl;
    }

    popupData() {
        return Object.keys(this.json).map(key => {
            return {name: key, value: this.json[key]}
        })
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class HtsgetReader {

    constructor(config, genome) {
        this.config = config;
        this.genome = genome;
        if (config.format) {
            this.format = config.format.toUpperCase();
        } else {
            throw Error('Format is required, and must be either "bam" or "cram"')
        }
        if (!(this.format === "BAM" || this.format === "VCF")) {
            throw Error(`htsget format ${config.format} is not supported`)
        }
    }

    async readHeaderData() {
        const url = `${getUrl(this.config)}?class=header&format=${this.format}`;
        const ticket = await igvxhr$1.loadJson(url, buildOptions(this.config));
        return await this.loadUrls(ticket.htsget.urls)
    }

    async readData(chr, start, end) {
        const url = `${getUrl(this.config)}?format=${this.format}&referenceName=${chr}&start=${Math.floor(start)}&end=${Math.ceil(end)}`;
        const ticket = await igvxhr$1.loadJson(url, buildOptions(this.config));
        return this.loadUrls(ticket.htsget.urls)
    }

    async loadUrls(urls) {

        const promiseArray = [];
        for (let urlData of urls) {

            if (urlData.url.startsWith('data:')) {
                // this is a data-uri
                promiseArray.push(Promise.resolve(dataUriToBytes(urlData.url)));

            } else {
                const options = {headers: urlData.headers || {}};
                promiseArray.push(igvxhr$1.loadArrayBuffer(urlData.url, options));
            }
        }
        const arrayBuffers = await Promise.all(promiseArray);
        return concatArrays(arrayBuffers)
    }


    static async inferFormat(config) {
        try {
            const url = getUrl(config);
            const headerURL = `${url}${url.includes("?") ? "&" : "?"}class=header`;
            const ticket = await igvxhr$1.loadJson(headerURL, buildOptions(config));
            if (ticket.htsget) {
                const format = ticket.htsget.format;
                if (!(format === "BAM" || format === "VCF")) {
                    throw Error(`htsget format ${format} is not supported`)
                }
                config.format = format.toLowerCase();
                config.sourceType = "htsget";
                if (!config.name) {
                    config.name = await getFilename$1(config.url);
                }
            }
        } catch (e) {
            // Errors => this is not an htsget source, not an application error.  Ignore
        }
    }
}

/**
 * Extract the full url from the config.  Striving for backward compatibility, "endpoint" and "id" are deprecated.
 *
 * @param config
 */
function getUrl(config) {
    if (config.url && config.endpoint && config.id) {
        return config.url + config.endpoint + config.id    // Deprecated
    } else if (config.endpoint && config.id) {
        return config.endpoint + config.id                // Deprecated
    } else if (config.url) {
        if (config.url.startsWith("htsget://")) {
            return config.url.replace("htsget://", "https://")    // htsget -> http not supported
        } else {
            return config.url
        }
    } else {
        throw Error("Must specify either 'url', or 'endpoint' and 'id")
    }


}

/**
 * Concatenate a list of array buffers, returning an UInt8Array
 * @param arrayBuffers
 */
function concatArrays(arrayBuffers) {

    let len = 0;
    for (let a of arrayBuffers) {
        len += a.byteLength;
    }

    let offset = 0;
    const newArray = new Uint8Array(len);
    for (let buf of arrayBuffers) {
        const a = new Uint8Array(buf);
        newArray.set(a, offset);
        offset += a.length;
    }

    return newArray
}

function dataUriToBytes(dataUri) {

    const split = dataUri.split(',');
    const info = split[0].split(':')[1];
    let dataString = split[1];

    if (info.indexOf('base64') >= 0) {
        dataString = atob(dataString);
    } else {
        dataString = decodeURI(dataString);
    }

    const bytes = new Uint8Array(dataString.length);
    for (var i = 0; i < dataString.length; i++) {
        bytes[i] = dataString.charCodeAt(i);
    }

    return bytes
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 The Regents of the University of California
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class HtsgetVariantReader extends HtsgetReader {

    constructor(config, genome) {
        super(config, genome);
        this.parser = new VcfParser();
    }

    async readHeader() {
        if (!this.header) {
            let data = await this.readHeaderData();
            if (isgzipped$1(data)) {
                data = ungzip_blocks$1(data);
            }

            const dataWrapper = getDataWrapper$1(data);
            this.header = await this.parser.parseHeader(dataWrapper, this.genome);
            this.chrAliasTable = this.header.chrAliasTable;
        }
        return this.header
    }

    async readFeatures(chr, start, end) {

        if (this.config.format && this.config.format.toUpperCase() !== "VCF") {
            throw Error(`htsget format ${this.config.format} is not supported`)
        }

        if (!this.chrAliasTable) {
            await this.readHeader();
        }

        let queryChr = this.chrAliasTable.has(chr) ? this.chrAliasTable.get(chr) : chr;

        let data = await this.readData(queryChr, start, end);
        if (isgzipped$1(data)) {
            data = ungzip_blocks$1(data);
        }

        const dataWrapper = getDataWrapper$1(data);

        return this.parser.parseFeatures(dataWrapper)

        //  return dataWrapper;

    }

}

/**
 * A data/feature source helper class for managing chromosome aliasing.  Maps reference sequence names to aliases
 * used by the feature source (e.g. chr20 -> 20).
 */
class ChromAliasManager {

    chrAliasTable = new Map()

    /**
     * @param sequenceNames - Sequence names defined by the data source (e.g. bam or feature file)
     * @param genome        - Reference genome object.
     */
    constructor(sequenceNames, genome) {
        this.sequenceNames = new Set(sequenceNames);
        this.genome = genome;
    }

    async getAliasName(chr) {
        if (!this.genome) {
            return chr   // A no-op manager, used in testing.
        }

        if (!this.chrAliasTable.has(chr)) {
            const aliasRecord = await this.genome.getAliasRecord(chr);
            if (!aliasRecord) {
                this.chrAliasTable.set(chr, undefined);  // No know alias, record to prevent searching again
            } else {
                let alias;
                const aliases = Object.keys(aliasRecord)
                    .filter(k => k !== "start" && k !== "end")
                    .map(k => aliasRecord[k])
                    .filter(a => this.sequenceNames.has(a));
                if (aliases.length > 0) {
                    alias = aliases[0];
                }

                this.chrAliasTable.set(chr, alias);  // alias may be undefined => no alias exists. Setting prevents repeated attempts
            }
        }

        return this.chrAliasTable.get(chr)
    }
}

class BaseFeatureSource {

    constructor(genome) {
        this.genome = genome;
    }


    // Return the next feature whose start is > position.
    async nextFeature(chr, position, direction, visibilityWindow) {

        let chromosomeNames = this.genome.chromosomeNames || [chr];
        let idx = chromosomeNames.indexOf(chr);
        if (idx < 0) return // This shouldn't happen

        // Look ahead (or behind) in 10 kb intervals, but no further than visibilityWindow
        const window = Math.min(10000, visibilityWindow || 10000);
        let queryStart = direction ? position : Math.max(position - window, 0);
        while (idx < chromosomeNames.length && idx >= 0) {
            chr = chromosomeNames[idx];
            const chromosome = this.genome.getChromosome(chr);
            const chromosomeEnd = chromosome.bpLength;
            while (queryStart < chromosomeEnd && queryStart >= 0) {
                let queryEnd = direction ? queryStart + window : Math.min(position, queryStart + window);
                const featureList = await this.getFeatures({chr, start: queryStart, end: queryEnd, visibilityWindow});
                if (featureList) {

                    const compare = (o1, o2) => o1.start - o2.start + o1.end - o2.end;
                    const sortedList = Array.from(featureList);
                    sortedList.sort(compare);

                    // Search for next or previous feature relative to centers.  We use a linear search because the
                    // feature is likely to be near the first or end of the list
                    let idx = direction ? 0 : sortedList.length - 1;
                    while(idx >= 0 && idx < sortedList.length) {
                        const f = sortedList[idx];
                        const center = (f.start + f.end) / 2;
                        if(direction) {
                            if(center > position) return f
                            idx++;
                        } else {
                            if(center < position) return f
                            idx--;
                        }
                    }
                }
                queryStart = direction ? queryEnd : queryStart - window;
            }
            if (direction) {
                idx++;
                queryStart = 0;
                position = 0;
            } else {
                idx--;
                if (idx < 0) break
                const prevChromosome = this.genome.getChromosome(chromosomeNames[idx]);
                position = prevChromosome.bpLength;
                queryStart = position - window;
            }
        }
    }

    async previousFeature(chr, position, direction, visibilityWindow) {

        let chromosomeNames = this.genome.chromosomeNames || [chr];
        let idx = chromosomeNames.indexOf(chr);
        if (idx < 0) return // This shouldn't happen

        // Look ahead (or behind) in 10 kb intervals, but no further than visibilityWindow
        const window = Math.min(10000, visibilityWindow || 10000);
        let queryStart = direction ? position : Math.max(position - window, 0);
        while (idx < chromosomeNames.length && idx >= 0) {
            chr = chromosomeNames[idx];
            const chromosome = this.genome.getChromosome(chr);
            const chromosomeEnd = chromosome.bpLength;
            while (queryStart < chromosomeEnd && queryStart >= 0) {
                let queryEnd = Math.min(position, queryStart + window);
                const featureList = await this.getFeatures({chr, start: queryStart, end: queryEnd, visibilityWindow});
                if (featureList) {

                    const compare = (o1, o2) => o1.start - o2.start + o1.end - o2.end;
                    const sortedList = Array.from(featureList);
                    sortedList.sort(compare);

                    // Search for next or previous feature relative to centers.  We use a linear search because the
                    // feature is likely to be near the first or end of the list
                    let idx = direction ? 0 : sortedList.length - 1;
                    while(idx >= 0 && idx < sortedList.length) {
                        const f = sortedList[idx];
                        const center = (f.start + f.end) / 2;
                        if(direction) {
                            if(center > position) return f
                            idx++;
                        } else {
                            if(center < position) return f
                            idx--;
                        }
                    }
                }
                queryStart = direction ? queryEnd : queryStart - window;
            }
            if (direction) {
                idx++;
                queryStart = 0;
                position = 0;
            } else {
                idx--;
                if (idx < 0) break
                const prevChromosome = this.genome.getChromosome(chromosomeNames[idx]);
                position = prevChromosome.bpLength;
                queryStart = position - window;
            }
        }
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016 University of California San Diego
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const GZIP_FLAG = 0x1;

class TDFReader {

    constructor(config, genome) {
        this.config = config;
        this.genome = genome;
        this.path = config.url;
        this.groupCache = {};
        this.datasetCache = {};
    }


    async readHeader() {

        if (this.magic !== undefined) {
            return this   // Already read
        }

        let data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {range: {start: 0, size: 64000}}));
        let binaryParser = new BinaryParser$1(new DataView(data));
        this.magic = binaryParser.getInt();
        this.version = binaryParser.getInt();
        this.indexPos = binaryParser.getLong();
        this.indexSize = binaryParser.getInt();
        binaryParser.getInt();


        if (this.version >= 2) {
            let nWindowFunctions = binaryParser.getInt();
            this.windowFunctions = [];
            while (nWindowFunctions-- > 0) {
                this.windowFunctions.push(binaryParser.getString());
            }
        }

        this.trackType = binaryParser.getString();
        this.trackLine = binaryParser.getString();

        let nTracks = binaryParser.getInt();
        this.trackNames = [];
        while (nTracks-- > 0) {
            this.trackNames.push(binaryParser.getString());
        }
        this.genomeID = binaryParser.getString();
        this.flags = binaryParser.getInt();
        this.compressed = (this.flags & GZIP_FLAG) !== 0;

        // Now read index
        data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {
            range: {
                start: this.indexPos,
                size: this.indexSize
            }
        }));
        binaryParser = new BinaryParser$1(new DataView(data));
        this.datasetIndex = {};
        let nEntries = binaryParser.getInt();
        while (nEntries-- > 0) {
            const name = binaryParser.getString();
            const pos = binaryParser.getLong();
            const size = binaryParser.getInt();
            this.datasetIndex[name] = {position: pos, size: size};
        }

        this.groupIndex = {};
        nEntries = binaryParser.getInt();
        while (nEntries-- > 0) {
            const name = binaryParser.getString();
            const pos = binaryParser.getLong();
            const size = binaryParser.getInt();
            this.groupIndex[name] = {position: pos, size: size};
        }

        return this
    }

    async readDataset(chr, windowFunction, zoom) {

        const key = chr + "_" + windowFunction + "_" + zoom;

        if (this.datasetCache[key]) {
            return this.datasetCache[key]

        } else {
            await this.readHeader();
            const wf = (this.version < 2) ? "" : "/" + windowFunction;
            const zoomString = (chr.toLowerCase() === "all" || zoom === undefined) ? "0" : zoom.toString();

            let dsName;
            if (windowFunction === "raw") {
                dsName = "/" + chr + "/raw";
            } else {
                dsName = "/" + chr + "/z" + zoomString + wf;
            }
            const indexEntry = this.datasetIndex[dsName];

            if (indexEntry === undefined) {
                return undefined
            }

            const data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {
                range: {
                    start: indexEntry.position,
                    size: indexEntry.size
                }
            }));

            if (!data) {
                return undefined
            }

            const binaryParser = new BinaryParser$1(new DataView(data));
            let nAttributes = binaryParser.getInt();
            const attributes = {};
            while (nAttributes-- > 0) {
                attributes[binaryParser.getString()] = binaryParser.getString();
            }
            const dataType = binaryParser.getString();
            const tileWidth = binaryParser.getFloat();
            let nTiles = binaryParser.getInt();
            const tiles = [];
            while (nTiles-- > 0) {
                tiles.push({position: binaryParser.getLong(), size: binaryParser.getInt()});
            }

            const dataset = {
                name: dsName,
                attributes: attributes,
                dataType: dataType,
                tileWidth: tileWidth,
                tiles: tiles
            };

            this.datasetCache[key] = dataset;
            return dataset
        }
    }

    async readRootGroup() {

        const genome = this.genome;
        const rootGroup = this.groupCache["/"];
        if (rootGroup) {
            return rootGroup
        } else {

            const group = await this.readGroup("/");
            const names = group["chromosomes"];
            const maxZoomString = group["maxZoom"];

            // Now parse out interesting attributes.
            if (maxZoomString) {
                this.maxZoom = Number(maxZoomString);
            }

            const totalCountString = group["totalCount"];
            if (totalCountString) {
                group.totalCount = Number(totalCountString);
            }

            // Chromosome names
            const chrAliasTable = {};
            if (names) {
                names.split(",").forEach(function (chr) {
                    const canonicalName = genome.getChromosomeName(chr);
                    chrAliasTable[canonicalName] = chr;
                });
            }
            this.chrAliasTable = chrAliasTable;

            this.groupCache["/"] = group;
            return group
        }
    }

    async readGroup(name) {

        const group = this.groupCache[name];
        if (group) {
            return group
        } else {

            await this.readHeader();
            const indexEntry = this.groupIndex[name];
            if (indexEntry === undefined) {
                return undefined
            }

            const data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {
                range: {
                    start: indexEntry.position,
                    size: indexEntry.size
                }
            }));

            if (!data) {
                return undefined
            }

            const binaryParser = new BinaryParser$1(new DataView(data));
            const group = {name: name};
            let nAttributes = binaryParser.getInt();
            while (nAttributes-- > 0) {
                const key = binaryParser.getString();
                const value = binaryParser.getString();
                group[key] = value;
            }
            this.groupCache[name] = group;
            return group
        }
    }

    async readTiles(tileIndeces, nTracks) {

        tileIndeces.sort(function (a, b) {
            return a.position - b.position
        });

        tileIndeces = tileIndeces.filter(function (idx) {
            return idx.size > 0
        });

        if (tileIndeces.length === 0) {
            return []
        }

        const tiles = [];

        for (let indexEntry of tileIndeces) {

            const data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {
                range: {
                    start: indexEntry.position,
                    size: indexEntry.size
                }
            }));

            let tileData;
            try {
                tileData = this.compressed ? inflate_1$3(data).buffer : data;
            } catch (e) {
                console.error(e);
                continue
            }

            const binaryParser = new BinaryParser$1(new DataView(tileData));
            const type = binaryParser.getString();
            let tile;
            switch (type) {
                case "fixedStep":
                    tile = createFixedStep(binaryParser, nTracks);
                    break
                case "variableStep":
                    tile = createVariableStep(binaryParser, nTracks);
                    break
                case "bed":
                case "bedWithName":
                    tile = createBed(binaryParser, nTracks, type);
                    break
                default:
                    throw "Unknown tile type: " + type
            }
            tiles.push(tile);
        }
        return tiles
    }

    async readTile(indexEntry, nTracks) {

        let data = await igvxhr$1.loadArrayBuffer(this.path, buildOptions(this.config, {
            range: {
                start: indexEntry.position,
                size: indexEntry.size
            }
        }));

        if (this.compressed) {
            const plain = inflate_1$3(data);
            data = plain.buffer;
        }

        const binaryParser = new BinaryParser$1(new DataView(data));
        const type = binaryParser.getString();
        switch (type) {
            case "fixedStep":
                return createFixedStep(binaryParser, nTracks)
            case "variableStep":
                return createVariableStep(binaryParser, nTracks)
            case "bed":
            case "bedWithName":
                return createBed(binaryParser, nTracks, type)
            default:
                throw "Unknown tile type: " + type
        }
    }

}

function createFixedStep(binaryParser, nTracks) {
    const nPositions = binaryParser.getInt();
    const start = binaryParser.getInt();
    const span = binaryParser.getFloat();

    const data = [];
    let nt = nTracks;
    while (nt-- > 0) {
        let np = nPositions;
        const dtrack = [];
        while (np-- > 0) {
            dtrack.push(binaryParser.getFloat());
        }
        data.push(dtrack);
    }

    return {
        type: "fixedStep",
        start: start,
        span: span,
        data: data,
        nTracks: nTracks,
        nPositions: nPositions
    }
}

function createVariableStep(binaryParser, nTracks) {

    const tileStart = binaryParser.getInt();
    const span = binaryParser.getFloat();
    const nPositions = binaryParser.getInt();
    const start = [];

    let np = nPositions;
    while (np-- > 0) {
        start.push(binaryParser.getInt());
    }
    binaryParser.getInt();  // # of samples, ignored but should === nTracks

    const data = [];
    let nt = nTracks;
    while (nt-- > 0) {
        np = nPositions;
        const dtrack = [];
        while (np-- > 0) {
            dtrack.push(binaryParser.getFloat());
        }
        data.push(dtrack);
    }

    return {
        type: "variableStep",
        tileStart: tileStart,
        span: span,
        start: start,
        data: data,
        nTracks: nTracks,
        nPositions: nPositions
    }
}

function createBed(binaryParser, nTracks, type) {

    const nPositions = binaryParser.getInt();

    let n = nPositions;
    const start = [];
    while (n-- > 0) {
        start.push(binaryParser.getInt());
    }

    n = nPositions;
    const end = [];
    while (n-- > 0) {
        end.push(binaryParser.getInt());
    }

    binaryParser.getInt();  // # of samples, ignored but should === nTracks
    const data = [];
    let nt = nTracks;
    while (nt-- > 0) {
        let np = nPositions;
        const dtrack = [];
        while (np-- > 0) {
            dtrack.push(binaryParser.getFloat());
        }
        data.push(dtrack);
    }

    if (type === "bedWithName") {
        n = nPositions;
        const name = [];
        while (n-- > 0) {
            name.push(binaryParser.getString());
        }
    }

    return {
        type: type,
        start: start,
        end: end,
        data: data,
        nTracks: nTracks,
        nPositions: nPositions
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016 University of California San Diego
 * Author: Jim Robinson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class TDFSource extends BaseFeatureSource {

    #wgValues = {}
    searchable = false


    constructor(config, genome) {
        super(genome);
        this.genome = genome;
        this.reader = new TDFReader(config, genome);
        this.queryable = true;
    }

    async getFeatures({chr, start, end, bpPerPixel, windowFunction = "mean"}) {

        if (chr.toLowerCase() === "all") {
            return this.getWGValues(windowFunction, bpPerPixel)
        } else {
            return this._getFeatures(chr, start, end, bpPerPixel, windowFunction)
        }
    }
    async _getFeatures(chr, start, end, bpPerPixel, windowFunction) {
        const genomicInterval = new GenomicInterval(chr, start, end);
        const genome = this.genome;


        if (!this.rootGroup) {
            this.rootGroup = await this.reader.readRootGroup();
            if (!this.normalizationFactor) {
                const totalCount = this.rootGroup.totalCount;
                if (totalCount) {
                    this.normalizationFactor = 1.0e6 / totalCount;
                }
            }
        }

        genomicInterval.bpPerPixel = bpPerPixel;
        const zoom = zoomLevelForScale$1(chr, bpPerPixel, genome);
        let queryChr = this.reader.chrAliasTable[chr];
        let maxZoom = this.reader.maxZoom;
        if (queryChr === undefined) queryChr = chr;
        if (maxZoom === undefined) maxZoom = -1;

        const wf = zoom > maxZoom ? "raw" : windowFunction;
        const dataset = await this.reader.readDataset(queryChr, wf, zoom);
        if (dataset == null) {
            return []
        }

        const tileWidth = dataset.tileWidth;
        const startTile = Math.floor(start / tileWidth);
        const endTile = Math.floor(end / tileWidth);
        const NTRACKS = 1;   // TODO read this
        const tiles = await this.reader.readTiles(dataset.tiles.slice(startTile, endTile + 1), NTRACKS);
        const features = [];
        for (let tile of tiles) {
            switch (tile.type) {
                case "bed":
                    decodeBedTile(tile, chr, start, end, bpPerPixel, features);
                    break
                case "variableStep":
                    decodeVaryTile(tile, chr, start, end, bpPerPixel, features);
                    break
                case "fixedStep":
                    decodeFixedTile(tile, chr, start, end, bpPerPixel, features);
                    break
                default:
                    throw ("Unknown tile type: " + tile.type)
            }
        }
        features.sort(function (a, b) {
            return a.start - b.start
        });

        return features
    }

    get supportsWholeGenome() {
        return true
    }

    get windowFunctions() {
        return this.reader.windowFunctions
    }

    async getWGValues(windowFunction, bpPerPixel) {

        const cached = this.#wgValues[windowFunction];
        if (cached && cached.bpPerPixel > 0.8 * bpPerPixel && cached.bpPerPixel < 1.2 * bpPerPixel) {
            return cached.values
        } else {
            const wgFeatures = [];
            const genome = this.genome;
            const chrNames = this.genome.wgChromosomeNames;
            if (chrNames) {
                for (let c of genome.wgChromosomeNames) {
                    const len = genome.getChromosome(c).bpLength;
                    bpPerPixel = len / 1000;
                    const chrFeatures = await this._getFeatures(c, 0, len, bpPerPixel, windowFunction);
                    if (chrFeatures) {
                        for (let f of chrFeatures) {
                            const wg = Object.assign({}, f);
                            wg.chr = "all";
                            wg.start = genome.getGenomeCoordinate(f.chr, f.start);
                            wg.end = genome.getGenomeCoordinate(f.chr, f.end);
                            wg._f = f;
                            wgFeatures.push(wg);
                        }
                    }
                }
            }
            this.#wgValues[windowFunction] = {values: wgFeatures, bpPerPixel};
            return wgFeatures
        }
    }

}

function decodeBedTile(tile, chr, bpStart, bpEnd, bpPerPixel, features) {

    const nPositions = tile.nPositions;
    const starts = tile.start;
    const ends = tile.end;
    const data = tile.data[0];   // Single track for now
    for (let i = 0; i < nPositions; i++) {
        const s = starts[i];
        const e = ends[i];
        if (e < bpStart) continue
        if (s > bpEnd) break
        features.push({
            chr: chr,
            start: s,
            end: e,
            value: data[i]
        });
    }
}

function decodeVaryTile(tile, chr, bpStart, bpEnd, bpPerPixel, features) {

    const nPositions = tile.nPositions;
    const starts = tile.start;
    const span = tile.span;
    const data = tile.data[0];   // Single track for now
    for (let i = 0; i < nPositions; i++) {
        const s = starts[i];
        const e = s + span;
        if (e < bpStart) continue
        if (s > bpEnd) break
        features.push({
            chr: chr,
            start: s,
            end: e,
            value: data[i]
        });
    }
}

function decodeFixedTile(tile, chr, bpStart, bpEnd, bpPerPixel, features) {

    const nPositions = tile.nPositions;
    let s = tile.start;
    const span = tile.span;
    const data = tile.data[0];   // Single track for now

    for (let i = 0; i < nPositions; i++) {
        const e = s + span;
        if (s > bpEnd) break
        if (e >= bpStart) {
            if (!Number.isNaN(data[i])) {
                features.push({
                    chr: chr,
                    start: s,
                    end: e,
                    value: data[i]
                });
            }
        }
        s = e;
    }
}


var log2 = Math.log(2);

function zoomLevelForScale$1(chr, bpPerPixel, genome) {

    // Convert bpPerPixel to IGV "zoom" level.   This is a bit convoluted,  TDF is computed zoom levels assuming
    // display in a 700 pixel window.  The fully zoomed out view of a chromosome is zoom level "0".
    // Zoom level 1 is magnified 2X,  and so forth

    var chrSize = genome.getChromosome(chr).bpLength;

    return Math.ceil(Math.log(Math.max(0, (chrSize / (bpPerPixel * 700)))) / log2)
}

const RPTREE_HEADER_SIZE = 48;
const RPTREE_NODE_LEAF_ITEM_SIZE = 32;   // leaf item size
const RPTREE_NODE_CHILD_ITEM_SIZE = 24;  // child item size

class RPTree {

    static magic = 610839776
    littleEndian = true
    nodeCache = new Map()

    constructor(path, config, startOffset, loader) {

        this.path = path;
        this.config = config;
        this.startOffset = startOffset;
        this.loader = loader || igvxhr$1;
    }


    async init() {
        const binaryParser = await this.#getParserFor(this.startOffset, RPTREE_HEADER_SIZE);
        let magic = binaryParser.getInt();
        if (magic !== RPTree.magic) {
            binaryParser.setPosition(0);
            this.littleEndian = !this.littleEndian;
            binaryParser.littleEndian = this.littleEndian;
            magic = binaryParser.getInt();
            if (magic !== RPTree.magic) {
                throw Error(`Bad magic number ${magic}`)
            }
        }

        const blockSize = binaryParser.getUInt();
        const itemCount = binaryParser.getLong();
        const startChromIx = binaryParser.getUInt();
        const startBase = binaryParser.getUInt();
        const endChromIx = binaryParser.getUInt();
        const endBase = binaryParser.getUInt();
        const endFileOffset = binaryParser.getLong();
        const itemsPerSlot = binaryParser.getUInt();
        const reserved = binaryParser.getUInt();
        const rootNodeOffset = this.startOffset + RPTREE_HEADER_SIZE;
        this.header = {
            magic,
            blockSize,
            itemCount,
            startChromIx,
            startBase,
            endChromIx,
            endBase,
            endFileOffset,
            itemsPerSlot,
            reserved,
            rootNodeOffset
        };
        return this
    }

    async #getParserFor(start, size) {
        const data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {range: {start, size}}));
        return new BinaryParser$1(new DataView(data), this.littleEndian)
    }


    async findLeafItemsOverlapping(chrIdx1, startBase, chrIdx2, endBase) {

        const leafItems = [];
        const walkTreeNode = async (offset) => {
            const node = await this.readNode(offset);
            for (let item of node.items) {
                if (overlaps(item, chrIdx1, startBase, chrIdx2, endBase)) {
                    if (node.type === 1) {   // Leaf node
                        leafItems.push(item);
                    } else { // Non leaf node
                        await walkTreeNode(item.childOffset);
                    }
                }
            }
        };

        await walkTreeNode(this.header.rootNodeOffset);
        return leafItems
    }


    async readNode(offset) {

        const nodeKey = offset;
        if (this.nodeCache.has(nodeKey)) {
            return this.nodeCache.get(nodeKey)
        }

        let binaryParser = await this.#getParserFor(offset, 4);
        const type = binaryParser.getByte();
        const isLeaf = (type === 1);
        binaryParser.getByte();
        const count = binaryParser.getUShort();
        let bytesRequired = count * (isLeaf ? RPTREE_NODE_LEAF_ITEM_SIZE : RPTREE_NODE_CHILD_ITEM_SIZE);
        binaryParser = await this.#getParserFor(offset + 4, bytesRequired);

        const items = [];
        for (let i = 0; i < count; i++) {
            let item = {
                isLeaf: isLeaf,
                startChrom: binaryParser.getInt(),
                startBase: binaryParser.getInt(),
                endChrom: binaryParser.getInt(),
                endBase: binaryParser.getInt(),
                childOffset: binaryParser.getLong()
            };
            if (isLeaf) {
                item.dataSize =  binaryParser.getLong();
                item.dataOffset = item.childOffset;
            }
            items.push(item);
        }

        const node = {type, items};
        this.nodeCache.set(nodeKey, node);
        return node
    }

}

/**
 * Return true if {chrIdx1:startBase-chrIdx2:endBase} overlaps item's interval
 * @returns {boolean}
 */
function overlaps(item, chrIdx1, startBase, chrIdx2, endBase) {

    if (!item) {
        return false
    }

    return ((chrIdx2 > item.startChrom) || (chrIdx2 === item.startChrom && endBase >= item.startBase)) &&
        ((chrIdx1 < item.endChrom) || (chrIdx1 === item.endChrom && startBase <= item.endBase))


}

function getDecoder(definedFieldCount, fieldCount, autoSql, format) {
//biggenepred
    if ("biginteract" === format || (autoSql && ('chromatinInteract' === autoSql.table || 'interact' === autoSql.table))) {
        return decodeInteract
    } else {
        const standardFieldCount = definedFieldCount - 3;
        return function (feature, tokens) {

            if (standardFieldCount > 0) {
                feature.name = tokens[0];
            }
            if (standardFieldCount > 1) {
                feature.score = Number(tokens[1]);
            }
            if (standardFieldCount > 2) {
                feature.strand = tokens[2];
            }
            if (standardFieldCount > 3) {
                feature.cdStart = parseInt(tokens[3]);
            }
            if (standardFieldCount > 4) {
                feature.cdEnd = parseInt(tokens[4]);
            }
            if (standardFieldCount > 5) {
                if (tokens[5] !== "." && tokens[5] !== "0" && tokens[5] !== "-1") {
                    const c = IGVColor.createColorString(tokens[5]);
                    feature.color = c.startsWith("rgb") ? c : undefined;
                }
            }
            if (standardFieldCount > 8) {
                const exonCount = parseInt(tokens[6]);
                const exonSizes = tokens[7].split(',');
                const exonStarts = tokens[8].split(',');
                const exons = [];
                for (let i = 0; i < exonCount; i++) {
                    const eStart = feature.start + parseInt(exonStarts[i]);
                    const eEnd = eStart + parseInt(exonSizes[i]);
                    exons.push({start: eStart, end: eEnd});
                }
                findUTRs(exons, feature.cdStart, feature.cdEnd);
                feature.exons = exons;
            }

            if (autoSql) {
                // TODO -- these should be equal, validate?  fieldCount-definedFieldCount, as.fields.length, tokens.length-3
                const extraStart = definedFieldCount;
                for (let i = extraStart; i < fieldCount; i++) {
                    if (i < autoSql.fields.length) {

                        const name = autoSql.fields[i].name;

                        if (name === "exonFrames") {
                            const frameOffsets = tokens[i - 3].replace(/,$/, '').split(',');
                            for (let i = 0; i < feature.exons.length; i++) {
                                const exon = feature.exons[i];
                                const fo = parseInt(frameOffsets[i]);
                                if (fo != -1) exon.readingFrame = fo;
                            }
                        } else {
                            const value = tokens[i - 3];
                            feature[name] = value;
                        }
                    }
                }
            }
        }
    }

    //table chromatinInteract
// "Chromatin interaction between two regions"
//     (
//     string chrom;      "Chromosome (or contig, scaffold, etc.). For interchromosomal, use 2 records"
//     uint chromStart;   "Start position of lower region. For interchromosomal, set to chromStart of this region"
//     uint chromEnd;     "End position of upper region. For interchromosomal, set to chromEnd of this region"
//     string name;       "Name of item, for display"
//     uint score;        "Score from 0-1000"
//     double value;      "Strength of interaction or other data value. Typically basis for score"
//     string exp;        "Experiment name (metadata for filtering). Use . if not applicable"
//     string color;      "Item color.  Specified as r,g,b or hexadecimal #RRGGBB or html color name, as in //www.w3.org/TR/css3-color/#html4."
//     string region1Chrom;  "Chromosome of lower region. For non-directional interchromosomal, chrom of this region."
//     uint region1Start;  "Start position of lower/this region"
//     uint region1End;    "End position in chromosome of lower/this region"
//     string region1Name;  "Identifier of lower/this region"
//     string region1Strand; "Orientation of lower/this region: + or -.  Use . if not applicable"
//     string region2Chrom; "Chromosome of upper region. For non-directional interchromosomal, chrom of other region"
//     uint region2Start;  "Start position in chromosome of upper/this region"
//     uint region2End;    "End position in chromosome of upper/this region"
//     string region2Name; "Identifier of upper/this region"
//     string region2Strand; "Orientation of upper/this region: + or -.  Use . if not applicable"
//     )
    function decodeInteract(feature, tokens) {

        feature.chr1 = tokens[5];
        feature.start1 = Number.parseInt(tokens[6]);
        feature.end1 = Number.parseInt(tokens[7]);

        feature.chr2 = tokens[10];
        feature.start2 = Number.parseInt(tokens[11]);
        feature.end2 = Number.parseInt(tokens[12]);

        feature.name = tokens[0];
        feature.score = Number(tokens[1]);
        feature.value = Number(tokens[2]);
        feature.color = tokens[4] === '.' ? undefined : tokens[4] === "0" ? "rgb(0,0,0)" : tokens[4];

        return feature
    }

}

function findUTRs(exons, cdStart, cdEnd) {

    for (let exon of exons) {
        const end = exon.end;
        const start = exon.start;
        if (end < cdStart || start > cdEnd) {
            exon.utr = true;
        } else {
            if (cdStart >= start && cdStart <= end) {
                exon.cdStart = cdStart;
            }
            if (cdEnd >= start && cdEnd <= end) {
                exon.cdEnd = cdEnd;
            }
        }
    }
}

function scoreShade(score, color) {
    const alpha = Math.min(1, 0.11 + 0.89 * (score / 779));
    return alpha.toString()
}

function parseAutoSQL(str) {

    let table;
    const fields = [];
    let startDecoding = false;
    const lines = str.trim().split(/\s*[\r\n]+\s*/g);
    for (let line of lines) {
        line = line.trim();
        if (line.length > 0) {
            if (line.startsWith('#')) {
                continue
            } else if (line.startsWith('table')) {
                table = line.split(/\s+/)[1].trim();
            } else if (line.startsWith('(')) {
                startDecoding = true;
            } else if (line.startsWith(')')) ; else if (startDecoding) {
                const idx = line.indexOf(';');
                if (idx > 0) {
                    const tokens = line.substr(0, idx).split(/\s+/);
                    const description = line.substr(idx + 1).replace(/"/g, '').trim();
                    fields.push({
                        type: tokens[0],
                        name: tokens[1],
                        description: description
                    });
                }
            }
        }
    }
    return {
        table: table,
        fields: fields
    }
}

// This is a port of trix-js from the GMOD repository:  https://github.com/GMOD/trix-js,


// this is the number of hex characters to use for the address in ixixx, see
// https://github.com/GMOD/ixixx-js/blob/master/src/index.ts#L182
const ADDRESS_SIZE = 10;

class Trix {

    ixFile  // URL to the ix file
    ixxFile  // URL to the ixx file
    bufferCache = new Map()

    constructor(ixxFile, ixFile) {
        this.ixFile = ixFile;
        this.ixxFile = ixxFile;
    }

    /**
     * @param searchString
     * @param opts
     * @returns {Promise<Map<any, any>|undefined|*[]>}
     */
    async search(searchString, opts) {

        const searchWords = searchString.split(' ');

        // we only support a single search term
        const searchWord = searchWords[0].toLowerCase();
        const str = await this._getBuffer(searchWord, opts);
        if (!str) {
            return undefined
        }

        const lines = str
            .slice(0, str.lastIndexOf('\n'))
            .split('\n')
            .filter(f => !!f);

        const matches = [];
        for (let line of lines) {
            const word = line.split(' ')[0];
            const match = word.startsWith(searchWord);
            if (match) {
                matches.push(line);
            }
            // we are done scanning if we are lexicographically greater than the search string
            if (word.slice(0, searchWord.length) > searchWord) {
                break
            }
        }

        if(matches.length === 0) {
            return undefined
        } else {
            const results = new Map();
            for(let m of matches) {
                const [term, ...parts] = m.split(' ');
                results.set(term, parts.map(p => p.split(',')[0]));
            }
            return results
        }
    }

    async getIndex(opts) {
        if (!this.index) {
            this.index = await this._readIndex();
        }
        return this.index
    }

    async _readIndex(opts) {

        const file = await igvxhr$1.loadString(this.ixxFile);

        return file
            .split('\n')
            .filter(f => !!f)
            .map(line => {
                const p = line.length - ADDRESS_SIZE;
                const prefix = line.slice(0, p);
                const posStr = line.slice(p);
                const pos = Number.parseInt(posStr, 16);
                return [prefix, pos]          //as [string, number]
            })
    }

    async _getBuffer(searchWord, opts) {

        let start = 0;
        let end = 65536;
        const indexes = await this.getIndex(opts);
        for (let i = 0; i < indexes.length; i++) {
            const [key, value] = indexes[i];
            const trimmedEnd = Math.min(key.length, searchWord.length);
            const trimmedKey = key.slice(0, trimmedEnd);
            if (trimmedKey < searchWord) {
                start = value;
                end = value + 65536;
            }
        }

        // Return the buffer and its end position in the file.
        const len = end - start;
        if (len < 0) {
            return undefined
        }

        if(this.bufferCache.has(start)) {
            return this.bufferCache.get(start)
        } else {
            const buffer = await igvxhr$1.loadString(this.ixFile, {range: {start, size: len}});
            this.bufferCache.set(start, buffer);
            return buffer
        }

    }
}

class ChromTree {

    nameToId = new Map()
    idToName = new Map()

    constructor(path, config, startOffset, loader) {
        this.path = path;
        this.config = config;
        this.startOffset = startOffset;

        this.bpTree = new BPTree(path, config, startOffset, 'BPChromTree', loader);
    }

    async init() {
        return this.bpTree.init()
    }

    getItemCount() {
        return this.bpTree.getItemCount()
    }

    /**
     * Return the chromosome ID for the given name. This is the internal chromosome ID for the parent BB file only.
     * @param {string} chr - The chromosome name.
     * @returns {number|null} - The chromosome ID or null if not found.
     */
    async getIdForName(chr) {
        if (this.nameToId.has(chr)) {
            return this.nameToId.get(chr)
        } else {
            try {
                const result = await this.bpTree.search(chr);
                if (result) {
                    const id = result.id;
                    this.nameToId.set(chr, id);
                    return id
                } else {
                    return
                }
            } catch (error) {
                throw new Error(error)
            }
        }
    }

    /**
     * Return the chromosome name for the given ID. This is a potentially expensive operation as it involves
     * walking the tree until the leaf item for the given name is found. Currently it is used in only 2
     * situations:
     * (1) decoding features from a bigbed search-by-name query
     * (2) decoding bigwig data from the whole genome view
     * @param {number} id
     * @return {string|null}
     */
    async getNameForId(id) {
        if (this.idToName.has(id)) {
            return this.idToName.get(id)
        } else {
            const name = await this.searchForName(id);
            if (name !== null) {
                this.idToName.set(id, name);
                return name
            }
        }
        return null
    }

    /**
     * Perform a reverse search by traversing the tree starting at the given offset. This is potentially expensive
     * as it traverses the tree to find the name corresponding to the given ID.  It shoud not be used for large trees.
     *
     * @param {number} id - The ID to search for.
     * @returns {string|null} - The name corresponding to the ID, or null if not found.
     */
    async searchForName(id) {

        const reverseSearch = async (offset, id) => {

            const node = await this.bpTree.readTreeNode(offset);

            let found = null;

            if (node.type === 1) {
                // Leaf node
                for (const item of node.items) {
                    const key = item.key;
                    const itemId = item.value.id;
                    if (itemId === id) {
                        found = key;
                    }
                    // Cache the name and ID for future lookups
                    this.nameToId.set(key, itemId);
                    this.idToName.set(id, itemId);
                }
                return found
            } else {
                // Non-leaf node
                for (const item of node.items) {
                    found = await reverseSearch.call(this, item.offset, id);
                    if (found !== null) {
                        break
                    }
                }
            }
            return found
        };

        try {
            return reverseSearch.call(this, this.startOffset + 32, id)
        } catch (error) {
            throw new Error(error)
        }
    }

    /**
     * Return an estimated length of the genome, which might be the actual length if the number of contigs is small.
     * This is only used for calculating a default feature visibility window.
     *
     * @return {number}
     */
    async estimateGenomeSize() {
        try {
            const runningTotal = {total: 0, count: 0};
            await this.accumulateSize(this.startOffset + 32, runningTotal, 10000);
            const itemCount = this.getItemCount();
            return (itemCount / runningTotal.count) * runningTotal.total

        } catch (error) {
            console.error("Error estimating genome size", error);
            return -1
        }
    }

    async accumulateSize(offset, runningTotal, maxCount) {

        const node = await this.bpTree.readTreeNode(offset);

        if (node.type === 1) {
            // Leaf node
            for (const item of node.items) {
                const value = item.value;
                runningTotal.total += value.size;
                runningTotal.count += 1;
            }
        } else {
            // Non-leaf node.  Items are visited in random order to avoid biasing the estimate
            const shuffledItems = node.items.slice().sort(() => Math.random() - 0.5);
            for (const item of shuffledItems) {
                await this.accumulateSize(item.offset, runningTotal, maxCount);
                if (runningTotal.count > maxCount) {
                    break
                }
            }
        }
        return runningTotal
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


const BIGWIG_MAGIC_LTH$1 = 0x888FFC26; // BigWig Magic Low to High
const BIGWIG_MAGIC_HTL = 0x26FC8F66; // BigWig Magic High to Low
const BIGBED_MAGIC_LTH$1 = 0x8789F2EB; // BigBed Magic Low to High
const BIGBED_MAGIC_HTL = 0xEBF28987; // BigBed Magic High to Low
const BBFILE_HEADER_SIZE = 64;
const BBFILE_EXTENDED_HEADER_HEADER_SIZE = 64;
const BUFFER_SIZE = 512000;     //  buffer

class BWReader {

    chrAliasTable = new Map()
    rpTreeCache = new Map()

    constructor(config, genome) {
        this.path = config.url;
        this.format = config.format || "bigwig";
        this.genome = genome;
        this.config = config;
        this.bufferSize = BUFFER_SIZE;
        this.loader = isDataURL(this.path) ?
            new DataBuffer(decodeDataURI$1(this.path).buffer) :
            igvxhr$1;

        const trixURL = config.trixURL || config.searchTrix;
        if (trixURL) {
            this._trix = new Trix(`${trixURL}x`, trixURL);
        }

    }

    /**
     * Preload all the data for this bb file
     * @returns {Promise<void>}
     */
    async preload() {
        const data = await igvxhr$1.loadArrayBuffer(this.path);
        this.loader = new DataBuffer(data);
        for (let rpTree of this.rpTreeCache.values()) {
            rpTree.loader = this.loader;
        }
        if (this._searchTrees) {
            for (let bpTree of this._searchTrees) {
                bpTree.loader = this.loader;
            }
        }
    }

    async readWGFeatures(wgChromosomeNames, bpPerPixel, windowFunction) {

        await this.loadHeader();
        // Convert the logic to JavaScript
        let minID = Number.MAX_SAFE_INTEGER;
        let maxID = -1;
        let chr1;
        let chr2;

        for (const chr of wgChromosomeNames) {
            const id = await this.getIdForChr(chr);
            if (id === null || id === undefined) {
                continue
            }
            if (id < minID) {
                minID = id;
                chr1 = chr;
            }
            if (id > maxID) {
                maxID = id;
                chr2 = chr;
            }
        }

        return this.readFeatures(chr1, 0, chr2, Number.MAX_VALUE, bpPerPixel, windowFunction)
    }

    async readFeatures(chr1, bpStart, chr2, bpEnd, bpPerPixel, windowFunction = "mean") {

        if (!bpStart) bpStart = 0;
        if (!bpEnd) bpEnd = Number.MAX_SAFE_INTEGER;

        await this.loadHeader();

        const chrIdx1 = await this.getIdForChr(chr1);
        const chrIdx2 = await this.getIdForChr(chr2);

        if (chrIdx1 === undefined || chrIdx2 === undefined) {
            return []
        }

        let treeOffset;
        let decodeFunction;
        if (this.type === "bigwig") {
            // Select a biwig "zoom level" appropriate for the current resolution.
            const zoomLevelHeaders = await this.getZoomHeaders();
            let zoomLevelHeader = bpPerPixel ? zoomLevelForScale(bpPerPixel, zoomLevelHeaders) : undefined;
            if (zoomLevelHeader && windowFunction != "none") {
                treeOffset = zoomLevelHeader.indexOffset;
                decodeFunction = decodeZoomData;
            } else {
                treeOffset = this.header.fullIndexOffset;
                decodeFunction = decodeWigData;
            }
        } else {
            // bigbed, zoom data is not currently used in igv for bed type features
            treeOffset = this.header.fullIndexOffset;
            decodeFunction = getBedDataDecoder.call(this);
        }


        // Load the R Tree and fine leaf items
        const rpTree = await this.loadRPTree(treeOffset);
        const leafItems = await rpTree.findLeafItemsOverlapping(chrIdx1, bpStart, chrIdx2, bpEnd);
        if (!leafItems || leafItems.length === 0) {
            return []
        } else {

            // Consolidate leaf items and get all data at once
            let start = Number.MAX_VALUE;
            let end = 0;
            for (let item of leafItems) {
                start = Math.min(start, item.dataOffset);
                end = Math.max(end, item.dataOffset + item.dataSize);
            }
            const size = end - start;
            const arrayBuffer = await this.loader.loadArrayBuffer(this.config.url, buildOptions(this.config, {
                range: {
                    start: start,
                    size: size
                }
            }));

            // Parse data and return features
            const features = [];
            for (let item of leafItems) {
                const uint8Array = new Uint8Array(arrayBuffer, item.dataOffset - start, item.dataSize);
                let plain;
                const isCompressed = this.header.uncompressBuffSize > 0;
                if (isCompressed) {
                    plain = inflate_1$3(uint8Array);
                } else {
                    plain = uint8Array;
                }
                await decodeFunction.call(this, new DataView(plain.buffer), chrIdx1, bpStart, chrIdx2, bpEnd, features, windowFunction);
            }

            features.sort(function (a, b) {
                return a.start - b.start
            });

            return features
        }
    }

    /**
     * Return the ID for the given chromosome name.  If there is no direct match, search for a chromosome alias.
     *
     * @param chr
     * @returns {Promise<*>}
     */
    async getIdForChr(chr) {

        if (this.chrAliasTable.has(chr)) {
            chr = this.chrAliasTable.get(chr);
            if (!chr) {
                return undefined
            }
        }

        let chrIdx = await this.chromTree.getIdForName(chr);

        // Try alias
        if (chrIdx === undefined && this.genome) {
            const aliasRecord = await this.genome.getAliasRecord(chr);
            let alias;
            if (aliasRecord) {
                for (let k of Object.keys(aliasRecord)) {
                    if (k === "start" || k === "end") continue
                    alias = aliasRecord[k];
                    if (alias === chr) continue   // Already tried this
                    chrIdx = await this.chromTree.getIdForName(alias);
                    if (chrIdx !== undefined) {
                        break
                    }
                }
            }
            this.chrAliasTable.set(chr, alias);  // alias may be undefined => no alias exists. Setting prevents repeated attempts
        }
        return chrIdx
    }


    /**
     * Potentially searchable if a bigbed source.  Bigwig files are not searchable.
     * @returns {boolean}
     */
    get searchable() {
        return "bigbed" === this.type
    }

    /**
     * Search the extended BP tree for the search term, and return any matching features.  This only works
     * for BB sources with an "extended" BP tree for searching
     * @param term
     * @returns {Promise<void>}
     */
    async search(term) {
        if (!this.header) {
            await this.loadHeader();
        }
        if (!(this.header && this.header.extraIndexCount)) {
            return undefined
        }

        const region = await this._searchForRegions(term);   // Either 1 or no (undefined) reginos returned for now
        if (region) {
            const features = await this._loadFeaturesForRange(region.offset, region.length);
            if (features) {
                // Collect all matching features and return the largest
                const matching = features.filter(f => {
                    // We could use the searchIndex parameter to pick an attribute (column),  but we don't know
                    // the names of all the columns and if they match IGV names
                    // TODO -- align all feature attribute names with UCSC, an use specific column
                    for (let key of Object.keys(f)) {
                        const v = f[key];
                        if (isString$3(v) && v.toLowerCase() === term.toLowerCase()) {
                            return true
                        }
                    }
                    return false
                });
                if (matching.length > 0) {
                    return matching.reduce((l, f) => (l.end - l.start) > (f.end - f.start) ? l : f, matching[0])
                } else {
                    return undefined
                }
            }
        }
    }

    async _searchForRegions(term) {
        const searchTrees = await this.#getSearchTrees();
        if (searchTrees) {

            // Use a trix index if we have one to map entered term to indexed value in bb file
            if (this._trix) {
                const termLower = term.toLowerCase();
                const trixResults = await this._trix.search(termLower);
                if (trixResults && trixResults.has(termLower)) {   // <= exact matches only for now
                    term = trixResults.get(termLower)[0];
                }
            }

            // For now take the first match, we don't support multiple results
            for (let bpTree of searchTrees) {
                const result = await bpTree.search(term);
                if (result) {
                    return result
                }
            }
        }
    }

    async #getSearchTrees() {

        if (this._searchTrees === undefined &&
            this.header.extraIndexOffsets &&
            this.header.extraIndexOffsets.length > 0) {
            this._searchTrees = [];
            for (let offset of this.header.extraIndexOffsets) {
                const type = undefined;
                const bpTree = await BPTree.loadBpTree(this.path, this.config, offset, type, this.loader);
                this._searchTrees.push(bpTree);
            }
        }
        return this._searchTrees

    }

    async getZoomHeaders() {
        if (this.zoomLevelHeaders) {
            return this.zoomLevelHeaders
        } else {
            await this.loadHeader();
            return this.zoomLevelHeaders
        }
    }

    /**
     * The BB header consists of
     *  (1) the common header
     *  (2) the zoom headers
     *  (3) autosql
     *  (4) total summary block (version 2 and later)
     *
     *  In addition, we read the chromomsome B+ tree
     * @returns {Promise<*>}
     */
    async loadHeader() {

        if (this.header) {
            return this.header
        } else {
            let data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {
                range: {
                    start: 0,
                    size: BBFILE_HEADER_SIZE
                }
            }));

            let header;

            // Assume low-to-high unless proven otherwise
            this.littleEndian = true;

            const binaryParser = new BinaryParser$1(new DataView(data), this.littleEndian);
            let magic = binaryParser.getUInt();
            if (magic === BIGWIG_MAGIC_LTH$1) {
                this.type = "bigwig";
            } else if (magic === BIGBED_MAGIC_LTH$1) {
                this.type = "bigbed";
            } else {
                //Try big endian order
                this.littleEndian = false;

                binaryParser.littleEndian = false;
                binaryParser.position = 0;
                let magic = binaryParser.getUInt();

                if (magic === BIGWIG_MAGIC_HTL) {
                    this.type = "bigwig";
                } else if (magic === BIGBED_MAGIC_HTL) {
                    this.type = "bigbed";
                } else ;
            }
            // Table 5  "Common header for bigwig and bigbed files"
            header = {
                bwVersion: binaryParser.getUShort(),
                nZoomLevels: binaryParser.getUShort(),
                chromTreeOffset: binaryParser.getLong(),
                fullDataOffset: binaryParser.getLong(),
                fullIndexOffset: binaryParser.getLong(),
                fieldCount: binaryParser.getUShort(),
                definedFieldCount: binaryParser.getUShort(),
                autoSqlOffset: binaryParser.getLong(),
                totalSummaryOffset: binaryParser.getLong(),
                uncompressBuffSize: binaryParser.getInt(),
                extensionOffset: binaryParser.getLong()
            };

            // Read the next chunk containing zoom headers, autosql, and total summary if present.  TotalSummary size = 40 bytes
            const startOffset = BBFILE_HEADER_SIZE;
            const size = header.totalSummaryOffset > 0 ?
                header.totalSummaryOffset - startOffset + 40 :
                Math.min(header.fullDataOffset, header.chromTreeOffset) - startOffset;
            let range = {
                start: startOffset,
                size: size
            };
            data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {range: range}));
            const extHeaderParser = new BinaryParser$1(new DataView(data), this.littleEndian);

            // Load zoom headers, store in order of decreasing reduction level (increasing resolution)
            const nZooms = header.nZoomLevels;
            this.zoomLevelHeaders = [];
            this.firstZoomDataOffset = Number.MAX_SAFE_INTEGER;
            for (let i = 1; i <= nZooms; i++) {
                const zoomNumber = nZooms - i;
                const zlh = new ZoomLevelHeader(zoomNumber, extHeaderParser);
                this.firstZoomDataOffset = Math.min(zlh.dataOffset, this.firstZoomDataOffset);
                this.zoomLevelHeaders[zoomNumber] = zlh;
            }

            // Autosql
            if (header.autoSqlOffset > 0) {
                extHeaderParser.position = header.autoSqlOffset - startOffset;
                const autoSqlString = extHeaderParser.getString();
                if (autoSqlString) {
                    this.autoSql = parseAutoSQL(autoSqlString);
                }
            }

            // Total summary
            if (header.totalSummaryOffset > 0) {
                extHeaderParser.position = header.totalSummaryOffset - startOffset;
                this.totalSummary = new BWTotalSummary(extHeaderParser);
            }

            this.chromTree = new ChromTree(this.path, this.config, header.chromTreeOffset, this.loader);
            await this.chromTree.init();

            // Estimate feature density from dataCount (bigbed only)
            if ("bigbed" === this.type) {
                const dataCount = await this.#readDataCount(header.fullDataOffset);
                this.featureDensity = dataCount / await this.chromTree.estimateGenomeSize();
            }

            this.header = header;

            //extension
            if (header.extensionOffset > 0) {
                await this.loadExtendedHeader(header.extensionOffset);
            }
            return this.header
        }
    }

    async #readDataCount(offset) {
        const data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {
            range: {
                start: offset,
                size: 4
            }
        }));
        const binaryParser = new BinaryParser$1(new DataView(data), this.littleEndian);
        return binaryParser.getInt()
    }


    async loadExtendedHeader(offset) {

        let data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {
            range: {
                start: offset,
                size: BBFILE_EXTENDED_HEADER_HEADER_SIZE
            }
        }));
        let binaryParser = new BinaryParser$1(new DataView(data), this.littleEndian);
        binaryParser.getUShort();
        const extraIndexCount = binaryParser.getUShort();
        const extraIndexListOffset = binaryParser.getLong();
        if (extraIndexCount === 0) return

        let sz = extraIndexCount * (2 + 2 + 8 + 4 + 10 * (2 + 2));
        data = await this.loader.loadArrayBuffer(this.path, buildOptions(this.config, {
            range: {
                start: extraIndexListOffset,
                size: sz
            }
        }));
        binaryParser = new BinaryParser$1(new DataView(data), this.littleEndian);

        const type = [];
        const reserved = [];
        const indexOffset = [];
        for (let i = 0; i < extraIndexCount; i++) {

            type.push(binaryParser.getUShort());

            const fc = binaryParser.getUShort();

            indexOffset.push(binaryParser.getLong());
            reserved.push(binaryParser.getInt());

            for (let j = 0; j < fc; j++) {
                binaryParser.getUShort();

                //const field = this.autoSql.fields[fieldId]
                //console.log(field)

                reserved.push(binaryParser.getUShort());
            }
        }
        this.header.extraIndexCount = extraIndexCount;
        this.header.extraIndexOffsets = indexOffset;
    }

    async loadRPTree(offset) {

        let rpTree = this.rpTreeCache.get(offset);
        if (rpTree) {
            return rpTree
        } else {
            rpTree = new RPTree(this.path, this.config, offset, this.loader);
            await rpTree.init();
            this.rpTreeCache.set(offset, rpTree);
            return rpTree
        }
    }

    async getType() {
        await this.loadHeader();
        return this.type
    }

    async getTrackType() {
        await this.loadHeader();
        if (this.type === "bigwig") {
            return "wig"
        } else {
            return this.autoSql && ("interact" === this.autoSql.table || "chromatinInteract" === this.autoSql.table) ? "interact" : "annotation"
        }
    }

    /**
     * Directly load features given a file offset and size.  Added to support search index.
     * @param offset
     * @param size
     * @private
     */
    async _loadFeaturesForRange(offset, size) {

        const arrayBuffer = await this.loader.loadArrayBuffer(this.config.url, buildOptions(this.config, {
            range: {
                start: offset,
                size: size
            }
        }));

        const uint8Array = new Uint8Array(arrayBuffer);
        const plain = (this.header.uncompressBuffSize > 0) ? inflate_1$3(uint8Array) : uint8Array;
        const decodeFunction = getBedDataDecoder.call(this);
        const features = [];
        await decodeFunction.call(this, new DataView(plain.buffer), 0, 0, Number.MAX_SAFE_INTEGER, Number.MAX_SAFE_INTEGER, features);
        return features

    }
}


class ZoomLevelHeader {
    constructor(index, byteBuffer) {
        this.index = index;
        this.reductionLevel = byteBuffer.getUInt();
        this.reserved = byteBuffer.getInt();
        this.dataOffset = byteBuffer.getLong();
        this.indexOffset = byteBuffer.getLong();
    }
}

class BWTotalSummary {

    constructor(byteBuffer) {
        if (byteBuffer) {
            this.basesCovered = byteBuffer.getLong();
            this.minVal = byteBuffer.getDouble();
            this.maxVal = byteBuffer.getDouble();
            this.sumData = byteBuffer.getDouble();
            this.sumSquares = byteBuffer.getDouble();
            computeStats.call(this);
        } else {
            this.basesCovered = 0;
            this.minVal = 0;
            this.maxVal = 0;
            this.sumData = 0;
            this.sumSquares = 0;
            this.mean = 0;
            this.stddev = 0;
        }
    }
}

function computeStats() {
    let n = this.basesCovered;
    if (n > 0) {
        this.mean = this.sumData / n;
        this.stddev = Math.sqrt(this.sumSquares / (n - 1));

        let min = this.minVal < 0 ? this.mean - 2 * this.stddev : 0,
            max = this.maxVal > 0 ? this.mean + 2 * this.stddev : 0;

        this.defaultRange = {
            min: min,
            max: max
        };
    }
}

function zoomLevelForScale(bpPerPixel, zoomLevelHeaders) {
    let level;
    for (let i = 0; i < zoomLevelHeaders.length; i++) {
        const zl = zoomLevelHeaders[i];
        if (zl.reductionLevel < bpPerPixel) {
            level = zl;
            break
        }
    }
    return level
}


async function decodeWigData(data, chrIdx1, bpStart, chrIdx2, bpEnd, featureArray, windowFunction, littleEndian) {

    const binaryParser = new BinaryParser$1(data, littleEndian);
    const chromId = binaryParser.getInt();
    const blockStart = binaryParser.getInt();
    let chromStart = blockStart;
    let chromEnd = binaryParser.getInt();
    const itemStep = binaryParser.getInt();
    const itemSpan = binaryParser.getInt();
    const type = binaryParser.getByte();
    binaryParser.getByte();
    let itemCount = binaryParser.getUShort();

    if (chromId >= chrIdx1 && chromId <= chrIdx2) {

        let idx = 0;
        while (itemCount-- > 0) {
            let value;
            switch (type) {
                case 1:
                    chromStart = binaryParser.getInt();
                    chromEnd = binaryParser.getInt();
                    value = binaryParser.getFloat();
                    break
                case 2:
                    chromStart = binaryParser.getInt();
                    value = binaryParser.getFloat();
                    chromEnd = chromStart + itemSpan;
                    break
                case 3:  // Fixed step
                    value = binaryParser.getFloat();
                    chromStart = blockStart + idx * itemStep;
                    chromEnd = chromStart + itemSpan;
                    idx++;
                    break
            }

            if (chromId < chrIdx1 || (chromId === chrIdx1 && chromEnd < bpStart)) continue
            else if (chromId > chrIdx2 || (chromId === chrIdx2 && chromStart >= bpEnd)) break

            if (Number.isFinite(value)) {
                const chr = await this.chromTree.getNameForId(chromId);
                featureArray.push({chr: chr, start: chromStart, end: chromEnd, value: value});
            }
        }
    }
}

function getBedDataDecoder() {

    const minSize = 3 * 4 + 1;   // Minimum # of bytes required for a bed record
    const decoder = getDecoder(this.header.definedFieldCount, this.header.fieldCount, this.autoSql, this.format);
    return async function (data, chrIdx1, bpStart, chrIdx2, bpEnd, featureArray) {

        const binaryParser = new BinaryParser$1(data, this.littleEndian);
        while (binaryParser.remLength() >= minSize) {

            const chromId = binaryParser.getInt();
            const chr = await this.chromTree.getNameForId(chromId);
            const chromStart = binaryParser.getInt();
            const chromEnd = binaryParser.getInt();
            const rest = binaryParser.getString();
            if (chromId < chrIdx1 || (chromId === chrIdx1 && chromEnd < bpStart)) continue
            else if (chromId > chrIdx2 || (chromId === chrIdx2 && chromStart >= bpEnd)) break

            if (chromEnd > 0) {
                const feature = {chr: chr, start: chromStart, end: chromEnd};
                featureArray.push(feature);
                const tokens = rest.split("\t");
                decoder(feature, tokens);
            }
        }
    }
}

async function decodeZoomData(data, chrIdx1, bpStart, chrIdx2, bpEnd, featureArray, windowFunction, littleEndian) {

    const binaryParser = new BinaryParser$1(data, littleEndian);
    const minSize = 8 * 4;  // Minimum # of bytes required for a zoom record


    while (binaryParser.remLength() >= minSize) {
        const chromId = binaryParser.getInt();
        const chromStart = binaryParser.getInt();
        const chromEnd = binaryParser.getInt();
        const validCount = binaryParser.getInt();
        const minVal = binaryParser.getFloat();
        const maxVal = binaryParser.getFloat();
        const sumData = binaryParser.getFloat();
        binaryParser.getFloat();
        let value;
        switch (windowFunction) {
            case "min":
                value = minVal;
                break
            case "max":
                value = maxVal;
                break
            default:
                value = validCount === 0 ? 0 : sumData / validCount;
        }

        if (chromId < chrIdx1 || (chromId === chrIdx1 && chromEnd < bpStart)) continue
        else if (chromId > chrIdx2 || (chromId === chrIdx2 && chromStart >= bpEnd)) break


        if (Number.isFinite(value)) {
            const chr = await this.chromTree.getNameForId(chromId);
            featureArray.push({chr: chr, start: chromStart, end: chromEnd, value: value});


        }
    }
}

class DataBuffer {

    constructor(data) {
        this.data = data;
    }

    /**
     * igvxhr interface
     * @param ignore
     * @param options
     * @returns {any}
     */
    loadArrayBuffer(ignore, options) {
        const range = options.range;
        return range ? this.data.slice(range.start, range.start + range.size) : this.data
    }

    /**
     * BufferedReader interface
     *
     * @param requestedRange - byte rangeas {start, size}
     * @param fulfill - function to receive result
     * @param asUint8 - optional flag to return result as an UInt8Array
     */
    async dataViewForRange(requestedRange, asUint8) {
        const len = Math.min(this.data.byteLength - requestedRange.start, requestedRange.size);
        return asUint8 ?
            new Uint8Array(this.data, requestedRange.start, len) :
            new DataView(this.data, requestedRange.start, len)
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class BWSource extends BaseFeatureSource {

    queryable = true
    #wgValues = {}
    windowFunctions = ["mean", "min", "max", "none"]

    constructor(config, genome) {
        super(genome);
        this.reader = new BWReader(config, genome);
        this.genome = genome;
        this.format = config.format || "bigwig";
    }

    async getFeatures({chr, start, end, bpPerPixel, windowFunction}) {

        await this.reader.loadHeader();
        const isBigWig = this.reader.type === "bigwig";

        let features;
        if ("all" === chr.toLowerCase()) {
            const wgChromosomeNames = this.genome.wgChromosomeNames;
            features = isBigWig && wgChromosomeNames? await this.getWGValues(wgChromosomeNames, windowFunction, bpPerPixel) : [];
        } else {
            features = await this.reader.readFeatures(chr, start, chr, end, bpPerPixel, windowFunction);
        }

        if (!isBigWig) {
            pack(features);
        }
        return features
    }

    async getHeader() {
        return this.reader.loadHeader()
    }

    async defaultVisibilityWindow() {
        if (this.reader.type === "bigwig") {
            return -1
        } else {
            return this.reader.featureDensity ? Math.floor(10000 / this.reader.featureDensity) : -1
        }

    }

    async getWGValues(wgChromosomeNames, windowFunction, bpPerPixel) {

        const genome = this.genome;
        const cached = this.#wgValues[windowFunction];
        if (cached && cached.bpPerPixel > 0.8 * bpPerPixel && cached.bpPerPixel < 1.2 * bpPerPixel) {
            return cached.values
        } else {
            const features = await this.reader.readWGFeatures(wgChromosomeNames, bpPerPixel, windowFunction);
            let wgValues = [];
            for (let f of features) {
                const chr = f.chr;
                const offset = genome.getCumulativeOffset(chr);
                if (undefined === offset) continue
                const wgFeature = Object.assign({}, f);
                wgFeature.chr = "all";
                wgFeature.start = offset + f.start;
                wgFeature.end = offset + f.end;
                wgFeature._f = f;
                wgValues.push(wgFeature);
            }
            wgValues.sort((a, b) => a.start - b.start);
            this.#wgValues[windowFunction] = {values: wgValues, bpPerPixel};
            return wgValues
        }
    }

    supportsWholeGenome() {
        return this.reader.type === "bigwig"
    }

    async trackType() {
        return this.reader.getTrackType()
    }

    get searchable() {
        return this.reader.searchable
    }

    async search(term) {
        return this.reader.search(term)
    }
}

const shim = .01;
const colorStripWidth = 4;
const axesXOffset = colorStripWidth + 1;

function paintAxis(ctx, width, height, colorOrUndefined) {

    let axisMin = this.axisMin; //|| this.dataRange ? this.dataRange.min : 0
    let axisMax = this.axisMax; //|| this.dataRange ? this.dataRange.max : undefined
    if (undefined === axisMax && this.dataRange) {
        axisMin = this.dataRange.min || 0;
        axisMax = this.dataRange.max;
    }
    if (undefined === axisMax) {
        return
    }

    IGVGraphics.fillRect(ctx, 0, 0, width, height, {fillStyle: 'white'});
    if (colorOrUndefined) {
        IGVGraphics.fillRect(ctx, width - colorStripWidth - 2, 0, colorStripWidth, height, {fillStyle: colorOrUndefined});
    }

    const flipAxis = (undefined === this.flipAxis) ? false : this.flipAxis;

    const xTickStart = 0.95 * width - 8 - axesXOffset;
    const xTickEnd = 0.95 * width - axesXOffset;

    const properties =
        {
            font: 'normal 10px Arial',
            textAlign: 'right',
            fillStyle: 'black',
            strokeStyle: 'black',
        };

    // tick
    IGVGraphics.strokeLine(ctx, xTickStart, shim * height, xTickEnd, shim * height, properties);
    IGVGraphics.fillText(ctx, prettyPrint(flipAxis ? axisMin : axisMax), xTickStart + 4, shim * height + 12, properties);

    const y = (1.0 - shim) * height;

    // tick
    IGVGraphics.strokeLine(ctx, xTickStart, y, xTickEnd, y, properties);
    IGVGraphics.fillText(ctx, prettyPrint(flipAxis ? axisMax : axisMin), xTickStart + 4, y - 4, properties);

    // vertical axis
    IGVGraphics.strokeLine(ctx, xTickEnd, shim * height, xTickEnd, y, properties);

    function prettyPrint(number) {

        if (Number.isInteger(number)) {
            return number
        } else if (number % 1 === 0) {   // Number can be represented exactly as an integer
            return number
        } else if (Math.abs(number) >= 10) {
            return number.toFixed()
        } else if (Math.abs(number) >= 1) {
            return number.toFixed(1)
        } else if (Math.abs(number) >= 0.1) {
            return number.toFixed(2)
        } else {
            return number.toExponential(1)
        }
    }
}

const ColorScaleFactory = {

    fromJson: (obj) => {
        switch (obj.type) {
            case 'gradient':
                return new GradientColorScale(obj)
            case 'doubleGradient':
            case 'diverging':
                return new DivergingGradientScale(obj)
            default:
                throw Error("Unknown color scale type: " + obj)
        }
    },

    defaultGradientScale: function (min, max) {

        return new GradientColorScale({
            "type": "doubleGradient",
            "min": min,
            "max": max,
            "minColor": "rgb(46,56,183)",
            "maxColor": "rgb(164,0,30)"
        })
    },

    defaultDivergingScale: function (min, mid, max) {
        return new DivergingGradientScale({
            "type": "doubleGradient",
            "min": 0,
            "mid": 0.25,
            "max": 0.5,
            "minColor": "rgb(46,56,183)",
            "midColor": "white",
            "maxColor": "rgb(164,0,30)"
        })
    }
};

/**
 *
 * @param cs - object containing
 * 1) array of threshold values defining bin boundaries in ascending order
 * 2) array of colors for bins  (length == thresholds.length + 1)
 * @constructor
 */
class BinnedColorScale {
    constructor(cs) {
        this.thresholds = cs.thresholds;
        this.colors = cs.colors;
    }

    getColor(value) {

        for (let threshold of this.thresholds) {
            if (value < threshold) {
                return this.colors[this.thresholds.indexOf(threshold)]
            }
        }

        return this.colors[this.colors.length - 1]
    }
}


class GradientColorScale {
    constructor(config) {
        this.type = 'gradient';
        const fixed = {
            min: config.min !== undefined ? config.min : config.low,
            max: config.max !== undefined ? config.max : config.high,
            minColor: config.minColor || config.lowColor,
            maxColor: config.maxColor || config.highColor
        };
        this.setProperties(fixed);
    }

    setProperties({min, max, minColor, maxColor}) {
        this.type = 'gradient';
        this.min = min;
        this.max = max;
        this._lowColor = minColor;
        this._highColor = maxColor;
        this.lowComponents = IGVColor.rgbComponents(minColor);
        this.highComponents = IGVColor.rgbComponents(maxColor);
    }

    get minColor() {
        return this._lowColor
    }

    set minColor(c) {
        this._lowColor = c;
        this.lowComponents = IGVColor.rgbComponents(c);
    }

    get maxColor() {
        return this._highColor
    }

    set maxColor(c) {
        this._highColor = c;
        this.highComponents = IGVColor.rgbComponents(c);
    }

    getColor(value) {

        if (value <= this.min) return this.minColor
        else if (value >= this.max) return this.maxColor

        const frac = (value - this.min) / (this.max - this.min);
        const r = Math.floor(this.lowComponents[0] + frac * (this.highComponents[0] - this.lowComponents[0]));
        const g = Math.floor(this.lowComponents[1] + frac * (this.highComponents[1] - this.lowComponents[1]));
        const b = Math.floor(this.lowComponents[2] + frac * (this.highComponents[2] - this.lowComponents[2]));

        return "rgb(" + r + "," + g + "," + b + ")"
    }

    /**
     * Return a simple json-like object, not a literaly json string
     * @returns {{max, min, maxColor, minColor}}
     */
    toJson() {
        return {
            type: this.type,
            min: this.min,
            max: this.max,
            minColor: this.minColor,
            maxColor: this.maxColor
        }
    }

    clone() {
        return new GradientColorScale(this.toJson())
    }

}

class DivergingGradientScale {

    constructor(json) {
        this.type = 'diverging';
        this.lowGradientScale = new GradientColorScale({
            minColor: json.minColor || json.lowColor,
            maxColor: json.midColor,
            min: json.min !== undefined ? json.min : json.low,
            max: json.mid
        });
        this.highGradientScale = new GradientColorScale({
            minColor: json.midColor,
            maxColor: json.maxColor || json.highColor,
            min: json.mid,
            max: json.max !== undefined ? json.max : json.high
        });
    }

    getColor(value) {
        if (value < this.mid) {
            return this.lowGradientScale.getColor(value)
        } else {
            return this.highGradientScale.getColor(value)
        }
    }

    get min() {
        return this.lowGradientScale.min
    }

    set min(v) {
        this.lowGradientScale.min = v;
    }

    get max() {
        return this.highGradientScale.max
    }

    set max(v) {
        this.highGradientScale.max = v;
    }

    get mid() {
        return this.lowGradientScale.max
    }

    set mid(v) {
        this.lowGradientScale.max = v;
        this.highGradientScale.min = v;
    }

    get minColor() {
        return this.lowGradientScale.minColor
    }

    set minColor(c) {
        this.lowGradientScale.minColor = c;
    }

    get maxColor() {
        return this.highGradientScale.maxColor
    }

    set maxColor(c) {
        this.highGradientScale.maxColor = c;
    }

    get midColor() {
        return this.lowGradientScale.maxColor
    }

    set midColor(c) {
        this.lowGradientScale.maxColor = c;
        this.highGradientScale.minColor = c;
    }


    /**
     * Return a simple json-like object, not a literaly json string
     * @returns {{max, mid, min, maxColor, midColor, minColor}}
     */
    toJson() {
        return {
            type: this.type,
            min: this.min,
            mid: this.mid,
            max: this.max,
            minColor: this.minColor,
            midColor: this.midColor,
            maxColor: this.maxColor
        }
    }

    clone() {
        const json = this.toJson();
        return new DivergingGradientScale(json)
    }
}

class ConstantColorScale {
    constructor(color) {
        this.color = color;
    }

    getColor() {
        return this.color
    }
}

/*!
 * vanilla-picker v2.12.3
 * https://vanilla-picker.js.org
 *
 * Copyright 2017-2024 Andreas Borgen (https://github.com/Sphinxxxx), Adam Brooks (https://github.com/dissimulate)
 * Released under the ISC license.
 */
var classCallCheck$1 = function (instance, Constructor) {
  if (!(instance instanceof Constructor)) {
    throw new TypeError("Cannot call a class as a function");
  }
};

var createClass$1 = function () {
  function defineProperties(target, props) {
    for (var i = 0; i < props.length; i++) {
      var descriptor = props[i];
      descriptor.enumerable = descriptor.enumerable || false;
      descriptor.configurable = true;
      if ("value" in descriptor) descriptor.writable = true;
      Object.defineProperty(target, descriptor.key, descriptor);
    }
  }

  return function (Constructor, protoProps, staticProps) {
    if (protoProps) defineProperties(Constructor.prototype, protoProps);
    if (staticProps) defineProperties(Constructor, staticProps);
    return Constructor;
  };
}();

var slicedToArray$1 = function () {
  function sliceIterator(arr, i) {
    var _arr = [];
    var _n = true;
    var _d = false;
    var _e = undefined;

    try {
      for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) {
        _arr.push(_s.value);

        if (i && _arr.length === i) break;
      }
    } catch (err) {
      _d = true;
      _e = err;
    } finally {
      try {
        if (!_n && _i["return"]) _i["return"]();
      } finally {
        if (_d) throw _e;
      }
    }

    return _arr;
  }

  return function (arr, i) {
    if (Array.isArray(arr)) {
      return arr;
    } else if (Symbol.iterator in Object(arr)) {
      return sliceIterator(arr, i);
    } else {
      throw new TypeError("Invalid attempt to destructure non-iterable instance");
    }
  };
}();

String.prototype.startsWith = String.prototype.startsWith || function (needle) {
    return this.indexOf(needle) === 0;
};
String.prototype.padStart = String.prototype.padStart || function (len, pad) {
    var str = this;while (str.length < len) {
        str = pad + str;
    }return str;
};

var colorNames$1 = { cb: '0f8ff', tqw: 'aebd7', q: '-ffff', qmrn: '7fffd4', zr: '0ffff', bg: '5f5dc', bsq: 'e4c4', bck: '---', nch: 'ebcd', b: '--ff', bvt: '8a2be2', brwn: 'a52a2a', brw: 'deb887', ctb: '5f9ea0', hrt: '7fff-', chcT: 'd2691e', cr: '7f50', rnw: '6495ed', crns: '8dc', crms: 'dc143c', cn: '-ffff', Db: '--8b', Dcn: '-8b8b', Dgnr: 'b8860b', Dgr: 'a9a9a9', Dgrn: '-64-', Dkhk: 'bdb76b', Dmgn: '8b-8b', Dvgr: '556b2f', Drng: '8c-', Drch: '9932cc', Dr: '8b--', Dsmn: 'e9967a', Dsgr: '8fbc8f', DsTb: '483d8b', DsTg: '2f4f4f', Dtrq: '-ced1', Dvt: '94-d3', ppnk: '1493', pskb: '-bfff', mgr: '696969', grb: '1e90ff', rbrc: 'b22222', rwht: 'af0', stg: '228b22', chs: '-ff', gnsb: 'dcdcdc', st: '8f8ff', g: 'd7-', gnr: 'daa520', gr: '808080', grn: '-8-0', grnw: 'adff2f', hnw: '0fff0', htpn: '69b4', nnr: 'cd5c5c', ng: '4b-82', vr: '0', khk: '0e68c', vnr: 'e6e6fa', nrb: '0f5', wngr: '7cfc-', mnch: 'acd', Lb: 'add8e6', Lcr: '08080', Lcn: 'e0ffff', Lgnr: 'afad2', Lgr: 'd3d3d3', Lgrn: '90ee90', Lpnk: 'b6c1', Lsmn: 'a07a', Lsgr: '20b2aa', Lskb: '87cefa', LsTg: '778899', Lstb: 'b0c4de', Lw: 'e0', m: '-ff-', mgrn: '32cd32', nn: 'af0e6', mgnt: '-ff', mrn: '8--0', mqm: '66cdaa', mmb: '--cd', mmrc: 'ba55d3', mmpr: '9370db', msg: '3cb371', mmsT: '7b68ee', '': '-fa9a', mtr: '48d1cc', mmvt: 'c71585', mnLb: '191970', ntc: '5fffa', mstr: 'e4e1', mccs: 'e4b5', vjw: 'dead', nv: '--80', c: 'df5e6', v: '808-0', vrb: '6b8e23', rng: 'a5-', rngr: '45-', rch: 'da70d6', pgnr: 'eee8aa', pgrn: '98fb98', ptrq: 'afeeee', pvtr: 'db7093', ppwh: 'efd5', pchp: 'dab9', pr: 'cd853f', pnk: 'c0cb', pm: 'dda0dd', pwrb: 'b0e0e6', prp: '8-080', cc: '663399', r: '--', sbr: 'bc8f8f', rb: '4169e1', sbrw: '8b4513', smn: 'a8072', nbr: '4a460', sgrn: '2e8b57', ssh: '5ee', snn: 'a0522d', svr: 'c0c0c0', skb: '87ceeb', sTb: '6a5acd', sTgr: '708090', snw: 'afa', n: '-ff7f', stb: '4682b4', tn: 'd2b48c', t: '-8080', thst: 'd8bfd8', tmT: '6347', trqs: '40e0d0', vt: 'ee82ee', whT: '5deb3', wht: '', hts: '5f5f5', w: '-', wgrn: '9acd32' };

function printNum$1(num) {
    var decs = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 1;

    var str = decs > 0 ? num.toFixed(decs).replace(/0+$/, '').replace(/\.$/, '') : num.toString();
    return str || '0';
}

var Color$1 = function () {
    function Color(r, g, b, a) {
        classCallCheck$1(this, Color);


        var that = this;
        function parseString(input) {

            if (input.startsWith('hsl')) {
                var _input$match$map = input.match(/([\-\d\.e]+)/g).map(Number),
                    _input$match$map2 = slicedToArray$1(_input$match$map, 4),
                    h = _input$match$map2[0],
                    s = _input$match$map2[1],
                    l = _input$match$map2[2],
                    _a = _input$match$map2[3];

                if (_a === undefined) {
                    _a = 1;
                }

                h /= 360;
                s /= 100;
                l /= 100;
                that.hsla = [h, s, l, _a];
            } else if (input.startsWith('rgb')) {
                var _input$match$map3 = input.match(/([\-\d\.e]+)/g).map(Number),
                    _input$match$map4 = slicedToArray$1(_input$match$map3, 4),
                    _r = _input$match$map4[0],
                    _g = _input$match$map4[1],
                    _b = _input$match$map4[2],
                    _a2 = _input$match$map4[3];

                if (_a2 === undefined) {
                    _a2 = 1;
                }

                that.rgba = [_r, _g, _b, _a2];
            } else {
                if (input.startsWith('#')) {
                    that.rgba = Color.hexToRgb(input);
                } else {
                    that.rgba = Color.nameToRgb(input) || Color.hexToRgb(input);
                }
            }
        }

        if (r === undefined) ; else if (Array.isArray(r)) {
            this.rgba = r;
        } else if (b === undefined) {
            var color = r && '' + r;
            if (color) {
                parseString(color.toLowerCase());
            }
        } else {
            this.rgba = [r, g, b, a === undefined ? 1 : a];
        }
    }

    createClass$1(Color, [{
        key: 'printRGB',
        value: function printRGB(alpha) {
            var rgb = alpha ? this.rgba : this.rgba.slice(0, 3),
                vals = rgb.map(function (x, i) {
                return printNum$1(x, i === 3 ? 3 : 0);
            });

            return alpha ? 'rgba(' + vals + ')' : 'rgb(' + vals + ')';
        }
    }, {
        key: 'printHSL',
        value: function printHSL(alpha) {
            var mults = [360, 100, 100, 1],
                suff = ['', '%', '%', ''];

            var hsl = alpha ? this.hsla : this.hsla.slice(0, 3),
                vals = hsl.map(function (x, i) {
                return printNum$1(x * mults[i], i === 3 ? 3 : 1) + suff[i];
            });

            return alpha ? 'hsla(' + vals + ')' : 'hsl(' + vals + ')';
        }
    }, {
        key: 'printHex',
        value: function printHex(alpha) {
            var hex = this.hex;
            return alpha ? hex : hex.substring(0, 7);
        }
    }, {
        key: 'rgba',
        get: function get() {
            if (this._rgba) {
                return this._rgba;
            }
            if (!this._hsla) {
                throw new Error('No color is set');
            }

            return this._rgba = Color.hslToRgb(this._hsla);
        },
        set: function set(rgb) {
            if (rgb.length === 3) {
                rgb[3] = 1;
            }

            this._rgba = rgb;
            this._hsla = null;
        }
    }, {
        key: 'rgbString',
        get: function get() {
            return this.printRGB();
        }
    }, {
        key: 'rgbaString',
        get: function get() {
            return this.printRGB(true);
        }
    }, {
        key: 'hsla',
        get: function get() {
            if (this._hsla) {
                return this._hsla;
            }
            if (!this._rgba) {
                throw new Error('No color is set');
            }

            return this._hsla = Color.rgbToHsl(this._rgba);
        },
        set: function set(hsl) {
            if (hsl.length === 3) {
                hsl[3] = 1;
            }

            this._hsla = hsl;
            this._rgba = null;
        }
    }, {
        key: 'hslString',
        get: function get() {
            return this.printHSL();
        }
    }, {
        key: 'hslaString',
        get: function get() {
            return this.printHSL(true);
        }
    }, {
        key: 'hex',
        get: function get() {
            var rgb = this.rgba,
                hex = rgb.map(function (x, i) {
                return i < 3 ? x.toString(16) : Math.round(x * 255).toString(16);
            });

            return '#' + hex.map(function (x) {
                return x.padStart(2, '0');
            }).join('');
        },
        set: function set(hex) {
            this.rgba = Color.hexToRgb(hex);
        }
    }], [{
        key: 'hexToRgb',
        value: function hexToRgb(input) {

            var hex = (input.startsWith('#') ? input.slice(1) : input).replace(/^(\w{3})$/, '$1F').replace(/^(\w)(\w)(\w)(\w)$/, '$1$1$2$2$3$3$4$4').replace(/^(\w{6})$/, '$1FF');

            if (!hex.match(/^([0-9a-fA-F]{8})$/)) {
                throw new Error('Unknown hex color; ' + input);
            }

            var rgba = hex.match(/^(\w\w)(\w\w)(\w\w)(\w\w)$/).slice(1).map(function (x) {
                return parseInt(x, 16);
            });

            rgba[3] = rgba[3] / 255;
            return rgba;
        }
    }, {
        key: 'nameToRgb',
        value: function nameToRgb(input) {

            var hash = input.toLowerCase().replace('at', 'T').replace(/[aeiouyldf]/g, '').replace('ght', 'L').replace('rk', 'D').slice(-5, 4),
                hex = colorNames$1[hash];
            return hex === undefined ? hex : Color.hexToRgb(hex.replace(/\-/g, '00').padStart(6, 'f'));
        }
    }, {
        key: 'rgbToHsl',
        value: function rgbToHsl(_ref) {
            var _ref2 = slicedToArray$1(_ref, 4),
                r = _ref2[0],
                g = _ref2[1],
                b = _ref2[2],
                a = _ref2[3];

            r /= 255;
            g /= 255;
            b /= 255;

            var max = Math.max(r, g, b),
                min = Math.min(r, g, b);
            var h = void 0,
                s = void 0,
                l = (max + min) / 2;

            if (max === min) {
                h = s = 0;
            } else {
                var d = max - min;
                s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
                switch (max) {
                    case r:
                        h = (g - b) / d + (g < b ? 6 : 0);break;
                    case g:
                        h = (b - r) / d + 2;break;
                    case b:
                        h = (r - g) / d + 4;break;
                }

                h /= 6;
            }

            return [h, s, l, a];
        }
    }, {
        key: 'hslToRgb',
        value: function hslToRgb(_ref3) {
            var _ref4 = slicedToArray$1(_ref3, 4),
                h = _ref4[0],
                s = _ref4[1],
                l = _ref4[2],
                a = _ref4[3];

            var r = void 0,
                g = void 0,
                b = void 0;

            if (s === 0) {
                r = g = b = l;
            } else {
                var hue2rgb = function hue2rgb(p, q, t) {
                    if (t < 0) t += 1;
                    if (t > 1) t -= 1;
                    if (t < 1 / 6) return p + (q - p) * 6 * t;
                    if (t < 1 / 2) return q;
                    if (t < 2 / 3) return p + (q - p) * (2 / 3 - t) * 6;
                    return p;
                };

                var q = l < 0.5 ? l * (1 + s) : l + s - l * s,
                    p = 2 * l - q;

                r = hue2rgb(p, q, h + 1 / 3);
                g = hue2rgb(p, q, h);
                b = hue2rgb(p, q, h - 1 / 3);
            }

            var rgba = [r * 255, g * 255, b * 255].map(Math.round);
            rgba[3] = a;

            return rgba;
        }
    }]);
    return Color;
}();

var EventBucket$1 = function () {
    function EventBucket() {
        classCallCheck$1(this, EventBucket);

        this._events = [];
    }

    createClass$1(EventBucket, [{
        key: 'add',
        value: function add(target, type, handler) {
            target.addEventListener(type, handler, false);
            this._events.push({
                target: target,
                type: type,
                handler: handler
            });
        }
    }, {
        key: 'remove',
        value: function remove(target, type, handler) {
            this._events = this._events.filter(function (e) {
                var isMatch = true;
                if (target && target !== e.target) {
                    isMatch = false;
                }
                if (type && type !== e.type) {
                    isMatch = false;
                }
                if (handler && handler !== e.handler) {
                    isMatch = false;
                }

                if (isMatch) {
                    EventBucket._doRemove(e.target, e.type, e.handler);
                }
                return !isMatch;
            });
        }
    }, {
        key: 'destroy',
        value: function destroy() {
            this._events.forEach(function (e) {
                return EventBucket._doRemove(e.target, e.type, e.handler);
            });
            this._events = [];
        }
    }], [{
        key: '_doRemove',
        value: function _doRemove(target, type, handler) {
            target.removeEventListener(type, handler, false);
        }
    }]);
    return EventBucket;
}();

function parseHTML$1(htmlString) {

    var div = document.createElement('div');
    div.innerHTML = htmlString;
    return div.firstElementChild;
}

function dragTrack$1(eventBucket, area, callback) {
    var dragging = false;

    function clamp(val, min, max) {
        return Math.max(min, Math.min(val, max));
    }

    function onMove(e, info, starting) {
        if (starting) {
            dragging = true;
        }
        if (!dragging) {
            return;
        }

        e.preventDefault();

        var bounds = area.getBoundingClientRect(),
            w = bounds.width,
            h = bounds.height,
            x = info.clientX,
            y = info.clientY;

        var relX = clamp(x - bounds.left, 0, w),
            relY = clamp(y - bounds.top, 0, h);

        callback(relX / w, relY / h);
    }

    function onMouse(e, starting) {
        var button = e.buttons === undefined ? e.which : e.buttons;
        if (button === 1) {
            onMove(e, e, starting);
        } else {
            dragging = false;
        }
    }

    function onTouch(e, starting) {
        if (e.touches.length === 1) {
            onMove(e, e.touches[0], starting);
        } else {
            dragging = false;
        }
    }

    eventBucket.add(area, 'mousedown', function (e) {
        onMouse(e, true);
    });
    eventBucket.add(area, 'touchstart', function (e) {
        onTouch(e, true);
    });
    eventBucket.add(window, 'mousemove', onMouse);
    eventBucket.add(area, 'touchmove', onTouch);
    eventBucket.add(window, 'mouseup', function (e) {
        dragging = false;
    });
    eventBucket.add(area, 'touchend', function (e) {
        dragging = false;
    });
    eventBucket.add(area, 'touchcancel', function (e) {
        dragging = false;
    });
}

var BG_TRANSP$1 = 'linear-gradient(45deg, lightgrey 25%, transparent 25%, transparent 75%, lightgrey 75%) 0 0 / 2em 2em,\n                   linear-gradient(45deg, lightgrey 25%,       white 25%,       white 75%, lightgrey 75%) 1em 1em / 2em 2em';
var HUES$1 = 360;

var EVENT_KEY$1 = 'keydown',
    EVENT_CLICK_OUTSIDE$1 = 'mousedown',
    EVENT_TAB_MOVE$1 = 'focusin';

function $$1(selector, context) {
    return (context || document).querySelector(selector);
}

function stopEvent$1(e) {

    e.preventDefault();
    e.stopPropagation();
}
function onKey$1(bucket, target, keys, handler, stop) {
    bucket.add(target, EVENT_KEY$1, function (e) {
        if (keys.indexOf(e.key) >= 0) {
            if (stop) {
                stopEvent$1(e);
            }
            handler(e);
        }
    });
}

var Picker$1 = function () {
    function Picker(options) {
        classCallCheck$1(this, Picker);


        this.settings = {

            popup: 'right',
            layout: 'default',
            alpha: true,
            editor: true,
            editorFormat: 'hex',
            cancelButton: false,
            defaultColor: '#0cf'
        };

        this._events = new EventBucket$1();

        this.onChange = null;

        this.onDone = null;

        this.onOpen = null;

        this.onClose = null;

        this.setOptions(options);
    }

    createClass$1(Picker, [{
        key: 'setOptions',
        value: function setOptions(options) {
            var _this = this;

            if (!options) {
                return;
            }
            var settings = this.settings;

            function transfer(source, target, skipKeys) {
                for (var key in source) {
                    if (skipKeys && skipKeys.indexOf(key) >= 0) {
                        continue;
                    }

                    target[key] = source[key];
                }
            }

            if (options instanceof HTMLElement) {
                settings.parent = options;
            } else {

                if (settings.parent && options.parent && settings.parent !== options.parent) {
                    this._events.remove(settings.parent);
                    this._popupInited = false;
                }

                transfer(options, settings);

                if (options.onChange) {
                    this.onChange = options.onChange;
                }
                if (options.onDone) {
                    this.onDone = options.onDone;
                }
                if (options.onOpen) {
                    this.onOpen = options.onOpen;
                }
                if (options.onClose) {
                    this.onClose = options.onClose;
                }

                var col = options.color || options.colour;
                if (col) {
                    this._setColor(col);
                }
            }

            var parent = settings.parent;
            if (parent && settings.popup && !this._popupInited) {

                var openProxy = function openProxy(e) {
                    return _this.openHandler(e);
                };

                this._events.add(parent, 'click', openProxy);

                onKey$1(this._events, parent, [' ', 'Spacebar', 'Enter'], openProxy);

                this._popupInited = true;
            } else if (options.parent && !settings.popup) {
                this.show();
            }
        }
    }, {
        key: 'openHandler',
        value: function openHandler(e) {
            if (this.show()) {

                e && e.preventDefault();

                this.settings.parent.style.pointerEvents = 'none';

                var toFocus = e && e.type === EVENT_KEY$1 ? this._domEdit : this.domElement;
                setTimeout(function () {
                    return toFocus.focus();
                }, 100);

                if (this.onOpen) {
                    this.onOpen(this.colour);
                }
            }
        }
    }, {
        key: 'closeHandler',
        value: function closeHandler(e) {
            var event = e && e.type;
            var doHide = false;

            if (!e) {
                doHide = true;
            } else if (event === EVENT_CLICK_OUTSIDE$1 || event === EVENT_TAB_MOVE$1) {

                var knownTime = (this.__containedEvent || 0) + 100;
                if (e.timeStamp > knownTime) {
                    doHide = true;
                }
            } else {

                stopEvent$1(e);

                doHide = true;
            }

            if (doHide && this.hide()) {
                this.settings.parent.style.pointerEvents = '';

                if (event !== EVENT_CLICK_OUTSIDE$1) {
                    this.settings.parent.focus();
                }

                if (this.onClose) {
                    this.onClose(this.colour);
                }
            }
        }
    }, {
        key: 'movePopup',
        value: function movePopup(options, open) {

            this.closeHandler();

            this.setOptions(options);
            if (open) {
                this.openHandler();
            }
        }
    }, {
        key: 'setColor',
        value: function setColor(color, silent) {
            this._setColor(color, { silent: silent });
        }
    }, {
        key: '_setColor',
        value: function _setColor(color, flags) {
            if (typeof color === 'string') {
                color = color.trim();
            }
            if (!color) {
                return;
            }

            flags = flags || {};
            var c = void 0;
            try {

                c = new Color$1(color);
            } catch (ex) {
                if (flags.failSilently) {
                    return;
                }
                throw ex;
            }

            if (!this.settings.alpha) {
                var hsla = c.hsla;
                hsla[3] = 1;
                c.hsla = hsla;
            }
            this.colour = this.color = c;
            this._setHSLA(null, null, null, null, flags);
        }
    }, {
        key: 'setColour',
        value: function setColour(colour, silent) {
            this.setColor(colour, silent);
        }
    }, {
        key: 'show',
        value: function show() {
            var parent = this.settings.parent;
            if (!parent) {
                return false;
            }

            if (this.domElement) {
                var toggled = this._toggleDOM(true);

                this._setPosition();

                return toggled;
            }

            var html = this.settings.template || '<div class="picker_wrapper" tabindex="-1"><div class="picker_arrow"></div><div class="picker_hue picker_slider"><div class="picker_selector"></div></div><div class="picker_sl"><div class="picker_selector"></div></div><div class="picker_alpha picker_slider"><div class="picker_selector"></div></div><div class="picker_editor"><input aria-label="Type a color name or hex value"/></div><div class="picker_sample"></div><div class="picker_done"><button>Ok</button></div><div class="picker_cancel"><button>Cancel</button></div></div>';
            var wrapper = parseHTML$1(html);

            this.domElement = wrapper;
            this._domH = $$1('.picker_hue', wrapper);
            this._domSL = $$1('.picker_sl', wrapper);
            this._domA = $$1('.picker_alpha', wrapper);
            this._domEdit = $$1('.picker_editor input', wrapper);
            this._domSample = $$1('.picker_sample', wrapper);
            this._domOkay = $$1('.picker_done button', wrapper);
            this._domCancel = $$1('.picker_cancel button', wrapper);

            wrapper.classList.add('layout_' + this.settings.layout);
            if (!this.settings.alpha) {
                wrapper.classList.add('no_alpha');
            }
            if (!this.settings.editor) {
                wrapper.classList.add('no_editor');
            }
            if (!this.settings.cancelButton) {
                wrapper.classList.add('no_cancel');
            }
            this._ifPopup(function () {
                return wrapper.classList.add('popup');
            });

            this._setPosition();

            if (this.colour) {
                this._updateUI();
            } else {
                this._setColor(this.settings.defaultColor);
            }
            this._bindEvents();

            return true;
        }
    }, {
        key: 'hide',
        value: function hide() {
            return this._toggleDOM(false);
        }
    }, {
        key: 'destroy',
        value: function destroy() {
            this._events.destroy();
            if (this.domElement) {
                this.settings.parent.removeChild(this.domElement);
            }
        }
    }, {
        key: '_bindEvents',
        value: function _bindEvents() {
            var _this2 = this;

            var that = this,
                dom = this.domElement,
                events = this._events;

            function addEvent(target, type, handler) {
                events.add(target, type, handler);
            }

            addEvent(dom, 'click', function (e) {
                return e.preventDefault();
            });

            dragTrack$1(events, this._domH, function (x, y) {
                return that._setHSLA(x);
            });

            dragTrack$1(events, this._domSL, function (x, y) {
                return that._setHSLA(null, x, 1 - y);
            });

            if (this.settings.alpha) {
                dragTrack$1(events, this._domA, function (x, y) {
                    return that._setHSLA(null, null, null, 1 - y);
                });
            }

            var editInput = this._domEdit;
            {
                addEvent(editInput, 'input', function (e) {
                    that._setColor(this.value, { fromEditor: true, failSilently: true });
                });

                addEvent(editInput, 'focus', function (e) {
                    var input = this;

                    if (input.selectionStart === input.selectionEnd) {
                        input.select();
                    }
                });
            }

            this._ifPopup(function () {

                var popupCloseProxy = function popupCloseProxy(e) {
                    return _this2.closeHandler(e);
                };

                addEvent(window, EVENT_CLICK_OUTSIDE$1, popupCloseProxy);
                addEvent(window, EVENT_TAB_MOVE$1, popupCloseProxy);
                onKey$1(events, dom, ['Esc', 'Escape'], popupCloseProxy);

                var timeKeeper = function timeKeeper(e) {
                    _this2.__containedEvent = e.timeStamp;
                };
                addEvent(dom, EVENT_CLICK_OUTSIDE$1, timeKeeper);

                addEvent(dom, EVENT_TAB_MOVE$1, timeKeeper);

                addEvent(_this2._domCancel, 'click', popupCloseProxy);
            });

            var onDoneProxy = function onDoneProxy(e) {
                _this2._ifPopup(function () {
                    return _this2.closeHandler(e);
                });
                if (_this2.onDone) {
                    _this2.onDone(_this2.colour);
                }
            };
            addEvent(this._domOkay, 'click', onDoneProxy);
            onKey$1(events, dom, ['Enter'], onDoneProxy);
        }
    }, {
        key: '_setPosition',
        value: function _setPosition() {
            var parent = this.settings.parent,
                elm = this.domElement;

            if (parent !== elm.parentNode) {
                parent.appendChild(elm);
            }

            this._ifPopup(function (popup) {

                if (getComputedStyle(parent).position === 'static') {
                    parent.style.position = 'relative';
                }

                var cssClass = popup === true ? 'popup_right' : 'popup_' + popup;

                ['popup_top', 'popup_bottom', 'popup_left', 'popup_right'].forEach(function (c) {

                    if (c === cssClass) {
                        elm.classList.add(c);
                    } else {
                        elm.classList.remove(c);
                    }
                });

                elm.classList.add(cssClass);
            });
        }
    }, {
        key: '_setHSLA',
        value: function _setHSLA(h, s, l, a, flags) {
            flags = flags || {};

            var col = this.colour,
                hsla = col.hsla;

            [h, s, l, a].forEach(function (x, i) {
                if (x || x === 0) {
                    hsla[i] = x;
                }
            });
            col.hsla = hsla;

            this._updateUI(flags);

            if (this.onChange && !flags.silent) {
                this.onChange(col);
            }
        }
    }, {
        key: '_updateUI',
        value: function _updateUI(flags) {
            if (!this.domElement) {
                return;
            }
            flags = flags || {};

            var col = this.colour,
                hsl = col.hsla,
                cssHue = 'hsl(' + hsl[0] * HUES$1 + ', 100%, 50%)',
                cssHSL = col.hslString,
                cssHSLA = col.hslaString;

            var uiH = this._domH,
                uiSL = this._domSL,
                uiA = this._domA,
                thumbH = $$1('.picker_selector', uiH),
                thumbSL = $$1('.picker_selector', uiSL),
                thumbA = $$1('.picker_selector', uiA);

            function posX(parent, child, relX) {
                child.style.left = relX * 100 + '%';
            }
            function posY(parent, child, relY) {
                child.style.top = relY * 100 + '%';
            }

            posX(uiH, thumbH, hsl[0]);

            this._domSL.style.backgroundColor = this._domH.style.color = cssHue;

            posX(uiSL, thumbSL, hsl[1]);
            posY(uiSL, thumbSL, 1 - hsl[2]);

            uiSL.style.color = cssHSL;

            posY(uiA, thumbA, 1 - hsl[3]);

            var opaque = cssHSL,
                transp = opaque.replace('hsl', 'hsla').replace(')', ', 0)'),
                bg = 'linear-gradient(' + [opaque, transp] + ')';

            this._domA.style.background = bg + ', ' + BG_TRANSP$1;

            if (!flags.fromEditor) {
                var format = this.settings.editorFormat,
                    alpha = this.settings.alpha;

                var value = void 0;
                switch (format) {
                    case 'rgb':
                        value = col.printRGB(alpha);break;
                    case 'hsl':
                        value = col.printHSL(alpha);break;
                    default:
                        value = col.printHex(alpha);
                }
                this._domEdit.value = value;
            }

            this._domSample.style.color = cssHSLA;
        }
    }, {
        key: '_ifPopup',
        value: function _ifPopup(actionIf, actionElse) {
            if (this.settings.parent && this.settings.popup) {
                actionIf && actionIf(this.settings.popup);
            } else {
                actionElse && actionElse();
            }
        }
    }, {
        key: '_toggleDOM',
        value: function _toggleDOM(toVisible) {
            var dom = this.domElement;
            if (!dom) {
                return false;
            }

            var displayStyle = toVisible ? '' : 'none',
                toggle = dom.style.display !== displayStyle;

            if (toggle) {
                dom.style.display = displayStyle;
            }
            return toggle;
        }
    }]);
    return Picker;
}();

const style$1 = {
    display: 'flex',
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'transparent',
};

class Checkbox {

    constructor({selected, label, onchange}) {

        this.state = selected;
        this.onchange = onchange;
        this.elem = div$2({style: style$1});

        const svgDiv = div$2({
            style: {
                width: '14px',
                height: '14px',
                borderColor: 'gray',
                borderWidth: '1px',
                borderStyle: 'solid'
            }
        });
        this.svg = createIcon$1('check', (true === selected ? '#444' : 'transparent'));
        this.svg.style.width = '12px';
        this.svg.style.height = '12px';
        svgDiv.appendChild(this.svg);
        this.elem.appendChild(svgDiv);

        if (label) {
            const d = div$2({style: {marginLeft: '5px'}}); //{ class: 'igv-some-label-class' });
            d.textContent = label;
            this.elem.appendChild(d);
        }

        const handleClick = (e) => {
            e.preventDefault();
            e.stopPropagation();
            const newState = !this.state;
            this.selected = newState;
            if (typeof this.onchange === 'function') {
                this.onchange(newState);
            }
        };
        this.elem.addEventListener('click', handleClick);
        this.elem.addEventListener('touchend', handleClick);
    }

    set selected(selected) {
        this.state = selected;
        const p = this.svg.querySelector('path');
        p.setAttributeNS(null, 'fill', (true === selected ? '#444' : 'transparent'));
    }

    get selected() {
        return this.state;
    }

    onchange(handler) {
        this.onchange = handler;
    }


}

function paintLegend(legend, newColorScale) {

    const ctx = legend.getContext("2d");
    const w = legend.width;
    const step = (newColorScale.max - newColorScale.min) / w;
    for (let i = 0; i < w; i++) {
        const v = newColorScale.min + i * step;
        const color = newColorScale.getColor(v);
        ctx.fillStyle = color;
        ctx.fillRect(i, 0, 1, legend.height);
    }
}

/**
 *   Editor for color scales.  Supported types:
 *
 *   'gradient': {min, max, minColor, maxColor}
 *
 *   'diverging': {mid, midColor, lowGradientScale, highGradientScale}
 *
 *
 */
class ColorScaleEditor {

    static open(colorScale, parent, callback) {

        let newColorScale = colorScale.clone();

        const table = document.createElement('table');

        const legend = document.createElement('canvas');
        legend.style.height = "20px";
        legend.style.width = "100%";
        legend.style.marginTop = "10px";
        legend.style.border = "1px solid black";

        const minTextbox = new TextBoxRow({
            label: "Min value",
            value: newColorScale.min.toString(),
            onchange: (v) => {
                newColorScale.min = Number.parseFloat(v);
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(minTextbox.row);

        const midTextbox = new TextBoxRow({
            label: "Mid value",
            value: (newColorScale.mid || newColorScale.min).toString(),
            onchange: (v) => {
                newColorScale.mid = Number.parseFloat(v);
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(midTextbox.row);

        const maxTextbox = new TextBoxRow({
            label: "Max value",
            value: newColorScale.max.toString(),
            onchange: (v) => {
                newColorScale.max = Number.parseFloat(v);
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(maxTextbox.row);


        const colorElem = new ColorPickerRow({
            label: "Min color",
            value: newColorScale.minColor,
            onchange: (v) => {
                newColorScale.minColor = v;
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(colorElem.row);

        const midColorElem = new ColorPickerRow({
            label: "Mid color",
            value: newColorScale.midColor || newColorScale.minColor,
            onchange: (v) => {
                newColorScale.midColor = v;
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(midColorElem.row);

        const highColorElem = new ColorPickerRow({
            label: "Max color",
            value: newColorScale.maxColor,
            onchange: (v) => {
                newColorScale.maxColor = v;
                paintLegend(legend, newColorScale);
            }
        });
        table.appendChild(highColorElem.row);

        const divergingCheckbox = new Checkbox({
            selected: "diverging" === colorScale.type,
            label: "Diverging Scale",
            onchange: (diverging) => {
                if (diverging) {
                    // Converting from gradient to diverting
                    newColorScale.mid = newColorScale.min < 0 && newColorScale.max > 0 ? 0 : (newColorScale.min + newColorScale.max) / 2;
                    newColorScale.midColor = "rgb(255,255,255)";
                    newColorScale = new DivergingGradientScale(newColorScale);

                    midTextbox.value = newColorScale.mid;
                    midTextbox.show();

                    midColorElem.value = newColorScale.midColor;
                    midColorElem.show();

                    paintLegend(legend, newColorScale);
                } else {

                    // Converting from diverging to gradient
                    newColorScale = new GradientColorScale(newColorScale);
                    midTextbox.hide();
                    midColorElem.hide();
                    paintLegend(legend, newColorScale);
                }
            }
        });
        divergingCheckbox.elem.style.marginBottom = "20px";

        if('diverging' !== colorScale.type) {
            midTextbox.hide();
            midColorElem.hide();
        }

        const panel = document.createElement('div');
        panel.appendChild(divergingCheckbox.elem);
        panel.appendChild(table);
        panel.appendChild(legend);

        const okHandler = () => {
            if (callback) {
                callback(newColorScale);
            }

        };

        const config = {
            parent, // label: 'Multi-select',
            content: {elem: panel}, okHandler
        };
        const dialog = new Dialog(config);
        parent.appendChild(dialog.elem);
        show$1(dialog.elem);

        paintLegend(legend, newColorScale);

    }

}

class LabeledButtonRow {
    constructor({label, value, onchange}) {

        this.row = document.createElement('tr');
        const cell = document.createElement('td');
        this.row.appendChild(cell);

        const div = document.createElement('div');
        div.innerHTML = label;
        cell.appendChild(div);
    }

    hide() {
        this.row.style.display = 'none';
    }

    show() {
        this.row.style.display = 'table-row';
    }
}

class TextBoxRow extends LabeledButtonRow {

    constructor({label, value, onchange}) {
        super({label, value, onchange});

        const cell2 = document.createElement('td');
        this.row.appendChild(cell2);
        this.input = document.createElement('input');

        value = value || "0";
        this.input.value = purify.sanitize(value);

        cell2.appendChild(this.input);

        if (onchange) {
            this.input.addEventListener('change', (e) => onchange(this.input.value));
        }
    }

    get value() {
        return this.input.value
    }

    set value(v) {
        this.input.value = v;
    }
}

class ColorPickerRow extends LabeledButtonRow {

    constructor({label, value, onchange}) {
        super({label, value, onchange});

        const cell2 = document.createElement('td');
        this.row.appendChild(cell2);
        const colorButton = document.createElement('div');
        cell2.appendChild(colorButton);
        colorButton.style.width = "20px";
        colorButton.style.height = "20px";
        colorButton.style.border = "1px solid black";
        this.colorButton = colorButton;

        value = value || "white";
        colorButton.style.background = value;

        const picker = new Picker$1(colorButton);
        picker.setOptions({
            alpha: false, color: value
        });

        picker.onDone =  (color) => {
            colorButton.style.background = color.rgbString;
            if (onchange) {
                onchange(color.rgbString);
            }
        };
    }

    set value(c) {
        this.colorButton.style.background = c;
    }
}

class WigTrack extends TrackBase {

    static defaultColor = 'rgb(150, 150, 150)'

    static defaults = {
        height: 50,
        flipAxis: false,
        logScale: false,
        windowFunction: 'mean',
        graphType: 'bar',
        normalize: undefined,
        scaleFactor: undefined,
        overflowColor: `rgb(255, 32, 255)`,
        baselineColor: 'lightGray',
        summarize: true,
        visibilityWindow: undefined
    }

    constructor(config, browser) {
        super(config, browser);
    }

    init(config) {

        super.init(config);

        this.type = "wig";
        this.featureType = 'numeric';
        this.resolutionAware = true;
        this._paintAxis = paintAxis.bind(this);

        const format = config.format ? config.format.toLowerCase() : config.format;
        if (config.featureSource) {
            this.featureSource = config.featureSource;
            delete config.featureSource;
        } else if ("bigwig" === format) {
            this.featureSource = new BWSource(config, this.browser.genome);
        } else if ("tdf" === format) {
            this.featureSource = new TDFSource(config, this.browser.genome);
        } else {
            this.featureSource = FeatureSource(config, this.browser.genome);
        }

        // Override autoscale default
        if (config.max === undefined || config.autoscale === true) {
            this.autoscale = true;
        } else {
            this.dataRange = {
                min: config.min || 0,
                max: config.max
            };
        }

        if (config.colorScale) {
            this._colorScale = ColorScaleFactory.fromJson(config.colorScale);
        }

        // Override default height for heatmaps
        if ("heatmap" === config.graphType && !config.height) {
            this.height = 20;
        }
    }

    async postInit() {
        const header = await this.getHeader();
        if (this.disposed) return   // This track was removed during async load
        if (header) this.setTrackProperties(header);

        this._initialColor = this.color || this.constructor.defaultColor;
        this._initialAltColor = this.altColor || this.constructor.defaultColor;

    }

    get supportsWholeGenome() {
        return !this.config.indexURL && this.config.supportsWholeGenome !== false
    }

    get paintAxis() {
        // Supply do-nothing implementation for heatmaps
        return "heatmap" === this.graphType ? () => {
        } : this._paintAxis
    }

    get colorScale() {
        return this._colorScale
    }

    async getFeatures(chr, start, end, bpPerPixel) {

        const windowFunction = this.windowFunction;

        const features = await this.featureSource.getFeatures({
            chr,
            start,
            end,
            bpPerPixel,
            visibilityWindow: this.visibilityWindow,
            windowFunction
        });
        if (this.normalize && this.featureSource.normalizationFactor) {
            const scaleFactor = this.featureSource.normalizationFactor;
            for (let f of features) {
                f.value *= scaleFactor;
            }
        }
        if (this.scaleFactor) {
            const scaleFactor = this.scaleFactor;
            for (let f of features) {
                f.value *= scaleFactor;
            }
        }

        // Summarize features to current resolution.  This needs to be done here, rather than in the "draw" function,
        // for group autoscale to work.
        if (this.summarize && ("mean" === windowFunction || "min" === windowFunction || "max" === windowFunction)) {
            return summarizeData(features, start, bpPerPixel, windowFunction)
        } else {
            return features
        }
    }

    menuItemList() {
        const items = [];

        if ('heatmap' === this.graphType) {
            items.push('<hr>');
            items.push({
                label: 'Set color scale', click: function () {
                    ColorScaleEditor.open(this.colorScale, this.browser.columnContainer, (colorScale) => {
                        this._colorScale = colorScale;
                        this.trackView.repaintViews();
                    });
                }
            });
        } else if (this.flipAxis !== undefined) {
            items.push('<hr>');
            items.push({
                label: 'Flip y-axis',
                click: function () {
                    this.flipAxis = !this.flipAxis;
                    this.trackView.repaintViews();
                }
            });
        }

        items.push(...this.graphTypeItems());

        if (this.featureSource.windowFunctions) {
            items.push(...this.wigSummarizationItems());
        }

        items.push(...this.numericDataMenuItems());

        return items
    }

    wigSummarizationItems() {

        const windowFunctions = this.featureSource.windowFunctions;

        const menuItems = [];
        menuItems.push('<hr/>');
        menuItems.push("<div>Windowing function</div>");
        for (const wf of windowFunctions) {

            function clickHandler() {
                this.windowFunction = wf;
                this.trackView.updateViews();
            }

            menuItems.push({element:createCheckbox(wf, this.windowFunction === wf), click: clickHandler});
        }

        return menuItems
    }

    graphTypeItems() {

        const graphType = ['bar', 'line', 'points', 'heatmap'];

        const menuItems = [];
        menuItems.push('<hr/>');
        menuItems.push("<div>Graph type</div>");

        for (const gt of graphType) {

            function clickHandler() {
                this.graphType = gt;
                this.trackView.repaintViews();
            }

            menuItems.push({element:createCheckbox(gt, this.graphType === gt), click: clickHandler});
        }

        return menuItems
    }

    async getHeader() {

        if (typeof this.featureSource.getHeader === "function") {
            this.header = await this.featureSource.getHeader();
        }
        return this.header
    }

    // TODO: refactor to igvUtils.js
    getScaleFactor(min, max, height, logScale) {
        const minValue = (logScale === true) ? ((min < 0) ? -Math.log10(Math.abs(min) + 1) : Math.log10(Math.abs(min) + 1)) : min;
        const maxValue = (logScale === true) ? Math.log10(Math.abs(max) + 1) : max;
        const scale = height / (maxValue - minValue);
        return scale
    }

    computeYPixelValue(yValue, yScaleFactor) {
        return (this.flipAxis ? (yValue - this.dataRange.min) : (this.dataRange.max - yValue)) * yScaleFactor
    }

    computeYPixelValueInLogScale(yValue, yScaleFactor) {
        let maxValue = this.dataRange.max;
        let minValue = this.dataRange.min;
        minValue = (minValue < 0) ? -Math.log10(Math.abs(minValue) + 1) : Math.log10(Math.abs(minValue) + 1);
        maxValue = (maxValue < 0) ? -Math.log10(Math.abs(maxValue) + 1) : Math.log10(Math.abs(maxValue) + 1);

        yValue = (yValue < 0) ? -Math.log10(Math.abs(yValue) + 1) : Math.log10(yValue + 1);
        return ((this.flipAxis ? (yValue - minValue) : (maxValue - yValue)) * yScaleFactor)
    }

    draw(options) {

        const features = options.features;
        const ctx = options.context;
        const bpPerPixel = options.bpPerPixel;
        const bpStart = options.bpStart;
        const pixelWidth = options.pixelWidth;
        const pixelHeight = options.pixelHeight - 1;
        const bpEnd = bpStart + pixelWidth * bpPerPixel + 1;

        const scaleFactor = this.getScaleFactor(this.dataRange.min, this.dataRange.max, pixelHeight, this.logScale);
        const yScale = (yValue) => this.logScale
            ? this.computeYPixelValueInLogScale(yValue, scaleFactor)
            : this.computeYPixelValue(yValue, scaleFactor);

        if (features && features.length > 0) {

            if (this.dataRange.min === undefined) this.dataRange.min = 0;

            // Max can be less than min if config.min is set but max left to autoscale.   If that's the case there is
            // nothing to paint.
            if (this.dataRange.max > this.dataRange.min) {

                let lastPixelEnd = -1;
                let lastY;
                const y0 = yScale(0);

                for (let f of features) {

                    if (f.end < bpStart) continue
                    if (f.start > bpEnd) break

                    const x = (f.start - bpStart) / bpPerPixel;
                    if (Number.isNaN(x)) continue

                    let y = yScale(f.value);

                    const rectEnd = (f.end - bpStart) / bpPerPixel;
                    const width = rectEnd - x;

                    const color = options.alpha ? IGVColor.addAlpha(this.getColorForFeature(f), options.alpha) : this.getColorForFeature(f);

                    if (this.graphType === "line") {
                        if (lastY !== undefined) {
                            IGVGraphics.strokeLine(ctx, lastPixelEnd, lastY, x, y, {
                                "fillStyle": color,
                                "strokeStyle": color
                            });
                        }
                        IGVGraphics.strokeLine(ctx, x, y, x + width, y, {"fillStyle": color, "strokeStyle": color});
                    } else if (this.graphType === "points") {
                        const pointSize = this.config.pointSize || 3;
                        const px = x + width / 2;
                        IGVGraphics.fillCircle(ctx, px, y, pointSize / 2, {"fillStyle": color, "strokeStyle": color});

                        if (f.value > this.dataRange.max) {
                            IGVGraphics.fillCircle(ctx, px, pointSize / 2, pointSize / 2, 3, {fillStyle: this.overflowColor});
                        } else if (f.value < this.dataRange.min) {
                            IGVGraphics.fillCircle(ctx, px, pixelHeight - pointSize / 2, pointSize / 2, 3, {fillStyle: this.overflowColor});
                        }


                    } else if (this.graphType === "heatmap") {
                        if (!this._colorScale) {
                            // Create a default color scale.
                            this._colorScale = this.dataRange.min < 0 && this.dataRange.max > 0 ?
                                ColorScaleFactory.defaultDivergingScale(this.dataRange.min, 0, this.dataRange.max) :
                                ColorScaleFactory.defaultGradientScale(this.dataRange.min, this.dataRange.max);
                        }
                        const color = this._colorScale.getColor(f.value);
                        IGVGraphics.fillRect(ctx, x, 0, width, pixelHeight, {fillStyle: color});
                    } else {
                        // Default graph type (bar)
                        const height = Math.min(pixelHeight, y - y0);
                        IGVGraphics.fillRect(ctx, x, y0, width, height, {fillStyle: color});
                        if (f.value > this.dataRange.max) {
                            IGVGraphics.fillRect(ctx, x, 0, width, 3, {fillStyle: this.overflowColor});
                        } else if (f.value < this.dataRange.min) {
                            IGVGraphics.fillRect(ctx, x, pixelHeight - 2, width, 3, {fillStyle: this.overflowColor});
                        }

                    }
                    lastPixelEnd = x + width;
                    lastY = y;
                }

                // If the track includes negative values draw a baseline
                if (this.dataRange.min < 0) {
                    let maxValue = this.dataRange.max;
                    let minValue = this.dataRange.min;
                    minValue = (this.logScale === true) ? ((minValue < 0) ? -Math.log10(Math.abs(minValue) + 1) : Math.log10(Math.abs(minValue) + 1)) : minValue;
                    maxValue = (this.logScale === true) ? ((maxValue < 0) ? -Math.log10(Math.abs(maxValue) + 1) : Math.log10(Math.abs(maxValue) + 1)) : maxValue;
                    const ratio = maxValue / (maxValue - minValue);
                    const basepx = this.flipAxis ? (1 - ratio) * pixelHeight : ratio * pixelHeight;
                    IGVGraphics.strokeLine(ctx, 0, basepx, options.pixelWidth, basepx, {strokeStyle: this.baselineColor});
                }
            }
        }

        // Draw guidelines
        if (this.config.hasOwnProperty('guideLines')) {
            for (let line of this.config.guideLines) {
                if (line.hasOwnProperty('color') && line.hasOwnProperty('y') && line.hasOwnProperty('dotted')) {
                    let y = yScale(line.y);
                    let props = {
                        'strokeStyle': line['color'],
                        'strokeWidth': 2
                    };
                    if (line['dotted']) IGVGraphics.dashedLine(options.context, 0, y, options.pixelWidth, y, 5, props);
                    else IGVGraphics.strokeLine(options.context, 0, y, options.pixelWidth, y, props);
                }
            }
        }
    }

    popupData(clickState, features) {

        if (features === undefined) features = this.clickedFeatures(clickState);

        if (features && features.length > 0) {

            const genomicLocation = clickState.genomicLocation;
            const popupData = [];

            // Sort features based on distance from click
            features.sort(function (a, b) {
                const distA = Math.abs((a.start + a.end) / 2 - genomicLocation);
                const distB = Math.abs((b.start + b.end) / 2 - genomicLocation);
                return distA - distB
            });

            // Display closest 10
            const displayFeatures = features.length > 10 ? features.slice(0, 10) : features;

            // Resort in ascending order
            displayFeatures.sort(function (a, b) {
                return a.start - b.start
            });

            for (let selectedFeature of displayFeatures) {
                if (selectedFeature) {
                    if (popupData.length > 0) {
                        popupData.push('<hr/>');
                    }
                    let posString = (selectedFeature.end - selectedFeature.start) === 1 ?
                        numberFormatter$1(Math.floor(selectedFeature.start) + 1)
                        : numberFormatter$1(Math.floor(selectedFeature.start) + 1) + "-" + numberFormatter$1(Math.floor(selectedFeature.end));
                    popupData.push({name: "Position:", value: posString});
                    popupData.push({
                        name: "Value:&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;",
                        value: numberFormatter$1(selectedFeature.value.toFixed(4))
                    });
                }
            }
            if (displayFeatures.length < features.length) {
                popupData.push("<hr/>...");
            }

            return popupData

        } else {
            return []
        }
    }

    /**
     * Return color for feature.
     * @param feature
     * @returns {string}
     */

    getColorForFeature(f) {
        let c = (f.value < 0 && this.altColor) ? this.altColor : this.color || WigTrack.defaultColor;
        return (typeof c === "function") ? c(f.value) : c
    }


    getState() {
        const state = super.getState();
        if (this._colorScale) {
            state.colorScale = this._colorScale.toJson();
        }
        return state
    }

    /**
     * Called when the track is removed.  Do any needed cleanup here
     */
    dispose() {
        this.trackView = undefined;
    }

}

/**
 * Summarize wig data in bins of size "bpPerPixel" with the given window function.
 *
 * @param features  wig (numeric) data -- features cannot overlap, and are in ascending order by start position
 * @param startBP  bp start position for computing binned data
 * @param bpPerPixel  bp per pixel (bin)
 * @param windowFunction mean, min, or max
 * @returns {*|*[]}
 */
function summarizeData(features, startBP, bpPerPixel, windowFunction = "mean") {

    if (bpPerPixel <= 1 || !features || features.length === 0 || windowFunction === "none") {
        return features
    }

    // Assume features are sorted by position.  Wig features cannot overlap.  Note, UCSC "reductionLevel" == bpPerPixel
    const chr = features[0].chr;
    const binSize = bpPerPixel;
    const summaryFeatures = [];

    const finishBin = (bin) => {
        const start = startBP + bin.bin * binSize;
        const end = start + binSize;
        let value;
        switch (windowFunction) {
            case "mean":
                value = bin.sumData / bin.count;
                break
            case "max":
                value = bin.max;
                break
            case "min":
                value = bin.min;
                break
            default:
                throw Error(`Unknown window function: ${windowFunction}`)
        }
        const description = `${windowFunction} of ${bin.count} values`;
        summaryFeatures.push({chr, start, end, value, description});
    };

    let currentBinData;
    for (let f of features) {

        // Loop through bins this feature overlaps, updating the weighted sum for each bin or min/max,
        // depending on window function
        let startBin = Math.floor((f.start - startBP) / binSize);
        const endBin = Math.floor((f.end - startBP) / binSize);

        if (currentBinData && startBin === currentBinData.bin) {
            currentBinData.add(f);
            startBin++;
        }

        if (!currentBinData || endBin > currentBinData.bin) {

            if (currentBinData) {
                finishBin(currentBinData);
            }

            // Feature stretches across multiple bins.
            if (endBin > startBin) {
                const end = startBP + endBin * binSize;
                summaryFeatures.push({chr, start: f.start, end, value: f.value});
            }

            currentBinData = new SummaryBinData(endBin, f);
        }

    }
    if (currentBinData) {
        finishBin(currentBinData);
    }

    // Consolidate
    const c = [];
    let lastFeature = summaryFeatures[0];
    for (let f of summaryFeatures) {
        if (lastFeature.value === f.value && f.start <= lastFeature.end) {
            lastFeature.end = f.end;
        } else {
            c.push(lastFeature);
            lastFeature = f;
        }
    }
    c.push(lastFeature);

    return c

}

class SummaryBinData {
    constructor(bin, feature) {
        this.bin = bin;
        this.sumData = feature.value;
        this.count = 1;
        this.min = feature.value;
        this.max = feature.value;
    }

    add(feature) {
        this.sumData += feature.value;
        this.max = Math.max(feature.value, this.max);
        this.min = Math.min(feature.value, this.min);
        this.count++;
    }

    get mean() {
        return this.sumData / this.count
    }
}

const DEFAULT_MAX_WG_COUNT = 10000;

/**
 * feature source for "bed like" files (tab or whitespace delimited files with 1 feature per line: bed, gff, vcf, etc)
 *
 * @param config
 * @constructor
 */
class TextFeatureSource extends BaseFeatureSource {

    constructor(config, genome) {

        super(genome);

        this.config = config || {};
        this.genome = genome;
        this.sourceType = (config.sourceType === undefined ? "file" : config.sourceType);
        this.maxWGCount = config.maxWGCount || DEFAULT_MAX_WG_COUNT;
        this.windowFunctions = ["mean", "min", "max", "none"];

        const queryableFormats = new Set(["bigwig", "bw", "bigbed", "bb", "biginteract", "biggenepred", "bignarrowpeak", "tdf"]);

        this.queryable = config.indexURL || config.queryable === true;   // False by default, unless explicitly set
        if (config.reader) {
            // Explicit reader implementation
            this.reader = config.reader;
            this.queryable = config.queryable !== false;
        } else if (config.sourceType === "ga4gh") {
            throw Error("Unsupported source type 'ga4gh'")
        } else if ((config.type === "eqtl" || config.type === "qtl") && config.sourceType === "gtex-ws") {
            this.reader = new GtexReader(config);
            this.queryable = true;
        } else if ("htsget" === config.sourceType) {
            this.reader = new HtsgetVariantReader(config, genome);
            this.queryable = true;
        } else if (config.sourceType === 'ucscservice') {
            this.reader = new UCSCServiceReader(config.source);
            this.queryable = true;
        } else if (config.sourceType === 'custom') {
            this.reader = new CustomServiceReader(config.source);
            this.queryable = false !== config.source.queryable;
        } else if ('service' === config.sourceType) {
            this.reader = new FeatureFileReader(config, genome);
            this.queryable = true;
        } else {
            // File of some type (i.e. not a webservice)
            this.reader = new FeatureFileReader(config, genome);
            if (config.queryable !== undefined) {
                this.queryable = config.queryable;
            } else if (queryableFormats.has(config.format) || this.reader.indexed) {
                this.queryable = true;
            } else ;
        }

        // Flag indicating if features loaded by this source can be searched for by name or attribute, true by default
        this.searchable = config.searchable !== false;

    }

    async defaultVisibilityWindow() {
        if (this.reader && typeof this.reader.defaultVisibilityWindow === 'function') {
            return this.reader.defaultVisibilityWindow()
        }
    }

    async trackType() {
        const header = await this.getHeader();
        if (header) {
            return header.type
        } else {
            return undefined    // Convention for unknown or unspecified
        }
    }

    async getHeader() {
        if (!this.header) {

            if (this.reader && typeof this.reader.readHeader === "function") {
                const header = await this.reader.readHeader();
                if (header) {
                    this.header = header;
                    if (header.format) {
                        this.config.format = header.format;
                    }
                } else {
                    this.header = {};
                }
            } else {
                this.header = {};
            }
        }
        return this.header
    }

    /**
     * Required function for all data source objects.  Fetches features for the
     * range requested.
     *
     * This function is quite complex due to the variety of reader types backing it, some indexed, some queryable,
     * some not.
     *
     * @param chr
     * @param start
     * @param end
     * @param bpPerPixel
     */
    async getFeatures({chr, start, end, bpPerPixel, visibilityWindow, windowFunction}) {

        const isWholeGenome = ("all" === chr.toLowerCase());

        start = start || 0;
        end = end || Number.MAX_SAFE_INTEGER;

        // Various conditions that can require a feature load
        //   * view is "whole genome" but no features are loaded
        //   * cache is disabled
        //   * cache does not contain requested range
        // const containsRange = this.featureCache.containsRange(new GenomicInterval(queryChr, start, end))
        if ((isWholeGenome && !this.wgFeatures && this.supportsWholeGenome()) ||
            this.config.disableCache ||
            !this.featureCache ||
            !this.featureCache.containsRange(new GenomicInterval(chr, start, end))) {
            await this.loadFeatures(chr, start, end, visibilityWindow);
        }

        if (isWholeGenome) {
            if (!this.wgFeatures) {
                if (this.supportsWholeGenome()) {
                    if("wig" === this.config.type) {
                        const allWgFeatures = await computeWGFeatures(this.featureCache.getAllFeatures(), this.genome, 1000000);
                        this.wgFeatures = summarizeData(allWgFeatures, 0, bpPerPixel, windowFunction);
                    } else {
                        this.wgFeatures = await computeWGFeatures(this.featureCache.getAllFeatures(), this.genome, this.maxWGCount);
                    }
                } else {
                    this.wgFeatures = [];
                }
            }
            return this.wgFeatures
        } else {
            return this.featureCache.queryFeatures(chr, start, end)
        }
    }

    async findFeatures(fn) {
        return this.featureCache ? this.featureCache.findFeatures(fn) : []
    }

    supportsWholeGenome() {
        return !this.queryable   // queryable (indexed, web services) sources don't support whole genome view
    }

    // TODO -- experimental, will only work for non-indexed sources
    getAllFeatures() {
        if (this.queryable || !this.featureCache) {   // queryable sources don't support all features
            return []
        } else {
            return this.featureCache.getAllFeatures()
        }
    }


    async loadFeatures(chr, start, end, visibilityWindow) {

        await this.getHeader();

        const reader = this.reader;
        let intervalStart = start;
        let intervalEnd = end;

        // chr aliasing
        let queryChr = chr;
        if (!this.chrAliasManager && this.reader && this.reader.sequenceNames) {
            this.chrAliasManager = new ChromAliasManager(this.reader.sequenceNames, this.genome);
        }
        if (this.chrAliasManager) {
            queryChr = await this.chrAliasManager.getAliasName(chr);
        }

        // Use visibility window to potentially expand query interval.
        // This can save re-queries as we zoom out.  Visibility window <= 0 is a special case
        // indicating whole chromosome should be read at once.
        if ((!visibilityWindow || visibilityWindow <= 0) && this.config.expandQuery !== false) {
            // Whole chromosome
            const chromosome = this.genome ? this.genome.getChromosome(chr) : undefined;
            intervalStart = 0;
            intervalEnd = Math.max(chromosome ? chromosome.bpLength : Number.MAX_SAFE_INTEGER, end);
        } else if (visibilityWindow > (end - start) && this.config.expandQuery !== false) {
            let expansionWindow = Math.min(4.1 * (end - start), visibilityWindow);
            if(this.config.minQuerySize && expansionWindow < this.config.minQuerySize) {
                expansionWindow = this.config.minQuerySize;
            }
            intervalStart = Math.max(0, (start + end - expansionWindow) / 2);
            intervalEnd = intervalStart + expansionWindow;
        }

        let features = await reader.readFeatures(queryChr, intervalStart, intervalEnd);
        if (this.queryable === undefined) {
            this.queryable = reader.indexed;
        }

        const genomicInterval = this.queryable ?
            new GenomicInterval(chr, intervalStart, intervalEnd) :
            undefined;

        if (features) {

            // Assign overlapping features to rows
            if (this.config.format !== "wig" && this.config.type !== "junctions") {
                const maxRows = this.config.maxRows || Number.MAX_SAFE_INTEGER;
                packFeatures(features, maxRows);
            }

            // Note - replacing previous cache with new one.  genomicInterval is optional (might be undefined => includes all features)
            this.featureCache = new FeatureCache$1(features, this.genome, genomicInterval);

            // If track is marked "searchable"< cache features by name -- use this with caution, memory intensive
            if (this.searchable) {
                this.addFeaturesToDB(features, this.config);
            }
        } else {
            this.featureCache = new FeatureCache$1([], genomicInterval);     // Empty cache
        }
    }

    addFeaturesToDB(featureList, config) {
        if (!this.featureMap) {
            this.featureMap = new Map();
        }
        const searchableFields = config.searchableFields || ["name", "transcript_id", "gene_id", "gene_name", "id"];
        for (let feature of featureList) {
            for (let field of searchableFields) {
                let key;
                if(feature.hasOwnProperty(field)) {
                    key = feature[field];
                }
                else if (typeof feature.getAttributeValue === 'function') {
                    key = feature.getAttributeValue(field);
                }
                if (key) {
                    key = key.replaceAll(' ', '+').toUpperCase();
                    // If feature is already present keep largest one
                    if (this.featureMap.has(key)) {
                        const f2 = this.featureMap.get(key);
                        if (feature.end - feature.start < f2.end - f2.start) {
                            continue
                        }
                    }
                    this.featureMap.set(key, feature);
                }
            }
        }
    }

    search(term) {
        if (this.featureMap) {
            return this.featureMap.get(term.toUpperCase())
        }

    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2015 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * feature source for features supplied directly, as opposed to reading and parsing from a file or webservice
 *
 * @param config
 * @constructor
 */
class StaticFeatureSource extends BaseFeatureSource {

    constructor(config, genome) {

        super(genome);
        this.config = config;
        this.genome = genome;
        this.queryable = false;
        this.searchable = config.searchable !== false;  // searchable by default
        this.updateFeatures(config.features);
    }

    updateFeatures(features) {
        features = fixFeatures(features, this.genome);
        packFeatures(features);
        if (this.config.mappings) {
            mapProperties(features, this.config.mappings);
        }
        this.featureCache = new FeatureCache$1(features, this.genome);

        if (this.searchable || this.config.searchableFields) {
            this.addFeaturesToDB(features, this.config);
        }
    }

    /**
     * Required function for all data source objects.  Fetches features for the
     * range requested.
     *
     * This function is complex due to the variety of reader types backing it, some indexed, some queryable,
     * some not.
     *
     * @param chr
     * @param start
     * @param end
     * @param bpPerPixel
     */
    async getFeatures({chr, start, end, bpPerPixel, visibilityWindow}) {

        const genome = this.genome;
        const queryChr = genome ? genome.getChromosomeName(chr) : chr;
        const isWholeGenome = ("all" === queryChr.toLowerCase());

        // Various conditions that can require a feature load
        //   * view is "whole genome" but no features are loaded
        //   * cache is disabled
        //   * cache does not contain requested range
        if (isWholeGenome) {
            return computeWGFeatures(this.featureCache.getAllFeatures(), this.genome, this.maxWGCount)
        } else {
            return this.featureCache.queryFeatures(queryChr, start, end)
        }
    }

    //
    // supportsWholeGenome() {
    //    return true
    // }

    getAllFeatures() {
        return this.featureCache.getAllFeatures()
    }

    supportsWholeGenome() {
        return true
    }

    addFeaturesToDB(featureList, config) {
        if (!this.featureMap) {
            this.featureMap = new Map();
        }
        const searchableFields = config.searchableFields || ["name"];
        for (let feature of featureList) {
            for (let field of searchableFields) {
                let key;

                if (typeof feature.getAttributeValue === 'function') {
                    key = feature.getAttributeValue(field);
                }
                if (!key) {
                    key = feature[field];
                }
                if (key) {
                    key = key.replaceAll(' ', '+');
                    const current = this.featureMap.get(key.toUpperCase());
                    if (current && ((current.end - current.start) > (feature.end - feature.start))) continue
                    this.featureMap.set(key.toUpperCase(), feature);
                }
            }
        }
    }

    search(term) {
        if (this.featureMap) {
            return this.featureMap.get(term.toUpperCase())
        }
    }
}


/**
 * This function is used to apply properties normally added during parsing to  features supplied directly in the
 * config as an array of objects.   At the moment the only application is bedpe type features.
 * @param features
 */
function fixFeatures(features, genome) {

    if (genome) {
        for (let feature of features) {
            feature.chr = genome.getChromosomeName(feature.chr);
        }
    }

    return features
}


function mapProperties(features, mappings) {
    let mappingKeys = Object.keys(mappings);
    features.forEach(function (f) {
        mappingKeys.forEach(function (key) {
            f[key] = f[mappings[key]];
        });
    });
}

/**
 * Represents a Genbank file, which combines both annotations (features) and sequence.   The format combines both
 * sequence and annotations.
 *
 * Implements the Genome interface
 */
class Genbank {

    constructor({chr, locus, accession, aliases, features, sequence}) {
        this.chr = chr;
        this.locus = locus;
        this.accession = accession;
        this.aliases = aliases;
        this.features = features;
        this.sequence = sequence;
        this.bpLength = sequence.length;
    }


    toJSON() {
        return {
            gbkURL: this.url
        }
    }


    // Genome interface follows

    getSequenceRecord(chr) {
        //chr, 0, sequenceRecord.bpLength
        return {chr: this.chr, bpLength: this.bpLength}
    }

    get chromosomeNames() {
        return [this.chr]
    }

    getFirstChromosomeName() {
        return this.chr
    }

    get id() {
        return this.accession
    }
    get name() {
        return this.locus
    }

    get initialLocus() {
        return this.chr
    }

    // Genome interface follows
    get description() {
        return this.locus
    }

    get infoURL() {
        return this.url
    }

    showWholeGenomeView() {
        return false
    }

    getHomeChromosomeName() {
        return this.chr
    }

    getChromosomeName(chr) {
        return chr
    }

    getChromosomeDisplayName(str) {
        return this.chr
    }

    getChromosome(chr) {
        if (chr === this.chr) {
            return {
                name: this.chr,
                bpLength: this.bpLength
            }
        }
    }

    async loadChromosome(chr) {
        return this.getChromosome(chr)
    }

    async getAliasRecord(chr) {
        return undefined
    }

    getCytobands(chr) {
        return []
    }

    getChromosomes() {
        return [this.getChromosome(this.chr)]
    }

    get wgChromosomeNames() {
        return undefined
    }

    /**
     * Return the genome coordinate in kb for the give chromosome and position.
     * NOTE: This might return undefined if the chr is filtered from whole genome view.
     */
    getGenomeCoordinate(chr, bp) {
        if (chr === this.chr)
            return bp
    }

    /**
     * Return the chromosome and coordinate in bp for the given genome coordinate
     */
    getChromosomeCoordinate(genomeCoordinate) {
        return {chr: this.chr, position: genomeCoordinate}
    }


    /**
     * Return the offset in genome coordinates (kb) of the start of the given chromosome
     * NOTE:  This might return undefined if the chromosome is filtered from whole genome view.
     */
    getCumulativeOffset(chr) {
        return 0
    }

    /**
     * Return the nominal genome length, this is the length of the main chromosomes (no scaffolds, etc).
     */
    getGenomeLength() {
        return this.bpLength
    }


    async getSequence(chr, start, end) {
        if (chr === this.chr) {
            return this.sequence.substring(start, end)
        } else {
            return undefined
        }
    }

    /**
     * Return the first cached interval containing the specified region, or undefined if no interval is found.
     *
     * @param chr
     * @param start
     * @param end
     * @returns a SequenceInterval or undefined
     */
    getSequenceInterval(chr, start, end) {
        if (chr === this.chr) {
            return new SequenceInterval(this.chr, 0, this.sequence.length, this.sequence)
        } else {
            return undefined
        }
    }
}

const wsRegex = /\s+/;

const genbankCache = new Map();

async function loadGenbank(url) {
    let genbank = genbankCache.get(url);

    if (!genbank) {
        const data = await igvxhr$1.loadString(url, {});
        genbank = parseGenbank(data);
        genbank.url = url;
        genbankCache.set(url, genbank);
    }

    return genbank;
}


function parseGenbank(data) {


    if (!data) return null

    const dataWrapper = getDataWrapper$1(data);

    // Read locus
    let line = dataWrapper.nextLine();
    const tokens = line.split(/\s+/);
    if (tokens[0].toUpperCase() !== "LOCUS") {
        throw Error("Expected `LOCUS` line.  Found: " + line)
    }
    const locus = tokens[1].trim();

    // Loop until FEATURES section
    let accession, aliases;
    do {
        line = dataWrapper.nextLine();
        if (line.startsWith("ACCESSION")) {
            const tokens = line.split(wsRegex);
            if (tokens.length < 2) {
                throw Error("Genbank file missing ACCESSION number.")
            } else {
                accession = tokens[1].trim();
            }
        } else if (line.startsWith("ALIASES")) {
            // NOTE - this is an IGV extension
            const tokens = line.split(wsRegex);
            if (tokens.length > 1) {
                aliases = tokens[1].split(",");
            }

        }
    }
    while (line && !line.startsWith("FEATURES"))

    const chr = accession || locus;
    const features = parseFeatures(chr, dataWrapper);
    const sequence = parseSequence(dataWrapper);

    return new Genbank({chr, locus, accession, aliases, features, sequence})
}


/**
 * Read the origin section.   Example...
 * <p/>
 * ORIGIN
 * 1 gatcctccat atacaacggt atctccacct caggtttaga tctcaacaac ggaaccattg
 * 61 ccgacatgag acagttaggt atcgtcgaga gttacaagct aaaacgagca gtagtcagct
 * 121 ctgcatctga agccgctgaa gttctactaa gggtggataa catcatccgt gcaagaccaa
 *
 * @param reader
 */
function parseSequence(dataWrapper) {

    let nextLine;
    let sequence = "";

    while ((nextLine = dataWrapper.nextLine()) && !nextLine.startsWith("//")) {
        nextLine = nextLine.trim();
        const tokens = nextLine.split(/\s+/);
        for (let i = 1; i < tokens.length; i++) {
            sequence += tokens[i];
        }
    }
    return sequence
}

/**
 * FEATURES             Location/Qualifiers
 * source          1..105338
 * /organism="Homo sapiens"
 * /mol_type="genomic DNA"
 * /db_xref="taxon:9606"
 * /chromosome="10"
 * gene            1..105338
 * /gene="PTEN"
 * /note="Derived by automated computational analysis using
 * gene prediction method: BestRefseq."
 * /db_xref="GeneID:5728"
 * /db_xref="HGNC:9588"
 * /db_xref="HPRD:03431"
 * /db_xref="MIM:601728"
 * <p/>
 * CDS             join(1033..1111,30588..30672,62076..62120,67609..67652,
 * 69576..69814,88681..88822,94416..94582,97457..97681,
 * 101850..102035)
 * /gene="PTEN"
 *
 * @param reader
 * @throws IOException
 */
function parseFeatures(chr, dataWrapper) {

    //Process features until "ORIGIN" or end of file
    const features = [];
    let currentLocQualifier;
    let nextLine;
    let errorCount = 0;
    let f;

    do {
        nextLine = dataWrapper.nextLine();

        if (nextLine === "") {
            continue  // Not sure this is legal in a gbk file
        }

        if (!nextLine || nextLine.startsWith("ORIGIN")) {
            break
        }

        if (nextLine.length < 6) {
            if (errorCount < 10) {
                console("Unexpected line in genbank file (skipping): " + nextLine);
            }
            errorCount++;
            continue
        }

        if (nextLine.charAt(5) !== ' ') {

            let featureType = nextLine.substring(5, 21).trim();
            f = {
                chr: chr,
                type: featureType,
                attributes: {},
                getAttributeValue: function(key) {return this.attributes[key]}
            };
            currentLocQualifier = nextLine.substring(21);

            if (featureType.toLowerCase() !== "source") {
                features.push(f);
            }

        } else {
            let tmp = nextLine.substring(21).trim();
            if (tmp.length > 0)

                if (tmp.charCodeAt(0) === 47) {   // 47 == '/'
                    if (currentLocQualifier.charCodeAt(0) === 47) {
                        let tokens = currentLocQualifier.split("=", 2);
                        if (tokens.length > 1) {
                            let keyName = tokens[0].length > 1 ? tokens[0].substring(1) : "";
                            let value = stripQuotes(tokens[1]);
                            f.attributes[keyName] = value;

                        }
                    } else {
                        // Assumed to be a continuation of the location string.  There are many forms of this string,
                        // igv only supports "join()"

                        // Crude test for strand
                        const strand = currentLocQualifier.includes("complement") ? "-" : "+";
                        f.strand = strand;

                        let joinString = currentLocQualifier.replace("join", "")
                            .replace("order", "")
                            .replace("complement", "")
                            .replace("(", "")
                            .replace(")", "");

                        if (joinString.includes("..")) {
                            joinString = joinString.replace("<", "")
                                .replace(">", "");

                            const exons = createExons(joinString, chr, strand);
                            const firstExon = exons[0];
                            f.start = firstExon.start;
                            const lastExon = exons[exons.length - 1];
                            f.end = lastExon.end;
                            if (exons.length > 1) {
                                f.exons = exons;
                            }
                        } else {
                            // TODO Single locus for now, other forms possible
                            //  const start = parseInt(joinString) - 1;const end = start + 1;
                            f.start = parseInt(joinString) - 1;
                            f.end = f.start + 1;
                        }
                    }
                    currentLocQualifier = tmp;
                } else {
                    currentLocQualifier = currentLocQualifier + tmp;
                }
        }
    }
    while (true)

    return features
}

/**
 * Create a list of Exon objects from the Embl join string.  Apparently exons in embl
 * format are represented by a single CDS record.
 *
 * @param joinString
 */
function createExons(joinString, chr, strand) {

    const lociArray = joinString.split(",");
    const exons = [];
    joinString.includes("complement");

    for (const loci of lociArray) {
        const tmp = loci.split("..");
        let exonStart = 0; // - (isNegative ? 0 : 1);

        try {
            exonStart = parseInt(tmp[0]) - 1;
        } catch (e) {
            console.error(e); // Handle error appropriately
        }

        let exonEnd = exonStart + 1;
        if (tmp.length > 1) {
            exonEnd = parseInt(tmp[1]);
        }

        exons.push({
            chr: chr,
            start: exonStart,
            end: exonEnd,
            strand: strand
        });
    }
    exons.sort(function (a, b) {
        return a.start - b.start
    });

    return exons

}

function stripQuotes(value) {
    if (value.startsWith('"') && value.endsWith('"')) {
        value = value.substring(1, value.length - 2);
    }
    return value
}

class GenbankFeatureSource extends BaseFeatureSource {

    constructor(config, genome) {
        super(genome);
        this.config = config;
        this.searchable = true;
    }

    // Feature source interface
    async getFeatures({chr, start, end, bpPerPixel, visibilityWindow}) {
        if(!this.featureSource) {
            const gbk = await loadGenbank(this.config.url);
            this.featureSource = new StaticFeatureSource({
                genome: this.config.genome,
                features: gbk.features,
                searchableFields: ['gene', 'db_xref', 'locus_tag', 'transcript_id']
            });

        }
        return this.featureSource.getFeatures({chr, start, end})
    }
    supportsWholeGenome() {
        return false
    }

    search(term) {
        return this.featureSource.search(term)
    }
}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2015 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const bbFormats = new Set(['bigwig', 'bw', 'bigbed', 'bb', 'biginteract', 'biggenepred', 'bignarrowpeak']);

function FeatureSource(config, genome) {

    const format = config.format ? config.format.toLowerCase() : undefined;

    if (config.features) {
        return new StaticFeatureSource(config, genome)
    } else if (bbFormats.has(format)) {
        return new BWSource(config, genome)
    } else if ("tdf" === format) {
        return new TDFSource(config, genome)
    } else if ("gbk" === format) {
        return new GenbankFeatureSource(config, genome)
    } else {
        return new TextFeatureSource(config, genome)
    }
}

function getExonPhase(exon) {
    return (3 - exon.readingFrame) % 3
}

function getEonStart(exon) {
    return exon.cdStart || exon.start
}

function getExonEnd(exon) {
    return exon.cdEnd || exon.end
}

const aminoAcidSequenceRenderThreshold = 0.25;

/**
 * @param feature
 * @param bpStart  genomic location of the left edge of the current canvas
 * @param xScale  scale in base-pairs per pixel
 * @returns {{px: number, px1: number, pw: number, h: number, py: number}}
 */
function calculateFeatureCoordinates(feature, bpStart, xScale) {
    let px = (feature.start - bpStart) / xScale;
    let px1 = (feature.end - bpStart) / xScale;
    //px = Math.round((feature.start - bpStart) / xScale),
    //px1 = Math.round((feature.end - bpStart) / xScale),
    let pw = px1 - px;

    if (pw < 3) {
        pw = 3;
        px -= 1.5;
    }

    return {
        px: px,
        px1: px1,
        pw: pw
    }
}

/**
 *
 * @param feature
 * @param bpStart  genomic location of the left edge of the current canvas
 * @param xScale  scale in base-pairs per pixel
 * @param pixelHeight  pixel height of the current canvas
 * @param ctx  the canvas 2d context
 * @param options  genomic state
 */
function renderFeature(feature, bpStart, xScale, pixelHeight, ctx, options) {

    try {
        ctx.save();

        // Set ctx color to a known valid color.  If getColorForFeature returns an invalid color string it is ignored, and
        // this default will be used.
        ctx.fillStyle = this.color;
        ctx.strokeStyle = this.color;

        const color = this.getColorForFeature(feature);
        // const color = '+' === feature.strand ? 'rgba(135,206,235,0.5)' : 'rgba(255,20,147,0.5)'

        ctx.fillStyle = color;
        ctx.strokeStyle = color;

        let h;
        let py;
        if (this.displayMode === "SQUISHED" && feature.row !== undefined) {
            h = this.featureHeight / 2;
            py = this.margin + this.squishedRowHeight * feature.row;
        } else if (this.displayMode === "EXPANDED" && feature.row !== undefined) {
            h = this.featureHeight;
            py = this.margin + this.expandedRowHeight * feature.row;
        } else {  // collapsed
            h = this.featureHeight;
            py = this.margin;
        }

        const pixelWidth = options.pixelWidth;   // typical 3*viewportWidth

        const cy = py + h / 2;
        const h2 = h / 2;
        const py2 = cy - h2 / 2;

        const exonCount = feature.exons ? feature.exons.length : 0;
        const coord = calculateFeatureCoordinates(feature, bpStart, xScale);
        const step = this.arrowSpacing;
        const direction = feature.strand === '+' ? 1 : feature.strand === '-' ? -1 : 0;

        if (exonCount === 0) {
            // single-exon transcript
            const xLeft = Math.max(0, coord.px);
            const xRight = Math.min(pixelWidth, coord.px1);
            const width = xRight - xLeft;

            ctx.fillRect(xLeft, py, width, h);

            if (direction !== 0) {
                ctx.fillStyle = "white";
                ctx.strokeStyle = "white";
                for (let x = xLeft + step / 2; x < xRight; x += step) {
                    // draw arrowheads along central line indicating transcribed orientation
                    IGVGraphics.strokeLine(ctx, x - direction * 2, cy - 2, x, cy);
                    IGVGraphics.strokeLine(ctx, x - direction * 2, cy + 2, x, cy);
                }
                ctx.fillStyle = color;
                ctx.strokeStyle = color;
            }
        } else {

            // multi-exon transcript
            IGVGraphics.strokeLine(ctx, coord.px + 1, cy, coord.px1 - 1, cy); // center line for introns
            const xLeft = Math.max(0, coord.px) + step / 2;
            const xRight = Math.min(pixelWidth, coord.px1);
            for (let x = xLeft; x < xRight; x += step) {
                // draw arrowheads along central line indicating transcribed orientation
                IGVGraphics.strokeLine(ctx, x - direction * 2, cy - 2, x, cy);
                IGVGraphics.strokeLine(ctx, x - direction * 2, cy + 2, x, cy);
            }

            for (let i = 0; i < feature.exons.length; i++) {

                const exon = feature.exons[i];

                // draw the exons
                let ePx = Math.round((exon.start - bpStart) / xScale);
                let ePx1 = Math.round((exon.end - bpStart) / xScale);
                let ePw = Math.max(1, ePx1 - ePx);
                let ePxU;

                if (ePx + ePw < 0) {
                    continue  // Off the left edge
                }
                if (ePx > pixelWidth) {
                    break // Off the right edge
                }

                if (exon.utr) {
                    ctx.fillRect(ePx, py2, ePw, h2); // Entire exon is UTR
                } else {
                    if (exon.cdStart) {
                        ePxU = Math.round((exon.cdStart - bpStart) / xScale);
                        ctx.fillRect(ePx, py2, ePxU - ePx, h2); // start is UTR
                        ePw -= (ePxU - ePx);
                        ePx = ePxU;
                    }
                    if (exon.cdEnd) {
                        ePxU = Math.round((exon.cdEnd - bpStart) / xScale);
                        ctx.fillRect(ePxU, py2, ePx1 - ePxU, h2); // start is UTR
                        ePw -= (ePx1 - ePxU);
                        ePx1 = ePxU;
                    }

                    ePw = Math.max(ePw, 1);

                    ctx.fillRect(ePx, py, ePw, h);

                    if (exon.readingFrame !== undefined) {

                        if (options.bpPerPixel < aminoAcidSequenceRenderThreshold &&
                            options.sequenceInterval) {

                            const leftExon = i > 0 && feature.exons[i - 1].readingFrame !== undefined ? feature.exons[i - 1] : undefined;
                            const riteExon = i < feature.exons.length - 1 && feature.exons[i + 1].readingFrame !== undefined ? feature.exons[i + 1] : undefined;

                            renderAminoAcidSequence.call(this, ctx, feature.strand, leftExon, exon, riteExon, bpStart, options.bpPerPixel, py, h, options.sequenceInterval);
                        }
                    }

                    // Arrows
                    if (ePw > step + 5 && direction !== 0 && options.bpPerPixel > aminoAcidSequenceRenderThreshold) {
                        ctx.fillStyle = "white";
                        ctx.strokeStyle = "white";
                        for (let x = ePx + step / 2; x < ePx1; x += step) {
                            // draw arrowheads along central line indicating transcribed orientation
                            IGVGraphics.strokeLine(ctx, x - direction * 2, cy - 2, x, cy);
                            IGVGraphics.strokeLine(ctx, x - direction * 2, cy + 2, x, cy);
                        }
                        ctx.fillStyle = color;
                        ctx.strokeStyle = color;

                    }
                }
            }
        }

        if (options.drawLabel && this.displayMode !== "SQUISHED") {
            renderFeatureLabel.call(this, ctx, feature, coord.px, coord.px1, py, options.referenceFrame, options);
        }
    } finally {
        ctx.restore();
    }
}

function renderAminoAcidSequence(ctx, strand, leftExon, exon, riteExon, bpStart, bpPerPixel, y, height, sequenceInterval) {

    const aaColors =
        [
            'rgb(124,124,204)',
            'rgb(12, 12, 120)'
        ];


    ctx.save();

    const renderAminoAcidLetter = (strand, width, xs, y, aminoAcidLetter) => {

        if ('STOP' === aminoAcidLetter) {
            aminoAcidLetter = '*';
        }

        const aminoAcidLetterWidth = ctx.measureText(aminoAcidLetter).width;
        IGVGraphics.fillText(ctx, aminoAcidLetter, xs + (width - aminoAcidLetterWidth) / 2, y - 4, {fillStyle: '#ffffff'});
    };

    const doPaint = (strand, start, end, aminoAcidLetter, colorToggle, index) => {

        const xs = Math.round((start - bpStart) / bpPerPixel);
        const xe = Math.round((end - bpStart) / bpPerPixel);

        const width = xe - xs;

        let aaLetter;
        if (undefined === aminoAcidLetter) {

            if (sequenceInterval.hasSequence(start, end)) {

                const sequence = sequenceInterval.getSequence(start, end);
                if (sequence && 3 === sequence.length) {
                    const key = '+' === strand ? sequence : complementSequence(sequence.split('').reverse().join(''));
                    aaLetter = translationDict[key];
                }
            }

        } else {
            aaLetter = aminoAcidLetter;
        }

        if ('M' === aminoAcidLetter) {
            ctx.fillStyle = '#83f902';
        } else if ('M' === aaLetter && 0 === index) {
            ctx.fillStyle = '#83f902';
        } else if ('STOP' === aaLetter) {
            ctx.fillStyle = '#ff2101';
        } else {
            ctx.fillStyle = aaColors[colorToggle];
        }

        ctx.fillRect(xs, y, width, height);

        if (aaLetter) {
            ctx.save();
            renderAminoAcidLetter(strand, width, xs, y + height, aaLetter);
            ctx.restore();
        }

        const widthBP = end - start;
        return widthBP > 0 && widthBP < 3 ? {start, end} : undefined
    };

    const phase = getExonPhase(exon);
    let ss = getEonStart(exon);
    let ee = getExonEnd(exon);

    let bpTripletStart;
    let bpTripletEnd;

    let remainder;
    let aminoAcidBackdropColorCounter = 1;
    let colorToggle;
    let index;
    if ('+' === strand) {

        if (phase > 0) {
            ss += phase;
        }

        aminoAcidBackdropColorCounter = 1;
        for (index = 0, bpTripletStart = ss; bpTripletStart < ee; index++, bpTripletStart += 3) {
            colorToggle = aminoAcidBackdropColorCounter % 2;
            bpTripletEnd = Math.min(ee, bpTripletStart + 3);
            remainder = doPaint(strand, bpTripletStart, bpTripletEnd, undefined, aminoAcidBackdropColorCounter % 2, index);
            ++aminoAcidBackdropColorCounter;
        }

        if (phase > 0 || remainder) {

            const result = phase > 0
                ? getAminoAcidLetterWithExonGap.call(this, strand, phase, ss - phase, ss, remainder, leftExon, exon, riteExon, sequenceInterval)
                : getAminoAcidLetterWithExonGap.call(this, strand, undefined, undefined, undefined, remainder, leftExon, exon, riteExon, sequenceInterval);

            if (result) {
                const {left, rite} = result;

                if (left) {
                    doPaint(strand, ss - phase, ss, left.aminoAcidLetter, 0, undefined);
                }

                if (rite) {
                    doPaint(strand, remainder.start, remainder.end, rite.aminoAcidLetter, colorToggle, undefined);
                }

            }

        }

    } else {

        if (phase > 0) {
            ee -= phase;
        }

        aminoAcidBackdropColorCounter = 1;
        index = 0;
        for (index = 0, bpTripletEnd = ee; bpTripletEnd > ss; index++, bpTripletEnd -= 3) {
            colorToggle = aminoAcidBackdropColorCounter % 2;
            bpTripletStart = Math.max(ss, bpTripletEnd - 3);
            remainder = doPaint(strand, bpTripletStart, bpTripletEnd, undefined, aminoAcidBackdropColorCounter % 2, index);
            ++aminoAcidBackdropColorCounter;
        }

        if (phase > 0 || remainder) {

            const result = phase > 0
                ? getAminoAcidLetterWithExonGap.call(this, strand, phase, ee, ee + phase, remainder, leftExon, exon, riteExon, sequenceInterval)
                : getAminoAcidLetterWithExonGap.call(this, strand, undefined, undefined, undefined, remainder, leftExon, exon, riteExon, sequenceInterval);

            if (result) {
                const {left, rite} = result;

                if (rite) {
                    doPaint(strand, ee, ee + phase, rite.aminoAcidLetter, 0, undefined);
                }

                if (left) {
                    doPaint(strand, remainder.start, remainder.end, left.aminoAcidLetter, colorToggle, undefined);
                }

            }

        }

    }

    ctx.restore();
}

/**
 * @param ctx       the canvas 2d context
 * @param feature
 * @param featureX  feature start in pixel coordinates
 * @param featureX1 feature end in pixel coordinates
 * @param featureY  feature y-coordinate
 * @param referenceFrame  genomic state
 * @param options  options
 */
function renderFeatureLabel(ctx, feature, featureX, featureX1, featureY, referenceFrame, options) {

    try {
        ctx.save();

        const labelField = this.config.labelField ? this.config.labelField : 'name';
        let name = feature[labelField];
        if (name === undefined && feature.gene) name = feature.gene.name;
        if (name === undefined) name = feature.id || feature.ID;
        if (!name || name === '.') return

        let pixelXOffset = options.pixelXOffset || 0;
        const t1 = Math.max(featureX, -pixelXOffset);
        const t2 = Math.min(featureX1, -pixelXOffset + options.viewportWidth);
        let centerX = (t1 + t2) / 2;
        //let centerX = (featureX + featureX1) / 2

        let transform;
        if (this.displayMode === "COLLAPSED" && this.labelDisplayMode === "SLANT") {
            transform = {rotate: {angle: 45}};
        }
        const labelY = getFeatureLabelY(featureY, transform);

        let color = this.getColorForFeature(feature);
        let selected = this.browser.qtlSelections.hasPhenotype(feature.name);

        const geneFontStyle = {
            textAlign: "SLANT" === this.labelDisplayMode ? undefined : 'center',
            fillStyle: color,
            strokeStyle: color
        };

        const textMetrics = ctx.measureText(name);
        const xleft = centerX - textMetrics.width / 2;
        const xright = centerX + textMetrics.width / 2;
        const lastLabelX = options.rowLastLabelX[feature.row] || -Number.MAX_SAFE_INTEGER;
        if (options.labelAllFeatures || xleft > lastLabelX || selected) {
            options.rowLastLabelX[feature.row] = xright;

            ctx.clearRect(
                centerX - textMetrics.width / 2 - 1,
                labelY - textMetrics.actualBoundingBoxAscent - 1,
                textMetrics.width + 2,
                textMetrics.actualBoundingBoxAscent + textMetrics.actualBoundingBoxDescent + 2);
            IGVGraphics.fillText(ctx, name, centerX, labelY, geneFontStyle, transform);

        }

    } finally {
        ctx.restore();
    }
}

function getFeatureLabelY(featureY, transform) {
    return transform ? featureY + 20 : featureY + 25
}

function getAminoAcidLetterWithExonGap(strand, phase, phaseExtentStart, phaseExtentEnd, remainder, leftExon, exon, riteExon, sequenceInterval) {

    let ss;
    let ee;
    let stringA = '';
    let stringB = '';
    let triplet = '';

    const aminoAcidLetters = {left: undefined, rite: undefined};
    if ('+' === strand) {

        if (phase) {
            stringB = sequenceInterval.getSequence(phaseExtentStart, phaseExtentEnd);

            if (!stringB) {
                return undefined
            }

            [ss, ee] = [getExonEnd(leftExon) - (3 - phase), getExonEnd(leftExon)];
            stringA = sequenceInterval.getSequence(ss, ee);

            if (!stringA) {
                return undefined
            }

            triplet = stringA + stringB;
            aminoAcidLetters.left = {triplet, aminoAcidLetter: translationDict[triplet]};
        }

        if (remainder) {
            stringA = sequenceInterval.getSequence(remainder.start, remainder.end);

            if (!stringA) {
                return undefined
            }

            const ritePhase = getExonPhase(riteExon);
            const riteStart = getEonStart(riteExon);
            stringB = sequenceInterval.getSequence(riteStart, riteStart + ritePhase);

            if (!stringB) {
                return undefined
            }

            triplet = stringA + stringB;
            aminoAcidLetters.rite = {triplet, aminoAcidLetter: translationDict[triplet]};
        }

    } else {

        if (phase) {
            stringA = sequenceInterval.getSequence(phaseExtentStart, phaseExtentEnd);

            if (undefined === stringA) {
                return undefined
            }

            [ss, ee] = [getEonStart(riteExon), getEonStart(riteExon) + (3 - phase)];
            stringB = sequenceInterval.getSequence(ss, ee);

            if (undefined === stringB) {
                return undefined
            }

            triplet = stringA + stringB;
            triplet = complementSequence(triplet.split('').reverse().join(''));
            aminoAcidLetters.rite = {triplet, aminoAcidLetter: translationDict[triplet]};
        }

        if (remainder) {
            stringB = sequenceInterval.getSequence(remainder.start, remainder.end);

            if (undefined === stringB) {
                return undefined
            }

            const leftPhase = getExonPhase(leftExon);
            const leftEnd = getExonEnd(leftExon);
            stringA = sequenceInterval.getSequence(leftEnd - leftPhase, leftEnd);

            if (undefined === stringA) {
                return undefined
            }

            triplet = stringA + stringB;
            triplet = complementSequence(triplet.split('').reverse().join(''));
            aminoAcidLetters.left = {triplet, aminoAcidLetter: translationDict[triplet]};
        }
    }

    return aminoAcidLetters
}

// SNP constants

const codingNonSynonSet = new Set(['nonsense', 'missense', 'stop-loss', 'frameshift', 'cds-indel']);
const codingSynonSet = new Set(['coding-synon']);
const spliceSiteSet = new Set(['splice-3', 'splice-5']);
const untranslatedSet = new Set(['untranslated-5', 'untranslated-3']);


/**
 * Renderer for a UCSC snp track
 *
 * @param snp
 * @param bpStart  genomic location of the left edge of the current canvas
 * @param xScale  scale in base-pairs per pixel
 * @param pixelHeight  pixel height of the current canvas
 * @param ctx  the canvas 2d context
 */
function renderSnp(snp, bpStart, xScale, pixelHeight, ctx) {

    var coord = calculateFeatureCoordinates(snp, bpStart, xScale),
        py = this.margin,
        h,
        colorArrLength = this.snpColors.length,
        colorPriority;

    h = this.displayMode === "squished" ? this.squishedRowHeight : this.expandedRowHeight;

    switch (this.colorBy) {
        case 'function':
            colorPriority = colorByFunc(snp.func);
            break
        case 'class':
            colorPriority = colorByClass(snp['class']);
    }

    ctx.fillStyle = this.snpColors[colorPriority];
    ctx.fillRect(coord.px, py, coord.pw, h);

    // Coloring functions, convert a value to a priority

    function colorByFunc(theFunc) {
        var priorities;
        var funcArray = theFunc.split(',');
        // possible func values


        priorities = funcArray.map(function (func) {
            if (codingNonSynonSet.has(func) || spliceSiteSet.has(func)) {
                return colorArrLength - 1
            } else if (codingSynonSet.has(func)) {
                return colorArrLength - 2
            } else if (untranslatedSet.has(func)) {
                return colorArrLength - 3
            } else { // locusSet.has(func) || intronSet.has(func)
                return 0
            }
        });

        return priorities.reduce(function (a, b) {
            return Math.max(a, b)
        })
    }

    function colorByClass(cls) {
        if (cls === 'deletion') {
            return colorArrLength - 1
        } else if (cls === 'mnp') {
            return colorArrLength - 2
        } else if (cls === 'microsatellite' || cls === 'named') {
            return colorArrLength - 3
        } else { // cls === 'single' || cls === 'in-del' || cls === 'insertion'
            return 0
        }
    }
}

/**
 *
 * @param feature
 * @param bpStart  genomic location of the left edge of the current canvas
 * @param xScale  scale in base-pairs per pixel
 * @param pixelHeight  pixel height of the current canvas
 * @param ctx  the canvas 2d context
 */
function renderFusionJuncSpan(feature, bpStart, xScale, pixelHeight, ctx) {

    const rowHeight = (this.displayMode === "EXPANDED") ? this.expandedRowHeight : this.squishedRowHeight;
    let py = this.margin;
    if (this.displayMode !== "COLLAPSED" && feature.row !== undefined) {
        py += feature.row * rowHeight;
    }

    const cy = py + 0.5 * rowHeight;
    const topY = cy - 0.5 * rowHeight;
    const bottomY = cy + 0.5 * rowHeight;

    // draw the junction arc
    const junctionLeftPx = Math.round((feature.junction_left - bpStart) / xScale);
    const junctionRightPx = Math.round((feature.junction_right - bpStart) / xScale);

    ctx.beginPath();
    ctx.moveTo(junctionLeftPx, cy);
    ctx.bezierCurveTo(junctionLeftPx, topY, junctionRightPx, topY, junctionRightPx, cy);

    ctx.lineWidth = 1 + Math.log(feature.num_junction_reads) / Math.log(2);
    ctx.strokeStyle = 'blue';
    ctx.stroke();

    // draw the spanning arcs
    const spanningCoords = feature.spanning_frag_coords;
    for (let i = 0; i < spanningCoords.length; i++) {

        const spanningInfo = spanningCoords[i];
        const spanLeftPx = Math.round((spanningInfo.left - bpStart) / xScale);
        const spanRightPx = Math.round((spanningInfo.right - bpStart) / xScale);

        ctx.beginPath();
        ctx.moveTo(spanLeftPx, cy);
        ctx.bezierCurveTo(spanLeftPx, bottomY, spanRightPx, bottomY, spanRightPx, cy);

        ctx.lineWidth = 1;
        ctx.strokeStyle = 'purple';
        ctx.stroke();
    }
}

class FeatureTrack extends TrackBase {

    static defaultColor = 'rgb(0,0,150)'

    static defaults = {
        type: "annotation",
        maxRows: 1000, // protects against pathological feature packing cases (# of rows of overlapping feaures)
        displayMode: "EXPANDED", // COLLAPSED | EXPANDED | SQUISHED
        margin: 10,
        featureHeight: 14,
        useScore: false
    }

    constructor(config, browser) {
        super(config, browser);
    }

    init(config) {
        super.init(config);

        // Obscure option, not common or supoorted, included for backward compatibility
        this.labelDisplayMode = config.labelDisplayMode;

        if (config._featureSource) {
            this.featureSource = config._featureSource;
            delete config._featureSource;
        } else if ('blat' !== config.type) {
            this.featureSource = config.featureSource ?
                config.featureSource :
                FeatureSource(config, this.browser.genome);
        }

        if ("FusionJuncSpan" === config.type) {
            this.render = config.render || renderFusionJuncSpan;
            this.squishedRowHeight = config.squishedRowHeight || 50;
            this.expandedRowHeight = config.expandedRowHeight || 50;
            this.height = config.height || this.margin + 2 * this.expandedRowHeight;
        } else if ('snp' === config.type) {
            this.render = config.render || renderSnp;
            // colors ordered based on priority least to greatest
            this.snpColors = ['rgb(0,0,0)', 'rgb(0,0,255)', 'rgb(0,255,0)', 'rgb(255,0,0)'];
            this.colorBy = 'function';
            this.expandedRowHeight = config.expandedRowHeight || 10;
            this.squishedRowHeight = config.squishedRowHeight || 5;
            this.height = config.height || 30;
        } else {
            this.render = config.render || renderFeature;
            this.arrowSpacing = 30;
            // adjust label positions to make sure they're always visible
            monitorTrackDrag(this);
            this.squishedRowHeight = config.squishedRowHeight || 15;
            this.expandedRowHeight = config.expandedRowHeight || 30;
            this.height = config.height || this.margin + 2 * this.expandedRowHeight;

            // Set colorBy fields considering legacy options for backward compatibility
            if (config.colorBy) {
                if (config.colorBy.field) {
                    config.colorTable = config.colorBy.pallete || config.colorBy.palette;
                    config.colorBy = config.colorBy.field;
                }
                this.colorBy = config.colorBy;   // Can be undefined => default
                if (config.colorTable) {
                    this.colorTable = new ColorTable(config.colorTable);
                } else {
                    this.colorTable = new PaletteColorTable("Set1");
                }
            }
        }
    }

    async postInit() {

        if (typeof this.featureSource.getHeader === "function") {
            this.header = await this.featureSource.getHeader();
            if (this.disposed) return   // This track was removed during async load
        }

        // Set properties from track line
        if (this.header) {
            this.setTrackProperties(this.header);
        }

        if (this.visibilityWindow === undefined && typeof this.featureSource.defaultVisibilityWindow === 'function') {
            this.visibilityWindow = await this.featureSource.defaultVisibilityWindow();
        }

        this._initialColor = this.color || this.constructor.defaultColor;
        this._initialAltColor = this.altColor || this.constructor.defaultColor;

        return this

    }

    /**
     * Return true if this track can be searched for genome location by feature property.
     * This track is searchable if its featureSource is searchable.
     * @returns {boolean}
     */
    get searchable() {
        return this.featureSource.searchable
    }

    async search(locus) {
        if (this.featureSource && this.featureSource.searchable) {
            return this.featureSource.search(locus)
        } else {
            return undefined
        }
    }

    /**
     * Return boolean indicating if this track supports the whole genome view.  Generally this is non-indexed feature
     * tracks.
     *
     * @returns {*|boolean}
     */
    get supportsWholeGenome() {
        if (this.config.supportsWholeGenome !== undefined) {
            return this.config.supportsWholeGenome
        } else if (this.featureSource && typeof this.featureSource.supportsWholeGenome === 'function') {
            return this.featureSource.supportsWholeGenome()
        } else {
            if (this.visibilityWindow === undefined && (this.config.indexed === false || !this.config.indexURL)) {
                return true
            }
        }
    }

    async getFeatures(chr, start, end, bpPerPixel) {
        const visibilityWindow = this.visibilityWindow;
        return this.featureSource.getFeatures({chr, start, end, bpPerPixel, visibilityWindow})
    }

    /**
     * The required height in pixels required for the track content.   This is not the visible track height, which
     * can be smaller (with a scrollbar) or larger.
     *
     * @param features
     * @returns {*}
     */
    computePixelHeight(features) {

        if (this.displayMode === "COLLAPSED") {
            return this.margin + this.expandedRowHeight
        } else {
            let maxRow = 0;
            if (features && (typeof features.forEach === "function")) {
                for (let feature of features) {
                    if (feature.row && feature.row > maxRow) {
                        maxRow = feature.row;
                    }
                }
            }

            const height = this.margin + (maxRow + 1) * ("SQUISHED" === this.displayMode ? this.squishedRowHeight : this.expandedRowHeight);
            return height

        }
    };

    /**
     *                 context: ctx,
     *                 pixelXOffset,
     *                 pixelWidth,
     *                 pixelHeight,
     *                 pixelTop,
     *                 bpStart,
     *                 bpEnd: bpEnd,
     *                 bpPerPixel,
     *                 windowFunction: this.windowFunction,
     *                 referenceFrame: this.referenceFrame,
     *                 selection: this.selection,
     *                 viewport: this,
     *                 viewportWidth: this.$viewport.width()
     * @param options
     */
    draw(options) {

        const {features, context, bpPerPixel, bpStart, bpEnd, pixelWidth, pixelHeight, referenceFrame} = options;

        // If drawing amino acids fetch cached sequence interval.  It is not needed if track does not support AA, but
        // costs nothing since only a reference to a cached object is fetched.
        if (bpPerPixel < aminoAcidSequenceRenderThreshold) {
            // Restrict the range requested to the limits: 1-chromosome.bpLength
            const chromosome = this.browser.genome.getChromosome(referenceFrame.chr);
            const chromosomeEnd = chromosome.bpLength;
            options.sequenceInterval = this.browser.genome.getSequenceInterval(referenceFrame.chr,
                bpStart > 0 ? bpStart : 0, bpEnd > chromosomeEnd ? chromosomeEnd : bpEnd);
        }


        if (!this.isMergedTrack) {
            IGVGraphics.fillRect(context, 0, options.pixelTop, pixelWidth, pixelHeight, {'fillStyle': "rgb(255, 255, 255)"});
        }

        if (features) {

            const rowFeatureCount = [];
            options.rowLastX = [];
            options.rowLastLabelX = [];
            for (let feature of features) {
                if (this._filter && !this._filter(feature)) continue
                if (feature.start > bpStart && feature.end < bpEnd) {
                    const row = this.displayMode === "COLLAPSED" ? 0 : feature.row || 0;
                    if (!rowFeatureCount[row]) {
                        rowFeatureCount[row] = 1;
                    } else {
                        rowFeatureCount[row]++;
                    }
                    options.rowLastX[row] = -Number.MAX_SAFE_INTEGER;
                    options.rowLastLabelX[row] = -Number.MAX_SAFE_INTEGER;
                }
            }
            const maxFeatureCount = Math.max(1, Math.max(...(rowFeatureCount.filter(c => !isNaN(c)))));
            const pixelsPerFeature = pixelWidth / maxFeatureCount;

            let lastPxEnd = [];
            const selectedFeatures = [];
            for (let feature of features) {

                if (this._filter && !this._filter(feature)) continue
                if (feature.end < bpStart) continue
                if (feature.start > bpEnd) break

                if (this.displayMode === 'COLLAPSED' && this.browser.qtlSelections.hasPhenotype(feature.name)) {
                    selectedFeatures.push(feature);
                }

                const row = this.displayMode === 'COLLAPSED' ? 0 : feature.row;
                options.drawLabel = options.labelAllFeatures || pixelsPerFeature > 10;
                const pxEnd = Math.ceil((feature.end - bpStart) / bpPerPixel);
                const last = lastPxEnd[row];
                if (!last || pxEnd > last) {

                    this.render.call(this, feature, bpStart, bpPerPixel, pixelHeight, context, options);

                    // Ensure a visible gap between features
                    const pxStart = Math.floor((feature.start - bpStart) / bpPerPixel);
                    if (last && pxStart - last <= 0) {
                        context.globalAlpha = 0.5;
                        IGVGraphics.strokeLine(context, pxStart, 0, pxStart, pixelHeight, {'strokeStyle': "rgb(255, 255, 255)"});
                        context.globalAlpha = 1.0;
                    }
                    lastPxEnd[row] = pxEnd;
                }
            }

            // If any features are selected redraw them here.  This insures selected features are visible in collapsed mode
            for (let feature of selectedFeatures) {
                options.drawLabel = true;
                this.render.call(this, feature, bpStart, bpPerPixel, pixelHeight, context, options);
            }

        }

    };

    clickedFeatures(clickState) {

        const y = clickState.y - this.margin;
        const allFeatures = super.clickedFeatures(clickState);

        let row;
        switch (this.displayMode) {
            case 'SQUISHED':
                row = Math.floor(y / this.squishedRowHeight);
                break
            case 'EXPANDED':
                row = Math.floor(y / this.expandedRowHeight);
                break
            default:
                row = undefined;
        }

        return allFeatures.filter(function (feature) {
            return (row === undefined || feature.row === undefined || row === feature.row)
        })
    }

    /**
     * Return "popup data" for feature @ genomic location.  Data is an array of key-value pairs
     */
    popupData(clickState, features) {

        if (features === undefined) features = this.clickedFeatures(clickState);
        const genomicLocation = clickState.genomicLocation;
        const data = [];
        for (let feature of features) {

            // Whole genome hack, whole-genome psuedo features store the "real" feature in an _f field
            const f = feature._f || feature;

            const featureData = (typeof f.popupData === "function") ?
                f.popupData(genomicLocation) :
                this.extractPopupData(f);

            if (featureData) {

                if (data.length > 0) {
                    data.push("<hr/><hr/>");
                }

                // If we have an infoURL, find the name property and create the link.  We do this at this level
                // to catch name properties in both custom popupData functions and the generic extractPopupData function

                const infoURL = this.infoURL || this.config.infoURL;
                for (let fd of featureData) {
                    data.push(fd);
                    if (infoURL &&
                        fd.name &&
                        fd.name.toLowerCase() === "name" &&
                        fd.value &&
                        isString$3(fd.value) &&
                        !fd.value.startsWith("<")) {
                        const href = infoURL.replace("$$", feature.name);
                        fd.value = `<a target=_blank href=${href}>${fd.value}</a>`;
                    }
                }


                //Array.prototype.push.apply(data, featureData);

                // If we have clicked over an exon number it.
                // Disabled for GFF and GTF files if the visibility window is < the feature length since we don't know if we have all exons
                const isGFF = "gff" === this.config.format || "gff3" === this.config.format || "gtf" === this.config.format;
                if (f.exons && f.exons.length > 1) {
                    for (let i = 0; i < f.exons.length; i++) {
                        const exon = f.exons[i];
                        if (genomicLocation >= exon.start && genomicLocation <= exon.end) {
                            const exonNumber = isGFF ?
                                exon.number :
                                f.strand === "-" ? f.exons.length - i : i + 1;
                            if (exonNumber) {
                                data.push('<hr/>');
                                data.push({name: "Exon Number", value: exonNumber});
                            }
                            break
                        }
                    }
                }
            }
        }

        return data

    }

    menuItemList() {

        const menuItems = [];

        if (this.render === renderSnp) {
            menuItems.push('<hr/>');

            for (const colorScheme of ["function", "class"]) {

                function colorSchemeHandler() {
                    this.colorBy = colorScheme;
                    this.trackView.repaintViews();
                }

                menuItems.push({element:createCheckbox(`Color by ${colorScheme}`, colorScheme === this.colorBy), click: colorSchemeHandler});
            }
        }

        menuItems.push('<hr/>');

        const lut =
            {
                "COLLAPSED": "Collapse",
                "SQUISHED": "Squish",
                "EXPANDED": "Expand"
            };

        for (const displayMode of ["COLLAPSED", "SQUISHED", "EXPANDED"]) {

            function displayModeHandler() {
                this.displayMode = displayMode;
                this.config.displayMode = displayMode;
                this.trackView.checkContentHeight();
                this.trackView.repaintViews();
            }

            menuItems.push({element:createCheckbox(lut[displayMode], displayMode === this.displayMode), click: displayModeHandler});
        }

        return menuItems

    };


    contextMenuItemList(clickState) {

        const features = this.clickedFeatures(clickState);

        if (undefined === features || 0 === features.length) {
            return undefined
        }

        if (features.length > 1) {
            features.sort((a, b) => (b.end - b.start) - (a.end - a.start));
        }
        const f = features[0];   // The shortest clicked feature

        if ((f.end - f.start) <= 1000000) {
            const list = [{
                label: 'View feature sequence',
                click: async () => {
                    let seq = await this.browser.genome.getSequence(f.chr, f.start, f.end);
                    if (!seq) {
                        seq = "Unknown sequence";
                    } else if (f.strand === '-') {
                        seq = reverseComplementSequence(seq);
                    }
                    this.browser.alert.present(seq);

                }
            }];

            if (isSecureContext() && navigator.clipboard !== undefined) {
                list.push(
                    {
                        label: 'Copy feature sequence',
                        click: async () => {
                            let seq = await this.browser.genome.getSequence(f.chr, f.start, f.end);
                            if (!seq) {
                                seq = "Unknown sequence";
                            } else if (f.strand === '-') {
                                seq = reverseComplementSequence(seq);
                            }
                            try {
                                await navigator.clipboard.writeText(seq);
                            } catch (e) {
                                console.error(e);
                                this.browser.alert.present(`error copying sequence to clipboard ${e}`);
                            }
                        }
                    }
                );
            }
            list.push('<hr/>');
            return list
        } else {
            return undefined
        }
    }

    description() {

        // if('snp' === this.type) {
        if (renderSnp === this.render) {
            let desc = "<html>" + this.name + '<hr/>';
            desc += '<em>Color By Function:</em><br>';
            desc += '<span style="color:red">Red</span>: Coding-Non-Synonymous, Splice Site<br>';
            desc += '<span style="color:green">Green</span>: Coding-Synonymous<br>';
            desc += '<span style="color:blue">Blue</span>: Untranslated<br>';
            desc += '<span style="color:black">Black</span>: Intron, Locus, Unknown<br><br>';
            desc += '<em>Color By Class:</em><br>';
            desc += '<span style="color:red">Red</span>: Deletion<br>';
            desc += '<span style="color:green">Green</span>: MNP<br>';
            desc += '<span style="color:blue">Blue</span>: Microsatellite, Named<br>';
            desc += '<span style="color:black">Black</span>: Indel, Insertion, SNP';
            desc += "</html>";
            return desc
        } else {
            return super.description()
        }

    };

    /**
     * Return color for feature.
     * @param feature
     * @returns {string}
     */

    getColorForFeature(f) {

        const feature = f._f || f;    // f might be a "whole genome" wrapper

        let color;

        if (f.name && this.browser.qtlSelections.hasPhenotype(f.name)) {
            color = this.browser.qtlSelections.colorForGene(f.name);
        } else if (this.altColor && "-" === feature.strand) {
            color = (typeof this.altColor === "function") ? this.altColor(feature) : this.altColor;
        } else if (this.color) {
            color = (typeof this.color === "function") ? this.color(feature) : this.color;  // Explicit setting via menu, or possibly track line if !config.color
        } else if (this.colorBy) {
            const value = feature.getAttributeValue ?
                feature.getAttributeValue(this.colorBy) :
                feature[this.colorBy];
            color = this.colorTable.getColor(value);
        } else if (feature.color) {
            color = feature.color;   // Explicit color for feature
        }

        // If no explicit setting use the default
        if (!color) {
            color = FeatureTrack.defaultColor;   // Track default
        }

        if (feature.alpha && feature.alpha !== 1) {
            color = IGVColor.addAlpha(color, feature.alpha);
        } else if (this.useScore && feature.score && !Number.isNaN(feature.score)) {
            // UCSC useScore option, for scores between 0-1000.  See https://genome.ucsc.edu/goldenPath/help/customTrack.html#TRACK
            const min = this.config.min ? this.config.min : this.viewLimitMin ? this.viewLimitMin : 0;
            const max = this.config.max ? this.config.max : this.viewLimitMax ? this.viewLimitMax : 1000;
            const alpha = getAlpha(min, max, feature.score);
            feature.alpha = alpha;    // Avoid computing again
            color = IGVColor.addAlpha(color, alpha);
        }


        function getAlpha(min, max, score) {
            const binWidth = (max - min) / 9;
            const binNumber = Math.floor((score - min) / binWidth);
            return Math.min(1.0, 0.2 + (binNumber * 0.8) / 9)
        }

        return color
    }


    /**
     * Called when the track is removed.  Do any needed cleanup here
     */
    dispose() {
        this.trackView = undefined;
    }
}

/**
 * Monitors track drag events, updates label position to ensure that they're always visible.
 * @param track
 */
function monitorTrackDrag(track) {

    if (track.browser.on) {
        track.browser.on('trackdragend', onDragEnd);
        track.browser.on('trackremoved', unSubscribe);
    }

    function onDragEnd() {
        if (track.trackView && track.displayMode !== "SQUISHED") {
            // Repaint views to adjust feature name if center is moved out of view
            track.trackView.repaintViews();
        }
    }

    function unSubscribe(removedTrack) {
        if (track.browser.un && track === removedTrack) {
            track.browser.un('trackdragend', onDragEnd);
            track.browser.un('trackremoved', unSubscribe);
        }
    }

}

class RegionTableBase {
    constructor(config) {

        this.config = config;

        this.browser = config.browser;

        this.columnFormat = config.columnFormat;

        this.tableRowSelectionList = [];

        this.tableDOM = div$2({ class: 'igv-roi-table' });

        if(config.width) {
            let [ w ] = config.width.split('px');
            w = parseInt(w, 10);
            this.tableDOM.style.width = `${Math.min(w, 1600)}px`;
        }

        config.parent.appendChild(this.tableDOM);

        this.headerDOM = config;

        this.tableColumnTitles = this.tableDOM;

        this.tableRowContainer = this.tableDOM;

        this.footerDOM = config.gotoButtonHandler;

    }

    set headerDOM({ browser, parent, headerTitle, dismissHandler }) {

        // header
        const dom = div$2();
        this.tableDOM.appendChild(dom);

        // header title
        const div$1 = div$2();
        dom.appendChild(div$1);
        div$1.innerHTML = headerTitle;

        // table dismiss button
        const dismiss = div$2();
        dom.appendChild(dismiss);
        dismiss.appendChild(createIcon$1('times'));

        this.boundDismissHandler = mouseClickHandler.bind(this);

        dismiss.addEventListener('click', this.boundDismissHandler);

        function mouseClickHandler (event) {
            event.stopPropagation();
            dismissHandler();
        }

        const { y:y_root } = browser.root.getBoundingClientRect();
        const { y:y_parent } = parent.getBoundingClientRect();
        const constraint = -(y_parent - y_root);
        makeDraggable$1(this.tableDOM, dom, { minX:0, minY:constraint });

        this.tableDOM.style.display = 'none';

        this._headerDOM = dom;

    }

    set tableColumnTitles(tableDOM) {

        const tblColumnTitles = div$2({ class: 'igv-roi-table-column-titles' });
        tableDOM.appendChild(tblColumnTitles);

        for (const { label, width } of this.columnFormat) {
            const col = div$2();
            tblColumnTitles.appendChild(col);
            col.style.width = width;
            col.innerText = label;
        }

        this._tableColumnTitlesDOM = tblColumnTitles;

    }

    get tableColumnTitles() {
        return this._tableColumnTitlesDOM
    }

    set tableRowContainer(container) {

        const tblRowContainer = div$2({ class: 'igv-roi-table-row-container' });
        container.appendChild(tblRowContainer);

        this._tableRowContainerDOM = tblRowContainer;

    }

    get tableRowContainer() {
        return this._tableRowContainerDOM
    }

    set footerDOM(gotoButtonHandler) {

        const dom = div$2();
        this.tableDOM.appendChild(dom);

        // Go To Button
        const gotoButton = div$2({class: 'igv-roi-table-button'});
        dom.appendChild(gotoButton);

        gotoButton.id = 'igv-roi-table-view-button';
        gotoButton.textContent = 'Go To';
        gotoButton.style.pointerEvents = 'none';

        this._footerDOM = dom;

        this.gotoButton = gotoButton;

        this.boundGotoButtonHandler = gotoButtonHandler.bind(this);

        this.gotoButton.addEventListener('click', this.boundGotoButtonHandler);

    }

    tableRowDOMHelper(dom) {

        dom.addEventListener('mousedown', event => {
            event.stopPropagation();

            dom.classList.toggle('igv-roi-table-row-selected');
            dom.classList.contains('igv-roi-table-row-selected') ? dom.classList.remove('igv-roi-table-row-hover') : dom.classList.add('igv-roi-table-row-hover');

            this.setTableRowSelectionState(dom.classList.contains('igv-roi-table-row-selected'));
        });

        dom.addEventListener('mouseover', e => {
            dom.classList.contains('igv-roi-table-row-selected') ? dom.classList.remove('igv-roi-table-row-hover') : dom.classList.add('igv-roi-table-row-hover');
        });

        dom.addEventListener('mouseout', e => {
            dom.classList.remove('igv-roi-table-row-hover');
        });

    }

    clearTable() {
        const elements = this.tableRowContainer.querySelectorAll('.igv-roi-table-row');
        for (let el of elements) {
            el.remove();
        }
    }

    setTableRowSelectionState(isTableRowSelected) {
        isTableRowSelected ? this.tableRowSelectionList.push(1) : this.tableRowSelectionList.pop();
        this.gotoButton.style.pointerEvents = this.tableRowSelectionList.length > 0 ? 'auto' : 'none';
    }

    present() {
        this.tableDOM.style.left = `${ 0 }px`;

        const { y:y_root } = this.browser.root.getBoundingClientRect();
        const { y:y_parent } = this.config.parent.getBoundingClientRect();

        this.tableDOM.style.top  = `${ y_root - y_parent }px`;
        this.tableDOM.style.display = 'flex';
    }

    dismiss() {
        this.tableDOM.style.display = 'none';
    }

    isVisible() {
        return this.tableDOM.style.display !== 'none'
    }

    dispose() {

        this.tableDOM.innerHTML = '';
        this.tableDOM.remove();

        for (const key of Object.keys(this)) {
            this[key] = undefined;
        }

        document.removeEventListener('click', this.boundDismissHandler);

    }

}

class BlatTable extends RegionTableBase {

    constructor(config) {

        const cooked = Object.assign({ 'width':'1024px' }, config);
        super(cooked);

        this.descriptionDOM = config;

    }

    set descriptionDOM(config) {

        if (config.description) {

            let dom;

            // BLAT result for query sequence
            dom = div$2({ class: 'igv-roi-table-description' });
            this.tableDOM.insertBefore(dom, this.tableColumnTitles);
            dom.style.height = 'auto';
            dom.innerHTML = `BLAT result for query sequence:`;

            // CTAATCAtctacactggtttctactg ...
            dom = div$2({ class: 'igv-roi-table-description' });
            this.tableDOM.insertBefore(dom, this.tableColumnTitles);
            dom.style.height = 'auto';
            dom.style.maxHeight = '128px';
            dom.innerHTML = config.description;

            // Select one or more rows ...
            dom = div$2({ class: 'igv-roi-table-goto-explainer' });
            this.tableDOM.insertBefore(dom, this.tableColumnTitles);
            dom.innerHTML = `Select one or more rows and click Go To to view the regions`;

        }

    }

    tableRowDOM(record) {

        const dom = div$2({ class: 'igv-roi-table-row' });

        const pretty = record.map(item => isFinite(item) ? numberFormatter$1(item) : item);

        for (let i = 0; i < pretty.length; i++) {

            const el = div$2();
            dom.appendChild(el);

            const format = this.columnFormat[ i ];
            el.style.width = format.width || 'fit-content';
            el.innerText = pretty[ i ];
        }

        this.tableRowDOMHelper(dom);

        return dom
    }

    renderTable(records) {

        Array.from(this.tableRowContainer.querySelectorAll('.igv-roi-table-row')).forEach(el => el.remove());

        if (records.length > 0) {

            for (let record of records) {
                const row = this.tableRowDOM(record);
                this.tableRowContainer.appendChild(row);
            }

        }

    }

    static getColumnFormatConfiguration() {

        /*
        return [
            { label:         'chr', width: '60px' },
            { label:       'start', width: '100px' },
            { label:         'end', width: '100px' },
            { label:      'strand', width: '50px' },
            { label:       'score', width: '50px' },
            { label:       'match', width: '50px' },
            { label:   "mis-match", width: '70px' },
            { label:  "rep. match", width: '70px' },
            { label:         "N's", width: '32px' },
            { label: 'Q gap count', width: '90px' },
            { label: 'Q gap bases', width: '90px' },
            { label: 'T gap count', width: '90px' },
            { label: 'T gap bases', width: '90px' },
        ]
        */

        return [
            { label:         'chr', width: '7%' },
            { label:       'start', width: '12%' },
            { label:         'end', width: '12%' },
            { label:      'strand', width: '5%' },
            { label:       'score', width: '5%' },
            { label:       'match', width: '5%' },
            { label:   "mis-match", width: '7%' },
            { label:  "rep. match", width: '7%' },
            { label:         "N's", width: '3%' },
            { label: 'Q gap count', width: '9%' },
            { label: 'Q gap bases', width: '9%' },
            { label: 'T gap count', width: '9%' },
            { label: 'T gap bases', width: '9%' },
        ]
    }

    static gotoButtonHandler (event) {

        event.stopPropagation();

        const selectedRows = this.tableDOM.querySelectorAll('.igv-roi-table-row-selected');

        const loci = [];
        for (const row of selectedRows) {

            const record = [];
            row.querySelectorAll('div').forEach(el => record.push(el.innerText));

            const [ chr, start, end ] = record;
            loci.push(`${ chr }:${ start }-${ end }`);
        }

        for (const el of this.tableDOM.querySelectorAll('.igv-roi-table-row')) {
            el.classList.remove('igv-roi-table-row-selected');
        }

        this.setTableRowSelectionState(false);

        this.browser.search(loci.join(' '));

        // console.log(`browser search( ${loci.join(' ')} )`)

    }

}

/*
http://genome.ucsc.edu/cgi-bin/hgBlat
?userSeq=CTAATCAtctacactggtttctactgaaatgtctgttgtcatagacttaattgtgtcttcagatacagcagttctgttatttctgagttttacctggggcaagagaatctttagcaagtttaaaggcacctatatctggaatcacccctccctccagatgaatatcacagactctcccattaaaggtcttgccTTCCTTGATAGCATCATCACTCCA
&type=DNA
&db=hg38
&output=json
 */

//const blatServer = "https://genome.ucsc.edu/cgi-bin/hgBlat"
const defaultBlatServer$1 = "https://igv.org/services/blatUCSC.php";
//const blatServer = "http://localhost:8000/blatUCSC.php"


async function blat({url, userSeq, db}) {

    url = url || defaultBlatServer$1;

    if(!db) {
        throw Error("Blat database is not defined")
    }

    const results = await postData(url, userSeq, db);

    results.fields;

    const features = results.blat.map(decodePSL);

    return features
}

async function postData(url = "", userSeq, db) {

    const data = new URLSearchParams();
    data.append("userSeq", userSeq);
    data.append("db", db);

    const response = await fetch(url, { method: "post", body: data });
    return response.json(); // parses JSON response into native JavaScript objects
}

const maxSequenceSize$1 = 25000;
//const blatServer = "https://genome.ucsc.edu/cgi-bin/hgBlat"
const defaultBlatServer = "https://igv.org/services/blatUCSC.php";
//const blatServer = "http://localhost:8000/blatUCSC.php"

class BlatTrack extends FeatureTrack {

    constructor(config, browser) {
        super(config, browser);
        if (!this.name) {
            this.name = 'Blat Results';
        }
        this.sequence = config.sequence;
        this.table = undefined;

        // On initial creation features are fetched before track construction
        if(config.features) {
            this._features = config.features;
            this.featureSource = new StaticFeatureSource({features: config.features}, this.browser.genome);
            delete config.features;
        }
    }

    async postInit() {
        if(!this.featureSource) {
            // This will be the case when restoring from a session
            const db = this.browser.genome.ucscID;   // TODO -- blat specific property
            const url = this.browser.config["blatServerURL"];
            const features = await blat({url, userSeq: this.sequence, db});
            this._features = features;
            this.featureSource = new StaticFeatureSource({features}, this.browser.genome);
        }

        this._initialColor = this.color || this.constructor.defaultColor;
        this._initialAltColor = this.altColor || this.constructor.defaultColor;

    }

    openTableView() {

        if (undefined === this.table) {

            const rows = this._features.map(f => [
                this.browser.genome.getChromosomeDisplayName(f.chr),
                (f.start + 1),
                f.end,
                f.strand,
                f.score,
                f.matches,
                f.misMatches,
                f.repMatches,
                f.nCount,
                f.qNumInsert,
                f.qBaseInsert,
                f.tNumInsert,
                f.tBaseInsert
            ]);

            const config =
                {
                    browser: this.browser,
                    parent: this.browser.columnContainer,
                    headerTitle: this.config.title,
                    description: this.sequence,
                    dismissHandler: () => {
                        this.table.dismiss();
                        this.table.dispose();
                        this.table = undefined;
                    },
                    columnFormat: BlatTable.getColumnFormatConfiguration(),
                    gotoButtonHandler: BlatTable.gotoButtonHandler
                };

            this.table = new BlatTable(config);
            this.table.renderTable(rows);
        }

        this.table.present();

    }

    menuItemList() {

        const menuItems = super.menuItemList();

        menuItems.push('<hr/>');

        function click() {
            this.openTableView();
        }
        menuItems.push({ label: 'Open table view', click });

        return menuItems
    }


    /**
     * Track has been permanently removed.  Release resources and other cleanup
     */
    dispose() {
        super.dispose();
        // Release DOM element for table
        if (this.table) {
            this.table.popover.parentElement.removeChild(this.table.popover);
        }
    }
}


async function createBlatTrack({sequence, browser, name, title}) {

    if (sequence.length > maxSequenceSize$1) {
        browser.alert.present(`Sequence size exceeds maximum allowed length (${sequence.length} > ${maxSequenceSize$1})`);
        return
    }

    try {

        const db = browser.genome.ucscID;   // TODO -- blat specific property
        const url = browser.config["blatServerURL"] || defaultBlatServer;
        const features = await blat({url, userSeq: sequence, db});

        const trackConfig = {
            type: 'blat',
            name: name || 'blat results',
            title: title || 'blat results',
            sequence: sequence,
            altColor: 'rgb(176, 176, 236)',
            color: 'rgb(236, 176, 176)',
            searchable: false,
            features
        };

        const track = (await browser.loadTrackList([trackConfig]))[0];
        track.openTableView();

    } catch (e) {
        browser.alert.present(`Error performing blat search:  ${e}`);
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

const defaultSequenceTrackOrder = Number.MIN_SAFE_INTEGER;

const translationDict = {
    'TTT': 'F',
    'TTC': 'F',
    'TTA': 'L',
    'TTG': 'L',
    'CTT': 'L',
    'CTC': 'L',
    'CTA': 'L',
    'CTG': 'L',
    'ATT': 'I',
    'ATC': 'I',
    'ATA': 'I',
    'ATG': 'M',
    'GTT': 'V',
    'GTC': 'V',
    'GTA': 'V',
    'GTG': 'V',
    'TCT': 'S',
    'TCC': 'S',
    'TCA': 'S',
    'TCG': 'S',
    'CCT': 'P',
    'CCC': 'P',
    'CCA': 'P',
    'CCG': 'P',
    'ACT': 'T',
    'ACC': 'T',
    'ACA': 'T',
    'ACG': 'T',
    'GCT': 'A',
    'GCC': 'A',
    'GCA': 'A',
    'GCG': 'A',
    'TAT': 'Y',
    'TAC': 'Y',
    'TAA': 'STOP',
    'TAG': 'STOP',
    'CAT': 'H',
    'CAC': 'H',
    'CAA': 'Q',
    'CAG': 'Q',
    'AAT': 'N',
    'AAC': 'N',
    'AAA': 'K',
    'AAG': 'K',
    'GAT': 'D',
    'GAC': 'D',
    'GAA': 'E',
    'GAG': 'E',
    'TGT': 'C',
    'TGC': 'C',
    'TGA': 'STOP',
    'TGG': 'W',
    'CGT': 'R',
    'CGC': 'R',
    'CGA': 'R',
    'CGG': 'R',
    'AGT': 'S',
    'AGC': 'S',
    'AGA': 'R',
    'AGG': 'R',
    'GGT': 'G',
    'GGC': 'G',
    'GGA': 'G',
    'GGG': 'G'
};

const complement = {};
const t1 = ['A', 'G', 'C', 'T', 'Y', 'R', 'W', 'S', 'K', 'M', 'D', 'V', 'H', 'B', 'N', 'X'];
const t2 = ['T', 'C', 'G', 'A', 'R', 'Y', 'W', 'S', 'M', 'K', 'H', 'B', 'D', 'V', 'N', 'X'];
for (let i = 0; i < t1.length; i++) {
    complement[t1[i]] = t2[i];
    complement[t1[i].toLowerCase()] = t2[i].toLowerCase();
}

const DEFAULT_HEIGHT = 25;
const TRANSLATED_HEIGHT = 115;
const SEQUENCE_HEIGHT = 15;
const FRAME_HEIGHT = 25;
const FRAME_BORDER = 5;
const BP_PER_PIXEL_THRESHOLD = 1 / 10;

const bppSequenceThreshold = 10;

class SequenceTrack {


    constructor(config, browser) {

        this.config = config;
        this.browser = browser;
        this.type = "sequence";
        this.removable = config.removable === true;  // defaults to false
        this.name = config.name;
        this.id = config.id;
        this.sequenceType = config.sequenceType || "dna";             //   dna | rna | prot
        this.disableButtons = false;
        this.order = config.order || defaultSequenceTrackOrder;
        this.ignoreTrackMenu = false;

        this.reversed = config.reversed === true;
        this.frameTranslate = config.frameTranslate === true;
        this.height = this.frameTranslate ? TRANSLATED_HEIGHT : DEFAULT_HEIGHT;

        // Hack for backward compatibility
        if (config.url) {
            config.fastaURL = config.url;
        }

        if (!config.fastaURL) {
            // Mark this as the genome reference sequence ==> backward compatibility convention
            this.id = config.id || "sequence";
        }

    }

    menuItemList() {
        return [
            {
                name: this.reversed ? "Forward" : "Reverse",
                click: () => {
                    this.reversed = !this.reversed;
                    this.trackView.repaintViews();
                }
            },
            {
                name: this.frameTranslate ? "Close Translation" : "Three-frame Translate",
                click: () => {
                    this.frameTranslate = !this.frameTranslate;
                    if (this.frameTranslate) {
                        for (let vp of this.trackView.viewports) {
                            vp.setContentHeight(TRANSLATED_HEIGHT);
                        }
                        this.trackView.setTrackHeight(TRANSLATED_HEIGHT);
                    } else {
                        for (let vp of this.trackView.viewports) {
                            vp.setContentHeight(DEFAULT_HEIGHT);
                        }
                        this.trackView.setTrackHeight(DEFAULT_HEIGHT);
                    }
                    this.trackView.repaintViews();

                }
            }
        ]
    }

    contextMenuItemList(clickState) {
        const viewport = clickState.viewport;
        if (viewport.referenceFrame.bpPerPixel <= 1) {
            const pixelWidth = viewport.getWidth();
            const bpWindow = pixelWidth * viewport.referenceFrame.bpPerPixel;
            const chr = viewport.referenceFrame.chr;
            const start = Math.floor(viewport.referenceFrame.start);
            const end = Math.ceil(start + bpWindow);
            const items = [
                {
                    label: this.reversed ? 'View visible sequence (reversed)...' : 'View visible sequence...',
                    click: async () => {
                        let seq = await this.browser.genome.getSequence(chr, start, end);
                        if (!seq) {
                            seq = "Unknown sequence";
                        } else if (this.reversed) {
                            seq = reverseComplementSequence(seq);
                        }
                        this.browser.alert.present(seq);
                    }
                }
            ];

            if (isSecureContext()) {
                items.push({
                    label: 'Copy visible sequence',
                    click: async () => {
                        let seq = await this.browser.genome.getSequence(chr, start, end);
                        if (!seq) {
                            seq = "Unknown sequence";
                        } else if (this.reversed) {
                            seq = reverseComplementSequence(seq);
                        }
                        try {
                            await navigator.clipboard.writeText(seq);
                        } catch (e) {
                            console.error(e);
                            this.browser.alert.present(`error copying sequence to clipboard ${e}`);
                        }
                    }
                });
            }

            items.push({
                label: 'BLAT visible sequence',
                click: async () => {
                    let sequence = await this.browser.genome.getSequence(chr, start, end);
                    if (sequence) {
                        if (this.reversed) {
                            sequence = reverseComplementSequence(sequence);
                        }
                        const name = `blat: ${chr}:${start + 1}-${end}`;
                        const title = `blat: ${chr}:${start + 1}-${end}`;
                        createBlatTrack({sequence, browser: this.browser, name, title});
                    }
                }
            });


            items.push('<hr/>');

            return items
        } else {
            return undefined
        }
    }

    translateSequence(seq) {

        const threeFrame = [[], [], []];

        for (let fNum of [0, 1, 2]) {
            let idx = fNum;

            while ((seq.length - idx) >= 3) {
                let st = seq.slice(idx, idx + 3);
                if (this.reversed) {
                    st = st.split('').reverse().join('');
                }

                const aa = translationDict[st.toUpperCase()] || "";
                threeFrame[fNum].push({
                    codons: st,
                    aminoA: aa
                });
                idx += 3;
            }
        }

        return threeFrame
    }

    /**
     * Return the source for sequence.  If an explicit fasta url is defined, use it, otherwise fetch sequence
     * from the current genome
     * *
     * @returns {Promise<WrappedFasta|*>}
     */
    async getSequenceSource() {
        if (this.config.fastaURL) {
            if (!this.fasta) {
                this.fasta = new WrappedFasta(this.config, this.browser.genome);
                await this.fasta.init();
            }
            return this.fasta
        } else {
            return this.browser.genome
        }
    }

    async getFeatures(chr, start, end, bpPerPixel) {

        start = Math.floor(start);
        end = Math.floor(end);

        if (bpPerPixel && bpPerPixel > bppSequenceThreshold) {
            return null
        } else {
            const sequenceSource = await this.getSequenceSource();
            //const extent = expandRegion(start, end, 1e5)
            const sequence = await sequenceSource.getSequence(chr, start, end);
            return {
                bpStart: start,
                sequence: sequence
            }
        }
    }

    draw(options) {

        const ctx = options.context;

        if (options.features) {

            let sequence = options.features.sequence;
            if (!sequence) {
                return
            }

            if (this.reversed) {
                sequence = sequence.split('').map(function (cv) {
                    return complement[cv]
                }).join('');
            }

            const sequenceBpStart = options.features.bpStart;  // genomic position at start of sequence
            const bpEnd = 1 + options.bpStart + (options.pixelWidth * options.bpPerPixel);

            for (let bp = Math.floor(options.bpStart); bp <= bpEnd; bp++) {

                const seqIdx = Math.floor(bp - sequenceBpStart);

                if (seqIdx >= 0 && seqIdx < sequence.length) {

                    const offsetBP = bp - options.bpStart;
                    const aPixel = offsetBP / options.bpPerPixel;
                    const pixelWidth = 1 / options.bpPerPixel;
                    const baseLetter = sequence[seqIdx];
                    const color = this.fillColor(baseLetter.toUpperCase());

                    if (options.bpPerPixel > BP_PER_PIXEL_THRESHOLD) {
                        IGVGraphics.fillRect(ctx, aPixel, FRAME_BORDER, pixelWidth, SEQUENCE_HEIGHT - FRAME_BORDER, {fillStyle: color});
                    } else {
                        const textPixel = aPixel + 0.5 * (pixelWidth - ctx.measureText(baseLetter).width);


                        if ('y' === options.axis) {
                            ctx.save();
                            IGVGraphics.labelTransformWithContext(ctx, textPixel);
                            IGVGraphics.strokeText(ctx, baseLetter, textPixel, SEQUENCE_HEIGHT, {strokeStyle: color});
                            ctx.restore();
                        } else {
                            IGVGraphics.strokeText(ctx, baseLetter, textPixel, SEQUENCE_HEIGHT, {strokeStyle: color});
                        }

                    }
                }
            }

            if (this.frameTranslate) {

                let y = SEQUENCE_HEIGHT + 2 * FRAME_BORDER;
                const translatedSequence = this.translateSequence(sequence);

                for (let fNum = 0; fNum < translatedSequence.length; fNum++) {    // == 3, 1 for each frame

                    const aaSequence = translatedSequence[fNum];  // AA sequence for this frame

                    for (let idx = 0; idx < aaSequence.length; idx++) {

                        let color = 0 === idx % 2 ? 'rgb(160,160,160)' : 'rgb(224,224,224)';
                        const cv = aaSequence[idx];

                        const bpPos = sequenceBpStart + fNum + (idx * 3);
                        const bpOffset = bpPos - options.bpStart;
                        const p0 = Math.floor(bpOffset / options.bpPerPixel);
                        const p1 = Math.floor((bpOffset + 3) / options.bpPerPixel);
                        const pc = Math.round((p0 + p1) / 2);

                        if (p1 < 0) {
                            continue   // off left edge
                        } else if (p0 > options.pixelWidth) {
                            break      // off right edge
                        }

                        let aaLabel = cv.aminoA;
                        if (cv.aminoA.indexOf('STOP') > -1) {
                            color = 'rgb(255, 0, 0)';
                            aaLabel = 'STOP'; //Color blind accessible
                        } else if (cv.aminoA === 'M') {
                            color = 'rgb(0, 153, 0)';
                            aaLabel = 'START'; //Color blind accessible
                        }

                        IGVGraphics.fillRect(ctx, p0, y, p1 - p0, FRAME_HEIGHT, {fillStyle: color});

                        if (options.bpPerPixel <= 1 / 10) {
                            IGVGraphics.strokeText(ctx, aaLabel, pc - (ctx.measureText(aaLabel).width / 2), y + 15);
                        }
                    }
                    y += (FRAME_HEIGHT + FRAME_BORDER);
                }
            }
        }
    }

    get supportsWholeGenome() {
        return false
    }

    computePixelHeight(ignore) {
        this.height = this.frameTranslate ? TRANSLATED_HEIGHT : DEFAULT_HEIGHT;
        return this.height
    }

    fillColor(index) {

        if (this.color) {
            return this.color
        } else if ("dna" === this.sequenceType) {
            return this.browser.nucleotideColors[index] || 'gray'
        } else {
            return 'rgb(0, 0, 150)'
        }
    }

    /**
     * Return the current state of the track.  Used to create sessions and bookmarks.
     *
     * @returns {*|{}}
     */
    getState() {
        const config = {
            type: "sequence"
        };
        if (this.order !== defaultSequenceTrackOrder) {
            config.order = this.order;
        }
        if (this.reversed) {
            config.revealed = true;
        }
        return config
    }

}

/**
 * Wrapper for a Fasta object that does chr name alias translation.   This is not neccessary for the genome fasta,
 * as it defines the reference name, but can be neccessary if loading an additional fasta as a track
 *
 */
class WrappedFasta {

    constructor(config, genome) {
        this.config = config;
        this.genome = genome;
    }

    async init() {
        this.fasta = await loadSequence(this.config);
        this.chrNameMap = new Map();
        for (let name of this.fasta.chromosomeNames) {
            this.chrNameMap.set(this.genome.getChromosomeName(name), name);
        }
    }

    async getSequence(chr, start, end) {
        const chrName = this.chrNameMap.has(chr) ? this.chrNameMap.get(chr) : chr;
        return this.fasta.getSequence(chrName, start, end)
    }

}

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

class Viewport {

    constructor(trackView, viewportColumn, referenceFrame, width) {
        this.guid = guid$2();
        this.trackView = trackView;
        this.referenceFrame = referenceFrame;

        this.browser = trackView.browser;

        this.viewportElement = document.createElement('div');
        this.viewportElement.className = 'igv-viewport';
        viewportColumn.appendChild(this.viewportElement);

        if (trackView.track.height) {
            this.setHeight(trackView.track.height);
        }

        // Create an alert dialog for the sequence track to copy ref sequence to.
        if (trackView.track instanceof SequenceTrack) {
            this.alert = new AlertDialog$1(this.viewportElement);
        }

        this.contentTop = 0;
        this.contentHeight = this.viewportElement.clientHeight;

        this.setWidth(width);

        this.initializationHelper();
    }

    initializationHelper() {}

    showMessage(message) {
        if (!this.messageDiv) {
            this.messageDiv = document.createElement('div');
            this.messageDiv.className = 'igv-viewport-message';
            this.viewportElement.appendChild(this.messageDiv);
        }
        this.messageDiv.textContent = message;
        this.messageDiv.style.display = 'inline-block';
    }

    hideMessage() {
        if (this.messageDiv) {
            this.messageDiv.style.display = 'none';
        }
    }

    setTrackLabel(label) {}

    startSpinner() {}

    stopSpinner() {}

    checkZoomIn() {
        return true;
    }

    shift() {}

    setTop(contentTop) {
        this.contentTop = contentTop;
    }

    async loadFeatures() {
        return undefined;
    }

    clearCache() {}

    repaint() {}

    draw(drawConfiguration, features, roiFeatures) {
    }

    checkContentHeight(features) {
        const track = this.trackView.track;
        features = features || this.cachedFeatures;
        if (track.displayMode === 'FILL') {
            this.setContentHeight(this.viewportElement.clientHeight);
        } else if (typeof track.computePixelHeight === 'function') {
            if (features && features.length > 0) {
                const requiredContentHeight = track.computePixelHeight(features);
                if (requiredContentHeight !== this.contentHeight) {
                    this.setContentHeight(requiredContentHeight);
                }
            }
        }
    }

    getContentHeight() {
        return this.contentHeight;
    }

    setContentHeight(contentHeight) {
        this.contentHeight = contentHeight;
    }

    isLoading() {
        return false;
    }

    saveSVG() {}

    isVisible() {
        return this.viewportElement.clientWidth > 0;
    }

    setWidth(width) {
        this.viewportElement.style.width = `${width}px`;
    }

    getWidth() {
        return this.viewportElement.clientWidth;
    }

    setHeight(height) {
        this.viewportElement.style.height = `${height}px`;
    }

    getContentTop() {
        return this.contentTop;
    }

    containsPosition(chr, position) {
    }

    addMouseHandlers() {}

    dispose() {
        this.viewportElement.remove();

        // Nullify all properties to free memory
        for (const key in this) {
            if (this.hasOwnProperty(key)) {
                this[key] = undefined;
            }
        }
    }
}

/*!!
 *  Canvas 2 Svg v1.0.19
 *  A low level canvas to SVG converter. Uses a mock canvas context to build an SVG document.
 *
 *  Licensed under the MIT license:
 *  http://www.opensource.org/licenses/mit-license.php
 *
 *  Author:
 *  Kerry Liu
 *
 *  Copyright (c) 2014 Gliffy Inc.
 */

//helper function to format a string
function format(str, args) {
    var keys = Object.keys(args), i;
    for (i = 0; i < keys.length; i++) {
        str = str.replace(new RegExp("\\{" + keys[i] + "\\}", "gi"), args[keys[i]]);
    }
    return str
}

//helper function that generates a random string
function randomString(holder) {
    var chars, randomstring, i;
    if (!holder) {
        throw new Error("cannot create a random attribute name for an undefined object")
    }
    chars = "ABCDEFGHIJKLMNOPQRSTUVWXTZabcdefghiklmnopqrstuvwxyz";
    randomstring = "";
    do {
        randomstring = "";
        for (i = 0; i < 12; i++) {
            randomstring += chars[Math.floor(Math.random() * chars.length)];
        }
    } while (holder[randomstring])
    return randomstring
}

//helper function to map named to numbered entities
function createNamedToNumberedLookup(items, radix) {
    var i, entity, lookup = {}, base10;
    items = items.split(',');
    radix = radix || 10;
    // Map from named to numbered entities.
    for (i = 0; i < items.length; i += 2) {
        entity = '&' + items[i + 1] + ';';
        base10 = parseInt(items[i], radix);
        lookup[entity] = '&#' + base10 + ';';
    }
    //FF and IE need to create a regex from hex values ie &nbsp; == \xa0
    lookup["\\xa0"] = '&#160;';
    return lookup
}

//helper function to map canvas-textAlign to svg-textAnchor
function getTextAnchor(textAlign) {
    //TODO: support rtl languages
    var mapping = {"left": "start", "right": "end", "center": "middle", "start": "start", "end": "end"};
    return mapping[textAlign] || mapping.start
}

//helper function to map canvas-textBaseline to svg-dominantBaseline
function getDominantBaseline(textBaseline) {
    //INFO: not supported in all browsers
    var mapping = {
        "alphabetic": "alphabetic",
        "hanging": "hanging",
        "top": "text-before-edge",
        "bottom": "text-after-edge",
        "middle": "central"
    };
    return mapping[textBaseline] || mapping.alphabetic
}


/**
 * Return a new normalized vector of given vector
 */
function normalize(vector) {
    var len = Math.sqrt(vector[0] * vector[0] + vector[1] * vector[1]);
    return [vector[0] / len, vector[1] / len]
}


function intersectRect(rect1, rect2) {
    return (rect1.x < rect2.x + rect2.width &&
        rect1.x + rect1.width > rect2.x &&
        rect1.y < rect2.y + rect2.height &&
        rect1.y + rect1.height > rect2.y)
}


// Unpack entities lookup where the numbers are in radix 32 to reduce the size
// entity mapping courtesy of tinymce
const namedEntities = createNamedToNumberedLookup(
    '50,nbsp,51,iexcl,52,cent,53,pound,54,curren,55,yen,56,brvbar,57,sect,58,uml,59,copy,' +
    '5a,ordf,5b,laquo,5c,not,5d,shy,5e,reg,5f,macr,5g,deg,5h,plusmn,5i,sup2,5j,sup3,5k,acute,' +
    '5l,micro,5m,para,5n,middot,5o,cedil,5p,sup1,5q,ordm,5r,raquo,5s,frac14,5t,frac12,5u,frac34,' +
    '5v,iquest,60,Agrave,61,Aacute,62,Acirc,63,Atilde,64,Auml,65,Aring,66,AElig,67,Ccedil,' +
    '68,Egrave,69,Eacute,6a,Ecirc,6b,Euml,6c,Igrave,6d,Iacute,6e,Icirc,6f,Iuml,6g,ETH,6h,Ntilde,' +
    '6i,Ograve,6j,Oacute,6k,Ocirc,6l,Otilde,6m,Ouml,6n,times,6o,Oslash,6p,Ugrave,6q,Uacute,' +
    '6r,Ucirc,6s,Uuml,6t,Yacute,6u,THORN,6v,szlig,70,agrave,71,aacute,72,acirc,73,atilde,74,auml,' +
    '75,aring,76,aelig,77,ccedil,78,egrave,79,eacute,7a,ecirc,7b,euml,7c,igrave,7d,iacute,7e,icirc,' +
    '7f,iuml,7g,eth,7h,ntilde,7i,ograve,7j,oacute,7k,ocirc,7l,otilde,7m,ouml,7n,divide,7o,oslash,' +
    '7p,ugrave,7q,uacute,7r,ucirc,7s,uuml,7t,yacute,7u,thorn,7v,yuml,ci,fnof,sh,Alpha,si,Beta,' +
    'sj,Gamma,sk,Delta,sl,Epsilon,sm,Zeta,sn,Eta,so,Theta,sp,Iota,sq,Kappa,sr,Lambda,ss,Mu,' +
    'st,Nu,su,Xi,sv,Omicron,t0,Pi,t1,Rho,t3,Sigma,t4,Tau,t5,Upsilon,t6,Phi,t7,Chi,t8,Psi,' +
    't9,Omega,th,alpha,ti,beta,tj,gamma,tk,delta,tl,epsilon,tm,zeta,tn,eta,to,theta,tp,iota,' +
    'tq,kappa,tr,lambda,ts,mu,tt,nu,tu,xi,tv,omicron,u0,pi,u1,rho,u2,sigmaf,u3,sigma,u4,tau,' +
    'u5,upsilon,u6,phi,u7,chi,u8,psi,u9,omega,uh,thetasym,ui,upsih,um,piv,812,bull,816,hellip,' +
    '81i,prime,81j,Prime,81u,oline,824,frasl,88o,weierp,88h,image,88s,real,892,trade,89l,alefsym,' +
    '8cg,larr,8ch,uarr,8ci,rarr,8cj,darr,8ck,harr,8dl,crarr,8eg,lArr,8eh,uArr,8ei,rArr,8ej,dArr,' +
    '8ek,hArr,8g0,forall,8g2,part,8g3,exist,8g5,empty,8g7,nabla,8g8,isin,8g9,notin,8gb,ni,8gf,prod,' +
    '8gh,sum,8gi,minus,8gn,lowast,8gq,radic,8gt,prop,8gu,infin,8h0,ang,8h7,and,8h8,or,8h9,cap,8ha,cup,' +
    '8hb,int,8hk,there4,8hs,sim,8i5,cong,8i8,asymp,8j0,ne,8j1,equiv,8j4,le,8j5,ge,8k2,sub,8k3,sup,8k4,' +
    'nsub,8k6,sube,8k7,supe,8kl,oplus,8kn,otimes,8l5,perp,8m5,sdot,8o8,lceil,8o9,rceil,8oa,lfloor,8ob,' +
    'rfloor,8p9,lang,8pa,rang,9ea,loz,9j0,spades,9j3,clubs,9j5,hearts,9j6,diams,ai,OElig,aj,oelig,b0,' +
    'Scaron,b1,scaron,bo,Yuml,m6,circ,ms,tilde,802,ensp,803,emsp,809,thinsp,80c,zwnj,80d,zwj,80e,lrm,' +
    '80f,rlm,80j,ndash,80k,mdash,80o,lsquo,80p,rsquo,80q,sbquo,80s,ldquo,80t,rdquo,80u,bdquo,810,dagger,' +
    '811,Dagger,81g,permil,81p,lsaquo,81q,rsaquo,85c,euro', 32);


//Some basic mappings for attributes and default values.
const STYLES = {
    "strokeStyle": {
        svgAttr: "stroke", //corresponding svg attribute
        canvas: "#000000", //canvas default
        svg: "none",       //svg default
        apply: "stroke"    //apply on stroke() or fill()
    },
    "fillStyle": {
        svgAttr: "fill",
        canvas: "#000000",
        svg: null, //svg default is black, but we need to special case this to handle canvas stroke without fill
        apply: "fill"
    },
    "lineCap": {
        svgAttr: "stroke-linecap",
        canvas: "butt",
        svg: "butt",
        apply: "stroke"
    },
    "lineJoin": {
        svgAttr: "stroke-linejoin",
        canvas: "miter",
        svg: "miter",
        apply: "stroke"
    },
    "miterLimit": {
        svgAttr: "stroke-miterlimit",
        canvas: 10,
        svg: 4,
        apply: "stroke"
    },
    "lineWidth": {
        svgAttr: "stroke-width",
        canvas: 1,
        svg: 1,
        apply: "stroke"
    },
    "globalAlpha": {
        svgAttr: "opacity",
        canvas: 1,
        svg: 1,
        apply: "fill stroke"
    },
    "font": {
        //font converts to multiple svg attributes, there is custom logic for this
        canvas: "10px sans-serif"
    },
    "shadowColor": {
        canvas: "#000000"
    },
    "shadowOffsetX": {
        canvas: 0
    },
    "shadowOffsetY": {
        canvas: 0
    },
    "shadowBlur": {
        canvas: 0
    },
    "textAlign": {
        canvas: "start"
    },
    "textBaseline": {
        canvas: "alphabetic"
    },
    "lineDash": {
        svgAttr: "stroke-dasharray",
        canvas: [],
        svg: null,
        apply: "stroke"
    }
};

/**
 *
 * @param gradientNode - reference to the gradient
 * @constructor
 */
class CanvasGradient {
    constructor(gradientNode, ctx) {
        this.__root = gradientNode;
        this.__ctx = ctx;
    }

    /**
     * Adds a color stop to the gradient root
     */
    addColorStop(offset, color) {
        var stop = this.__ctx.__createElement("stop"), regex, matches;
        stop.setAttribute("offset", offset);
        if (color && color.indexOf("rgba") !== -1) {
            //separate alpha value, since webkit can't handle it
            regex = /rgba\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d?\.?\d*)\s*\)/gi;
            matches = regex.exec(color);
            stop.setAttribute("stop-color", format("rgb({r},{g},{b})", {r: matches[1], g: matches[2], b: matches[3]}));
            stop.setAttribute("stop-opacity", matches[4]);
        } else {
            stop.setAttribute("stop-color", color);
        }
        this.__root.appendChild(stop);
    }
}

class CanvasPattern {
    constructor(pattern, ctx) {
        this.__root = pattern;
        this.__ctx = ctx;
    }
}

/**
 * The mock canvas context
 * @param config - options include:
 * ctx - existing Context2D to wrap around
 * width - width of your canvas (defaults to 500)
 * height - height of your canvas (defaults to 500)
 * enableMirroring - enables canvas mirroring (get image data) (defaults to false)
 * document - the document object (defaults to the current document)
 */
class ctx {
    constructor(config) {

        if (!(this instanceof ctx)) {
            //did someone call this without new?
            return new ctx(config)
        }

        // clone config
        this.config = config;

        //setup options
        this.width = config.width;
        this.height = config.height;
        this.enableMirroring = config.enableMirroring || false;

        this.canvas = this;   ///point back to this instance!
        this.__document = document;

        // allow passing in an existing context to wrap around
        // if a context is passed in, we know a canvas already exist
        if (config.ctx) {
            this.__ctx = config.ctx;
        } else {
            this.__canvas = this.__document.createElement("canvas");
            this.__ctx = this.__canvas.getContext("2d");
        }

        // give this canvas a type
        this.isSVG = true;

        this.__setDefaultStyles();
        this.__stack = [this.__getStyleState()];
        this.__groupStack = [];

        // root svg element
        this.__root = this.__createElement("svg");
        this.__root.setAttribute("width", this.width);
        this.__root.setAttribute("height", this.height);

        // allow contents to overflow svg bbox
        this.__root.setAttribute('overflow', 'visible');

        // viewbox
        if (config.viewbox) {
            const str = config.viewbox.x + ' ' + config.viewbox.y + ' ' + config.viewbox.width + ' ' + config.viewbox.height;
            this.__root.setAttribute("viewBox", str);

            this.viewbox = config.viewbox;
        }

        // make sure we don't generate the same ids in defs
        this.__ids = {};

        // defs
        this.__defs = this.__createElement("defs");
        this.__root.appendChild(this.__defs);

        this.multiLocusGap = config.multiLocusGap;

        // svg background color
        let backdropConfig =
            {
                id: 'svg_output_backdrop',
                width: '100%',
                height: '100%',
                fill: config.backdropColor || 'white'
            };

        let backdropRect = this.__createElement('rect', backdropConfig);
        this.__root.appendChild(backdropRect);

        // root group
        this.__rootGroup = this.__createElement('g', {id: 'root-group'});
        this.__root.appendChild(this.__rootGroup);

        // point current element to root group
        this.__currentElement = this.__rootGroup;
    }

    setWidth(width) {

        this.width = width;
        this.__root.setAttribute("width", this.width);

        const str = this.config.viewbox.x + ' ' + this.config.viewbox.y + ' ' + width + ' ' + this.config.viewbox.height;
        this.__root.setAttribute("viewBox", str);

    };

    setHeight(height) {

        this.height = height;
        this.__root.setAttribute("height", this.height);

        const str = this.config.viewbox.x + ' ' + this.config.viewbox.y + ' ' + this.config.viewbox.width + ' ' + height;
        this.__root.setAttribute("viewBox", str);

    };

    /**
     * Creates the specified svg element
     * @private
     */
    __createElement(elementName, properties, resetFill) {

        if (typeof properties === "undefined") {
            properties = {};
        }

        let element = this.__document.createElementNS("http://www.w3.org/2000/svg", elementName);

        if (resetFill) {
            //if fill or stroke is not specified, the svg element should not display. By default SVG's fill is black.
            element.setAttribute("fill", "none");
            element.setAttribute("stroke", "none");
        }

        for (let key of Object.keys(properties)) {
            element.setAttribute(key, properties[key]);
        }

        return element
    };

    /**
     * Applies default canvas styles to the context
     * @private
     */
    __setDefaultStyles() {
        //default 2d canvas context properties see:http://www.w3.org/TR/2dcontext/
        var keys = Object.keys(STYLES), i, key;
        for (i = 0; i < keys.length; i++) {
            key = keys[i];
            this[key] = STYLES[key].canvas;
        }
    };

    /**
     * Applies styles on restore
     * @param styleState
     * @private
     */
    __applyStyleState(styleState) {
        var keys = Object.keys(styleState), i, key;
        for (i = 0; i < keys.length; i++) {
            key = keys[i];
            this[key] = styleState[key];
        }
    };

    /**
     * Gets the current style state
     * @return {Object}
     * @private
     */
    __getStyleState() {
        var i, styleState = {}, keys = Object.keys(STYLES), key;
        for (i = 0; i < keys.length; i++) {
            key = keys[i];
            styleState[key] = this[key];
        }
        return styleState
    };

    /**
     * Apples the current styles to the current SVG element. On "ctx.fill" or "ctx.stroke"
     * @param type
     * @private
     */
    __applyStyleToCurrentElement(type) {
        var currentElement = this.__currentElement;
        var currentStyleGroup = this.__currentElementsToStyle;
        if (currentStyleGroup) {
            currentElement.setAttribute(type, "");
            currentElement = currentStyleGroup.element;
            currentStyleGroup.children.forEach(function (node) {
                node.setAttribute(type, "");
            });
        }

        var keys = Object.keys(STYLES), i, style, value, id, regex, matches;
        for (i = 0; i < keys.length; i++) {
            style = STYLES[keys[i]];
            value = this[keys[i]];
            if (style.apply) {
                //is this a gradient or pattern?
                if (value instanceof CanvasPattern) {
                    //pattern
                    if (value.__ctx) {
                        //copy over defs
                        while (value.__ctx.__defs.childNodes.length) {
                            id = value.__ctx.__defs.childNodes[0].getAttribute("id");
                            this.__ids[id] = id;
                            this.__defs.appendChild(value.__ctx.__defs.childNodes[0]);
                        }
                    }
                    currentElement.setAttribute(style.apply, format("url(#{id})", {id: value.__root.getAttribute("id")}));
                } else if (value instanceof CanvasGradient) {
                    //gradient
                    currentElement.setAttribute(style.apply, format("url(#{id})", {id: value.__root.getAttribute("id")}));
                } else if (style && style.apply.indexOf(type) !== -1 && style.svg !== value) {
                    if ((style.svgAttr === "stroke" || style.svgAttr === "fill") && value && value.indexOf("rgba") !== -1) {
                        //separate alpha value, since illustrator can't handle it
                        regex = /rgba\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d?\.?\d*)\s*\)/gi;
                        matches = regex.exec(value);
                        currentElement.setAttribute(style.svgAttr, format("rgb({r},{g},{b})", {
                            r: matches[1],
                            g: matches[2],
                            b: matches[3]
                        }));
                        //should take globalAlpha here
                        var opacity = matches[4];
                        var globalAlpha = this.globalAlpha;
                        if (globalAlpha != null) {
                            opacity *= globalAlpha;
                        }
                        currentElement.setAttribute(style.svgAttr + "-opacity", opacity);
                    } else {
                        var attr = style.svgAttr;
                        if (keys[i] === 'globalAlpha') {
                            attr = type + '-' + style.svgAttr;
                            if (currentElement.getAttribute(attr)) {
                                //fill-opacity or stroke-opacity has already been set by stroke or fill.
                                continue
                            }
                        }
                        //otherwise only update attribute if right type, and not svg default
                        currentElement.setAttribute(attr, value);
                    }
                }
            }
        }
    };

    /**
     * Will return the closest group or svg node. May return the current element.
     * @private
     */
    __closestGroupOrSvg(node) {
        node = node || this.__currentElement;
        if (node.nodeName === "g" || node.nodeName === "svg") {
            return node
        } else {
            return this.__closestGroupOrSvg(node.parentNode)
        }
    };

    /**
     * Returns the serialized value of the svg so far
     * @param fixNamedEntities - Standalone SVG doesn't support named entities, which document.createTextNode encodes.
     *                           If true, we attempt to find all named entities and encode it as a numeric entity.
     * @return serialized svg
     */
    getSerializedSvg(fixNamedEntities) {
        var serialized = new XMLSerializer().serializeToString(this.__root),
            keys, i, key, value, regexp;

        //IE search for a duplicate xmnls because they didn't implement setAttributeNS correctly
        // xmlns = /xmlns="http:\/\/www\.w3\.org\/2000\/svg".+xmlns="http:\/\/www\.w3\.org\/2000\/svg/gi;
        // if (xmlns.test(serialized)) {
        //     serialized = serialized.replace('xmlns="http://www.w3.org/2000/svg','xmlns:xlink="http://www.w3.org/1999/xlink');
        // }

        if (fixNamedEntities) {
            keys = Object.keys(namedEntities);
            //loop over each named entity and replace with the proper equivalent.
            for (i = 0; i < keys.length; i++) {
                key = keys[i];
                value = namedEntities[key];
                regexp = new RegExp(key, "gi");
                if (regexp.test(serialized)) {
                    serialized = serialized.replace(regexp, value);
                }
            }
        }

        return serialized
    };


    /**
     * Returns the root svg
     * @return
     */
    getSvg() {
        return this.__root
    };

    /**
     * Will generate a group tag.
     */
    saveWithTranslationAndClipRect(id, tx, ty, width, height, clipYOffset) {

        // clip rect
        const clip_id = `${id}_clip_rect`;
        let clipPath = this.__createElement('clipPath', {id: clip_id});

        this.__defs.appendChild(clipPath);

        const config =
            {
                x: '0',
                y: clipYOffset.toString(),
                width: width.toString(),
                height: height.toString()
            };

        clipPath.appendChild(this.__createElement('rect', config));

        const group = this.__createElement("g");
        group.setAttribute('transform', format('translate({x},{y})', {x: tx, y: ty}));
        group.setAttribute('clip-path', format('url(#{id})', {id: clip_id}));

        const parent = this.__closestGroupOrSvg();
        parent.appendChild(group);
        this.__groupStack.push(parent);

        this.__currentElement = group;
        this.__stack.push(this.__getStyleState());
    };

    save() {
        var group = this.__createElement("g");
        var parent = this.__closestGroupOrSvg();
        this.__groupStack.push(parent);
        parent.appendChild(group);
        this.__currentElement = group;
        this.__stack.push(this.__getStyleState());
    };

    /**
     * Sets current element to parent, or just root if already root
     */
    restore() {
        this.__currentElement = this.__groupStack.pop();
        this.__currentElementsToStyle = null;
        //Clearing canvas will make the poped group invalid, currentElement is set to the root group node.
        if (!this.__currentElement) {
            this.__currentElement = this.__root.childNodes[1];
        }
        var state = this.__stack.pop();
        this.__applyStyleState(state);
    };

    /**
     * Helper method to add transform
     * @private
     */
    __addTransform(t) {
        //if the current element has siblings, add another group
        var parent = this.__closestGroupOrSvg();
        if (parent.childNodes.length > 0) {
            if (this.__currentElement.nodeName === "path") {
                if (!this.__currentElementsToStyle) this.__currentElementsToStyle = {element: parent, children: []};
                this.__currentElementsToStyle.children.push(this.__currentElement);
                this.__applyCurrentDefaultPath();
            }

            var group = this.__createElement("g");
            parent.appendChild(group);
            this.__currentElement = group;
        }

        var transform = this.__currentElement.getAttribute("transform");
        if (transform) {
            transform += " ";
        } else {
            transform = "";
        }
        transform += t;
        this.__currentElement.setAttribute("transform", transform);
    };

    addTrackGroupWithTranslationAndClipRect(id, tx, ty, width, height, clipYOffset) {

        // clip rect
        const clip_id = id + '_clip_rect';
        let clipPath = this.__createElement('clipPath', {id: clip_id});

        this.__defs.appendChild(clipPath);
        clipPath.appendChild(this.__createElement('rect', {
            x: '0',
            y: clipYOffset.toString(),
            width: width.toString(),
            height: height.toString()
        }));

        let group = this.__createElement('g');
        this.__rootGroup.appendChild(group);

        group.setAttribute('transform', format('translate({x},{y})', {x: tx, y: ty}));
        group.setAttribute('id', (id + '_group'));

        // add clip rect
        group.setAttribute('clip-path', format('url(#{id})', {id: clip_id}));

        this.__currentElement = group;
    };

    /**
     *  scales the current element
     */
    scale(x, y) {
        if (y === undefined) {
            y = x;
        }
        this.__addTransform(format("scale({x},{y})", {x: x, y: y}));
    };

    /**
     * rotates the current element
     */
    rotate(angle) {
        var degrees = (angle * 180 / Math.PI);
        this.__addTransform(format("rotate({angle},{cx},{cy})", {angle: degrees, cx: 0, cy: 0}));
    };

    /**
     * translates the current element
     */
    translate(x, y) {
        this.__addTransform(format("translate({x},{y})", {x: x, y: y}));
    };

    /**
     * applies a transform to the current element
     */
    transform(a, b, c, d, e, f) {
        this.__addTransform(format("matrix({a},{b},{c},{d},{e},{f})", {a: a, b: b, c: c, d: d, e: e, f: f}));
    };

    /**
     * Create a new Path Element
     */
    beginPath() {
        var path, parent;

        // Note that there is only one current default path, it is not part of the drawing state.
        // See also: https://html.spec.whatwg.org/multipage/scripting.html#current-default-path
        this.__currentDefaultPath = "";
        this.__currentPosition = {};

        path = this.__createElement("path", {}, true);
        parent = this.__closestGroupOrSvg();
        parent.appendChild(path);
        this.__currentElement = path;
    };

    /**
     * Helper function to apply currentDefaultPath to current path element
     * @private
     */
    __applyCurrentDefaultPath() {
        var currentElement = this.__currentElement;
        if (currentElement.nodeName === "path") {
            currentElement.setAttribute("d", this.__currentDefaultPath);
        } else {
            console.error("Attempted to apply path command to node", currentElement.nodeName);
        }
    };

    /**
     * Helper function to add path command
     * @private
     */
    __addPathCommand(command) {
        this.__currentDefaultPath += " ";
        this.__currentDefaultPath += command;
    };

    /**
     * Adds the move command to the current path element,
     * if the currentPathElement is not empty create a new path element
     */
    moveTo(x, y) {
        if (this.__currentElement.nodeName !== "path") {
            this.beginPath();
        }

        // creates a new subpath with the given point
        this.__currentPosition = {x: x, y: y};
        this.__addPathCommand(format("M {x} {y}", {x: x, y: y}));
    };

    /**
     * Closes the current path
     */
    closePath() {
        if (this.__currentDefaultPath) {
            this.__addPathCommand("Z");
        }
    };

    /**
     * Adds a line to command
     */
    lineTo(x, y) {
        this.__currentPosition = {x: x, y: y};
        if (this.__currentDefaultPath && this.__currentDefaultPath.indexOf('M') > -1) {
            this.__addPathCommand(format("L {x} {y}", {x: x, y: y}));
        } else {
            this.__addPathCommand(format("M {x} {y}", {x: x, y: y}));
        }
    };

    /**
     * Add a bezier command
     */
    bezierCurveTo(cp1x, cp1y, cp2x, cp2y, x, y) {
        this.__currentPosition = {x: x, y: y};
        this.__addPathCommand(format("C {cp1x} {cp1y} {cp2x} {cp2y} {x} {y}",
            {cp1x: cp1x, cp1y: cp1y, cp2x: cp2x, cp2y: cp2y, x: x, y: y}));
    };

    /**
     * Adds a quadratic curve to command
     */
    quadraticCurveTo(cpx, cpy, x, y) {
        this.__currentPosition = {x: x, y: y};
        this.__addPathCommand(format("Q {cpx} {cpy} {x} {y}", {cpx: cpx, cpy: cpy, x: x, y: y}));
    };

    /**
     * Adds the arcTo to the current path
     *
     * @see http://www.w3.org/TR/2015/WD-2dcontext-20150514/#dom-context-2d-arcto
     */
    arcTo(x1, y1, x2, y2, radius) {
        // Let the point (x0, y0) be the 